/*
 * $Id: liberror.c,v 1.9 1998/02/17 09:51:57 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/liberror/liberror.c,v $
 * $Revision: 1.9 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdarg.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

#include <modem_defs.h>
#include "liberror.h"


/* Maximum length of error string */
#define errMsgBufLength 100

/* Default error handler */
static void defaultErrorFunction( const char* fmt );

/* Static buffer that contains error message */
static char _errMsgBuf[errMsgBufLength];

/* 
 * errorFuncPtr points to the error handler.
 * The default value is to point to defaultErrorFunction
 */
static _errorFuncPtr errorFuncPtr = defaultErrorFunction;

/*
 * Default error handler. Just write the error message to stderr
 */
static void defaultErrorFunction( const char* msg )
{
   fprintf( stderr, "%s\n", msg );
}

/*
 * This function should be called whenever there is an error condition
 * in your program.
 * It sets up an error message according to the format given.
 * On some systems where the '%m' option is not recognized by
 * the printf functions, we handle this option ourselves.
 *
 * vsnprintf is used to asure _errMsgBuf will not overflow.
 *
 */
void error( const char* fmt, ... )
{
/* Does printf recognize the "%m" format string */
#ifndef PRINTF_HAS_M_FORMAT
   va_list ap;
   char* str;
   char* ptr2;
   int   saved_errno;
   
   saved_errno = errno;
   
   va_start( ap, fmt );
   
   str = (char*)malloc( strlen( fmt ) + 1 );
   ptr2 = str;
   /*
    * subsitue strerror(saved_errno) for all non escaped occurences
    * of '%m'
    */   
   while( *fmt )
   {
      switch( *fmt )
      {
         case '%':
            if( *(fmt + 1) == 'm' )
            { /* non escaped '%m' found. Substitute strerror(saved_errno) */
               *ptr2 = '\0';
               str = (char* )realloc( str, strlen( str ) + 
                              strlen( strerror(saved_errno) ) + 1 );
               strcat( str, strerror(saved_errno) );
               ptr2 = str + strlen( str );
               fmt++;
               fmt++;
            }
            else
            if( *(fmt + 1) == '%' )
            { /* escaped '%' character */
               *ptr2++ = *fmt++;
               *ptr2++ = *fmt++;
               break;
            }
            else
               *ptr2++ = *fmt++;
            break;
         default:
            *ptr2++ = *fmt++;
            break;
      }
   }
   *ptr2 = '\0';
   
   /* create formatted error message */
   vsnprintf( _errMsgBuf, sizeof( _errMsgBuf), str, ap );

   free( str );
   
   /* Call error handler */
   errorFuncPtr( _errMsgBuf );
   
   va_end( ap );

#else
/*
 * '%m' option is recognized by this system
 */
   va_list ap;
   va_start( ap, fmt );
   
   /* create formatted error message */
   vsnprintf( _errMsgBuf, sizeof( _errMsgBuf), fmt, ap );
   
   /* Call error handler */
   errorFuncPtr( _errMsgBuf );
   
   va_end( ap );
#endif
}

/*
 * Change error handler to f
 */
void setErrorFunc( _errorFuncPtr f )
{
   errorFuncPtr = f;
}

/*
 * Return current error handler
 */
const _errorFuncPtr getErrorFunc()
{
   return errorFuncPtr;
}

/*
 * Return time and date in representation of
 * current locale.
 *
 * A pointer to static buffer is returend.
 */
const char* timeStamp()
{
   struct tm *tm;
   time_t now;
   static char time_stamp[60];
   char*  format = "[%c]";

   time( &now );
   tm = localtime( &now );
   
   strftime( time_stamp, sizeof( time_stamp ), format, tm );
   return time_stamp;
}

/*
 * EOF liberror/liberror.c
 */
