/*
 * $Id: libhelp.c,v 1.20 1998/02/17 09:51:59 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/libhelp/libhelp.c,v $
 * $Revision: 1.20 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <signal.h>

#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include <modem_defs.h>
#include <liberror/liberror.h>
#include <libport/libport.h>
#include "libhelp.h"

/* Save sigaction for SIGCHLD */
static struct sigaction savedSigChld;

/* Save pid of help viewer */
static pid_t helpViewerPid = -1;

/*
 * helpSignalHandler
 *
 * Signal handler used to catch termination of help viewer
 * It is installed from displayHelp when a new help viewer 
 * needs to be executed. 
 * This function restores the  SIGCHLD setting
 * to the setting before displayHelp was called.
 *
 */ 
static void helpSignalHandler( int sig )
{
   int result;

   TEMP_FAILURE_RETRY( result,
      waitpid( helpViewerPid, NULL, WNOHANG ) );
   if( result == -1 )
   {
      FAIL( "waitpid" );
      exit( 1 );
   }

   if( result == 0 )
   {
      /* SIGCHLD for other process received */
      if( savedSigChld.sa_handler != SIG_IGN &&
          savedSigChld.sa_handler != SIG_DFL )
         savedSigChld.sa_handler( sig );
   }
   else
   {
      /* restore SIGCHLD handlers */
      sigaction( SIGCHLD, &savedSigChld, NULL );
      helpViewerPid = -1;
   }
}

static void helpErrorFunc( const char* msg )
{
   fprintf( stderr, "%s\n", msg );
}

void displayHelp( const char* helpTag )
{
   int      result;
   char     help_cmd[512];
   struct   sigaction act;
   sigset_t block;
   sigset_t omask;
   
   /* Install signal handler and save old settings in savedSigChld */
   sigaction( SIGCHLD, NULL, &savedSigChld );
   sigaction( SIGCHLD, NULL, &act );
   act.sa_handler = helpSignalHandler;
#ifdef SA_RESTART
   act.sa_flags   = SA_RESTART;
#endif
   if( sigaction( SIGCHLD, &act, NULL ) == -1 )
   {
      FAIL( "sigaction" );
      exit( 1 );
   }

   /* Block receipt of SIGCHLD */
   if( sigemptyset( &block ) == -1 )
   {
      FAIL( "sigemptyset" );
      exit( 1 );
   }
   if( sigaddset( &block, SIGCHLD ) == -1 )
   {
      FAIL( "sigaddset" );
      exit( 1 );
   }
   if( sigprocmask( SIG_BLOCK, &block, &omask ) == -1 )
   {
      FAIL( "sigprocmask" );
      exit( 1 );
   }

   /* We fork imediately so that executing the helpviewer
    * will not block the current process
    */
   helpViewerPid = fork();

   switch( helpViewerPid )
   {
      case -1:
         FAIL( "fork" );
         exit( 1 );
      case 0:
         /* Child */

         /* Ignore SIGCHLD here */
         act.sa_handler = SIG_DFL;
         if( sigaction( SIGCHLD, &act, NULL ) == -1 )
         {
            FAIL( "sigaction" );
            exit( 1 );
         }
         /* Restore signal mask */
         if( sigprocmask( SIG_SETMASK, &omask, NULL ) == -1 )
         {
            FAIL( "sigprocmask" );
            exit( 1 );
         }
         /* 
          * Since we are probably using X as user interface, we
          * get into trouble when a child process tries to use its
          * parent X interface. We therefore set a new error handler
          */
         setErrorFunc( helpErrorFunc );

         /* Construct help command */
         sprintf( help_cmd, "%s %s '%s(%s/%s.%s)'",
            NETSCAPE,
            REMOTE,
            OPEN_URL,
            HELP_URL, 
            helpTag, 
            HELP_EXT );
         /* Execute help command */
         result = system( help_cmd );

         /* Is there a running netscape process ? */
         if( result != 0 )
         {
            /* No, we start one ourselves */
            
            /* Construct alternative help command */
            sprintf( help_cmd, "%s %s/%s.%s &", 
               NETSCAPE,
               HELP_URL,
               helpTag,
               HELP_EXT );
            fprintf( stderr, "starting: %s\n", help_cmd );

            /* Execute netscape progam */
            result = system( help_cmd );
            if( result != 0 )
            {
               FAIL1( "system", help_cmd );
               _exit( 1 );
            }
         }
         _exit( 0 );
         
         
      default:
         /* Parent */

         /* Restore signal mask */
         if( sigprocmask( SIG_SETMASK, &omask, NULL) == -1 )
         {
            FAIL( "sigprocmask" );
            exit( 1 );
         }
         return;
   }
}
/*
 * EOF libhelp/libhelp.c
 */
