/*
 * $Id: libmclient.c,v 1.13 1998/02/17 09:52:02 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/libmclient/libmclient.c,v $
 * $Revision: 1.13 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/time.h>
#include <time.h>
#include <errno.h>
#include <pwd.h>
#include <unistd.h>
#include <string.h>

#include <libtcpip/libtcpip.h>
#include <liberror/liberror.h>
#include <libport/libport.h>
#include <modem_defs.h>
#include "libmclient.h"
#include "../modemd/request.h" /* for request */

static int modemClientFd = -1;


/*
 * Wait for broadcast reply from modem server
 */
static int modemClientBroadcast( int fd )
{
   fd_set rfds;
   struct timeval timeout;
   int retval;
   
   while( 1 )
   {
      timeout.tv_sec = SERVER_TIMEOUT;
      timeout.tv_usec = 0;
   
      FD_ZERO( &rfds );
      FD_SET( fd, &rfds );
      retval = select( FD_SETSIZE, &rfds, NULL, NULL, &timeout );
       
      if( retval < 0 )
      {
         if( errno == EINTR )
            continue;
         FAIL( "select" );
      }
      return retval;
   }
}
                           
int modemClientConnect( const char* host )
{
   int modemd_servers = 0;
   sockStruct sock_tcp;
   sockStruct sock_udp;
   sockStruct server;
   request req;
   int result;
   int dummy;
   const char* serverName;
   
      
   /* Initialise sockets */
   sockSetProtocol( &sock_tcp, TCP );
   sockSetProtocol( &sock_udp, UDP );
   
   if( sockSetPortToService( &sock_udp, UDP_SERVICE ) == -1 )
   {
      fprintf( stderr, "Unknown service: %s, using default port: %d\n",
         UDP_SERVICE, UDP_DEFAULT_PORT );
      sockSetPort( &sock_udp, UDP_DEFAULT_PORT );
   }

   if( sockSetPortToService( &sock_tcp, TCP_SERVICE ) == -1 )
   {
      fprintf( stderr, "Unknown service: %s, using default port: %d\n",
         TCP_SERVICE, TCP_DEFAULT_PORT );
      sockSetPort( &sock_tcp, TCP_DEFAULT_PORT );
   }

   if( sockClient( &sock_udp, host ) == -1 )
   {
      FAIL( "udp" );
      exit( 1 );
   }
   
   if( sockSetOption( &sock_udp, SO_BROADCAST, 1 ) < 0 )
   {
      FAIL( "sockSetOption" );
      exit( 1 );
   }
   
   /* Send broadcast message to the network. Send username as message */
   result = sockSend( &sock_udp, &dummy, sizeof( dummy ) );
   if( result == -1 )
   {
      FAIL( "sockSend" );
      exit( 1 );
   }
   
   if( result == 0 )
      exit( 0 );

   while( 1 )
   {
      /* Wait for a reply from server */
      if( modemClientBroadcast( sock_udp.fd ) == 0 )
      {
         if( modemd_servers == 0 )
            error( "No servers found on the net" );
         else
            error( "No modems available" );
         exit( 1 );
      }
      
      /* Read request from server */
      if( sockRecv( &sock_udp, &server, &req, sizeof( req ) ) == -1 )
      {
         FAIL( "recvfrom" );
         exit( 1 );
      }
      
      modemd_servers++;
      /* Get name of server who replied */
      serverName = sockConnectedTo( &server );
      if( sockClient( &sock_tcp, serverName ) == -1 )
      {
         FAIL( "tcp" );
         exit( 1 );
      }
      
      /* Connect to that server */
      if( sockConnect( &sock_tcp ) == -1 )
      {
         FAIL( "connect" );
         exit( 1 );
      }
      
      /* Send request to that server */
      TEMP_FAILURE_RETRY( result, write( sock_tcp.fd, &req, sizeof(req) ) );
      if( result < 0 )
      {
         FAIL( "write" );
         exit( 1 );
      }
      if( result == 0 )
         exit( 0 );


      /* return */
      result = protConnectRequest( sock_tcp.fd );
      if( result < 0 )
      {
         FAIL( "protConnectRequest" );
         exit( 1 );
      }
      if( result == PROT_REQUEST_OK )
      {
         modemClientFd = sock_tcp.fd;
         return sock_tcp.fd;
      }
      else
      {
         /* Modem server is unable to handle our requests */
         close( sock_tcp.fd );
      }
   }
}
         
void modemClientSetHandler( modemDataType type , modemHandler handler )
{
   protSetHandler( type, handler );
}


int modemClientWrite( void* data, size_t length )
{
   int bytes;
   if( modemClientFd == -1 )
      return -1;
   bytes = protSend( PROT_DATA, modemClientFd, data, length );
   return bytes;
}

int  modemClientRead()
{
   int bytes;
   if( modemClientFd == -1 )
      return -1;
   bytes = protReceive( modemClientFd );
   return bytes;
}


void  modemClientDisconnect()
{
   if( modemClientFd == -1 )
      return;
   modemClientHangup();
   protSend( PROT_COMMAND | MDM_EXIT, modemClientFd, NULL, 0 );
   
   close( modemClientFd );
}

void modemClientHangup()
{
   protSend( PROT_COMMAND | MDM_HANGUP, modemClientFd, NULL, 0 );
}


void  modemClientDial( const char* number )
{
   protSend( PROT_COMMAND | MDM_DIAL, 
             modemClientFd, number,
             strlen( number ) );
}
/*
 * EOF libmclient/libmclient.c
 */
