/*
 * $Id: libtcpip.c,v 1.12 1998/02/17 09:52:20 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/libtcpip/libtcpip.c,v $
 * $Revision: 1.12 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <limits.h>
#include <sys/utsname.h>
#include <strings.h>

#include "libtcpip.h"
#include <liberror/liberror.h>

#define UDP_PROT "udp"
#define TCP_PROT "tcp"
#define BACKLOG   3 /* Max. number of pending connections */

/* Set type of protocol to use */
int sockSetProtocol( sockStruct* sock, sockProtocol prot )
{
   if( sock == NULL )
      return -1;
   sock->prot = prot;
   return 0;
}

/* Set port to use */
int sockSetPort( sockStruct* sock, int port )
{
   if( sock == NULL )
      return -1;
   sock->port = htons( port );
   return 0;
}

/* Set port to service as described in /etc/services */
int sockSetPortToService( sockStruct* sock, char* service )
{
   struct servent *serv;
   char* prot;
   
   /* Wich protocol do we use */
   switch( sock->prot )
   {
      case UDP:
         prot = UDP_PROT;
         break;
      case TCP:
         prot = TCP_PROT;
         break;
      default:
         return -1;
   }   
   if( sock == 0 )
      return -1;

   /* get the port from /etc/services */
   serv = getservbyname( service, prot );
   if( serv == NULL )
      return -1;
   sock->port = serv->s_port;
   return 0;
}

/* Private function
 * 
 * Create new socket
 * Protocol and port should already be set
 */
static int sockMakeSocket( sockStruct* sock )
{
   int type;
   if( sock == NULL )
      return -1;

   /* Which protocol do we use */
   switch( sock->prot )
   {
      case UDP:
         type = SOCK_DGRAM;
         break;
      case TCP:
         type = SOCK_STREAM;
         break;
      default:
         return -1;
   }
   /* Create new socket */
   sock->fd = socket( AF_INET, type, 0 );
   if( sock->fd < 0 )
      return -1;
   sock->name.sin_family = AF_INET;
   sock->name.sin_port = sock->port;

   return 0;
}

/*
 * sock should be a server
 */
int sockServer( sockStruct* sock )
{
   /* Make new socket */
   if( sockMakeSocket( sock ) == -1 )
      return -1;
   sock->name.sin_addr.s_addr = htonl( INADDR_ANY );
   if( bind( sock->fd, (struct sockaddr *)&(sock->name), 
             sizeof( sock->name) ) < 0 )
      return -1;
   return 0;
}

/* 
 * sock should be a client.
 *
 */
int sockClient( sockStruct* sock, const char* host )
{
   struct hostent *hostinfo;

   /* Make new socket */
   if( sockMakeSocket( sock ) == -1 )
      return -1;
   
   /* Get addressinformation of host */
   hostinfo = gethostbyname( host );
   if( hostinfo == NULL )
      return -1;

   /* save address of host */
   sock->name.sin_addr = *(struct in_addr *)hostinfo->h_addr;

   return 0;
}

/*
 * Set socket option
 */
int sockSetOption( sockStruct* sock, int option, int _on )
{
   int on = _on;
   if( sock == NULL )
      return -1;
   return setsockopt( sock->fd, SOL_SOCKET, option, (const char*)&on, sizeof( on ) );
}

/*
 * Listen.
 */
int sockListen( sockStruct* sock )
{
   if( sock == NULL )
      return -1;
   return listen( sock->fd, BACKLOG );
}

/*
 * Accept
 *
 * Save information of new client in client sockStruct
 */
int sockAccept( sockStruct* sock, sockStruct* client )      
{
   int size;
   
   if( sock == NULL || client == NULL )
      return -1;
      
   size = sizeof( client->name);
   client->fd = accept( sock->fd, (struct sockaddr*)&(client->name), &size );
   
   if( client->fd == -1 )
      return -1;
   client->port = client->name.sin_port;
   return 0;
}

/*
 * Connect
 * 
 * The host to connect to is given as argument to sockClient
 */
int sockConnect( sockStruct* sock )
{
   int result;
   if( sock == NULL )
      return -1;
   result = connect( 
      sock->fd, (struct sockaddr *)&(sock->name), sizeof( sock->name ) );
   return result;
}

/* 
 * Who are we connected to
 */
const char* sockConnectedTo( sockStruct* sock )
{
   struct hostent* host;
   static char host_name[256];
   host = 
      gethostbyaddr( (char*)&sock->name.sin_addr, 
                  sizeof( struct in_addr ), AF_INET );
   strncpy( host_name, host->h_name, sizeof host_name );
   host_name[sizeof host_name - 1] = '\0';
   return host_name;
}

/*
 * Send data
 */      
int sockSend( sockStruct* sock, void* data, size_t size )
{
   if( sock == NULL )
      return -1;
   return
      sendto( sock->fd, data, size, 0, (struct sockaddr*)&sock->name, sizeof( sock->name ) );
}

/* 
 * Receive data
 *
 * Return sockProt with information of remote host. In this way,
 * we can receive broadcast replies in a clean way.
 */
int sockRecv( sockStruct* sock, sockStruct* result, void* data, size_t size )
{
   int fromlen;
   int nbytes;
      
   if( sock == NULL || result == NULL )
      return -1;
   fromlen = sizeof( sock->name );

   nbytes = 
      recvfrom( sock->fd, data, size, 0, (struct sockaddr*)&result->name, &fromlen );

   result->fd = sock->fd;
   result->prot = sock->prot;
   result->port = result->name.sin_port;
   return nbytes;
}

/*
 * EOF libtcpip/libtcpip.c
 */
