/*
 * $Id: ttyfuncs.c,v 1.24 1998/02/17 09:52:24 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/libtty/ttyfuncs.c,v $
 * $Revision: 1.24 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
/* 
 * Parts of this file are based on dip by Fred N. van Kempen.
 *
 *
 * dip      A program for handling dialup IP connecions.
 *    UNIX terminal I/O support functions.  This file takes
 *    care of opening, setting up and maintaining the line,
 *    and it takes care of allocating the buffers and init-
 *    ializing their control structures.
 *
 * 
 *
 *    Author:    Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *    Copyright 1988-1993 MicroWalt Corporation
 *    Lock file stuff stolen from Taylor UUCP
 *    Copyright (C) 1991, 1992 Ian Lance Taylor
 * 
 *    This program is free software; you can redistribute it
 *    and/or  modify it under  the terms of  the GNU General
 *    Public  License as  published  by  the  Free  Software
 *    Foundation;  either  version 2 of the License, or  (at
 *    your option) any later version.
 */   
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <termios.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <pwd.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include <libport/libport.h>

/* This might not be there */
#ifndef CRTSCTS
#define CRTSCTS 0
#endif

#include "ttyfuncs.h"

/*
 * Datastructure used for terminals.
 * Currently only one terminal can be used at the same time
 * so the structure is not really needed
 */
typedef struct
{
   int fd;
   char* dev;
   int   locked;
   struct termios   tty_saved;
   struct termios   tty_current;
} _ttyData;

static struct {
  char   *speed;
  int code;
} tty_speeds[] = {         /* table of usable baud rates */
  { "0",        B0      },
  { "50",   B50   }, { "75",  B75   },
  { "110",  B110  }, { "300", B300  }, 
  { "600",  B600  }, { "1200",   B1200 },
  { "2400", B2400 }, { "4800",   B4800 },
  { "9600", B9600 },
#ifdef B14400
  { "14400",   B14400   },
#endif
#ifdef B19200
  { "19200",   B19200   },
#endif
#ifdef B38400
  { "38400",   B38400   },
#endif
#ifdef B57600
  { "57600",   B57600   },
#endif
#ifdef B115200
  { "115200",  B115200  },
#endif
  { NULL,   0  }
};


static _ttyData* ttyData = NULL;

/*
 * return name of terminal in use
 */
char* tty_device_name()
{
   if( ttyData == NULL )
      return NULL;
   return ttyData->dev;
}

/*
 * Allocate new tty structure 
 */
static _ttyData* tty_alloc()
{
   _ttyData* tty = (_ttyData*)malloc( sizeof( _ttyData ) );
   if( tty == NULL )
   {
      FAIL( "malloc" );
      exit( 1 );
   }
   tty->fd = -1;
   tty->dev = NULL;
   tty->locked = 0;
   return tty;
}

/*
 * Free memeory used for tty structure
 */
static void tty_free( _ttyData* tty )
{
   if( tty == NULL )
      return;
   free( tty->dev );
   free( tty );
}

/* 
 * Is dev already locked
 */
int tty_already_locked(char *dev) /* nam - complete path to lock file */
{
  int  i = 0, pid = 0;
  FILE *fd = NULL;
   char* lock_file = lock_file_name( dev );
  /* Does the lock file on our device exist? */
   fd = fopen( lock_file, "r" );
   if( fd == NULL )
      return 0;

  /* Yes, the lock is there.  Now let's make sure */
  /* at least there's no active process that owns */
  /* that lock.                                   */
  i = fscanf(fd, "%d", &pid);
  fclose(fd);
  
  if (i != 1) /* Lock file format's wrong! Kill't */
    return 0;

  /* We got the pid, check if the process's alive */
  if (kill(pid, 0) == 0)      /* it found process */
      return 1;           /* Yup, it's running... */

  /* Dead, we can proceed locking this device...  */
  return 0;
}

/* 
 * Lock the tty 
 */
static int
tty_lock( _ttyData* tty, int mode)
{
   static char* lock_file;
   struct passwd *pw;
   pid_t ime;
   FILE *fd; 
#if HAVE_V2_LOCKFILES
   int cwrote;
#endif /* HAVE_V2_LOCKFILES */

   if( tty == NULL )
      return -1;
   lock_file = lock_file_name( tty->dev );

   if (mode == 1)
   {  
      /* lock */
      if (tty_already_locked( tty->dev ) == 1)
      {
         FAILn( "attempt to use already locked tty %s", lock_file );            
         exit( 1 );
      }
      if ((fd = fopen(lock_file, "w")) == (FILE *)0)
      {
         FAIL1( "fopen", lock_file );
         exit( 1 );
      }
      ime = getpid();
#ifdef HAVE_V2_LOCKFILES
      TEMP_FAILURE_RETRY( cwrote, write (fileno(fd), &ime, sizeof(ime)) );
#else
      fprintf(fd, "%10d\n", (int)ime);
#endif
      fclose(fd);
       
      /* Make sure UUCP owns the lockfile.  Required by some packages. */
      if ((pw = getpwnam( _UID_UUCP )) == NULL) 
      {
         FAILn( "lock: UUCP user %s unknown!", _UID_UUCP);
         return 0;  /* keep the lock anyway */
      }
      chown(lock_file, pw->pw_uid, pw->pw_gid);
      tty->locked = 1;
   } 
   else
   if (mode == 2)
   { 
      /* re-acquire a lock after a fork() */
      if (tty->locked != 1) 
      {
         FAILn( "tty_lock reaquire: lock was not saved!", "");
         exit( 1 );
      }
      if ((fd = fopen(lock_file, "w")) == (FILE *)0)
      {
         FAIL1( "fopen", lock_file );
         exit( 1 );
      }
      ime = getpid();
#ifdef HAVE_V2_LOCKFILES
      TEMP_FAILURE_RETRY( cwrote, write (fileno(fd), &ime, sizeof(ime)) );
#else
      fprintf(fd, "%10d\n", (int)ime);
#endif
      fclose(fd);
      chmod(lock_file, 0444);
      chown(lock_file, getuid(), getgid());
      return 0;
   } 
   else
   {   
      /* unlock */
      if ( tty->locked != 1)
      {
         FAILn( "tty_lock: lock was not saved?!", "" );
         return 0;
      }
      if ((fd = fopen(lock_file, "w")) == NULL )
      {
         FAIL1( "fopen", lock_file );
         exit( 1 );
      }
      if (unlink(lock_file) < 0)
      {
         FAIL1( "unlink", lock_file );
         tty->locked = 0;
         exit( 1 );
      }
      tty->locked = 0;
   }
   return 0;
}

/* Find a serial speed code in the table. */
static int
tty_find_speed(char *speed)
{
   int i;

   i = 0;
   while (tty_speeds[i].speed != NULL) 
   {
      if (! strcmp(tty_speeds[i].speed, speed)) 
      {
         return(tty_speeds[i].code);
      }
      i++;
   }
   return -EINVAL;
}

/* Set the number of stop bits. */
static int
tty_set_stopbits(struct termios *tty, char *stopbits)
{
  switch(*stopbits) {
   case '1':
      tty->c_cflag &= ~CSTOPB;
      break;

   case '2':
      tty->c_cflag |= CSTOPB;
      break;

   default:
      return -EINVAL;
  }
  return 0;
}

/* Set the number of data bits. */
static int
tty_set_databits(struct termios *tty, char *databits)
{
  tty->c_cflag &= ~CSIZE;
  switch(*databits) {
   case '5':
      tty->c_cflag |= CS5;
      break;

   case '6':
      tty->c_cflag |= CS6;
      break;

   case '7':
      tty->c_cflag |= CS7;
      break;

   case '8':
      tty->c_cflag |= CS8;
      break;

   default: 
      return -EINVAL;
  }
  return 0;
}  

/* Set the type of parity encoding. */
static int
tty_set_parity(struct termios *tty, char *parity)
{
  switch(toupper(*parity)) {
   case 'N':
      tty->c_cflag &= ~(PARENB | PARODD);
      break;  

   case 'O':
      tty->c_cflag &= ~(PARENB | PARODD);
      tty->c_cflag |= (PARENB | PARODD); 
      break;

   case 'E':
      tty->c_cflag &= ~(PARENB | PARODD);
      tty->c_cflag |= (PARENB);
      break;

   default:
      return -EINVAL;
  }
  return 0;
}

/* Set the line speed of a terminal line. */
static int
tty_set_speed(struct termios *tty, char *speed)
{
   speed_t spd;
   spd = tty_find_speed( speed );

   if( cfsetispeed( tty, spd ) == -1 )
   {
      FAIL( "setispeed" );
      exit( 1 );
   }
   if( cfsetospeed( tty, spd ) == -1 )
   {
      FAIL( "cfsetospeed" );
      exit( 1 );
   }

   tty->c_cflag |= CREAD;
   tty->c_cflag &= ~(CLOCAL);

   return 0;
}
 
/* Put a terminal line in a transparent state. */
int
tty_set_raw(struct termios *tty)
{
   int i;
   speed_t ospeed;
   speed_t ispeed;
   
   for(i = 0; i < NCCS; i++)
      tty->c_cc[i] = '\0';

   tty->c_cc[VMIN] = 1;
   tty->c_cc[VTIME] = 5;
   tty->c_iflag = IGNBRK ; /*| IGNPAR;*/
   tty->c_oflag = 0;
   tty->c_lflag = 0;

   ospeed = cfgetospeed( tty );
   ispeed = cfgetispeed( tty );
   
   tty->c_cflag = CRTSCTS | HUPCL | CREAD;

   cfsetispeed( tty, ispeed );
   cfsetospeed( tty, ospeed );

   return 0;
}

/* Fetch the state of a terminal. */
static int
tty_get_state( int fd, struct termios* tty )
{
   if (fd >= 0) 
   {
      if (tcgetattr(fd, tty ) < 0) 
      {
         FAIL( "tcgetattr" );
         exit( 1 );
      }
      return 0;
   }
   return -1;
}
 
/* Set the state of a terminal. */
static int
tty_set_state( int fd, struct termios *tty)
{
   if (fd >= 0) 
   {
      if (tcsetattr(fd, TCSANOW, tty) < 0) 
      {
         FAIL( "tcsetattr" );
         exit( 1 );
      }
      return 0;
   }
   return -1; 
}  

/* Return the TTY link's file descriptor. */
int
tty_askfd()
{  
   if( ttyData == NULL )
      return -1;
   return ttyData->fd;
}

/* Set the number of databits a terminal line. */
int
tty_databits(char *bits)
{  
   if( ttyData == NULL )
      return -1;
   if (tty_set_databits(&ttyData->tty_current, bits) < 0) 
      return -1;
   return tty_set_state(ttyData->fd, &ttyData->tty_current);
}

/* Set the number of stopbits of a terminal line. */
int
tty_stopbits(char *bits)
{  
   if( ttyData == NULL )
      return -1;
   if (tty_set_stopbits(&ttyData->tty_current, bits) < 0) 
      return -1;
   return tty_set_state(ttyData->fd, &ttyData->tty_current);
}

/* Set the type of parity of a terminal line. */
int
tty_parity(char *type)
{  
   if( ttyData == NULL )
      return -1;
   if (tty_set_parity(&ttyData->tty_current, type) < 0) 
      return -1;
   return tty_set_state( ttyData->fd, &ttyData->tty_current);
}   

/* Set the line speed of a terminal line. */
int
tty_speed(char *speed)
{
   if (tty_set_speed(&ttyData->tty_current, speed) < 0) 
      return -1;

   return tty_set_state( ttyData->fd, &ttyData->tty_current);
}

/* Hangup the line. */
int
tty_hangup(void)
{
   struct sigaction act;
   struct sigaction saved;
   struct termios trm;
   struct termios old;
   
   if( ttyData == NULL )
      return -1;
   if( ttyData->fd == -1 )
      return 0;

   /* Ignore SIGHUP during hangup. Afterward we reset the SIGHUP signal */
   if( sigaction( SIGHUP, NULL, &saved ) == -1 ||
       sigaction( SIGHUP, NULL, &act   ) == -1 )
   {
      FAIL( "sigaction" ); 
      exit( 1 );
   }
   act.sa_handler = SIG_IGN;
   if( sigaction( SIGHUP, &act, NULL ) == -1 )
   {
      FAIL( "sigaction" );
      exit( 1 );
   }

   if ( tcgetattr( ttyData->fd, &trm ) == -1 )
   {
      FAIL( "tcgetattr" );
      exit( 1 );
   }
   if( tcgetattr( ttyData->fd, &old ) == -1 )
   {
      FAIL( "tcgetattr" );
      exit( 1 );
   }
   
   if( cfsetospeed( &trm, B0 ) == -1 )
   {
      FAIL( "cfsetospeed" );
      exit( 1 );
   }
   
   if( tcsetattr( ttyData->fd, TCSANOW, &trm ) == -1 )
   {
      FAIL( "tcsetattr" );
      exit( 1 );
   }
   
   sleep( 1 );
   
   /* restore terminal settings */
   tcsetattr( ttyData->fd, TCSANOW, &old );
   
   /* restore SIGHUP */
   if( sigaction( SIGHUP, &saved, NULL ) == -1 )
   {
      FAIL( "sigaction" );
      exit( 1 );
   }
   
   return 0;
}
   
/* Flush input on the terminal. */
int
tty_flush( int fd)
{
   return(0);
}

/* Reset terminal settings */
int tty_reset()
{
   int result;
   if( ttyData == NULL )
      return -1;
   if( ttyData->fd < 0 )
      return -1;
   result = tty_set_state( ttyData->fd, &ttyData->tty_saved );
   return result;
}


/* Close down a terminal line. */
int
tty_close(void)
{
   int result;
   if( ttyData == NULL )
      return -1;

   if( ttyData->fd < 0 )
      return -1;

   close( ttyData->fd );
   ttyData->fd = -1;
   result = tty_lock( ttyData, 0);
   tty_free( ttyData );
   ttyData = NULL;
      
   return result;
}

/* Open and initialize a terminal line. */
int
tty_open(char *name)
{
   int flags;

   ttyData = tty_alloc();
   if( ttyData == NULL )
   {
      FAIL( "tty_alloc" );
      exit( 1 );
   }
   
   ttyData->fd = -1;
   
   /* Try opening the TTY device. */
   if( name != NULL )
   {
      ttyData->dev = strdup(  make_path( name ) );
      /* Now - can we lock it? */
      if( tty_lock( ttyData, 1) ) 
      {
         return -1;
      }
      ttyData->fd = open( ttyData->dev, O_RDWR | O_NONBLOCK );
      if( ttyData->fd < 0 ) 
      {
         tty_lock( ttyData, 0);
         FAIL1( "open", ttyData->dev );
         exit( 1 );
      }
   } 
   else
      return -1;
   
   
   /* Fetch the current state of the terminal. */
   if (tty_get_state( ttyData->fd, &ttyData->tty_saved) < 0) 
   {
      return -1;
   }
   
   memcpy ((char *)&ttyData->tty_current,
      (char *)&ttyData->tty_saved, sizeof(struct termios));

   /* Put this terminal line in a 8-bit transparent mode. */
   if (tty_set_raw( &ttyData->tty_current ) < 0) 
   {
      return -1;
   }   

   /* If we are running in MASTER mode, set the default speed. */
   if ((name != NULL) && (tty_set_speed(&ttyData->tty_current, "38400") != 0)) 
   {
      FAILn( "tty_open: cannot set 38400 bps!", "");
      exit( 1 );
   }   

  /* Set up a completely 8-bit clean line. */
   if (tty_set_databits(&ttyData->tty_current, "8") ||
       tty_set_stopbits(&ttyData->tty_current, "1") || 
       tty_set_parity(&ttyData->tty_current, "N")) 
   {  
      FAILn( "libtty: tty_open: cannot set 8N1 mode!", "" );
      exit( 1 );
   }   

   /* Set the new line mode. */
   if( tty_set_state( ttyData->fd, &ttyData->tty_current) < 0) 
      return -1;
   /* Clear the NDELAY flag now (line is in CLOCAL) */
   flags = fcntl( ttyData->fd, F_GETFL, 0);
   fcntl( ttyData->fd, F_SETFL, flags & ~O_NONBLOCK );

  /* OK, all done.  Lock this terminal line. */
   return ttyData->fd;
}

void tty_sendbreak(void)
{
   if( ttyData == NULL )
      return;

   if (ttyData->fd >= 0)
      tcsendbreak(ttyData->fd, 0);
}
 
/*
 * creates absolute path name from path, possibly appending
 * /dev/ in front of path
 */
char* make_path( char* path )
{
   static char buf[_POSIX_PATH_MAX];
   /* absolute pathname */
   if( path[0] == '/' )
      return path;
   sprintf( buf, "/dev/%s", path );
   return buf;   
}

/*
 * create valid lock file name e.g.
 * /var/spool/LCK..cua1 from dev
 *
 */
char* lock_file_name( char* dev )
{
   static char lock_file[_POSIX_PATH_MAX];
   dev = basename( dev );
   sprintf( lock_file, "%s/LCK..%s", _PATH_LOCKD, dev );
   return lock_file;
}

/*
 * EOF libtty/ttyfuncs.c
 */
