/*
 * $Id: ttyutil.c,v 1.12 1998/02/17 09:52:25 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/libtty/ttyutil.c,v $
 * $Revision: 1.12 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/ioctl.h>

#include <libconf/libconf.h>
#include <modem_defs.h>
#include "ttyutil.h"
#include "ttyfuncs.h"

/* Option struct for tty's */
struct _tty_option
{
   char* name;
   int (*func)(char* );
};

/* 
 * These options are currently understood.
 * You can set them in tghe configurationfile
 */
static struct _tty_option tty_options[] = {
  { "databits",   tty_databits },
  { "parity",      tty_parity },
  { "speed",      tty_speed },
  { "stopbits",   tty_stopbits },
  { NULL,          NULL }
};

/* Name of terminal in use */
char* ttyDev()
{
   return tty_device_name();
}

/* Set terminal in raw mode */
int ttySetRaw( int fd )
{
   struct termios tty;
   if( tcgetattr( fd, &tty ) < 0 )
      return -1;

   tty_set_raw( &tty );

   if( tcsetattr( fd, TCSANOW, &tty ) < 0 )
      return -1;
   
   return 0;
}

/* Close terminal */
int ttyClose()
{
   return tty_close();
}

/* Reset terminal settings */
int ttyReset()
{
   return tty_reset();
}

int ttyHangup()
{
   return tty_hangup();
}

/* 
 * Find a free terminal
 * Check all terminals defined in the conf file,
 * Return name of first one that's free
 */
static char* ttyFindFreeTty( conf_file* cf )
{
   grp* group;
   group = goto_first_group( cf );
   while( group != NULL )
   {
      if( !tty_already_locked( group_name( group ) ) )
         return group_name( group );
      group = goto_next_group( cf );
   }
   return NULL;
}

/*
 * Read ans set options from config file for terminal
 */
static int ttyConfigure( char* dev, conf_file* cf )
{
   int status = 0;
   item* item;
   char* key;
   int i;
   
   goto_group( cf, dev );
   item = goto_first_item( cf );
   while( item != NULL )
   {
      key = item_key( item );
      i = 0;
      while( tty_options[i].name != NULL )
      {
         if( strcasecmp( tty_options[i].name, key ) == 0 )
            break;
         i++;
      }   
      if( tty_options[i].name != NULL )
      {
         status = tty_options[i].func( item_value( item ) );
         if( status < 0 )
         {
            error( "ttyConfigure: %m" );
            return -1;
         }
      }
      item = goto_next_item( cf );
   }
   return 0;
}

/*
 * Is there a tty defined in confFile available */
int ttyAvailable( char* confFile )
{
   conf_file* cf;
   int available;
   
   cf = open_conf( confFile );
   if( cf == NULL )
      return 0;
   if( ttyFindFreeTty( cf ) == NULL )
      available = 0;
   else
      available = 1;
   close_conf( cf );
   
   return available;
}

/*
 * Open and configure one of the tty's defined in the confFile
 */
int ttyOpen( const char* confFile )
{
   conf_file* cf;
   int   ttyfd;
   char* tty_name;
      
   cf = open_conf( confFile );
   if( cf == NULL )
      return -1;
   tty_name = ttyFindFreeTty( cf );
   if( tty_name == NULL )
   {
      close_conf( cf );
      return -1;
   }
   
   ttyfd = tty_open( tty_name );
   if( ttyfd == -1 )
   {
      close_conf( cf );
      return -1;
   }
   if( ttyConfigure( tty_name, cf ) == -1 )
   {
      tty_close();
      close_conf( cf );
      return -1;
   }
   close_conf( cf );
   return ttyfd;
}   
   
/*
 * EOF libtty/ttyutil.c
 */
