/*
 * $Id: daemon.c,v 1.11 1998/02/17 09:53:11 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/modemd/daemon.c,v $
 * $Revision: 1.11 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/ioctl.h>
/*#include <sys/file.h> */
#include <sys/stat.h> /* for umask */
#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <libport/libport.h>
#include <liberror/liberror.h>
#include <libport/libport.h>
#include <modem_defs.h>
#include "log.h"
#include "configuration.h"

static int Fork( void );
static int CloseFileDescriptors( void );
static int DetachTTY( void );
static int Chdir( const char* serverDir );
static int Umask( int Umask );
static int OpenDummys( void );
static int LockFile( const char* lockFile );

/*
 * Function: startup_daemon
 *
 * Description:
 *   Create daemon process by forking, and let parent exit
 *
 * Arguments:
 *   - int   daemon   Indicates whether or not we go in
 *                     background by forking.
 * Returns:
 *   - int   0     On success
 *
 *
 * Steps taken in this function:
 *
 * 1) fork, and let parent exit  (if daemon == 0)
 * 2) detach from controlling tty (if daemon == 0 )
 * 3) Chdir to known directory ('server_dir' in config file)
 *    default: SERVER_DIR
 * 4) Set umask ( 'server_umask' in config file )
 *    default: UMASK
 * 5) Create lock file ('server_lock_file'  in conf file)
 *    default: LOCKFILE
 * 6) Close all file descriptors (if daemon == 0 )
 * 7) Open dummy file descriptors (if daemon == 0 )
 *    (/dev/null) for stdin, stdout
 *
 */
int startup_daemon( int daemon)
{
   int    umask;
   const char* serverDir;
   const char* lockFile;
   
   umask     = get_server_umask();
   serverDir = get_server_dir();
   lockFile  = get_server_lock_file();

   /* step 1 */
   if( daemon )
      Fork();
   
   /* step 2 */
   if( daemon )
      DetachTTY();
   
   /* step 3 */
   Chdir( serverDir );
   
   /* step 4 */
   Umask( umask );
   
   
   /* step 5 */
   if( daemon )
      CloseFileDescriptors();

   startLog( daemon );
   
   /* step 6 */
   if( daemon )
      OpenDummys();

   /* step 7 */
   LockFile( lockFile );
   
   
   return 0;
}


/*
 * Function: Fork
 *
 * Description:
 *   Step 1 in startup_daemon function.
 *   fork and let parent exit.
 *
 * Arguments:
 *   - None
 *
 * Returns:
 *   - int   0   On success
 *
 * Remarks:
 *   This function exits on error
 *
 *
 */
static int Fork( void )
{
   switch( fork() )
   {
      case 0: /* child */
         break;

      case -1:
         FAIL( "fork" );
         exit( 1 );
      default: /* parent */
         exit(0);   /* parent exists*/
   }
   return 0;
}

/*
 * Function: CloseFileDescriptors
 *
 * Description:
 *   Step 5 in startup_daemon function.
 *   close all filedescriptors of current process.
 *
 * Arguments:
 *   - None
 *
 * Returns:
 *   - int   0   On success (this is always the case)
 *
 */
static int CloseFileDescriptors( void )
{
   int i;
   
   for( i = sysconf( _SC_OPEN_MAX ); i >= 0; --i )
      close( i );
   
   return 0;
}

/*
 * Function: DetachTTY
 *
 * Description:
 *   Step 2 in startup_daemon function.
 *   Detach from tty.
 *   Don't receive signals from process that started server
 *
 * Arguments:
 *   - None
 *
 * Returns:
 *   - int   0   On success (always the case)
 *
 */
static int DetachTTY( void )
{
   pid_t pgrp;
   pgrp = setsid();
   
   if( pgrp == -1 )
   {
      FAIL( "setsid" );
      exit( 1 );
   }
   return 0;
}

/*
 * Function: Chdir
 *
 * Description:
 *   Step 3 in startup_daemon function.
 *   Change to known directory where server runs.
 *
 * Arguments:
 *   - const char*   serverDir   Dir. to chdir to.
 *
 * Returns:
 *   - int     0        On success (always the case)
 *
 * Remarks:
 *   This function exits on error
 *
 *
 */
static int Chdir( const char* serverDir )
{
   if( chdir( serverDir ) < 0 )
   {
      FAIL1( "chdir", serverDir );
      exit( 1 );
   }

   return 0;
}

/*
 * Function: Umask
 *
 * Description:
 *   Step 4 in startup_daemon function.
 *   Change umask of current process.
 *
 * Arguments:
 *   - int    umask   The umask to set.
 *
 * Returns:
 *   - int   0    On success (Always the case)
 *
 */
static int Umask( int Umask )
{
   umask( Umask );

   return 0;
}


/*
 * Function: OpenDummys
 *
 * Description:
 *   Step 6 in startup_daemon function.
 *   filedescr. 0, 1, 2 are closed. Since some standard
 *   functions get confused when one of these are closed,
 *   we use dummy's for them.
 *
 * Arguments:
 *   - None
 *
 * Returns:
 *   - int   0   On success (always the case)
 *
 */
static int OpenDummys( void )
{
   int fd;
   
   fd = open( "/dev/null", O_RDWR );   /* stdin */
   if( fd == -1 )
   {
      FAIL1( "open", "/dev/null" );
      exit( 1 );
   }
   if( dup( fd ) == -1   ||  /* stdout */
       dup( fd ) == -1 )     /* stderr */
   {
      FAIL( "dup" );
      exit( 1 );
   }
   return 0;
}

/*
 * Function: LockFile
 *
 * Description:
 *   Step 7 in startup_daemon function.
 *   Create lockfile, to avoid multiple copies,
 *   and write pid in lockfile.
 *
 * Arguments:
 *   - const char*   lockfile   Filename of lockfile to use
 *
 * Returns:
 *   - int     0       On success (always the case)
 *
 * Remarks:
 *   This function exits on error.
 *
 */
static int LockFile( const char* lockFile )
{
   int   lf;
   char  buf[10];
   int   result;
   struct flock lock;

   /* Fill in flock structure */
   lock.l_type = F_WRLCK;     /* Exclusive write lock */
   lock.l_start = 0;          /* from offset 0 ... */
   lock.l_whence = SEEK_SET;
   lock.l_len = 0;            /* ... until eof */
   lock.l_pid = getpid();   
   
   lf = open( lockFile, O_RDWR | O_CREAT, 0640 );
   if( lf < 0 )      /* error opening file */
   {
      FAIL1( "open", lockFile );
      exit( 1 );
   }

   /* Check whether lock file is already locked */
   if( fcntl( lf, F_GETLK, &lock ) == -1 )
   {
      FAIL1( "fcntl", lockFile );
      exit( 1 );
   }

   /* Alread locked! modemd server is already running */
   if( lock.l_type != F_UNLCK )
   {
      error( "server already started" );
      exit( 1 );
   }

   /* Set lock */
   lock.l_type = F_WRLCK;
   if( fcntl( lf, F_SETLK, &lock ) == -1 )
   {
      FAIL1( "fcntl", lockFile );
      exit( 1 );
   }
   
   /* Record process id. */
   sprintf( buf, "%-6d", (int)getpid() );
   TEMP_FAILURE_RETRY( result, write( lf, buf, strlen( buf ) ) );
   if( result == -1 )
   {
      FAIL( "write" );
      exit( 1 );
   }    

   return 0;
}

/*
 * EOF modemd/daemon.c
 */
