/*
 * $Id: main.c,v 1.14 1998/02/17 09:53:13 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/modemd/main.c,v $
 * $Revision: 1.14 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>

#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include <sys/stat.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include <modem_defs.h>

#ifdef HAS_GETOPT_H
#include <getopt.h>
#endif
extern char* optarg;

#include <libtcpip/libtcpip.h>
#include <liberror/liberror.h>
#include "log.h"
#include "modemd.h"
#include "configuration.h"

/* Do we want to become a daemon process */
static int DaemonMode = 1;
static char hostName[40];

#define USAGE_MSG \
   "modemd (" ## VERSION ## ")\n"\
   "usage:\n" \
   "      modemd [-option ...]\n" \
   "\n" \
   "where options include:\n"\
   "     -h           print out this message\n"\
   "     -n           no daemon mode.\n"\
   "     -v           displays version information\n"\
   "\n"

#define VERSION_MSG \
   "modemd version " ## VERSION ## "\n"

static char* opt_string = "hnv";

static void usage() 
{
   fprintf( stdout, USAGE_MSG );
}

static void version()
{
   fprintf( stdout, VERSION_MSG );
}

/*
 * Creates pathname p
 */
static int dirMake( const char* p )
{
   char *s;
   char* ptr;

   struct stat buf;
   int    result;
   
   s = strdup( p );
   if( s == NULL )
   {
      FAILn( "strdup: %s", "not enough memory" );
      exit( 1 );
   }
   
   ptr = s;
   
   do
   {
      /* find next component from path */
      ptr = strchr( ptr + 1, '/' );
      
      /* 
       * Make pathname from component.
       * a/b becomes a,
       * a   remains a
       */
      
      if( ptr != NULL ) 
         *ptr = '\0';

      /* check whether component exists */
      result = stat( s, &buf );
      
      if( result == -1 )
      {
         /* No, then create  it */
         if( errno == ENOENT )
         {
            result = mkdir( s, S_IRWXU | S_IRWXG  | S_IXOTH | S_IROTH );
            if( result == -1 )
            {
               /* mkdir failed for some raison return */
               FAIL1( "mkdir", s );
               free( s );
               exit( 1 );
            }
         }
         else
         {
            /* some other error during stat function call => abort */
            FAIL1( "stat", s );
            free( s );
            exit( 1 );
         }
      }
      
      /* restore path name */
      if( ptr != NULL )
         *ptr = '/';
   /* coninue until we've created all the path components */
   } while( ptr != NULL );

   /* free used memory */
   free( s );
   return 0;
}
      
   
/*
 * Shutdown the server
 */
static void serverShutdown()
{
   exit( 0 );
}

/*
 * Create an error string containing a timestamp, the name
 * of the host the server is running on and the error message.
 */
static void errMsg( const char* _msg )
{
   fprintf( ERR_LOG, "%s[%s] %s\n", timeStamp(), hostName, _msg );
}   

/*
 * Signal handler.
 * On receipt of a SIGTERM, the server will exit.
 */
static void signal_handler( int sig )
{
   switch( sig )
   {
      case SIGCHLD:
         break;
      case SIGTERM:
         error( "caught SIGTERM, shutting down" );
         serverShutdown();
         break;
   }
}

void main( int argc, char* argv[] )
{
   int c;
   struct sigaction act;
   
   /* Set signals */
   sigemptyset( &act.sa_mask );
   act.sa_handler = SIG_IGN;
#ifdef SA_RESTART
   act.sa_flags   = SA_RESTART;
#endif
#ifdef SA_NOCLDWAIT
   act.sa_flags |= SA_NOCLDWAIT;
#endif
   sigaction( SIGCHLD, &act, NULL );
   sigaction( SIGHUP,  &act, NULL );
   sigaction( SIGINT,  &act, NULL );
   sigaction( SIGQUIT, &act, NULL );

   act.sa_handler = signal_handler;
   sigaction( SIGTERM, &act, NULL );
   
   gethostname( hostName, sizeof(hostName ) );   
   setErrorFunc( errMsg );

   while( 1 )
   {
      c = getopt( argc, argv, opt_string );
      if( c == -1 )
         break;
      switch( c )
      {
         case 'h' :
            usage();
            exit( 0 );
         case 'n' :
            DaemonMode = 0;
            break;
         case 'v' :
            version();
            exit( 0 );
         default:
            exit( 1 );
            break;
      }
   }
   
   init_configuration();
   
   if( !DaemonMode )
   {
      sigemptyset( &act.sa_mask );
      act.sa_handler = serverShutdown;
      sigaction( SIGINT,  &act, NULL );
   }
   
   if( dirMake( get_server_dir() ) == -1 )
      exit( 1 );
   startup_daemon( DaemonMode );      
   server();
}
/*
 * EOF modemd/main.c
 */
