/*
 * $Id: service.c,v 1.30 1998/02/17 09:53:20 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/modemd/service.c,v $
 * $Revision: 1.30 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <time.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include <libtty/libtty.h>
#include <libtcpip/libtcpip.h>
#include <libport/libport.h>

#ifndef WEXITSTATUS
#define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8 )
#endif

#ifndef WIFEXITED
#define WIFEXITED(stat_val) (((stat_val) & 256) == 0 )
#endif


/* for connectStr and speedStr */
#define USE_MODEM_REPLY
#include <libmodem/libmodem.h>
#include <libconf/libconf.h>
#include <liberror/liberror.h>
#include <libprotocol/libprotocol.h>
#include <modem_defs.h>
#include "log.h"
#include "syscmd.h"
#include "configuration.h"

static void serviceSignalHandler( int );
static void serviceExit( void );
static void serviceModem2Client( void );
static void serviceClient2Modem( protType, void*, size_t );
static void serviceMainLoop( void );
static void serviceCommandHandler( protType, void*, size_t );
static void serviceSetup( const char*, const char*, int );
static void serviceErrorFunc( const char* err );
static void serviceDial( const char* );
static void serviceHangup();
static char serviceReadModem( void /*int, void*, size_t */);
static int  serviceCheckNumber( void );

struct
{
   int    modem;            /* Modem device */
   int    allocated;        /* 0/1: no modem has/has not been alocated */
   int    client;           /* Socket */
   int    connected;        /* 0=not connected, 1=connected */
   char*  remoteHost;       /* Name of host connected to us */
   char*  remoteUser;       /* Name of user connected to us */
   char*  localHost;        /* Name of host server is running on */
   char*  number;           /* Phone number to dial */
   time_t connected_since;  /* Time the modem connection started */
} Data;

static char buf[protBufSize];
static jmp_buf saved_env;

/*
 * Fork and start handling client on remote host
 */
void handle_client( int sock, const char* host, const char* uName )
{
   /* Setup the service and start the event loop */
   serviceSetup( host, uName, sock );
   serviceMainLoop();
}

/*
 * Allocate and initialize modem device.
 * Save information about connection and do some other stuff.
 */
static void serviceSetup( const char* host, const char* uName, int socket )
{
   char tmp[40];
   struct sigaction act;
   int    result;
   
   /* Asure clean exit */
   atexit( serviceExit );
   
   /* Start new session and become session leader */
   if( setsid() == -1 )
   {
      FAIL( "setsid" );
      exit( 1 );
   }

   /* Save remoteHost, remoteUser and local hostname */
   gethostname( tmp, sizeof tmp );
   Data.localHost  = strdup( tmp );
   Data.remoteHost = strdup( host );
   Data.remoteUser = strdup( uName );

   /* no modem allocated yet */
   Data.allocated = 0; 

   /* Initial value for modem file desc */
   Data.modem = -1;

   /* Start in disconnected state */
   Data.connected = 0;

   /* Save fdesc of client */
   Data.client = socket;
   
   /* Allocate new modem device defined in MODEM_CONFIG_FILE and in
    * configuration file as: modem_config_file. Modem device becomes
    * our controlling terminal.
    */
   Data.modem = ttyOpen( get_modem_config_file() );
   if( Data.modem == -1 )
   {
      if( protRequestConfirm( Data.client, PROT_REQUEST_FAILED ) < 0 )
      {
         FAIL( "protRequestConfirm" );
         exit( 1 );
      }
      exit( 0 );
   }

#if defined(TIOCSTTY) && !defined(CIBAUD)
         /*
          * 4.3+BSD way to acquire controlling terminal
          * !CIBAUD to avoid doing this under SunOS
          * from Advanced Pogramming in the UNIX Environment
          */
         if( ioctl( Data.modem, TIOCSCTTY, NULL ) < 0 )
         {
            FAIL( "ioctl(TIOCSCTTY)" );
            exit( 1 );
         }
#endif
   /* 
    * Initialize and reset modem using configuration file
    * MODEM_CONFIG_FILE or modem_config_file in configuration file
    */
   if( mdm_init( Data.modem, ttyDev(), get_modem_config_file() ) == -1 )
   {
      result =protRequestConfirm( Data.client, PROT_REQUEST_FAILED );
      if( result < 0 )
      {
         FAIL( "protRequestConfirm" );
         exit( 1 );
      }
      if( result == 0 )
         exit( 0 );

      FAIL( "modem init" );
      exit( 1 );
   }

   /* 
    * Install protocol handlers for PROT_DATA and
    * PROT_COMMAND events.
    */
   protSetHandler( PROT_DATA,    serviceClient2Modem );
   protSetHandler( PROT_COMMAND, serviceCommandHandler );

   sigemptyset( &act.sa_mask );
   act.sa_handler = serviceSignalHandler;
#ifdef SA_RESTART
   act.sa_flags   = SA_RESTART;
#endif
   sigaction( SIGQUIT, &act, NULL );
   sigaction( SIGTERM, &act, NULL );
   sigaction( SIGHUP,  &act, NULL );
   sigaction( SIGPIPE, &act, NULL );

   /* Install correct error handler */
   setErrorFunc( serviceErrorFunc );

   Data.allocated = 1;

   /* Write log message to access logfile */
   fprintf( ACCESS_LOG, "%s[%s] Connect from: \"%s@%s\"\n",
      timeStamp(),
      Data.localHost,
      Data.remoteUser,
      Data.remoteHost );
   fflush( ACCESS_LOG );

   /* Modem allocated, we're now ready to serve our client */
   result = protRequestConfirm( Data.client, PROT_REQUEST_OK );
   if( result < 0 )
   {
      FAIL( "protRequestConfirm" );
      exit( 1 );
   }
   if( result == 0 )
      exit( 0 );
}



/*
 * Main eventhandler of service 
 */
static void serviceMainLoop()
{
   int retval;
   struct timeval timeout;
   fd_set fds;

   while( 1 )
   {
      /* Save state for non-local jump in serviceReadModem */
      if( setjmp( saved_env ) == 1 )
         /* Non-local jump: start over again */
         continue;

      /* Setup for select */
      FD_ZERO( &fds );
      FD_SET( Data.client, &fds );
      /* 
       * Only add modem device to descriptor 
       * list when we are connected.
       */
      if( Data.connected )
         FD_SET( Data.modem, &fds );
      
      /* client timeout in minutes, not seconds */
      timeout.tv_sec = CLIENT_TIMEOUT * 60;
      timeout.tv_usec = 0;
      TEMP_FAILURE_RETRY( retval, 
                          select( FD_SETSIZE, &fds, NULL, NULL, &timeout ) );
      if( retval < 0 )
      {
         FAIL( "select" );
         exit( 1 );
      }

      /*
       * Timeout.
       * User is not connected and didn't do anything for 
       * CLIENT_TIMEOUT seconds. Free modem resources and
       * exit.
       */
      if( retval == 0 )
      {
         if( !Data.connected )
         {
            error( "you didn't do anything, bye" );
            exit( 1 );
         }
         continue;
      }

      /* 
       * Data arrived from client. Let protocol
       * routines handle this.
       */
      if( FD_ISSET( Data.client, &fds ) )
      {
         retval = protReceive( Data.client );

         /* EOF reached, client has closed connection */
         if( retval == 0 )
         {
            exit( 0 );
         }
         if( retval < 0 )
         {
            FAIL( "protReceive" );
            exit( 1 );
         }

      }
      /* Data arrived from modem */
      if( FD_ISSET( Data.modem, &fds ) )
         serviceModem2Client();
   }
}

/*
 * Read data from modem and send it to client
 */
static void serviceModem2Client()
{
   int bytes;

   /* Read data from modem */
   bytes = modemRead( Data.modem, buf, sizeof buf );
   if( bytes < 0 )
   {
      FAIL( "modemRead" );
      exit( 1 );
   }
   if( bytes == 0 )
      exit( 0 );

   /* Send data as PROT_DATA to client */
   bytes = protSend( PROT_DATA, Data.client, buf, bytes );

   /* EOF */
   if( bytes == 0 )
      exit( 0 );
      
   if( bytes < 0 )
   {
      FAIL( "protSend" );
      exit( 1 );
   }
}

/* 
 * Protocol handler for PROT_DATA.
 * Just send the data to the modem.
 */
static void serviceClient2Modem( protType type, void* data, size_t length )
{
   int bytes;

   /* Only send to modem when we are connected */
   if( !Data.connected )
      return;
   /* Send the data to modem */
   bytes = modemWrite( Data.modem, data, length );
   if( bytes < 0 )
   {
      FAIL( "modemWrite" );
      exit( 1 );
   }
   if( bytes == 0 )
      exit( 0 );
}

/*
 * Error handler
 * All errors are send as PROT_ERROR messages to client.
 */
static void serviceErrorFunc( const char* err )
{
   int result;
   struct sigaction saved;
   struct sigaction act;

   /* 
    * A SIGPIPE will normally terminate the program. We do not want
    * the program to terminate without having sent the error message.
    * We therefore ignore the signal. Writing the message when the
    * connection has been closed by the remote site will now result
    * in an error, in which case we send the message to stderr.
    */
   sigaction( SIGPIPE, NULL, &saved );
   sigaction( SIGPIPE, NULL, &act );
    
   act.sa_handler = SIG_IGN;
   sigaction( SIGPIPE, &act, NULL );

   /* Send message to client (inc. terminating '\0') */
   result = protSend( PROT_ERROR, Data.client, err, strlen( err ) + 1 );

   /* 
    * Send to remote site failed. Perhaps remote site has closed
    * connection we send the error message to stderr instead.
    */
   if( result <= 0 )
   {
      fprintf( stderr, "%s\n", err );
   }

   /* Restore SIGPIPE */
   sigaction( SIGPIPE, &saved, NULL );
}   

/*
 * Protocol handler for PROT_COMMAND
 * According to the subtype of the protocol message,
 * dial a number or hangup the modem
 */
static void serviceCommandHandler( protType type, void* data, size_t length )
{
   switch( protSubTypeOf( type ) )
   {
      case MDM_DIAL: 
         /* Dial a number; data contains number */
         serviceDial( data );
         break;
      case MDM_HANGUP:
         /* Hangup the modem */
         serviceHangup();
         break;
      case MDM_EXIT:
         /* Quit */
         exit( 0 );
         
   }
}

static void serviceSignalHandler( int sig )
{
   switch( sig )
   {
      case SIGPIPE:
         exit( 0 );
      case SIGQUIT:
      case SIGTERM:
         exit( 1 );
      case SIGHUP:
         /* Modem hangup */
         /* No more writes to modem device */
         Data.modem = -1;

         /* Inform user */
         error( "Modem disconnected, exiting!" );
         exit( 0 );
   }
}

/*
 * Exit routine
 */
static void serviceExit()
{
   static int exiting = 0;
   /* 
    * In error situations, it can happen that
    * this routine gets called recursively. So
    * we need a sort of semaphore to assure that
    * serviceHangup and mdm_reset are called just 
    * once.
    */
   if( exiting == 0 )
   {
      /* Set semaphore */
      exiting++;

      /* Hangup the modem */
      if( Data.connected )
         serviceHangup();
    
      /* Only do the following when a modem was allocated */
      if( Data.allocated == 1 )
      {
         /*
          * Write a message to the acces log indicating the disconnection
          * of the client. When no modem was allocated we forget this user
          */
         fprintf( ACCESS_LOG, "%s[%s] Disconnect from: \"%s@%s\"\n",
            timeStamp(),
            Data.localHost,  
            Data.remoteUser, 
            Data.remoteHost );
         fflush( ACCESS_LOG );
      }
      
      /* Reset modem device and terminal unless Data.modem is invalid 
       * after SIGHUP has been received.
       */
      if( Data.modem != -1 )
      {
         /* Reset the modem */
         mdm_reset( Data.modem );
         
         ttyReset();

         ttyHangup();
      }
      
      /* Close and unlock modem device */
      ttyClose();

      shutdown( Data.client, 2 );

      if( close( Data.client ) == -1 )
      {
         FAIL( "close" );
         exit( 1 );
      }

      closeLog();   
   }
}

/*
 * This routine checks whether a user/host is allowed 
 * to dial a number.
 * The routine executes a program (VERIFY_NUMBER or as defined in
 * configuration file as verify_number) that
 * actually does the check.
 *
 * The program is executed with the following arguments:
 *
 * argv:       0              1            2         3
 *       VERIFY_NUMBER   remoteUser   remoteHost   number
 *
 * where VERIFY_NUMBER is the name of the program.
 * The program should return 0 when the user has permission
 * to dial the number and 1 otherwise.
 *
 * Function return:
 *   1   The user does not have permission to dial the number,
 *   0   The user does have permission.
 *
 */
static int serviceCheckNumber( void )
{
   int retval;
   /* 
    * Execute VERIFY_NUMBER script
    */
   retval = sys_cmd( get_verify_number(),
                     Data.remoteUser,
                     Data.remoteHost,
                     Data.number,
                     NULL );
   if( retval == -1 )
   {
      FAIL1( "sys_cmd", get_verify_number() );
      exit( 1 );
   }
   /* Abnormal termination */
   if( !WIFEXITED( retval ) )
      exit( 1 );

   if( WIFEXITED( retval ) )
      return  WEXITSTATUS( retval );
   return 1;
}
   
/* 
 * This routine dials a phonenumber.
 */
static void serviceDial( const char* number )
{
   int result;
   /* If already connected, first disconnect */
   if( Data.connected )
      serviceHangup();

   /* Save number to dial */
   Data.number = strdup( number );

   /* Check if user has permission to dial the number */
   if( serviceCheckNumber() == 1)
   {
      error( "No permission to dial %s", number );
      exit( 1 );
   }
   /* Dial the number */
   mdm_dial( Data.modem, number );

   /* Wait on any reply from the modem */
   result = matchText( connectStr, MATCH_ALL, serviceReadModem );

   switch( result )
   { 
      /* CONNECT */
      case 0: 
         /* Save current time */   
         time( &Data.connected_since );

         /* We're connected now */
         Data.connected = 1;
         /* We are connected, get connectspeed from modem */
         result = matchText( speedStr, MATCH_ALL, serviceReadModem );

         /* Send speed as PROT_REPLY to client */
         result = protSend( PROT_REPLY | MDM_CONNECT, Data.client,
                      speedStr[result],
                      strlen( speedStr[result] ) + 1 );
         /* Result is evalutated below */
         break;

      /* NO CARRIER */
      case 1:
         result = 
            protSend( PROT_REPLY | MDM_NO_CARIER, Data.client, NULL, 0 );
         /* Result is evalutated below */
         break;

      /* NO DIALTONE */
      case 2:
         result = 
            protSend( PROT_REPLY | MDM_NO_DIALTONE, Data.client, NULL, 0 );
         /* Result is evalutated below */
         break;

      /* ERROR */
      case 3:
         result = 
            protSend( PROT_REPLY | MDM_ERROR, Data.client, NULL, 0 );
         /* Result is evalutated below */
         break;

      /* NO ANSWER */
      case 4:
         result =
            protSend( PROT_REPLY | MDM_NO_ANSWER, Data.client, NULL, 0 );
         /* Result is evalutated below */
         break;

      /* BUSY */
      case 5:
         result = 
            protSend( PROT_REPLY | MDM_BUSY, Data.client, NULL, 0 );
         /* Result is evalutated below */
         break;
      default:
         error( "UNKNOWN result: %d", result );
         exit( 1 );
   }
   /* result value of protSend is evaluated here */

   /* EOF */
   if( result == 0 )
      exit( 0 );
   /* Error */
   if( result < 0 )
   {
      FAIL( "protSend" );
      exit( 1 );
   }
}

/*
 * Hangup the modem and write a log message to
 * ACCOUNT_LOG
 * The log message is as follows:
 *
 * remoteUser@remoteHost,modemDevice,number,localHost,[startTime],[endTime]
 *
 */
static void serviceHangup()
{
   char       buf[100];
   struct tm* tm;

   /* Hangup modem deice, unless Data.modem is invalid after
    * SIGHUP has been received.
    */
   if( Data.modem != -1 )
      mdm_hangup( Data.modem );

   /* No log message when not connected */
   if( !Data.connected )
      return;

   tm = localtime( &Data.connected_since );
   strftime( buf, sizeof(buf), "[%c]", tm );
   fprintf( ACCOUNT_LOG, "%s@%s,%s,%s,%s,%s,%s\n",
      Data.remoteUser,
      Data.remoteHost,
      ttyDev(),
      Data.number,
      Data.localHost,
      buf,
      timeStamp()
      );
   fflush( ACCOUNT_LOG );

   /* We're not connected anymore */
   Data.connected = 0;
}

/*
 * Function that reads a character from modem.
 * This function is used by matchText
 *
 * This function will be interrupted when data from
 * the client arrives. In that case, the function 
 * longjumps to the loop in serviceMainLoop().
 * 
 * By doing so, the process does not block when waiting
 * for input from the modem.
 *
 */
static char serviceReadModem()
{
   int retval;
   char c;
   fd_set fds;

   /* Make set of descriptors */
   FD_ZERO( &fds );
   FD_SET( Data.modem,  &fds );
   FD_SET( Data.client, &fds );
   /* Wait on input on any of the descriptors in fds */
   TEMP_FAILURE_RETRY( retval,
                      select( FD_SETSIZE, &fds, NULL, NULL, NULL ) );

   if( retval == -1 )
   {
      FAIL( "select" );
      exit( 1 );
   }

   /* Input arrived from client, longjump tp serviceMainLoop() */
   if( FD_ISSET( Data.client, &fds ) )
   {
      longjmp( saved_env, 1 );
   }
   /* 
    * Data ready from modem, we can no savely 
    * read that data and return.
    */
   TEMP_FAILURE_RETRY( retval, read( Data.modem, &c, 1 ) );
   if( retval < 0 )
   {
      FAIL( "read" );
      exit( 1 );
   }

   if( retval == 0 )
   {
      exit( 0 );
   }

   /* Send data as PROT_DATA to client */
   retval = protSend( PROT_DATA, Data.client, &c, 1 );
   if( retval == 0 )
      exit( 0 );
   if( retval < 0 )
   {
      FAIL( "protSend" );
      exit( 1 );
   }
   return c;
}

/*
 * EOF modemd/service.c
 */
