/*
 * $Id: main.c,v 1.6 1998/02/17 09:53:25 mdejonge Exp $
 *
 *   $Source: /home/mdejonge/CVS/projects/modem/modemlogin/main.c,v $
 * $Revision: 1.6 $
 *    Author: Merijn de Jonge
 *     Email: mdejonge@wins.uva.nl
 * 
 *  
 * 
 * This file is part of the modem communication package.
 * Copyright (C) 1996-1998  Merijn de Jonge
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * 
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>


#include <libui/libui.h>
#include <libmodem/libmodem.h>
#include <libphonebook/libphonebook.h>
#include <liberror/liberror.h>
#include <libport/libport.h>
#include <modem_defs.h>
#include "login.h"

#ifdef HAS_GETOPT_H
#include <getopt.h>
#endif
extern char* optarg;

#define USAGE_MSG \
  "modemlogin (" ## VERSION ## ")\n"\
  "usage:\n" \
  "      modemlogin [-toolkitoption ...] [-option ...]\n" \
  "\n" \
  "where options include:\n"\
  "     -e name      use login information from phone books entry name\n" \
  "     -h           print out this message\n" \
  "     -l           don't prompt for login/password when it can be\n" \
  "                  determined from the entry in your phone book\n" \
  "     -m dev       use modem device dev\n" \
  "     -v           displays version information\n" \
  "\n" 

#define VERSION_MSG \
   "modemlogin version " ## VERSION ## "\n"

#define WAIT_LOGIN_STRING 1
#define WAIT_PWD_STRING   2
#define WAIT_SUCC_STRING  3


static void initStart( int argc, char* argv[] );
static void initSetLogin( char* name );

static void usage();
static void version();
static void getLoginCancel();
static void getLoginDone( char*, char* );
static void doLoginInit();
static void doLoginError( char* );
static void doLoginTimeout();
static void doLoginCancel();
static void clearInput( int fd );
static int  myRead( int fd );
static void modemloginError( const char* msg );


static int state = 0;

/* Is X interface already started */
static int   interface = 0;
static int   timeout;
static char* login = NULL;
static char* password = NULL;
static int   auto_login = 0;
static int   deviceFd; /* fdesc of opened device*/
/* Command line options used by xmlogin */
static char* opt_string = "e:hlm:v";

/* 
 * Error Handler 
 */
static void modemloginError( const char* msg )
{
   /* Use interface when possible to display error */
   if( interface )
      errorDialogPopup( (char*)msg );
   else
      fprintf( stderr, "ERROR: %s\n", msg );
}      
   

/* Handle command line options */
static void initStart( int argc, char* argv[] )
{
   int c;

   while( 1 )
   {
      c = getopt( argc, argv, opt_string );
      if( c == -1 )
         break;
      switch( c )
      {
         case 'e' :
            /* Use login from phonebookentry optarg */
            initSetLogin( optarg );
            break;
         case 'h' :
            usage();
            exit( 0 );
         case 'l' :
            auto_login = 1;
            break;
         case 'm':
            /* Use specified device instead of stdin */
            deviceFd = open( optarg, O_RDWR );
            if( deviceFd == -1 )
            {
               FAIL1( "open", optarg );
               exit( 1 );
            }
            break;
         case 'v' :
            version();
            exit( 0 );
         default:
            exit( 1 );
            break;
      }
   }
}

static void usage()
{
   fprintf( stdout, USAGE_MSG );
}

static void version()
{
   fprintf( stdout, VERSION_MSG );
}

/*
 * Set login and password from phonebookentry name
 */
static void initSetLogin( char* name )
{
   phonebook_entry entry;
   phonebook* book;
   int result;
   
   /* Open the phonebook */
   book = phonebookOpen( PHONEBOOK );
   if( book == NULL )
      return;

   /* Get entry name from phonebook */
   result = phonebookGetEntryByName( book, &entry, name );
   if( result )
   {
      /* Set login */
      if( entry.login != NULL )
         login = strdup( entry.login );
      /* Set password */
      if( entry.password != NULL )
         password = strdup( entry.password );
   }
   phonebookClose( book );
}

      
/*
 * Start getting login and password from user
 */
static void getLoginStart()
{
   loginDialogInfoRec info;

   /* Set values in info rec */
   info.user           = login;
   info.pw             = password;
   info.cancelCallback = getLoginCancel;
   info.okCallback     = getLoginDone;

   /* Initialise login dialiog */
   loginDialogInit( &info );
   
   /* Pop the dialog up */
   loginDialogPopup();
}


/*
 * User pressed cancel button
 */
static void getLoginCancel()
{
   
   exit( 1 );
}

/*
 * Got user and password from user.
 * Start loging in
 */
static void getLoginDone( char* user, char* pw )
{
   if( login != NULL )
      free( login );

   if( password != NULL )
      free( password );

   login = strdup( user );
   password = strdup( pw );
   
   doLoginInit();
}

/*
 * User canceled the login process
 */
static void doLoginCancel()
{
   /* Popdown the dialog */
   loginMsgDialogPopdown();
   /* Startover again */
   getLoginStart();
}

/*
 * timed out
 */
static void doLoginTimeout() 
{
   doLoginError( "Timeout" );

   /* Popdown the dialog */
   loginMsgDialogPopdown();
   /* Startover again */
   getLoginStart();
}

/*
 * Start process of loging in
 */
static void doLoginInit()
{
   loginMsgDialogInfoRec info;
   int result;
   
   /* Set values of info rec */
   info.user = login;
   info.timeout = timeout;
   info.cancelCallback = doLoginCancel;
   info.timeoutCallback = doLoginTimeout;
   
   /* Initialize dialog */
   loginMsgDialogInit( &info );

   /* Popup the dialog */
   loginMsgDialogPopup();
         
   /* Set state to wait for login string */
   state = WAIT_LOGIN_STRING;
   
   /* Start receiving input from modem device */
   interfaceSetInput( deviceFd, myRead );

   /* Write some linefeeds to get the login prompt. */
   TEMP_FAILURE_RETRY( result, write( deviceFd, "\r\r", 2 ) );
   if( result < 0 )
   {
      FAIL( "write" );
      exit( 1 );
   }
   
   /* Discard all input */
   clearInput( deviceFd );
   
   /* Yet another linefeed to get prompt back */
   TEMP_FAILURE_RETRY( result, write( deviceFd, "\r", 1 ) );
   if( result < 0 )
   {
      FAIL( "write" );
      exit( 1 );
   }
   /* Initialize match-routines */
   matchInit();
}

/*
 * Read a byte and try to match
 * a string according to state.
 */
static int myRead( int fd )
{
   int status;
   char buf[100];
   if( state == 0 )
      return 1;
   TEMP_FAILURE_RETRY( status, read( fd, buf, 1 ) );

   if( status <= 0 )
      return status;

   switch( state )
   {
      case 0:
         return 0;
      case WAIT_LOGIN_STRING:
         /* Wait for login prompt */
         status = matchOneChar( loginString, *(char*)buf, MATCH_ALL );
         if( status == MATCH_INCONCLUSIVE )
            return 1;
         /* Got login prompt; send username */
         sprintf( buf, "%s\r", login );
         TEMP_FAILURE_RETRY( status, 
            write( deviceFd, buf, strlen( buf ) ) );
         if( status < 0 )
         {
            FAIL( "write" );
            exit( 1 );
         }
         /* Go to next state */ 
         state = WAIT_PWD_STRING;
         /* Initialize match routines */
         matchInit();
         break;       
      case WAIT_PWD_STRING:
         /* Wait for password prompt */
         status = matchOneChar( pwdString, *(char*)buf, MATCH_ALL );
         if( status == MATCH_INCONCLUSIVE )
            return 1;
         /* Got password prompt 
          * 
          * Next state checks whether login succeded.
          * We need to discard all input, before 
          * we can continue.
          */ 
         clearInput( fd );
         /* Send password */
         sprintf( buf, "%s\r", password );
         TEMP_FAILURE_RETRY( status, 
            write( deviceFd, buf, strlen( buf ) ) );
         if( status < 0 )
         {
            FAIL( "write" );
            exit( 1 );
         }
         /* Go to next state */
         state = WAIT_SUCC_STRING;
         matchInit();
         break;       
      case WAIT_SUCC_STRING:
         /* Try to find out whether login succeded */
         status = matchOneChar( failureString, *(char*)buf, MATCH_BEGINNING );
         if( status == MATCH_INCONCLUSIVE )
            return 1;
         /* 
          * No match means no error message, thus
          * login succeded.
          */
         if( status == MATCH_NO_MATCH )
         {
            /* Logged in */
            loginMsgDialogPopdown();
            exit( 0 );
         }
         /* 
          * We got an recognized message, indicating
          * a login failure.
          */
         else if ( status >= 0 )
         { /* Login incorrect */
            clearInput(fd);
            state = 0;
            loginMsgDialogPopdown();
            doLoginError( "Login incorrect" );
            /* Start over again */
            getLoginStart( login, password );
        }
         break;       
   }
   return 1;
}

/*
 * Discard all input
 */   
static void clearInput( int fd )
{
   fd_set fds;
   struct timeval timeout;
   int retval;
   char c;
         
   while( 1 )
   {
      FD_ZERO( &fds );
      FD_SET( fd, &fds );
      timeout.tv_sec = 0; 
      timeout.tv_usec = 0;
      retval = select( FD_SETSIZE, &fds, NULL, NULL, &timeout );
      /* As long as there is data read a byte */
      if( retval && FD_ISSET( fd, &fds ) )
      {
         TEMP_FAILURE_RETRY( retval, read( fd, &c, 1 ) );
         if( retval < 0 )
         {
            FAIL( "read" );
            exit( 1 );
         } 
         else
         if( retval == 0 )
            return;
      }
      else
         return;
   }
}



/*
 * Indicate error 
 */
static void doLoginError( char* msg )
{
   errorDialogPopup( msg );
}
   
void main( int argc, char* argv[] )
{
   /* set error handler */
   setErrorFunc( modemloginError );

   /* Default device to use is stdin */
   deviceFd = STDIN_FILENO;
   
   /* Create interface */
   interfaceCreate( &argc, argv, "Modemlogin" );


   /* Handle commandline options */
   initStart( argc, argv );

   timeout = 60;

   /* Create dialogs */
   loginDialogCreate();
   loginMsgDialogCreate();
   errorDialogCreate();

   /* Start loging in */
   if( auto_login == 1 &&
       login != NULL && password != NULL )
       getLoginDone( strdup( login ), strdup( password ) );
   else
   getLoginStart( login, password );
   interface = 1;
   interfaceRun();
}

/*
 * EOF modemlogin/modemlogin.c
 */
