/*
SMDL 
----
A program for Linux to download DSP code to the Digicom Connection 96+/144+.
See smdl.doc for more information.
*/


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>

#define RAM_DATA 0
#define RAM_CODE 1

#define VERSION		"1.02"
#define UART_DLL	0
#define UART_DLM	1
#define UART_LCR	3
#define UART_MCR	4
#define UART_LCR_DLAB	0x80
#define UART_MCR_OUT2	0x08
#define UART_MCR_OUT1	0x04
#define PORT_LEN 	8

unsigned long port_base;
char rom_name[14] = "Just a filler";
int comport = 0, pbase[4] = {0x3f8,0x2f8,0x3e8,0x2e8};

void checkioperm(void);
void die (const char *s);
void clearioperm(void);

/* PORT_OUT */

inline static void port_out (unsigned short offset, char value)
{
__asm__ volatile ("outb %0,%1"
		  : :"a" ((char) value),"d" ((unsigned short)
		  			     (port_base + offset)));
}


/* PORT_IN */
inline static char port_in (unsigned short offset)
{
	char _v;
__asm__ volatile ("inb %1,%0"
		  :"=a" (_v):"d" ((unsigned short) (port_base + offset)));
 	return _v;
}


/* SELECT_RAM_TYPE */
int select_ram_type (int type)
{
	static int current_type = -1;
	if (current_type < 0)
		current_type = port_in (7) & 1;
	if (current_type != type)
		{
		  port_out (7, 0x18);
		  port_out (7, 0x19);
		}
	current_type = port_in (7) & 1;
	return (current_type != type);
}


/* PORT_HUNT */
void port_hunt(void)
{
	int x;
	short found = 0;
	
	for(x = 0; x <= 3; x++)
		{
		  
		  port_base = pbase[x];
		  comport = x + 1;
		  
		  checkioperm();
		  
		  if (((port_in (7) ^ port_in (7)) & 0x80) != 0)
		   {	
		  	found = 1;
		  	break;
		   }	

		  clearioperm();

		 }

	if (found == 0)
		port_base = 0;
}


/* NO_HUNT */
void no_hunt(void)
{
	checkioperm();	
	if (((port_in (7) ^ port_in (7)) & 0x80) == 0)
		port_base = 0;
}


/* CHECKIOPERM */
void checkioperm(void)
{
	if (ioperm (port_base, PORT_LEN, 1))
                 die("Couldn't get I/O perms.\n");
}


/* CLEARIOPERM */
void clearioperm(void)
{
	if (ioperm(port_base, PORT_LEN, 0))
		die("Couldn't clear I/O perms.\n");
}


/* HUNT_NAME */
void hunt_name(long int size, unsigned char *buffer)
{
	char temp[5] = "Null";
	int i, j, found = 1;

	for(i = 0; i < size; i++)
		{
		 if (buffer[i] == '/')
		 	{
		 	  for(j = 0; j <= 3; j++)
		 	  	 {
		 	  	  i++;
		 	  	  temp[j] = buffer[i];
		 	  	 } 
			  	
			  found = strcmp(temp, "????");

			  if (found == 0)
			  	{
			  	 for(j = 0; j <= 12; j++)
			  	 	 rom_name[j] = buffer[i + j - 17];
				}
			}			
		 if (found == 0) 
		 	break;
		}
}
		

/* DIE */
void die (const char *s)
{
	if (s)
	 {
	  fprintf (stderr, "%s", s);
	 }
	exit(1);
}


/* PDIE */
void pdie (const char *s)
{
	perror(s);
	exit(1);
}


/* USAGE */
void usage(char *name)
{
	printf("\nDownloads DSP file to DSI Connection 96+/14.4+  v"VERSION"\n");
	printf("COM port is automatically searched for if not specified.\n\n");
	printf("Usage:  %s [-s -p [port]] dsp_file\n", name);
	printf("        Port is in the range 1 - 4.\n\n");
	exit(0);
}


/***************/
/** MAIN BODY **/
/***************/

void main (int argc, char *argv[])
{
	int i, dohunt = 1, silent = 0;
	long int file_length;
	FILE *fp;
	char *file;
	unsigned char *buf, *p;
	int old_lcr, old_mcr;
	struct stat filestat;
	
	for(i = 1; i < argc; i++)
	{
		 if (argv[i][0] == '-')
		 {
		  switch (argv[i][1])
			{
			 case 's':
			 case 'S':
				silent = 1;
			        break;
			 case 'p':
			 case 'P':
				comport = atoi(argv[i + 1]);
				if (comport < 1 || comport > 4)
					usage(argv[0]);
				port_base = pbase[comport - 1];
				dohunt = 0;
				i += 1;
				break;
			 default:
				usage(argv[0]);
			}
		 }
	}
	
	if (argc < 2 || argc > 5)
		usage(argv[0]);
	
	if (argc == 2)
		dohunt = 1;

	file = argv[argc - 1];
	fp = fopen(file, "r");
	if (fp == NULL)
		pdie(file);
	
	if (fread(&file_length, 4, 1, fp) != 1)
		pdie(file);
	
	i = stat(file,&filestat);
	
	if (i != 0) 
		pdie(file);
	
	if (file_length != (filestat.st_size - 4))
		die("Invalid DSP file.\n");
	
	buf = malloc(file_length);
	if (buf == NULL)
		die("Not enough memory.\n");
	
	if (fread(buf, 1, file_length, fp) != file_length)
		pdie(file);
	
	fclose(fp);
	
	p = buf;
	
	if (dohunt == 1)
		port_hunt();
	else
		no_hunt();
			
	if (port_base == 0)
	        die("C96+/14.4+ not found or not responding.\n");
	         
	old_mcr = port_in(UART_MCR);
	old_lcr = port_in(UART_LCR);
	
	port_out(UART_LCR , UART_LCR_DLAB);
	port_out(UART_MCR, UART_MCR_OUT1 | UART_MCR_OUT2);
	
	port_out(UART_DLL, 0x31);
	port_out(UART_DLM, 0x8c);
	
	port_out(7, 0x10);
	port_out(7, 0x19);
	
	if ((port_in(7) & 0x1a) != 0x18)
		die("DSP bus not granted.\n");
	
	if (select_ram_type (RAM_CODE))
		die("Could not select program bus.\n");
	
	for(i = 0; i < 0x3800; i++)
		{
		 port_out(0, *p++);
		 port_out(1, *p++);
		 port_out(2, *p++);
		}
		
	if (select_ram_type (RAM_DATA))
		die("Could not select data bus.\n");
	
	for (i = 0; i < 0x800; i++)
		{
		 port_out(1, *p++);
		 port_out(2, 0);
		 port_out(1, *p++);
		 port_out(2, *p++);
		}
	
	for (i = 0; i < ((file_length - 0xc000) / 2); i++)
		{
		 port_out(1, *p++);
		 port_out(2, *p++);
		}
	
	port_out(7, 0x11);
	port_out(7, 0x03);
	
	port_out(UART_MCR, old_mcr & ~UART_MCR_OUT1);
	port_out(UART_LCR, old_lcr);
	
	clearioperm();
	
	if (silent == 0)
		{
		 hunt_name(file_length, buf);
		 printf("%s downloaded successfully onto COM%d.\n", rom_name, comport);
		}

	exit(0);
}		
