// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of ccscript.
// 
// The exception is that, if you link the ccscript library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccscript library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccscript.  If you copy code from other releases into a copy of
// ccscript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own for ccscript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.

#include <cc++/config.h>
#include <cc++/misc.h>
#include <cc++/slog.h>
#include <cc++/url.h>
#include <cc++/file.h>
#include <cc++/export.h>
#include <cstdlib>
#include <cstdio>
#include "bayonnescript.h"

#ifndef	WIN32
#include "bayonneconfig.h"
#ifdef	HAVE_REGEX_H
#include <regex.h>
#endif
#endif

#if !defined(__GNUG__) || defined(__STRICT_ANSI__)
#define FIXED_ARRAY     1
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

static long tens[] = {1, 10, 100, 1000, 10000, 100000, 1000000, 10000000};

Script::Attr *ScriptInterp::attr = NULL;
Script::Test *ScriptInterp::test = NULL;
Script::Fun *ScriptInterp::ifun = NULL;

static void logerror(const char *script, unsigned id, const char *msg)
{
	slog.error() << script;
	if(id)
		slog() << "(" << id << ")";
	slog() << ": " << msg << endl;
}

static void adjustValue(char *buffer, int diff)
{
	int value = atoi(buffer);

	sprintf(buffer, "%d", value + diff);
}

static int mapicmp(const char *s1, const char *s2)
{
        for(;;)
        {
                if(!*s1 || !*s2)
                        return *s2 - *s1;
                if(tolower(*s1) != tolower(*s2))
                        if(*s1 != '?' && *s2 != '?')
                                return tolower(*s2) - tolower(*s1);
                ++s1;
                ++s2;
        }
}

static int mapnicmp(const char *s1, const char *s2, size_t n)
{
        while(n--)
        {
                if(!*s1 || !*s2)
                        return *s2 - *s1;
                if(tolower(*s1) != tolower(*s2))
                        if(*s1 != '?' && *s2 != '?')
                                return tolower(*s2) - tolower(*s1);
                ++s1;
                ++s2;
        }
        return 0;
}


ScriptInterp::ScriptInterp(ScriptCommand *cmdset, unsigned sym, size_t pg, const char *id) :
ScriptSymbol(sym, pg, id)
{
	once = true;
	warn = false;
	trace = false;
	steps = 1;
	signalmask = 0;
	stack = 0;
	cmd = cmdset;
	image = NULL;
	memset(temps, 0, sizeof(temps));
	datasource = NULL;
	recmode = NULL;

	for(tempidx = 0; tempidx < SCRIPT_TEMP_SPACE; ++tempidx)
		temps[tempidx] = new char[getSymbolSize() + 1];
	tempidx = 0;
	symsize = sym;
	pgsize = pg;

	if(id)
		snprintf(_idname, sizeof(_idname), "scr/%s", id);
	else
		snprintf(_idname, sizeof(_idname), "scr/%p", (void *)this);
}

ScriptInterp::~ScriptInterp()
{
	for(tempidx = 0; tempidx < SCRIPT_TEMP_SPACE; ++tempidx)
		if(temps[tempidx])
			delete[] temps[tempidx];
}

void ScriptInterp::clearStack(void)
{
	unsigned indexes[SCRIPT_STACK_SIZE];
	unsigned idx = 0, len = 0;
	char values[SCRIPT_STACK_SIZE * 6];

	while(stack)
	{
		if(script[stack - 1].script != script[stack].script)
			break;
		pull();
		indexes[idx++] = script[stack].index;
	}
	snprintf(values, 4, "%d", idx);
	setSymbol("script.stack", values);
	values[1] = 0;
	while(idx--)
	{
		snprintf(values + len, sizeof(values) - len, ",%d", indexes[idx]);
		len = (unsigned)strlen(values);
	}
	setSymbol("script.index", values + 1);
}

unsigned long ScriptInterp::getScriptMask(void)
{
	ScriptCommand *cmd = getCommand();
	unsigned sp = script[stack].base;
	unsigned long mask = 0;

	while(sp < stack)
	{
		mask |= (script[sp].script->mask & script[sp].line->mask & cmd->imask);
		++sp;
	}
	mask |= script[stack].script->mask;
	return mask;
}

const char *ScriptInterp::getVariable(const char *id)
{
	Symbol *sym;

	if(!id)
		return NULL;

	if(*id == '&' || *id == '&')
		++id;

	if(recmode)
		sym = recmode->getEntry(id, 0);
	else if(strchr(id, '.'))
		sym = getEntry(id, 0);
	else
		sym = getLocal(id, 0);

	if(!sym)
		return NULL;

	if(sym->flags.initial)
		return NULL;

	return readSymbol(sym);
}

bool ScriptInterp::setVariable(const char *id, unsigned size, const char *value)
{
	Symbol *sym;
	long val;

	if(!id)
		return false;

	if(*id == '%' || *id == '&')
		++id;

	if(!size)
		size = getSymbolSize();

	if(recmode)
		sym = recmode->getEntry(id, size);
        else if(strchr(id, '.'))
                sym = getEntry(id, size);
        else
                sym = getLocal(id, size);

	if(!sym)
		return false;

	if(!value)
		return true;

	switch(sym->flags.type)
	{
	case ARRAY:
	case SEQUENCE:
	case CACHE:
	case FIFO:
	case STACK:
		return postSymbol(sym, value);
        case COUNTER:
                val = atoi(value);
                snprintf(sym->data, sym->flags.size + 1, "%ld", --val);
                break;
	default:
		if(sym->flags.readonly)
			return false;
                strncpy(sym->data, value, sym->flags.size);
                sym->data[sym->flags.size] = 0;
        }
	sym->flags.initial = false;
        if(sym->flags.commit)
                commit(sym);

	return true;
}

char ScriptInterp::getPackToken(void)
{
	char *sym = getSymbol("script.token");
	if(!sym)
		sym = ",";

	if(!*sym)
		sym = ",";

	return *sym;
}

Script::Name *ScriptInterp::getScriptCopy(const char *name)
{
	char buffer[256];
	Name *scr;

	snprintf(buffer, 255, "%s::%p", name, (void *)this);
	scr = image->dupScript(name, buffer);
	if(scr && !strcmp(script[stack].script->name, name))
		script[stack].script = scr;
	return scr;
}

unsigned long ScriptInterp::getMask(void)
{
	return script[stack].line->mask & script[stack].mask;
}

Script::Name *ScriptInterp::getScriptImage(const char *name)
{
	char buffer[256];
	Name *scr = image->getScript(name);

	if(!scr)
		return NULL;

	if(scr->mode == Name::mCOPIED)
	{
		snprintf(buffer, 255, "%s::%p", name, (void *)this);
		scr = image->getScript(buffer);
	}
	return scr;
}


bool ScriptInterp::scrRestart(void)
{
	const char *mem = getMember();
	unsigned base = script[stack].base;
	Name *alt = NULL;
	const char *opt = getValue(NULL);
	bool local = false;

	if(opt)
		alt = getScriptImage(opt);

	if(!mem)
		mem = "script";

	if(!stricmp(mem, "clear"))
	{
		mem = "script";
		cleardigits(true);
	}
	else if(!stricmp(mem, "cleardigit"))
		cleardigits(false);

	clearStack();

	if(!stricmp(mem, "state") || !stricmp(mem, "script") ||!stricmp(mem, "function"))
	{
		local = true;
		initKeywords(0);
		goto jump;
	}

	if(!stricmp(mem, "base"))
	{
		while(stack > base)
			pull();

		goto jump;
	}

	while(stack)
		pull();


	script[stack].script = getScriptImage(getSymbol("script.home"));

jump:
	if(alt)
		script[stack].script = alt;

	script[stack].advance = true;
	script[stack].caseflag = script[stack].tranflag = false;
	script[stack].line = script[stack].first = script[stack].script->first;
	script[stack].index = 0;
	if(!local)
	{
		script[stack].mask = getScriptMask();
		evtGoto();
	}
	if(script[stack].script->access == Name::aFUNCTION)
		script[stack].tranflag = true;
	return true;
}

bool ScriptInterp::scrEnable(void)
{
	char buffer[256];
	char *name = getKeyword("name");
	Name *scr;
	unsigned long mask, id;
	char *cp;

	if(!name)
		name = getValue(NULL);

	if(!name)
	{
		advance();
		return true;
	}

	if(!strncmp(name, "::", 2))
	{
		setString(buffer, sizeof(buffer), script[stack].script->name);
		cp = strstr(buffer, "::");
		if(cp)
			*cp = 0;
		strcat(buffer, name);
		name = buffer;
	}
	scr = getScriptCopy(name);
	if(!scr)
	{
		error("script-not-found");
		return true;
	}

	while(NULL != (name = getValue(NULL)))
	{
		id = cmd->getTrapId(name);
		mask = cmd->getTrapMask(name);
		if(!mask)
		{
			error("handler-invalid");
			return true;
		}
		if(!scr->trap[id])
		{
			error("handler-not-found");
			return true;
		}
		scr->mask |= mask;
	}
	advance();
	return true;
}

bool ScriptInterp::scrDisable(void)
{
	char buffer[256];
	char *name = getKeyword("name");
	Name *scr;
	unsigned long mask, id;
	char *cp;

	if(!name)
		name = getValue(NULL);

	if(!name)
	{
		advance();
		return true;
	}

	if(!strncmp(name, "::", 2))
	{
		setString(buffer, sizeof(buffer), script[stack].script->name);
		cp = strstr(buffer, "::");
		if(cp)
			*cp = 0;
		strcat(buffer, name);
		name = buffer;
	}
	scr = getScriptCopy(name);
	if(!scr)
	{
		error("script-not-found");
		return true;
	}

	while(NULL != (name = getValue(NULL)))
	{
		id = cmd->getTrapId(name);
		mask = cmd->getTrapMask(name);
		if(!mask)
		{
			error("handler-invalid");
			return true;
		}
		if(!scr->trap[id])
		{
			error("handler-not-found");
			return true;
		}
		scr->mask &= ~mask;
	}
	advance();
	return true;
}

void ScriptInterp::rewindTemp(void)
{
	tempidx = 0;
}

void ScriptInterp::setTemp(const char *value)
{
	snprintf(temps[tempidx++], getSymbolSize() + 1, "%s", value);
	if(tempidx >= SCRIPT_TEMP_SPACE)
		tempidx = 0;
}

void ScriptInterp::advance(void)
{
	if(script[stack].advance)
		script[stack].line = script[stack].line->next;
}

void ScriptInterp::error(const char *errmsg)
{
	char evtname[80];
	setSymbol("script.error", errmsg);
	snprintf(evtname, sizeof(evtname), "error:%s", errmsg);
	if(event(evtname))
		return;

	if((script[stack].script->mask & 0x02) && script[stack].script->trap[1])
		trap(1);
	else
		advance();
}

void ScriptInterp::trap(const char *trapid)
{
	unsigned trap = cmd->getTrapId(trapid);
	if(!trap)
	{
		if(!image)
			return;

		if(!stricmp(trapid, "first") || !stricmp(trapid, "top"))
		{
			script[stack].advance = true;
			script[stack].tranflag = script[stack].caseflag = false;
			script[stack].line = script[stack].first;
			return;
		}
	}
	ScriptInterp::trap(trap);
}

void ScriptInterp::trap(unsigned id)
{
	Line *trap = NULL;
	unsigned base = script[stack].base;

	if(!image)
		return;

	if(getGlobalTrap(id))
		return;

	// we can inherit traps at lower levels

	for(;;)
	{
		trap = script[stack].script->trap[id];
		if(trap == script[stack].first)
		{
			advance();
			return;
		}
		if(!trap && !cmd->isInherited(id))
		{
			advance();
			return;
		}	
		if(trap || stack == base)
			break;

		pull();
	}

	// when doing a trap, always unwind loop or recursive stack frames

	clearStack();
	script[stack].advance = true;
	script[stack].tranflag = script[stack].caseflag = false;
	script[stack].line = script[stack].first = trap;
}

bool ScriptInterp::step(void)
{
//	unsigned long mask, cmask;
	bool rtn = false, sync;
	Line *next, *line;
	unsigned count = steps;

	if(!image)
		return true;

	script[stack].index = 0;
	if(!script[stack].line)
		goto exit;

	// auto return code

/*
	cmask = script[stack].line->cmask;
	if(cmask)
	{
		if((cmask & script[stack].mask) != cmask)
		{
			skip = true;
			advance();
			goto retry;
		}
	}
*/

	sync = script[stack].line->sync;
	if(sync)
		setExclusive(true);

trans:
	line = script[stack].line;
	next = line->next;
	rtn = execute(line->method);
	if(sync)
	{
		setExclusive(false);
		goto exit;
	}
	if(!rtn || !script[stack].line)
		goto exit;
	
	if(!script[stack].advance)
		goto exit;

	if((script[stack].tranflag && !trace) || recmode)
	{
		count = 0;
		script[stack].index = 0;
		goto trans;
	}

	if(count-- && script[stack].line == next && !trace)
	{
		script[stack].index = 0;
		goto trans;
	}

exit:
	while(!script[stack].line && stack)
	{
		if(script[stack - 1].local == script[stack].local)
			break;
		pull();
		if(script[stack].line)
			advance();
	}

	if(!script[stack].line)
	{
		if(initialized)
			exit();
		return false;
	}

	if(!initialized && !rtn)
	{
		slog.error() << "script: " << script[stack].script->filename
			<< "(" << line->lnum << "): "
			<< line->cmd << " invalid for initialization" << endl;
		if(script[stack].line == line)
			script[stack].line = next;
		if(next)
			return true;
		return false;
	}

	return rtn;
}

bool ScriptInterp::event(const char *name, bool inhereted)
{
	Name::Event *evt, *top = script[stack].script->events;
	unsigned base = script[stack].base;
	const char *chkname = name;
	unsigned current = stack;
	bool found = false;
#ifdef	HAVE_REGEX_H
	regex_t *regex;
#endif

retry:
	evt = script[current].script->events;
	while(evt)
	{
		switch(evt->type)
		{
		case '@':
			if(!stricmp(evt->name, chkname))
				found = true;
			break;
#ifdef	HAVE_REGEX_H
		case '~':
			regex = new regex_t;
                	memset(regex, 0, sizeof(regex_t));

                	if(!regcomp(regex, evt->name, REG_ICASE|REG_NOSUB|REG_NEWLINE))
				if(!regexec(regex, chkname, 0, NULL, 0))
					found = true;

                        regfree(regex);
                        delete regex;
			break;
#endif
		}

		if(found)
			break;

		evt = evt->next;
	}

	if(!evt && NULL != (chkname = strchr(chkname, ':')))
	{
		++chkname;
		goto retry;
	}

	if(evt)
	{
		while(stack > current)
			pull();

		clearStack();
		script[stack].advance = true;
		script[stack].tranflag = script[stack].caseflag = false;
		script[stack].line = script[stack].first = evt->line;

		// fast call for events that simply branch

		if(evt->line && (evt->line->method == &ScriptInterp::scrGoto ||
		        evt->line->method == &ScriptInterp::scrReturn ||
			evt->line->method == &ScriptInterp::scrRestart ||
			evt->line->method == &ScriptInterp::scrBegin))
		{
			script[stack].index = 0;
			(this->*(evt->line->method))();
		}
		return true;
	}


	while(current > base && script[current].script->events == top && inhereted)
		--current;

	if(script[current].script->events != top)
	{
		top = script[current].script->events;
		chkname = name;
		goto retry;
	}

	return false;
}

bool ScriptInterp::signal(const char *trapname)
{
	Line *line = script[stack].line;
	if(!image)
		return true;

	unsigned long mask = cmd->getTrapMask(trapname);

	mask &= line->mask;
	mask &= script[stack].mask;

	if(!mask)
		return false;

	stop(mask);
	trap(trapname);
	line = getScript();

	// fast call interface for signals that simply branch

	if(line && (line->method == &ScriptInterp::scrGoto ||
		line->method == &ScriptInterp::scrReturn ||
		line->method == &ScriptInterp::scrRestart ||
		line->method == &ScriptInterp::scrBegin))
	{
		script[stack].index = 0;
		(this->*(line->method))();
	}

	return true;
}

bool ScriptInterp::signal(unsigned id)
{
	Line *line = script[stack].line;
	if(!image)
		return true;

	if(id >= TRAP_BITS)
		return false;

	unsigned long mask = cmd->getTrapMask(id);
	mask &= script[stack].mask;
	mask &= script[stack].line->mask;

	if(!mask)
	{
		signalmask |= id;
		return false;
	}

	stop(mask);
	trap(id);
	line = getScript();

	// fast call interface for signals that simply branch

	if(line && (line->method == &ScriptInterp::scrGoto ||
		line->method == &ScriptInterp::scrReturn ||
		line->method == &ScriptInterp::scrRestart ||
		line->method == &ScriptInterp::scrBegin))
	{
		script[stack].index = 0;
		(this->*(line->method))();
	}

	return true;
}

Script::Symbol *ScriptInterp::getLocal(const char *id, unsigned size)
{
	Symbol *sym = NULL;
	char sesname[65];

	if(recmode)
		sym = recmode->getEntry(id, size);
	else if(script[stack].local)
		sym = script[stack].local->getEntry(id, size);

	if(!sym)
		sym = getEntry(id, size);

	if(!sym && !size)
	{
		snprintf(sesname, sizeof(sesname), "session.%s", id);
		sym = getEntry(sesname);
	}

        if(!sym && !size)
        {
                snprintf(sesname, sizeof(sesname), "script.%s", id);
                sym = getEntry(sesname);
        }

	return sym;
}

bool ScriptInterp::scrSkip(void)
{
	char *val;
	char *id = getValue(NULL);
	Line *line;
	unsigned argc;
	bool cf = false;

	if(!id)
	{
		advance();
		if(script[stack].line)
			if(script[stack].line->method == &ScriptInterp::scrCase)
				cf = true;

		advance();
		while(script[stack].line && cf)
		{
			if(script[stack].line->method != &ScriptInterp::scrCase)
				break;

			advance();
		}
		return true;
	}

	script[stack].line = script[stack].first;
	while(NULL != (line = script[stack].line))
	{
		advance();
		if(line->method != &ScriptInterp::scrLabel)
			continue;

		argc = 0;
		while(argc < line->argc)
		{
			val = getContent(line->args[argc++]);
			if(!stricmp(val, id))
				return true;
		}
	}
	return true;
}

bool ScriptInterp::scrData(void)
{
	while(script[stack].line->method == &ScriptInterp::scrData)
	{
		advance();
		if(!script[stack].line)
			return true;
	}
	return true;
}

bool ScriptInterp::scrOnce(void)
{
	if(getOnce())
		return scrGoto();
	advance();
	return true;
}

bool ScriptInterp::scrSelect(void)
{
	char buffer[128];
	const char *mem = getMember();
	const char *opt;
	Line *prev = script[stack].line;
	Line *line;

	while(NULL != (opt = getValue(NULL)))
	{
		if(mem)
		{
			snprintf(buffer, sizeof(buffer), "%s:%s", mem, opt);
			opt = buffer;
		}
		if(event(opt))
		{
			line = script[stack].line;
			script[stack].line = prev;
			initKeywords(0);
			script[stack].line = line;
			return true;
		}
	}
	error("event-missing");
	return true;
}

bool ScriptInterp::scrGoto(void)
{
	if(!stricmp(script[stack].line->cmd, "goto.clear"))
		cleardigits(true);
	else if(!stricmp(script[stack].line->cmd, "goto.cleardigit"))
		cleardigits(false);
		
	script[stack].advance = true;
	script[stack].tranflag = false;
	return intGoto(true);
}

bool ScriptInterp::intGoto(bool evflag)
{
	ScriptCommand *cmd = getCommand();
	char namebuf[256];
	char *label = getOption(NULL);
	char *ext;
	size_t len;
	bool pvt = true;
	Name *scr = getObject();
	bool shortflag = true;
	Line *next, *prev;
	unsigned base = script[stack].base;
	bool fun = false;
	bool isfun = false;
	unsigned long mask = script[stack].line->mask & script[stack].mask & cmd->imask;

	if(!stricmp(script[stack].line->cmd, "call"))
		fun = true;

	if(scr->access == Name::aFUNCTION)
		isfun = true;

	if(label && *label != '@' && *label != '{' && evflag)
		label = getContent(label);

	if(!label)
	{
		error("branch-failed");
		return true;
	}

	if(strstr(label, "::"))
		shortflag = false;

	if(*label == '^' && evflag)
	{
		prev = script[stack].line;
		if(!signal(++label))
		{
			error("trap-invalid");
			return true;
		}
		next = script[stack].line;
		script[stack].line = prev;
		initKeywords(0);
		script[stack].line = next;
		return true;
	}

	if((*label == '@' || *label == '{') && evflag)
	{
		prev = script[stack].line;
		if(event(++label))
		{
			next = script[stack].line;
			script[stack].line = prev;
			initKeywords(0);
			script[stack].line = next;
			return true;
		}
		advance();
		return true;
	}

	len = strlen(label);

retry:
	if(shortflag)
	{
		snprintf(namebuf, sizeof(namebuf), "%s", script[stack].script->name);
		ext = strstr(namebuf, "::");
		if(ext)
			*ext = 0;
		len = strlen(namebuf);
		snprintf(namebuf + len, sizeof(namebuf) - len, "::%s", label);
		scr = getScriptImage(namebuf);
		if(scr)
		{
			pvt = false;
			goto script;
		}
		shortflag = false;
		goto retry;

	}
	else if(!strncmp(label, "::", 2))
	{
		pvt = false;
		setString(namebuf, sizeof(namebuf), script[stack].script->name);
		ext = strstr(namebuf, "::");
		if(ext)
			setString(ext, sizeof(namebuf) + namebuf - ext, label);
		else
			addString(namebuf, sizeof(namebuf), label);
		label = namebuf;
	}
	else if(fun || isfun)
	{
		setString(namebuf, sizeof(namebuf), script[stack].script->name);
		ext = strstr(namebuf, "::");
		if(ext)
			setString(ext + 2, sizeof(namebuf) + namebuf - ext - 2, label);
		else
		{
			addString(namebuf, sizeof(namebuf), "::");
			addString(namebuf, sizeof(namebuf), label);
		}
		scr = getScriptImage(namebuf);
		if(scr)
		{
			pvt = false;
			goto script;
		}
	}
	scr = getScriptImage(label);
script:
	if(!scr)
	{
		error("script-not-found");
		return true;
	}
	if(pvt && scr->access == Name::aPRIVATE)
	{
		error("script-private");
		return true;
	}
	if(!isfun && !fun && scr->access == Name::aFUNCTION && scr != script[stack].script)
	{
		error("script-function");
		return true;
	}
	if(scr->mode == Name::mDATA)
	{
		error("script-data");
		return true;
	}

	if(evflag)
		initKeywords(0);

	if(scr->access != Name::aFUNCTION)
		isfun = false;

	if(isfun && evflag)
		clearStack();
	else while(evflag && stack > base)
		pull();

	once = true;

	script[stack].caseflag = false;
	script[stack].script = scr;
	script[stack].line = script[stack].first = scr->first;
	script[stack].index = 0;
	if(evflag && isfun && stack)
	{	
		mask = script[stack - 1].line->mask & script[stack - 1].mask & cmd->imask;
		script[stack].mask = (mask | scr->mask);
		evtGoto();
	}
	else if(evflag)
	{
		script[stack].mask = getScriptMask();
		evtGoto();
	}
	else
		script[stack].mask = (mask | scr->mask);

	return true;
}

bool ScriptInterp::scrSwap(void)
{
	Symbol *s1, *s2;
	s1 = initVariable(getSymbolSize());
	s2 = initVariable(getSymbolSize());
	bool b1 = false, b2 = false;

	if(!s1 || !s2)
	{
		error("symbol-not-found");
		return true;
	}
	if(script[stack].local)
	{
		if(strchr(s1->id, '.'))
			b1 = true;

		if(strchr(s2->id, '.'))
			b2 = true;

		if(b1 != b2)
		{
			error("symbol-mixed-contexts");
			return true;
		}
		if(!b1)
		{
			script[stack].local->swapSymbol(s1->id, s2->id);
			advance();
			return true;
		}
	}
	swapSymbol(s1->id, s2->id);
	advance();
	return true;
}

bool ScriptInterp::scrMap(void)
{
	Line *line;
	Name *scr;
	char *cp;
	const char *prefix = getMember();
	char *value = getKeyword("table");
	char namebuf[256];
	size_t len;
	bool find = false;

	enum
	{
		MAP_PREFIX,
		MAP_SUFFIX,
		MAP_ABSOLUTE,
		MAP_VALUE
	}	mapmode = MAP_ABSOLUTE;

	if(!prefix)
		prefix = getKeyword("match");

	if(prefix)
        {
                if(!strnicmp(prefix, "pre", 3))
                        mapmode = MAP_PREFIX;
                else if(!strnicmp(prefix, "suf", 3))
                        mapmode = MAP_SUFFIX;
		else if(!strnicmp(prefix, "end", 3))
			mapmode = MAP_SUFFIX;
		else if(!strnicmp(prefix, "val", 3))
			mapmode = MAP_VALUE;
                else if(!strnicmp(prefix, "abs", 3))
                        mapmode = MAP_ABSOLUTE;
        }

	if(!value)
		value = script[stack].script->name;
	else if(!strnicmp(value, "::", 2))
        {
		setString(namebuf, sizeof(namebuf), script[stack].script->name);
		addString(namebuf, sizeof(namebuf), value);
                value = namebuf;
        }

        scr = getScriptImage(value);
        if(!scr)
        {
                error("no-source-to-read");
                return true;
        }
	line = scr->first;

	value = getValue("*");
	len = strlen(value);
	
	while(line)
	{
		if(line->method != &ScriptInterp::scrData)
		{
			line = line->next;
			continue;
		}	
		cp = strchr(line->cmd, '.');
		if(!cp)
		{
			line = line->next;
			continue;
		}
		++cp;

		switch(mapmode)
		{
		case MAP_VALUE:
			if(atol(cp) == atol(value))
				find = true;
			break;
		case MAP_ABSOLUTE:
			if(!mapicmp(cp, value))
				find = true;
			break;
		case MAP_PREFIX:
			if(!mapnicmp(cp, value, len))
				find = true;
			break;
		case MAP_SUFFIX:
			if(strlen(cp) > len)
				break;

			cp = cp + strlen(cp) - len;
			if(!mapicmp(cp, value))
				find = true;
			break;
		}
						
		if(find)
			break;

		line = line->next;
	}

	if(line)
	{
		setLine(line);
		return scrGoto();		
	}

	error("no-map-data");
	return true;
}	

bool ScriptInterp::setData(const char *name)
{
	char namebuf[256];
	Name *scr;
	char *tok;

	if(!name)
		name = script[stack].script->name;
	
	if(!strnicmp(name, "::", 2))
	{
		setString(namebuf, sizeof(namebuf), script[stack].script->name);
		tok = strstr(namebuf, "::");
		if(!tok)
			tok = namebuf + strlen(namebuf);
		setString(tok, namebuf + sizeof(namebuf) - tok, name);
		name = namebuf;
	}
	scr = getScriptImage(name);
	if(!scr)
		return false;

	script[stack].read = scr->first;
	return true;
}

bool ScriptInterp::scrRead(void)
{
	Symbol *sym = NULL;
	Line *rd;
	const char *mem = getMember();
	unsigned argc = 0;
	char *value = NULL;
	size_t offset = 0;
	bool packed = false;
	unsigned size = getSymbolSize();
	const char *sz = getKeyword("size");
	char packtoken = ',';
	unsigned row = 0;
	unsigned col = 0;
	unsigned count = 0;

	if(!mem)
		mem = "";

	if(sz)
		size = atoi(sz);

	if(!strnicmp(mem, "pack", 4))
	{
		packed = true;
		packtoken = *getSymbol("script.token");
		value = getKeyword("token");
		if(value)
			packtoken = *value;
	}

	if(!stricmp(mem, "from"))
	{
		value = getKeyword("table");
		if(!value)
			value = getValue(script[stack].script->name);

		if(!setData(value))
		{
			error("no-source-to-read");
			return true;
		}
		advance();
		return true;
	}

	value = getKeyword("row");
	if(value)
		row = atoi(value);

	value = getKeyword("col");
	if(value)
		col = atoi(value);

	value = getKeyword("count");
	if(!value)
		value = getKeyword("limit");
	if(value)
		count = atoi(value);

	value = getKeyword("from");
	if(!value)
		value = getKeyword("table");

	if(value)
		if(!setData(value))
		{
			error("no-source-to-read");
			return true;
		}

rows:
	rd = script[stack].read;
	while(rd)
	{
		if(rd->method == &ScriptInterp::scrData)
			break;
		rd = rd->next;
	}
	if(!rd)
	{
		script[stack].read = NULL;
		error("end-of-data");
		return true;
	}
	else
		script[stack].read = rd->next;

	if(row--)
		goto rows;

	while(argc < rd->argc)
	{
		value = getContent(rd->args[argc++]);
		if(!value)
			break;

		if(col)
		{
			--col;
			continue;
		}

		if(!packed || !sym)
		{
			if(packed || sz)
				sym = initVariable(size);
			else
				sym = initVariable((unsigned)strlen(value));
			if(!sym)
				break;
			if(sym->flags.readonly)
				continue;

			if(packed)
				sym->data[0] = 0;
		}

		if(packed)
		{
			if(offset)
				sym->data[offset++] = packtoken;
			snprintf(sym->data + offset, sym->flags.size - offset, "%s", value);
			offset = strlen(sym->data);
		}
		else
		{
			snprintf(sym->data, sym->flags.size + 1, "%s", value);
			sym->flags.initial = false;
			if(sym->flags.commit)
				commit(sym);
		}
		if(count)
			if(!--count)
				break;
	}
	if(sym && packed)
	{
		sym->flags.initial = false;
		if(sym->flags.commit)
			commit(sym);
	}

	advance();
	return true;
}

bool ScriptInterp::scrRepeat(void)
{
	unsigned short loop = script[stack].line->loop;
	Line *line;
	int index = script[stack].index;
	int count;

	script[stack].index = 0;
	count = atoi(getValue("0"));

	if(index >= count)
	{
		line = script[stack].line->next;
		while(line)
		{
			if(line->loop == loop)
			{
				script[stack].line = line;
				advance();
				return true;
			}
			line = line->next;
		}
		error("loop-overflow");
		return true;
	}
	script[stack].index = ++index;
	if(!push())
		return true;

	advance();
	return true;
}

bool ScriptInterp::scrFordata(void)
{
	Symbol *sym;
	unsigned short loop = script[stack].line->loop;
	unsigned index = script[stack].index;
	char *table = getKeyword("table");
	Line *rd, *line;
	unsigned argc = 0;
	unsigned size = getSymbolSize();
	const char *sz = getKeyword("size");
	const char *value;
	unsigned row = 0;
	unsigned limit=0;

	if(sz)
		size = atoi(sz);

	script[stack].index = 0;
	if(!table)
		table = getValue(NULL);

	value = getKeyword("count");
	if(!value)
		value = getKeyword("limit");
	if(value)
	{
		limit = atoi(value);
		value = getKeyword("index");
		if(!value)
			value = getKeyword("row");
		if(value)
			limit += atoi(value);
	}

	if(!index)
	{
		value = getKeyword("index");
		if(!value)
			value = getKeyword("row");
		if(value)
			row = atoi(value);
		if(!setData(table))
			script[stack].read = NULL;

	}
	rd = script[stack].read;
row:
	while(rd)
	{
		if(rd->method == &ScriptInterp::scrData)
			break;
		rd = rd->next;
	}

	if(row && rd)
	{
		rd = rd->next;
		goto row;
	}

	if(!rd || (limit && index >= limit))
	{
                line = script[stack].line->next;
                while(line)
                {
                        if(line->loop == loop)
                        {
                                script[stack].line = line;
                                advance();
                                return true;
                        }
                        line = line->next;
                }
                error("loop-overflow");
                return true;
	}

	script[stack].read = rd->next;
	while(argc < rd->argc)
	{
		value = getContent(rd->args[argc++]);
		if(!value)
			break;

		if(sz)
			sym = initVariable(size);
		else
			sym = initVariable((unsigned)strlen(value));

		if(!sym)
			break;
		if(sym->flags.readonly)
			continue;

		snprintf(sym->data, sym->flags.size + 1, "%s", value);
		sym->flags.initial = false;
		if(sym->flags.commit)
			commit(sym);
	}
	if(!push())
	{
		error("stack-overflow");
		return true;
	}

	advance();
	return true;
}

bool ScriptInterp::scrForarray(void)
{
	Symbol *sym;
	unsigned short loop = script[stack].line->loop;
	Line *line;
	unsigned index = script[stack].index;
	const char *kw, *value;
	unsigned limit = 0;
	unsigned ix = 0;
	
	value = getKeyword("count");
	if(!value)
		value = getKeyword("limit");
	if(value)
	{
		limit = atoi(value);
		value = getKeyword("index");
		if(value)
			ix = atoi(value);
		if(ix)
			--ix;
		limit += ix;
	}

	script[stack].index = 0;
	value = getOption(NULL);

	if(!index)
	{
		kw = getKeyword("index");
		if(kw)
		{
			index = atoi(kw);
			if(index)
				--index;
		}
	}

	if(!value)
	{
		error("array-not-defined");
		return true;
	}

	sym = getLocal(value);
	if(!sym)
	{
		error("array-not-found");
		return true;
	}

	if(sym->flags.type != ARRAY)
	{
		error("forarray-not-array");
		return true;
	}

	if(index >= (unsigned)sym->data[1] || (limit && index >= limit))
	{

		line = script[stack].line->next;
		while(line)
		{
			if(line->loop == loop)
			{
				script[stack].line = line;
				advance();
				return true;
			}
			line = line->next;
		}
		error("loop-overflow");
		return true;
	}
	sym->data[0] = index;
	script[stack].index = ++index;
	if(!push())
	{
		error("stack-overflow");
		return true;
	}
	advance();
	return true;
}

bool ScriptInterp::scrFordatasource(void)
{
	unsigned short loop = script[stack].line->loop;
	Line *line;
	unsigned index = script[stack].index;
	const char *kw;
	unsigned limit = 0, ix = 0;

	script[stack].index = 0;

	kw = getKeyword("count");
	if(!kw)
		kw = getKeyword("limit");
	if(kw)
	{
		limit = atoi(kw);
		kw = getKeyword("index");
		if(!kw)
			kw = getKeyword("row");
		if(kw)
			ix = atoi(kw);
		if(ix)
			--ix;
		limit += ix;
	}

	if(!index)
	{
		kw = getKeyword("index");
		if(!kw)
			kw = getKeyword("row");
		if(kw)
		{
			index = atoi(kw);
			if(index)
				--index;
		}
	}

	if((limit && index >= limit) || !datasource->fetch(this, index))
 	{
		line = script[stack].line->next;
		while(line)
		{
			if(line->loop == loop)
			{
				script[stack].line = line;
				advance();
				return true;
			}
			line = line->next;
		}
		error("loop-overflow");
		return true;
	}
	script[stack].index = ++index;
	if(!push())
	{
		error("stack-overflow");
		return true;
	}
	advance();
	return true;
}


bool ScriptInterp::scrForeach(void)
{
	Symbol *sym;
	unsigned short loop = script[stack].line->loop;
	Line *line;
	int index = script[stack].index;
	char packtoken = getPackToken();
	unsigned len = 0;
	char *kw = getKeyword("token");
	unsigned size = getSymbolSize();
	const char *sz = getKeyword("size");
	const char *value;

	if(sz)
		size = atoi(sz);

	if(kw && *kw)
		packtoken = *kw;

	script[stack].index = 0;

	if(!index)
	{
		kw = getKeyword("index");
		if(kw)
			index = atoi(kw);
	}

	sym = initVariable(size);

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	sym->data[sym->flags.size] = 0;

	value = getValue();
	if(!value)
	{
		error("list-missing");
		return true;
	}

	while(value[index] && (len--) > 1)
	{
		while(value[index] && value[index] != packtoken)
			++index;
		if(value[index] == packtoken)
			++index;
	}

	if(!value[index])
	{
		line = script[stack].line->next;
		while(line)
		{
			if(line->loop == loop)
			{
				script[stack].line = line;
				advance();
				return true;
			}
			line = line->next;
		}
		error("loop-overflow");
		return true;
	}
	len = 0;
	while(value[index] && value[index] != packtoken && len < sym->flags.size)
		sym->data[len++] = value[index++];
	if(value[index] == packtoken)
		++index;
	sym->data[len] = 0;
	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);

	script[stack].index = index;
	if(!push())
	{
		error("stack-overflow");
		return true;
	}

	advance();
	return true;
}

bool ScriptInterp::scrFor(void)
{
	Symbol *sym;
	unsigned short loop = script[stack].line->loop;
	Line *line;
	int index = script[stack].index;
	const char *opt = getMember();
	unsigned size = getSymbolSize();
	const char *kw = getKeyword("size");

	if(kw)
		size = atoi(kw);

	if(opt && !stricmp(opt, "data"))
		return scrFordata();

	if(opt && !stricmp(opt, "each"))
		return scrForeach();

	if(opt && !stricmp(opt, "array"))
		return scrForarray();

        if(opt && !stricmp(opt, "index"))
                return scrForarray();

	if(opt && datasource)
		if(datasource->isId(opt))
			return scrFordatasource();

	if(opt)
		goto failed;

	script[stack].index = 0;

	if(!index)
	{
		kw = getKeyword("index");
		if(kw)
			index = atoi(kw);
	}

	if(!index)
		++index;

	sym = initVariable(size);
	char *value;

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	sym->data[sym->flags.size] = 0;
	script[stack].index = index;
	value = getValue(NULL);
	if(!value)
	{
failed:
		line = script[stack].line->next;
		while(line)
		{
			if(line->loop == loop)
			{
				script[stack].line = line;
				advance();
				return true;
			}
			line = line->next;
		}
		error("loop-overflow");
		return true;
	}

	strncpy(sym->data, value, sym->flags.size);
	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);

	if(!push())
		return true;

	advance();
	return true;
}

bool ScriptInterp::scrDo(void)
{
	unsigned short loop = script[stack].line->loop;
	Line *line;

	script[stack].index = 0;	// always reset

	if(script[stack].line->argc)
	{
		if(!conditional())
		{
			line = script[stack].line->next;
			while(line)
			{
				if(line->loop == loop)
				{
					script[stack].line = line;
					advance();
					return true;
				}
				line = line->next;
			}
			error("loop-overflow");
			return true;
		}
	}

	if(!push())
		return true;

	advance();
	return true;
}

bool ScriptInterp::scrLoop(void)
{
	unsigned short loop;

	if(stack < 1)
	{
		error("stack-underflow");
		return true;
	}

	loop = script[stack - 1].line->loop;
	if(!loop)
	{
		error("stack-not-loop");
		return true;
	}

	if(script[stack].line->argc)
	{
		if(!conditional())
		{
			script[stack - 1] = script[stack];
			--stack;
			advance();
			return true;
		}
	}

	--stack;
	return execute(script[stack].line->method);
}

bool ScriptInterp::scrContinue(void)
{
	Line *line;
	unsigned short loop;

	if(script[stack].line->argc)
	{
		if(!conditional())
		{
			advance();
			return true;
		}
	}

	if(stack < 1)
	{
		error("stack-underflow");
		return true;
	}

	loop = script[stack - 1].line->loop;
	line = script[stack].line->next;

	if(!loop)
	{
		error("stack-not-loop");
		return true;
	}

	while(line)
	{
		if(line->loop == loop)
		{
			script[stack].line = line;
			return true;
		}
		line = line->next;
	}
	error("loop-overflow");
	return true;
}

bool ScriptInterp::scrBreak(void)
{
	Line *line;
	unsigned short loop;

	if(script[stack].line->argc)
	{
		if(!conditional())
		{
			advance();
			return true;
		}
	}

	if(stack < 1)
	{
		error("stack-underflow");
		return true;
	}

	loop = script[stack - 1].line->loop;
	line = script[stack].line->next;

	if(!loop)
	{
		error("stack-not-loop");
		return true;
	}

	while(line)
	{
		if(line->loop == loop)
		{
			--stack;
			script[stack].line = line;
			advance();
			return true;
		}
		line = line->next;
	}
	error("loop-overflow");
	return true;
}

bool ScriptInterp::scrStruct(void)
{
	const char *id = getOption(NULL);
	char *target = getOption(NULL);
	ScriptSymbol *syms;
	Symbol *sym = NULL, *node, *nn, *last, **record = NULL;
	unsigned len;

	if(recmode)
		recmode->record = NULL;

	recmode = NULL;

	if(id && !target)
	{
		if(strchr(id, '.'))
			recmode = (ScriptSymbol *)this;
		else
			recmode = script[stack].local;
		if(!recmode)
			recmode = (ScriptSymbol *)this;
		recmode->setStruct(id); 
	}

	if(!id || !target)
	{
		advance();
		return true;
	}

	if(strchr(id, '.'))
		syms = (ScriptSymbol *)this;
	else
		syms = script[stack].local;

	if(!syms)
		syms = (ScriptSymbol *)this;

	sym = syms->getEntry(id, 0);

	if(sym && sym->flags.type != RECORD)
	{
		error("invalid-source-object");
		return true;
	}	

	while(sym && target)
	{
		if(*target == '@' || *target == '{')
		{
			target = getContent(target);
			if(!target)
				break;
		}
		if(strchr(target, '.'))
			syms = (ScriptSymbol *)this;
		else
			syms = script[stack].local;
		if(!syms)
			syms = (ScriptSymbol *)this;
		
		memcpy(&node, sym->data + 1, sizeof(Symbol *));
		if(syms->setStruct(target))
		{
			nn = syms->getEntry(target, 0);
			if(nn)
				record = (Symbol **)(nn->data + 1);
			while(node && record)
			{
				len = sizeof(Symbol) + node->flags.size;
				nn = (Symbol *)syms->alloc(len);
				memcpy(nn, node, len);
				nn->next = NULL;
				if(*record)
				{
					last = *record;
					for(;;)
					{
						if(!last->next)
						{
							last->next = nn;
							break;
						}
						last = last->next;
					}
				}
				else
					*record = nn;	
				node = node->next;
			}			
		}
		syms->setStruct(NULL);
		target = getOption(NULL);
	}

	advance();
	return true;
}

bool ScriptInterp::scrBegin(void)
{
	const char *mem = getMember();

	if(!mem)
		mem = "none";

	if(!stricmp(mem, "if"))
	{
		script[stack].tranflag = true;
		return scrIf();
	}

	if(script[stack].tranflag)
	{
		error("begin-already-in-transaction");
		return true;
	}

	script[stack].tranflag = true;
	advance();
	return true;
}

bool ScriptInterp::scrEnd(void)
{
	if(!script[stack].tranflag)
	{
		error("end-not-in-transaction");
		return true;
	}
	script[stack].tranflag = false;
	advance();
	return true;
}

bool ScriptInterp::scrEndcase(void)
{
	script[stack].caseflag = script[stack].tranflag = false;
	advance();
	return true;
}

bool ScriptInterp::scrEndif(void)
{
	script[stack].tranflag = false;
	advance();
	return true;
}

bool ScriptInterp::scrLabel(void)
{
	advance();
	return true;
}

bool ScriptInterp::scrElse(void)
{
	int level = 0;
	Line *line;

	advance();

	while(NULL != (line = script[stack].line))
	{
		advance();

		if(line->method == &ScriptInterp::scrThen)
			++level;
		else if(line->method == &ScriptInterp::scrEndif)
		{
			if(!level)
				return true;
		}
	}
	return true;
}

bool ScriptInterp::scrThen(void)
{
	int level = 0;
	Line *line;

	advance();

	while(NULL != (line = script[stack].line))
	{
		advance();

		if(line->method == &ScriptInterp::scrThen)
			++level;
		else if(line->method == &ScriptInterp::scrElse)
		{
			if(!level)
				return true;
		}
		else if(line->method == &ScriptInterp::scrEndif)
		{
			if(!level)
				return true;
			--level;
		}
	}
	return true;
}

bool ScriptInterp::scrIfThen(void)
{
	if(!conditional())
		advance();

	advance();
	return true;
}

bool ScriptInterp::scrCase(void)
{
	unsigned short loop = 0xffff;
	Line	*line;

	if(!script[stack].caseflag)
		if(conditional() || !script[stack].line->argc)
		{
			script[stack].caseflag = true;
			advance();
			while(script[stack].line)
			{
				if(script[stack].line->method == &ScriptInterp::scrCase)
					advance();
				else
					return true;
			}
			return true;
		}

	if(stack && script[stack].line->loop)
		loop = script[stack - 1].line->loop;

	advance();
	while(NULL != (line = script[stack].line))
	{
		if(line->loop == loop)
			return true;

		if(line->method == &ScriptInterp::scrCase && !script[stack].caseflag)
			return true;

		if(line->method == &ScriptInterp::scrEndcase)
			return true;

		advance();
	}
	return true;
}

bool ScriptInterp::ifGoto(void)
{
        if(script[stack].index < script[stack].line->argc)
                return scrGoto();
        advance();
        if(script[stack].line->method == &ScriptInterp::scrThen)
                advance();
        return true;
}

bool ScriptInterp::scrIf(void)
{
	if(conditional())
		return ifGoto();
	advance();
	return true;
}

bool ScriptInterp::scrLock(void)
{
	const char *id = getKeyword("id");
	const char *member = getMember();

	if(!id)
		id = getValue(NULL);

	if(!id)
	{
		error("no-lock-id");
		return true;
	}

	if(!member)
		member = "wait";

	if(!stricmp(member, "try"))
	{
		if(!locks.lock(this, id))
			error("lock-busy");
		else
			advance();
		return true;
	}

	if(!stricmp(member, "unlock") || !stricmp(member, "ulock") || !stricmp(member, "end"))
	{
		if(!locks.unlock(this, id))
			error("lock-invalid");
		else
			advance();
		if(!stricmp(member, "end"))
			script[stack].tranflag = false;
		return true;
	}

	if(!locks.lock(this, id))
		return true;

	if(!stricmp(member, "begin"))
		script[stack].tranflag = true;

	advance();
	return true;
}

bool ScriptInterp::scrCall(void)
{
	Line *line = getScript();
	unsigned index = script[stack].index;
	int id = 0;
	Symbol *sym;
	char symname[8];
	char idname[33];
	char *arg, *opt;
	const char *member = getMember();
	bool trans = false, local = false;
	ScriptSymbol *ref, *syms = script[stack].local;
	const char *name = getObject()->name;
	char namebuf[65];
	char *n;
	size_t len;

	if(!strnicmp(script[stack].line->cmd, "call", 4))
		trans = true;

	if(!strnicmp(script[stack].line->cmd, "source", 6))
	{
		trans = true;
		if(!member)
			member = "local";
	}

	if(!member)
		member = "none";

	if(!push())
		return true;

	// gosub sets new stack base

	script[stack].tranflag = trans;

	if(!strnicmp(script[stack].line->cmd, "gosub", 5))
		script[stack].base = stack;

	if(!stricmp(member, "global") || !stricmp(member, "public"))
		script[stack].local = NULL;
	else if(stricmp(member, "local") && stricmp(member, "protected"))
	{
		snprintf(idname, sizeof(idname), "%s:%d", _idname, stack);
		script[stack].local = new ScriptSymbol(symsize, pgsize, idname);
		local = true;
		script[stack].local->setConst("0", name);
	}
	else
		return intGoto();

	sprintf(symname, "%d", id++);
	if(local)
		script[stack].local->setConst(symname, name);
	else
	{
		sym = getLocal(symname, symsize);
		if(sym)
		{
			sym->flags.initial = false;
			setString(sym->data, sym->flags.size + 1, name);
		}
	}

	getValue(NULL);	// skip label

	while(NULL != (arg = getOption(NULL)))
	{
		sprintf(symname, "%d", id++);
		if(local && *arg != '&')
		{
			--stack;
			arg = getContent(arg);
			++stack;
			if(!arg)
				arg = "";
			script[stack].local->setConst(symname, arg);
		}
		else if(*arg != '&')
		{
			arg = getContent(arg);
			if(!arg)
				arg = "";
			setVariable(symname, (unsigned)strlen(arg), arg);
		}
		else
		{
			sym = NULL;
			++arg;
			if(syms)
				sym = syms->getEntry(arg, 0);
			if(sym)
				ref = syms;
			else
				ref = this;
			len = strlen(arg) + 1;
			sym = getLocal(symname, (unsigned)len - 1 + sizeof(ref));
			if(!sym->flags.initial)
				continue;
			enterMutex();
			*((ScriptSymbol **)(sym->data)) = ref;
			setString(sym->data + sizeof(ref), len, arg);
			sym->flags.initial = false;
			sym->flags.readonly = true;
			sym->flags.type = REF;
			leaveMutex();
		}
	}


	script[stack].index = index;
	script[stack].tranflag = trans;

	index = 0;
	while(index < line->argc)
	{
		opt = line->args[index++];
		if(*opt != '=')
			continue;
		if(*(++opt) == '%')
			++opt;
		else if(*opt == '.')
		{
			snprintf(namebuf, sizeof(namebuf), "%s", script[stack].script->name);
			n = strchr(namebuf, ':');
			if(n)
				*n = 0;
			snprintf(namebuf + strlen(namebuf), sizeof(namebuf) - strlen(namebuf), "%s", opt);
			opt = namebuf;
		}
		arg = line->args[index++];
		if(local && *arg != '&')
		{
			--stack;
			arg = getContent(arg);
			if(!arg)
				arg = "";
			++stack;
			script[stack].local->setConst(opt, arg);
		}
		else if(*arg != '&')
		{
			arg = getContent(arg);
			if(!arg)
				arg = "";
			setVariable(opt, (unsigned)strlen(arg), arg);
		}
		else
		{
			sym = NULL;
			if(syms)
				sym = syms->getEntry(arg, 0);
			if(sym)
				ref = syms;
			else
				ref = this;
			len = strlen(arg) + 1;
			sym = getLocal(opt, (unsigned)len - 1 + sizeof(ref));
			if(!sym->flags.initial)
				continue;
			enterMutex();
			*((ScriptSymbol **)(sym->data)) = ref;
			setString(sym->data + sizeof(ref), len, arg);
			sym->flags.initial = false;
			sym->flags.readonly = true;
			sym->flags.type = REF;
			leaveMutex();
		}
	}

	return intGoto();
}

void ScriptInterp::clrLocal(void)
{
	if(!script[stack].local)
		return;

	if(stack && script[stack - 1].local == script[stack].local)
	{
		script[stack].local = NULL;
		return;
	}
	delete script[stack].local;
	script[stack].local = NULL;
}

void ScriptInterp::setLocal(void)
{
	char idname[33];
	clrLocal();
	snprintf(idname, sizeof(idname), "%s:%d", _idname, stack);
	script[stack].local = new ScriptSymbol(symsize, pgsize, idname);
}

bool ScriptInterp::scrReturn(void)
{
	Line *line = getScript();
	char *label = getOption(NULL), *ext, *var;
	char namebuf[256];
	unsigned len;
	unsigned argc = 0;
	Name *scr;
	const char *member = getMember();
	bool exitflag = false;
	bool localflag = false;
	bool topflag = false;
	bool errflag = false;
	char *n;

	if(label && *label != '@' && *label != '{')
		label = getContent(label);

	if(!member)
		member = "0";

	if(!stricmp(member, "exit"))
		exitflag = true;
	else if(!stricmp(member, "local"))
		localflag = true;
	else if(!stricmp(member, "top"))
	{
		localflag = true;
		topflag = true;
	}
	else if(!stricmp(member, "cleardigit"))
		cleardigits(false);
	else if(!stricmp(member, "clear"))
		cleardigits(true);

	len = atoi(member);
	if(!len)
		len = getSymbolSize();

	tempidx = 0;
	while(argc < line->argc)
	{
		if(*line->args[argc++] != '=')
			continue;

		snprintf(temps[tempidx], getSymbolSize() + 1,
			"%s", getContent(line->args[argc]));

//		line->args[argc] = temps[tempidx];
		if(tempidx++ >= SCRIPT_TEMP_SPACE)
			tempidx = 0;
		++argc;
	}

	do {
		if(!pull())
		{
			errflag = true;
			if(localflag)
				break;
			if(exitflag)
				scrExit();
			return true;
		}

	} while(script[stack].line->loop != 0 || topflag);

	argc = 0;
	tempidx = 0;
	while(argc < line->argc)
	{
		var = line->args[argc++];
		if(*var != '=')
			continue;

		++argc;
		if(*(++var) == '%')
			++var;
		else if(*var == '.')
		{
			snprintf(namebuf, sizeof(namebuf), "%s", script[stack].script->name);
			n = strchr(namebuf, ':');
			if(n)
				*n = 0;
			snprintf(namebuf + strlen(namebuf), sizeof(namebuf) - strlen(namebuf), "%s", var);
			var = namebuf;
		}
		ext = temps[tempidx++];
		if(tempidx >= SCRIPT_TEMP_SPACE)
			tempidx = 0;
		setVariable(var, len, ext);
	}

retry:
	if(!label)
	{
		if(!errflag)
			advance();
		return true;
	}

	if(!*label)
	{
		if(!errflag)
			advance();
		return true;
	}
	if(*label == '@' || *label == '{')
	{
		if(event(label + 1))
			return true;
	}
	if(*label == '^')
	{
		if(!signal(++label))
		{
			error("trap-invalid");
			return true;
		}
		return true;
	}
        len = (unsigned)strlen(label);
        if(!strncmp(label, "::", 2))
        {
		setString(namebuf, sizeof(namebuf), script[stack].script->name);
                ext = strstr(namebuf, "::");
                if(ext)
			setString(ext, sizeof(namebuf) + namebuf - ext, label);
                else
			addString(namebuf, sizeof(namebuf), label);
                label = namebuf;
        }
        else if(label[len - 1] == ':')
        {
		setString(namebuf, sizeof(namebuf), script[stack].script->name);
                ext = strstr(namebuf, "::");
                if(ext)
			*ext = 0;

		addString(namebuf, sizeof(namebuf), "::");
		addString(namebuf, sizeof(namebuf), label);

                label = namebuf;
		len = (unsigned)strlen(label);
		label[len - 1] = 0;
        }

        scr = getScriptImage(label);
        if(!scr)
        {
		label = getValue(NULL);
		if(label)
			goto retry;
                error("script-not-found");
                return true;
        }
	clearStack();
        once = true;
	script[stack].advance = true;
	script[stack].caseflag = script[stack].tranflag = false;
        script[stack].script = scr;
        script[stack].line = script[stack].first = scr->first;
        script[stack].index = 0;
	script[stack].mask = getScriptMask();
	evtGoto();
        return true;
}

bool ScriptInterp::scrExit(void)
{
	while(stack)
		pull();

	script[stack].line = NULL;
	return true;
}

bool ScriptInterp::scrRemove(void)
{
	Symbol *sym = initVariable();
	char *val;

	if(!sym)
	{
		error("symbol-missing");
		return true;
	}

	if(sym->flags.type != FIFO &&
	   sym->flags.type != STACK &&
	   sym->flags.type != SEQUENCE &&
	   sym->flags.type != CACHE)
	{
		error("symbol-invalid");
		return true;
	}

	while(NULL != (val = getValue()))
		removeSymbol(sym, val);

	advance();
	return true;
}

bool ScriptInterp::scrClear(void)
{
	Symbol *sym = initVariable();

	while(sym)
	{
		if(sym->flags.type == ARRAY)
			sym->data[0] = sym->data[1] = 0;
		else if(sym->flags.type == FIFO || sym->flags.type == SEQUENCE || sym->flags.type == STACK || sym->flags.type == CACHE)
			sym->data[1] = sym->data[2] = 0;
		else if(!sym->flags.readonly)
		{
			sym->data[0] = 0;
			sym->flags.initial = true;
			if(sym->flags.commit)
				commit(sym);
		}
		sym = initVariable();
	}
	advance();
	return true;
}

bool ScriptInterp::scrIndex(void)
{
	const char *idx = getMember();
	unsigned char pos;
	const char *opt;
	Symbol *sym;

	if(!idx)
		idx = getKeyword("pos");
	if(!idx)
		idx = getValue(NULL);

	if(!idx)
	{
		error("index-missing-value");
		return true;
	}
	pos = atoi(idx);
	if(pos)
		--pos;
	while(NULL != (opt = getOption(NULL)))
	{
		sym = getLocal(opt);
		if(!sym)
			continue;
		if(sym->flags.type != ARRAY)
			continue;
		sym->data[0] = pos;
	}
	advance();
	return true;
}

bool ScriptInterp::scrPostList(void)
{
	const char *opt;
	char *vars[65];
	unsigned count = 0, idx;
	Symbol *sym;

	while(NULL != (opt = getOption(NULL)) && count < 64)
	{
		if(!stricmp(opt, "-eq"))
			break;

		if(*opt == '%' || *opt == '&')
			++opt;

		vars[count++] = (char *)opt;
	}

	while(NULL != (opt = getValue(NULL)))
	{
		idx = 0;
		while(idx < count)
		{
			sym = getLocal(vars[idx++]);
			if(!sym)
				continue;
			postSymbol(sym, opt);
		}
	}
	advance();
	return true;
}

bool ScriptInterp::scrPost(void)
{
	Symbol *sym;
	char *opt = getOption(NULL);
	if(!opt)
	{
		error("symbol-missing");
		return true;
	}

	if(*opt != '%')
	{
		error("symbol-invalid");
		return true;
	}

	sym = getLocal(++opt, 0);
	if(!sym)
	{
		error("symbol-missing");
		return true;
	}

	if(sym->flags.type != FIFO && sym->flags.type != SEQUENCE && sym->flags.type != STACK && sym->flags.type != CACHE && sym->flags.type != ARRAY)
	{
		error("symbol-type-invalid");
		return true;
	}

	while(NULL != (opt = getValue(NULL)))
		postSymbol(sym, opt);

	advance();
	return true;
}

bool ScriptInterp::scrArray(void)
{
	unsigned short rec = getSymbolSize() - 10;
	unsigned char count;
	const char *kw = getKeyword("count");
	const char *mem = getMember();
	int rtn;

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	while(NULL != (mem = getOption(NULL)))
	{
		if(!stricmp(mem, "-eq"))
		{
			script[stack].index = 0;
			return scrPostList();
		}
		if(strchr(mem, '.') || !script[stack].local)
			rtn = makeArray(mem, count, rec);
		else
			rtn = script[stack].local->makeArray(mem, count,  rec);
		if(!rtn)
		{
			error("array-make-failed");
			return true;
		}
	}
	advance();
	return true;
}

bool ScriptInterp::scrFifo(void)
{
	unsigned char rec = getSymbolSize() - 10;
	unsigned char count;
	const char *kw = getKeyword("count");
	const char *mem = getMember();
	int rtn;

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	while(NULL != (mem = getOption(NULL)))
	{
		if(!stricmp(mem, "-eq"))
		{
			script[stack].index = 0;
			return scrPostList();
		}
		if(strchr(mem, '.') || !script[stack].local)
			rtn = makeFifo(mem, count, rec);
		else
			rtn = script[stack].local->makeFifo(mem, count,  rec);
		if(!rtn)
		{
			error("fifo-make-failed");
			return true;
		}
	}
	advance();
	return true;
}

bool ScriptInterp::scrCounter(void)
{
	const char *mem = getMember();
	const char *id;
	int rtn;

	if(!mem)
		mem = "1";

	while(NULL != (id = getOption(NULL)))
	{
		if(!stricmp("-eq", id))
		{
			script[stack].index = 0;
			return scrNumber();
		}	
		if(strchr(id, '.') || !script[stack].local)
			rtn = makeCounter(id);
		else
			rtn = script[stack].local->makeCounter(id);
		if(!rtn)
		{
			error("counter-make-failed");
			return true;
		}
		setVariable(id, 0, mem);
	}
	advance();
	return true;
}

bool ScriptInterp::scrStack(void)
{
	unsigned char rec = getSymbolSize() - 10;
	unsigned char count;
	const char *mem = getMember();
	const char *kw = getKeyword("count");

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	while(NULL != (mem = getOption(NULL)))
	{
		if(!stricmp(mem, "-eq"))
		{
			script[stack].index = 0;
			return scrPostList();
		}
		if(!makeStack(mem, count, rec))
		{
			error("stack-make-failed");
			return true;
		}
	}
	advance();
	return true;
}


bool ScriptInterp::scrSequence(void)
{
	unsigned char rec = getSymbolSize() - 10;
	unsigned char count;
	const char *mem = getMember();
	const char *kw = getKeyword("count");
	int rtn;

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	mem = getOption(NULL);
	if(!mem)
	{
		error("symbol-missing");
		return true;
	}

	if(strchr(mem, '.') || !script[stack].local)
		rtn = makeSequence(mem, count, rec);
	else
		rtn = script[stack].local->makeSequence(mem, count, rec);

	if(!rtn)
	{
		error("sequence-make-failed");
		return true;
	}
	--script[stack].index;
	scrPost();
	return true;
}

bool ScriptInterp::scrCache(void)
{
	unsigned char rec = getSymbolSize() - 10;
	unsigned char count;
	const char *mem = getMember();
	const char *kw = getKeyword("count");
	int rtn;

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	mem = getOption(NULL);
	if(!mem)
	{
		error("symbol-missing");
		return true;
	}

	if(strchr(mem, '.') || !script[stack].local)
		rtn = makeCache(mem, count, rec);
	else
		rtn = script[stack].local->makeCache(mem, count, rec);

	if(!rtn)
	{
		error("cache-make-failed");
		return true;
	}
	--script[stack].index;
	scrPost();
	return true;
}

bool ScriptInterp::scrSize(void)
{
	Symbol *sym;
	int size = getSymbolSize();
	const char *opt;

        if(!strnicmp(script[stack].line->cmd, "var", 3))
        {
		opt = getKeyword("size");
		if(!opt)
			opt = getMember();
		if(opt)
			size = atoi(opt);
        }
	else
		size = atoi(getValue("0"));

	opt = getOption(NULL);
	if(!opt)
		opt = getKeyword("name");

	if(!opt)
	{
		error("symbol-not-specified");
		return true;
	}

	if(!size)
	{
		error("symbol-no-size");
		return true;
	}

	while(opt)
	{
		if(*opt == '@')
		{
			opt = getVariable(++opt);
			if(!opt)
			{
				opt = getOption(NULL);
				continue;
			}
		}
		else if(*opt != '%')
		{
			error("symbol-not-constant");
			return true;
		}

		if(*opt == '%')
			++opt;
		if(strchr(opt, '.') || !script[stack].local)
			sym = getEntry(opt, size);
		else
			sym = script[stack].local->getEntry(opt, size);
		opt = getKeyword("value");
		if(opt)
			setVariable(sym->id, size, opt);
		opt = getOption(NULL);
	}

	advance();
	return true;
}

bool ScriptInterp::scrDecimal(void)
{
	script[stack].decimal = atoi(getValue("0"));
	advance();
	return true;
}

bool ScriptInterp::scrError(void)
{
	error(getValue("no-arguments"));
	return true;
}

bool ScriptInterp::scrLoadable(void)
{
	Line *line = getScript();
	const char *err;
	ScriptModule *mod = ScriptModule::find(line->cmd);

	if(!mod)
	{
		error("module-not-found");
		return true;
	}
	err = mod->parseScript(this, line);
	if(err)
	{
		error(err);
		return true;
	}
	if(line != getScript())
		return true;

	advance();
	return true;
}

bool ScriptInterp::scrRef(void)
{
	Symbol *sym = NULL;
	char *ref;
	const char *mem = getKeyword("save");
	char refbuf[256];
	unsigned len = 0, sz = getSymbolSize();
	const char *prefix = getKeyword("prefix");
	const char *suffix = getKeyword("suffix");
	const char *size = getKeyword("Size");

	if(!mem)
		mem = getKeyword("var");

	if(mem)
		if(*mem == '&')
			++mem;

        if(!strnicmp(script[stack].line->cmd, "set", 3))
        {
		if(size)
			sz = atoi(size);
		sym = initVariable(sz);
		if(!sym)
		{
			error("ref-no-symbol");
			return true;
		}
		ref = "ref-invalid-variable";
		if(sym->flags.type == NORMAL && sym->flags.initial)
			ref = NULL;
		else if(sym->flags.type == REF)
			ref = NULL;
		if(ref)
		{
			error(ref);
			return true;
		}
        }
	
	if(prefix)
	{
		snprintf(refbuf, sizeof(refbuf), "%s", prefix);
		len = (unsigned)strlen(refbuf);
	}
	
	while(NULL != (ref = getValue(NULL)) && len < 250)
	{
		if(*ref == '%')
			++ref;

		if(script[stack].line->argc <= script[stack].index && !mem)
			break;

		if(len && !prefix)
			snprintf(refbuf + len, sizeof(refbuf) - len,
				".%s", ref);
		else
			snprintf(refbuf + len, sizeof(refbuf) - len,
				"%s", ref);
		len = (unsigned)strlen(refbuf);
		prefix = NULL;
	}
	if(suffix)
	{
		snprintf(refbuf + len, sizeof(refbuf) - len, "%s", suffix);
		len = (unsigned)strlen(refbuf);
	}

	if(mem)
		ref = (char *)mem;

	if(!sym)
	{
		sym = getLocal(ref, len + sizeof(ScriptSymbol *));
		if(!sym->flags.initial)
		{
			error("alias-invalid-reference-object");
			return true;
		}
	}
	enterMutex();
	*((ScriptSymbol **)(sym->data)) = this;
	len = sym->flags.size - 4 + 1;
	snprintf(sym->data + sizeof(ScriptSymbol *), len, "%s", refbuf);  
	sym->flags.readonly = true;
	sym->flags.initial = false;
	sym->flags.type = REF;
	leaveMutex();
	advance();
	return true;
}

bool ScriptInterp::scrAlias(void)
{
	char *sym;
	char *src;

	while(NULL != (sym = getOption(NULL)))
	{
		if(*sym != '%' && !isalpha(*sym))
		{
			error("alias-not-symbol");
			return true;
		}
		src = getOption(NULL);
		if(*src && *src != '&')
			src = getContent(src);
		else if(src)
			++src;

		if(!src)
		{
			error("alias-no-source");
			return true;
		}

		if(!setAlias(sym, src))
		{
			error("alias-failure");
			return true;
		}
	}

	advance();
	return true;
}

bool ScriptInterp::scrConst(void)
{
	char *sym = NULL;
	char *val;
	unsigned max = getSymbolSize();
	max = 1024;
	char buffer[1024];
	int count = 0;
	unsigned idx = 0, len = 0;
	Line *line = getScript();
	bool fconst = false;
	char symname[5];
	unsigned id = 1;

	if(!stricmp(script[stack].line->cmd, "fconst"))
		fconst = true;
	else
		sym = getOption(NULL);

	buffer[0] = 0;
	if(!sym)
	{
		while(idx < line->argc)
		{
			val = line->args[idx++];
			if(*val != '=')
				continue;
			if(*(++val) == '%')
				++val;
			setConst(val, getContent(line->args[idx++]));
			++count;
		}

		if(fconst)
			goto scan;

		if(count)
			advance();
		else
			error("const-not-specified");
		return true;
	}

scan:
	while(NULL != (val = getValue(NULL)) && len < max)
	{
		if(fconst)
		{
			snprintf(symname, sizeof(symname), "%d", id++);
			if(script[stack].local)
				script[stack].local->setConst(symname, val);
			else
				setVariable(symname, sizeof(val), val);
			continue;
		}

		strncpy(buffer + len, val, max - len);
		buffer[max] = 0;
		len = (unsigned)strlen(buffer);
	}

	if(fconst)
	{
		advance();
		return true;
	}

	if(script[stack].local && !strchr(sym, '.'))
		script[stack].local->setConst(sym, buffer);
	else if(!setConst(sym, buffer))
	{
		error("const-not-set");
		return true;
	}
	advance();
	return true;
}


bool ScriptInterp::scrDup(void)
{
	const char *id;
	Symbol *src, *dup;
	while(NULL != (src = initVariable()))
	{
		dup = initVariable(src->flags.size);
		if(!dup)
		{
			error("no-target");
			return true;

		}
		enterMutex();
		if(!dup->flags.initial)
		{
			leaveMutex();
			error("target-exists");
			return true;
		}
		id = dup->id;
		memcpy(dup, src, sizeof(Symbol) + src->flags.size);
		dup->id = id;
		leaveMutex();
	}
	advance();
	return true;
}

bool ScriptInterp::scrNumber(void)
{
	unsigned prec = 0;
	const char *mem = getMember();
	Symbol *sym;
	const char *content;
	char *p;
	char fmt[13];
	long iNumber, hNumber, lNumber;
	char dec = *getSymbol("script.decimal");
	bool hex = false;
	Property *prop = NULL;

	if(mem)
	{
		prop = Property::find(mem);
		prec = atoi(mem);
		if(!stricmp("hex", mem))
			hex = true;
	}
	else
		prec = script[stack].decimal;

	if(prec > 7)
		prec = 7;

	snprintf(fmt, sizeof(fmt), "%s%d%s", "%ld.%0", prec, "ld");

	while(NULL != (content = getOption(NULL)))
	{
		if(!stricmp(content, "-eq"))
			break;

		if(*content == '%')
			++content;

		if(prop)
			sym = getLocal(content, prop->getPropertySize());
		else
			sym = getLocal(content, 11);
		if(!sym)
			continue;

		if(!sym->flags.initial)
			continue;

		if(sym->flags.readonly)
			continue;

		if(hex)
			snprintf(sym->data, sym->flags.size + 1, "0x00000000");
		else if(prop)
			prop->setProperty(sym->data, "", sym->flags.size);
		else if(prec)
			snprintf(sym->data, sym->flags.size + 1, fmt, 0, 0);
		else
			snprintf(sym->data, sym->flags.size + 1, "0");
		p = strchr(sym->data, '.');
		if(p)
			*p = dec;
		sym->flags.initial = false;
		if(sym->flags.commit)
			commit(sym);
	}

	if(!content)
	{
		advance();
		return true;
	}

	if(getExpression(&iNumber, 1, prec, prop) != 1)
	{
		error("bad-expression");
		return true;
	}

	script[stack].index= 0;

	while(NULL != (content = getOption(NULL)))
	{
		if(!stricmp(content, "-eq"))
			break;
		
		if(*content == '%')
			++content;

		sym = getLocal(content, 0);

		if(!sym)
			continue;

		if(sym->flags.readonly)
			continue;

		if(prop)
		{
			prop->setValue(sym->data, sym->flags.size, iNumber);
			if(sym->flags.commit)
				commit(sym);
			continue;
		}

		hNumber = iNumber / tens[prec];
		lNumber = iNumber % tens[prec];
		if(lNumber < 0)
			lNumber = -lNumber;
		if(hex)
			snprintf(sym->data, sym->flags.size + 1, "0x%08lx", iNumber);
		else if(prec)
			snprintf(sym->data, sym->flags.size + 1, fmt, hNumber, lNumber);
		else
			snprintf(sym->data, sym->flags.size + 1, "%ld", iNumber);

		p = strchr(sym->data, '.');
		if(p)
			*p = dec;

		if(sym->flags.commit)
			commit(sym);
	}
	advance();
	return true;
}

/*
static int keyindex(const char *keyword)
{
        unsigned key = 0;

        while(*keyword)
                key ^= key << 1 ^ (*(keyword++) & 0x1f);

        return key % KEYWORD_INDEX_SIZE;
}
*/

bool ScriptInterp::scrSet(void)
{
	Property *prop = NULL;
	Symbol *sym;
	int size = 0;
	bool first = true, iflag = false;
	unsigned len, rlen = 0;
	const char *value;
	bool number = false;
	unsigned sz = 0, idx = 0, count = 0;
	Line *line = getScript();

	enum
	{
		VAL_FILL,
		VAL_RIGHT,
		VAL_NONE,
		VAL_MIN,
		VAL_MAX,
		VAL_SPLIT,
		VAL_CENTER,
		VAL_ADD
	}	minmax = VAL_NONE;

	if(!strnicmp(script[stack].line->cmd, "init", 4))
	{
		iflag = true;
		value = getMember();
	}
	else if(!strnicmp(script[stack].line->cmd, "add", 3))
	{
		minmax = VAL_ADD;
		value = NULL;
	}
	else
		value = getMember();

	if(value)
	{
		if(!stricmp(value, "ref"))
			return scrRef();
		else if(!stricmp(value, "size"))
			size = atoi(getValue("0"));
		else if(!stricmp(value, "min"))
			minmax = VAL_MIN;
		else if(!stricmp(value, "max"))
			minmax = VAL_MAX;
		else if(!stricmp(value, "right"))
			minmax = VAL_RIGHT;
		else if(!stricmp(value, "fill") || !stricmp(value, "left"))
			minmax = VAL_FILL;
		else if(!stricmp(value, "split"))
			minmax = VAL_SPLIT;
		else if(!stricmp(value, "add") || !stricmp(value, "append"))
			minmax = VAL_ADD;
		else if(!stricmp(value, "center"))
			minmax = VAL_CENTER;
		else if(!strnicmp(value, "val", 3) || !strnicmp(value, "num", 3))
		{
			number = true;
			size = 11;
		}
		else if(NULL ==(prop = Property::find(value)))
			size = atoi(value);
	}
	else
	{
		value = getKeyword("justify");
		if(!value)
			value = "";
		if(!stricmp(value, "right"))
			minmax = VAL_RIGHT;
		else if(!stricmp(value, "center"))
			minmax = VAL_CENTER;
		else if(!stricmp(value, "left"))
			minmax = VAL_FILL;
	}
	if(minmax == VAL_ADD)
	{
		sym = initVariable(sz);
		if(!sym)
		{
			error("no-sym-to-add");
			return true;
		}
	}
	else
	{
		value = getKeyword("size");
		if(value)
			sz = atoi(value);

		if(!size && prop)
			size = prop->getPropertySize();

		if(!size)
			size = getSymbolSize();

		if(!sz)
			sz = size;
		sym = initVariable(sz);
	}
	if(!sym)
	{
		while(idx < line->argc)
		{
			value = line->args[idx++];
			if(*value != '=')
				continue;
			if(*(++value) == '%')
				++value;
			++count;
			if(strchr(value, '.'))
				sym = getEntry(value, size);
			else
				sym = getLocal(value, size);
			value = line->args[idx++];
			if(!sym)
				continue;
			if(iflag && !sym->flags.initial)
				continue;
			setVariable(sym->id, size, value);
		}
		if(count)
			advance();
		else
			error("symbol-not-found");
		return true;
	}

	switch(sym->flags.type)
	{
	case ARRAY:
	case FIFO:
	case STACK:
	case CACHE:
	case SEQUENCE:
		script[stack].index = 0;
		return scrPost();
	default:
		break;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	if(iflag && !sym->flags.initial)
	{
		advance();
		return true;
	}

	sym->data[sym->flags.size] = 0;
	if(minmax == VAL_ADD)
		len = (unsigned)strlen(sym->data);
	else
		len = 0;

	while((minmax == VAL_RIGHT || minmax == VAL_CENTER) && rlen < sym->flags.size)
		sym->data[rlen++] = ' ';

	while(NULL != (value = getValue(NULL)))
	{

		first = false;

		if(len >= sym->flags.size)
			break;

		if(minmax == VAL_MIN && atoi(value) >= atoi(sym->data) && len)
			continue;

		if(minmax == VAL_MAX && atoi(value) <= atoi(sym->data) && len)
			continue;
	
		if(minmax == VAL_RIGHT || minmax == VAL_CENTER)
		{
			rlen -= (unsigned)strlen(value);
			if(minmax == VAL_CENTER && rlen > 0)
				rlen /= 2;

			if(rlen < 0)
			{
				value -= rlen;
				rlen = 0;
			}
			strncpy(sym->data + rlen, value, strlen(value));
			if(minmax == VAL_CENTER)
				rlen = sym->flags.size;
		}
		else if(minmax == VAL_NONE || minmax == VAL_FILL || minmax == VAL_SPLIT || minmax == VAL_ADD)
			strncpy(sym->data + len, value, sym->flags.size - len);
		else
			strncpy(sym->data, value, sym->flags.size);
		sym->data[sym->flags.size] = 0;
		len = (unsigned)strlen(sym->data) - rlen;
		if(minmax == VAL_SPLIT)
		{
			rlen = len;
			while(rlen < sym->flags.size)
				sym->data[rlen++] = ' ';
			minmax = VAL_RIGHT;
			len = 0;
		}
	}

	while(len < sym->flags.size && minmax == VAL_FILL)
		sym->data[len++] = ' ';
	
	sym->data[len + rlen] = 0;	

	if(number)
		sprintf(sym->data, "%ld", atol(sym->data));
	else if(prop)
	{
		setString(temps[tempidx], getSymbolSize() + 1, sym->data);
		prop->setProperty(sym->data, temps[tempidx], sym->flags.size);
	}

	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);

	advance();
	return true;
}

bool ScriptInterp::scrDump(void)
{
	Symbol *index[64];
	unsigned pos = 0;
	unsigned count = gather(index, 63, getValue(NULL), NULL);	

	while(pos < count)
	{
		slog.debug() << index[pos]->id << " = " << index[pos]->data << endl;
		++pos;
	}
	advance();
	return true;
}

bool ScriptInterp::scrGather(void)
{
	Symbol *sym = initVariable(getSymbolSize());
	char *suffix = getValue(NULL);
	Name *list[33];
	int count = 0, idx;

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	count = image->gather(suffix, list, 32);
	if(!count)
	{
		error("no-scripts-found");
		return true;
	}
	sym->data[0] = 0;
	for(idx = 0; idx < count; ++idx)
	{
		if(idx)
			strcat(sym->data, ",");
		strcat(sym->data, list[idx]->name);
	}
	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);
	advance();
	return true;
}


long ScriptInterp::getRealValue(double d, unsigned prec)
{
	char buf[20];
	char *cp;
	char lval[9];
	unsigned count;
	long rval;
	snprintf(buf, sizeof(buf), "%f", d);

	rval = atol(buf) * tens[prec];
	cp = strchr(buf, '.');
	if(!cp)
		return rval;
	count = (unsigned)strlen(++cp);
	if(count > prec)
		count = prec;
	strcpy(lval, "00000000");
	strncpy(lval, cp, count);
	lval[prec] = 0;
	if(rval < 0)
		return rval - atol(lval);
	return rval + atol(lval);
}

long ScriptInterp::getIntValue(const char *text, unsigned prec, Property *p)
{
	Fun *fun = ifun;
	unsigned count;
	char *arg;
	long *ival, rval;
	char lval[9];
	const char *orig = text;
	char dec = *getSymbol("script.decimal");

	if(p && p->isProperty(text))
	{
		rval = p->getValue(text);
		return rval * tens[prec];
	}

	if(!isalpha(*text))
	{
		if(!strnicmp("0x", text, 2))
		{
			rval = strtol(text, NULL, 16);
			return rval * tens[prec];
		}
		rval = atol(text);
		rval *= tens[prec];
		text = strchr(text, '.');
		if(!text)
			text = strrchr(orig, ',');
		if(!text)
			text = strrchr(orig, dec);
		if(!text)
			return rval;
		count = (unsigned)strlen(++text);
		if(count > prec)
			count = prec;
		strcpy(lval, "00000000");
		strncpy(lval, text, count);
		lval[prec] = 0;
		if(rval < 0)
			return rval - atol(lval);

		return rval + atol(lval);
	}	

	while(NULL != fun)
	{
		if(!stricmp(fun->id, text))
			break;
		fun = fun->next;
	}

	if(!fun)
		return 0;

	if(!fun->args)
		return fun->fn(NULL, prec);

	arg = getValue(NULL);
	if(!arg)
		return 0;

	if(stricmp(arg, "("))
		return 0;

	ival = new long[fun->args];

	count = getExpression(ival, fun->args, prec);
	if(count != fun->args)
		return 0;

	rval = fun->fn(ival, prec);	
	delete[] ival;
	return rval;
}	

double ScriptInterp::getDouble(long value, unsigned prec)
{
        return (double)value / (double)tens[prec];
}

long ScriptInterp::getInteger(long value, unsigned prec)
{
	return value / tens[prec];
}

long ScriptInterp::getTens(unsigned prec)
{
	return tens[prec];
}

int ScriptInterp::getExpression(long *vals, int max, unsigned prec, Property *p)
{
	char estack[32];
	unsigned esp = 0;
	long vstack[32], val;
	const char *value;
	char **expr;
	int count = 0;
	long nval;

	static char *elist[] = {"*", "+", "-", "/", "%", NULL};

	vstack[0] = 0;

	while(NULL != (value = getValue(NULL)))
	{
		expr = elist;
		while(*expr)
		{
			if(!stricmp(*expr, value))
				break;
			++expr;
		}
		if(*expr)
			estack[esp] = *value;
		else
			estack[esp] = 0;

		if(!stricmp(value, "("))
		{
			if(esp < 31)
			{
				++esp;
				vstack[esp] = 0;
				continue;
			}
			return -1;
		}

		if(!stricmp(value, ","))
		{
			if(esp)
				return -1;
			if(count < max)
				*(vals++) = vstack[0];
			vstack[0] = 0;
			++count;
			continue;
		}
		if(!stricmp(value, ")"))
		{
			if(!esp)
			{
				if(count < max)
					*(vals++) = vstack[0];
				return ++count;
			}

			switch(estack[--esp])
			{
			case '+':
				vstack[esp] = vstack[esp] + vstack[esp + 1];
				break;
			case '*':
				vstack[esp] = vstack[esp] * vstack[esp + 1] / tens[prec];
				break;
			case '/':
				if(vstack[esp + 1] == 0)
					return -1;
				vstack[esp] = vstack[esp] * tens[prec] / vstack[esp + 1];
				break;
			case '-':
				vstack[esp] = vstack[esp] - vstack[esp + 1];
				break;
			case '%':
				vstack[esp] = vstack[esp] % vstack[esp + 1];
				break;
			default:
				vstack[esp] = vstack[esp + 1];
			}
			if(p)
			{
				nval = p->adjustValue(getInteger(vstack[esp], prec));
				vstack[esp] = nval * tens[prec];
			}
			continue;
		}
	
		if(!*expr)
		{
			vstack[esp] = getIntValue(value, prec, p);
			continue;
		}

		value = getValue("0");
		if(!stricmp(value, "("))
		{
			if(esp > 31)
				return -1;
			vstack[++esp] = 0;
			continue;
		}

		val = getIntValue(value, prec, p);

		switch(estack[esp])
		{
		case '+':
			vstack[esp] = vstack[esp] + val;
			break;
		case '-':
			vstack[esp] = vstack[esp] - val;
			break;
		case '/':
			if(!val)
				return -1;
			vstack[esp] = vstack[esp] * tens[prec] / val;
			break;
		case '*':
			vstack[esp] = vstack[esp] * val / tens[prec];
			break;
		case '%':
			vstack[esp] = vstack[esp] % atol(value);
			break;
		}
		if(p)
		{
			nval = p->adjustValue(getInteger(vstack[esp], prec));
			vstack[esp] = nval * tens[prec];
		}
	}
	if(count < max)
	{
		if(p)
			*(vals++) = p->adjustValue(getInteger(vstack[esp], prec));
		else
			*(vals++) = vstack[esp];
	}
	if(!esp)
		return ++count;
	else
		return -1;
}

bool ScriptInterp::conditional(void)
{
	Line *line = script[stack].line;
	char *joiner;
	bool rtn;
	bool andfalse = false;
	bool ortrue = false;

	for(;;)
	{
		rtn = expConditional();
		if(script[stack].index < line->argc)
			joiner = line->args[script[stack].index];
		else
			joiner = "";

		if(!stricmp(joiner, "and"))
		{
			if(!rtn)
				andfalse = true;
		}
		else if(!stricmp(joiner, "or"))
		{
			if(rtn)
				ortrue = true;
		}
		else
			break;

		++script[stack].index;
	}
	if(andfalse)
		return false;

	if(ortrue)
		return true;

	return rtn;
}
		

bool ScriptInterp::expConditional(void)
{
#ifdef	HAVE_REGEX_H
	regex_t *regex;
	bool rtn;
#endif

	Test *node = NULL;
	char *v1, *op = NULL, *v2;
	char n1[12], n2[12];
	size_t l1, l2;
	long ival;

	// no first parm, invalid

	v1 = getOption(NULL);
	if(!v1)
		return false;

	if(*v1 == '-' || *v1 == '!')
	{
		node = ScriptInterp::test;
		while(node)
		{
			if(!stricmp(node->id, v1 + 1))
				break;
			node = node->next;
		}
		if(node)
			op = (char *)node->id;
	}

	if(!strcmp(v1, "("))
	{
		getExpression(&ival, 1, script[stack].decimal);
		snprintf(n1, sizeof(n1), "%ld", ival);
		v1 = n1;
	}
	else if(!stricmp(v1, "-script") || !stricmp(v1, "!script"))
		op = v1;
	else if(!stricmp(v1, "-defined") || !stricmp(v1, "!defined"))
		op = v1;
	else if(!stricmp(v1, "-empty") || !stricmp(v1, "!empty"))
		op = v1;
	else if(!stricmp(v1, "-module") || !stricmp(v1, "-installed") || !stricmp(v1, "-has"))
		op = "-module";
	else if(!stricmp(v1, "!module") || !stricmp(v1, "!installed"))
		op = "!module";
	else
		v1 = getContent(v1);

	// sym/label by itself, invalid

	if(!op)
		op = getValue(NULL);

	if(!op)
	{
		script[stack].index = 0;
		if(v1 && *v1)
			return true;
		return false;
	}

	// ifdef sym ... format assumed

	v2 = getOption(NULL);

	if(!v2)
	{
		script[stack].index = 1;
		if(v1 && *v1)
			return true;
		return false;
	}

	if(node)
	{
		if(*v1 == '!')
			return !node->handler(this, getContent(v2));
		else
			return node->handler(this, getContent(v2));
	}

	if(!stricmp(op, "-script"))
	{
		if(getScriptImage(getContent(v2)))
			return true;

		return false;
	}

	if(!stricmp(op, "!script"))
	{
		if(!getScriptImage(getContent(v2)))
			return true;
	
		return false;
	}

	if(!stricmp(op, "-module"))
	{
		if(cmd->getHandler(getContent(v2)))
			return true;
		return false;
	}

	if(!stricmp(op, "!module"))
	{
		if(cmd->getHandler(getContent(v2)))
			return false;
		return true;
	}

	if(!stricmp(op, "-defined"))
	{
		if(getVariable(v2))
			return true;
		return false;
	}

	if(!stricmp(op, "!defined"))
	{
		if(getVariable(v2))
			return false;
		return true;
	}

	if(!stricmp(op, "-empty"))
	{
		v2 = getContent(v2);
		if(!v2)
			return true;

		if(!*v2)
			return true;

		return false;
	}

	if(!stricmp(op, "!empty"))
	{
		v2 = getContent(v2);
		if(!v2)
			return false;

		if(!*v2)
			return false;

		return true;
	}

	if(!strcmp(v2, "("))
	{
		getExpression(&ival, 1, script[stack].decimal);
		snprintf(n2, sizeof(n2), "%ld", ival);
		v2 = n2;
	}
	else
		v2 = getContent(v2);

	if(!v1)
		v1 = "";

	if(!v2)
		v2 = "";

	if(!stricmp(op, "=") || !stricmp(op, "-eq"))
	{
		if(atol(v1) == atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, "<>") || !stricmp(op, "-ne"))
	{
		if(atol(v1) != atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, "==") || !stricmp(op, ".eq."))
	{
		if(!stricmp(v1, v2))
			return true;

		return false;
	}

	if(!stricmp(op, "!=") || !stricmp(op, ".ne."))
	{
		if(stricmp(v1, v2))
			return true;

		return false;
	}


	if(!stricmp(op, "$") || !stricmp(op, ".in."))
	{
		if(strstr(v2, v1))
			return true;
		return false;
	}

	if(!stricmp(op, "!$"))
	{
		if(strstr(v2, v1))
			return false;

		return true;
	}

#ifdef	HAVE_REGEX_H
	if(!stricmp(op, "~") || !stricmp(op, "!~"))
	{
		script[stack].tranflag = false;
		rtn = false;
		regex = new regex_t;
		memset(regex, 0, sizeof(regex_t));

		if(regcomp(regex, v2, REG_ICASE|REG_NOSUB|REG_NEWLINE))
		{
			regfree(regex);
			delete regex;
			return false;
		}

		if(regexec(regex, v1, 0, NULL, 0))
			rtn = false;
		else
			rtn = true;

		regfree(regex);
		delete regex;
		if(*op == '!')
			return !rtn;
		return rtn;
	}
#endif

	if(!stricmp(op, "$<") || !stricmp(op, "$+") || !stricmp(op, ".prefix."))
	{
		if(!strnicmp(v1, v2, strlen(v1)))
			return true;
		return false;
	}

	if(!stricmp(op, "$>") || !stricmp(op, "$-") || !stricmp(op, ".suffix."))
	{
		l1 = strlen(v1);
		l2 = strlen(v2);
		if(l1 <= l2)
			if(!strnicmp(v1, v2 + l2 - l1, l1))
				return true;

		return false;
	}

	if(!stricmp(op, "<") || !stricmp(op, "-lt"))
	{
		if(atol(v1) < atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, ".le."))
	{
		if(stricmp(v1, v2) <= 0)
			return true;
		return false;
	}

	if(!stricmp(op, ".ge."))
	{
		if(stricmp(v1, v2) >= 0)
			return true;
		return false;
	}

	if(!stricmp(op, "<=") || !stricmp(op, "=<") || !stricmp(op, "-le"))
	{
		if(atol(v1) <= atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, ">") || !stricmp(op, "-gt"))
	{
		if(atol(v1) > atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, ">=") || !stricmp(op, "=>") || !stricmp(op, "-ge"))
	{
		if(atol(v1) >= atol(v2))
			return true;
		return false;
	}

	// if no op, assume ifdef format

	script[stack].index = 1;
	if(*v1)
		return true;

	return false;
}

bool ScriptInterp::push(void)
{
	if(stack >= (SCRIPT_STACK_SIZE - 1))
	{
		error("stack-overflow");
		return false;
	}

	script[stack + 1] = script[stack];
	script[stack + 1].caseflag = script[stack + 1].tranflag = false;
	script[stack + 1].advance = true;
	++stack;
	return true;
}

bool ScriptInterp::pull(void)
{
	if(!stack)
	{
		error("stack-underflow");
		return false;
	}

	if(script[stack - 1].local != script[stack].local)
	{
		if(script[stack].local)
			delete script[stack].local;
	}

	--stack;
	return true;
}

bool ScriptInterp::getOnce(void)
{
	bool ret = once;
	once = false;
	return ret;
}

void ScriptInterp::setLine(Line *line)
{
	script[stack].line = line;
	script[stack].index = 0;
}

bool ScriptInterp::redirect(const char *scriptname)
{
	Name *scr;
	char namebuf[128];
	char *ext;

	if(!strncmp(scriptname, "::", 2))
	{
		setString(namebuf, sizeof(namebuf), script[stack].script->name);
		ext = strstr(namebuf, "::");
		if(ext)
			*ext = 0;
		addString(namebuf, sizeof(namebuf), scriptname);
	}
	else
		setString(namebuf, sizeof(namebuf), scriptname);

	scr = getScriptImage(namebuf);
	if(scr)
	{
		clearStack();
		script[stack].script = scr;
		script[stack].line = script[stack].first = scr->first;
		script[stack].mask = getScriptMask();
		evtGoto();
		return true;
	}
	return false;
}

void ScriptInterp::initRuntime(Name *scr)
{
	while(stack)
		pull();
	script[stack].script = scr;
        script[stack].line = script[stack].first = script[stack].script->first;
        script[stack].index = 0;
        script[stack].read = NULL;
        script[stack].caseflag = script[stack].tranflag = false;
        script[stack].advance = true;
        script[stack].decimal = 0;
        script[stack].base = 0;
        script[stack].mask = script[stack].script->mask;
}

bool ScriptInterp::attach(const char *scriptname)
{
	ScriptImage::InitialList *ilist;
	ScriptModule *mod;
	Name *scr, *init;
	stack = 0;
	lckcount = 0;
	cmd->enterMutex();
	image = cmd->active;
	initialized = false;

	if(!image)
	{
		cmd->leaveMutex();
		return false;
	}
	script[stack].local = NULL;
	scr = getScriptImage(scriptname);
	if(scr && scr->access == Name::aPUBLIC)
	{
		ilist = image->ilist;
		while(ilist)
		{
			setVariable(ilist->name, ilist->size, ilist->value);
			ilist = ilist->next;
		}
		setSymbol("script.home", scriptname);
		mod = ScriptModule::first;
		while(mod)
		{
			mod->moduleAttach(this);
			mod = mod->next;
		}
		initialize();
		init = image->index[SCRIPT_INDEX_SIZE];
		while(init)
		{
			initRuntime(init);
			while(step())
				Thread::yield();	
			init = init->next;
		}
		initRuntime(scr);
		++image->refcount;
		if(script[stack].line->method == &ScriptInterp::scrGoto)
			(this->*(script[stack].line->method))();
		cmd->leaveMutex();
		initialized = true;
		return true;
	}
	cmd->leaveMutex();
	once = true;
	signalmask = 0;
	logerror(scriptname, getId(), "missing; attach failed");
	return false;
}

void ScriptInterp::detach(void)
{
	ScriptModule *mod;
	char scrname[65];
	char *sp;

	recmode = NULL;
	record = NULL;

	snprintf(scrname, sizeof(scrname), "%s", script[0].script->name);
	sp = strchr(scrname, ':');
	if(sp)
		*sp = 0;	

	if(!image)
		return;

	cmd->enterMutex();
	--image->refcount;
	mod = ScriptModule::first;
	while(mod)
	{
		mod->moduleDetach(this, scrname);
		mod = mod->next;
	}
	if(image)
		if(!image->refcount && image != cmd->active)
			delete image;

	cmd->leaveMutex();
	image = NULL;

	while(stack)
		pull();
	locks.release(this);
}

unsigned ScriptInterp::initKeywords(unsigned size)
{
	unsigned idx = 0;
	unsigned count = 0;
	Line *line = script[stack].line;
	char *opt, *cp;
	Symbol *sym;
	char namebuf[65];
	char *n;
	unsigned len;

	if(!size)
		size = getSymbolSize();

	while(idx < line->argc)
	{
		opt = line->args[idx++];
		if(*opt != '=')
			continue;
		if(*(++opt) == '%')
			++opt;
		else if(*opt == '.')
		{
			snprintf(namebuf, sizeof(namebuf), "%s", script[stack].script->name);
			n = strchr(namebuf, ':');
			if(n)
				*n = 0;
			snprintf(namebuf + strlen(namebuf), sizeof(namebuf) - strlen(namebuf), "%s", opt);
			opt = namebuf;
		}
		++count;
		cp = getContent(line->args[idx++]);
		if(!cp)
			continue;

		if(*cp == '&')
		{
			++cp;
			len = (unsigned)strlen(cp) + 1;
			sym = getLocal(opt, len - 1 + sizeof(ScriptSymbol *));
			if(!sym->flags.initial)
				continue;

			enterMutex();
			*((ScriptSymbol **)(sym->data)) = this;
			setString(sym->data + sizeof(ScriptSymbol *), len, cp);
			sym->flags.initial = false;
			sym->flags.readonly = true;
			sym->flags.type = REF;
			leaveMutex();
			continue;
		}

		if(script[stack].local == NULL || strchr(opt, '.'))
		{
			setSymbol(opt, size);
			setSymbol(opt, cp);
			continue;
		}

		script[stack].local->setConst(opt, cp);
		script[stack].local->setSymbol(opt, cp);
	}
	return count;
}

char *ScriptInterp::getKeyword(const char *kw)
{
	unsigned idx = 0;
	Line *line = script[stack].line;
	char *opt;
	while(idx < line->argc)
	{
		opt = line->args[idx++];
		if(*opt == '=')
		{
			if(!strnicmp(kw, opt + 1, strlen(kw)))
				return getContent(line->args[idx]);
			++idx;
		}
	}
	return NULL;
}

char *ScriptInterp::getOption(const char *def)
{
	for(;;)
	{
		if(script[stack].index >= script[stack].line->argc)
			return (char *)def;

		if(*script[stack].line->args[script[stack].index] != '=')
			break;

		script[stack].index += 2;
	}

	return script[stack].line->args[script[stack].index++];
}

Script::Symbol *ScriptInterp::initVariable(unsigned size)
{
	char *opt;
	Symbol *sym;
	char alt[128];
	char *cp;
	bool local = true;

	for(;;)
	{
		if(script[stack].index >= script[stack].line->argc)
			return NULL;

		opt = script[stack].line->args[script[stack].index++];

		if(*opt != '=')
			break;

		++script[stack].index;
	}

	if(*opt != '%' && *opt != '@')
		return NULL;

	if(*opt == '@')
	{
		if(strchr(++opt, '.'))
			sym = getEntry(opt);
		else
			sym = getLocal(opt);

		if(sym)
			opt = sym->data;
		else
			return NULL;
	}
	else
		++opt;

        if(strchr(opt + 1, '#'))
        {
                snprintf(alt, sizeof(alt), "%s",  opt);
                cp = strchr(alt + 1, '#');
                *(cp++) = 0;
                sym = getLocal(cp);
                if(!sym)
			return NULL;

                snprintf(alt + strlen(alt), sizeof(alt) - strlen(alt), ".%s", sym->data);
                opt = alt;
        }

	if(strchr(opt, '.'))
		local = false;

	if(!local)
		return getEntry(opt, size);
	else
		return getLocal(opt, size);
}

void ScriptInterp::missing(const char *sym)
{
//	if(!warn)
//		return;

	if(*sym == '@' || *sym == '%' || *sym == '&')
		++sym;

	slog.warn() << "script: " << script[stack].script->filename 
		<< "(" << script[stack].line->lnum << "): "
		<< sym << " undefined" << endl;
}


char *ScriptInterp::getContent(char *opt)
{
	Property *prop;
	Symbol *sym;
	char *ext = strrchr(opt, '.');
	char *buf, *dec;
	unsigned pos;
	char *cp;
	unsigned slen = getSymbolSize() + 1;
	char *subref = NULL;

	char alt[128];


	if(!opt)
		return NULL;

/*
	if(*opt == '!' && script[stack].line->prescan)
	{
		opt = temps[tempidx++];
		if(tempidx >= SCRIPT_TEMP_SPACE)
			tempidx = 0;
		return opt;
	}
*/

	if(*opt == '&' && strchr(opt + 1, '#'))
	{
		cp = temps[tempidx++];
		if(tempidx >= SCRIPT_TEMP_SPACE)
			tempidx = 0;

		snprintf(cp, getSymbolSize(), "%s", opt);
		opt = cp;
		cp = strchr(opt + 1, '#');
		*(cp++) = 0;
		sym = getLocal(cp);
		if(!sym)
		{
			missing(cp);
			return NULL;
		}
		snprintf(opt + strlen(opt), getSymbolSize() - strlen(opt), ".%s", sym->data);
		return opt;
	}

	if(opt[0] == '%' && !opt[1])
		return opt;

	if(opt[0] == '{')
		return ++opt;

	if(*opt != '%' && *opt != '@')
		return opt;

	if(*opt == '@')
	{
		subref = strrchr(opt, ':');
		sym = getLocal(++opt);
		if(!sym)
		{
			missing(opt);
			return NULL;
		}
		opt = sym->data;
		ext = strchr(opt, '.');
	}
	else
		++opt;

	if(strchr(opt + 1, '#'))
	{
		snprintf(alt, sizeof(alt), "%s",  opt);
		cp = strchr(alt + 1, '#');
		*(cp++) = 0;
		sym = getLocal(cp);
		if(!sym)
		{
			missing(cp);
			return NULL;
		}

		snprintf(alt + strlen(alt), sizeof(alt) -
				strlen(alt), ".%s", sym->data);
		opt = alt;
		ext = NULL;
	}

	sym = getLocal(opt);
	if(sym)
	{
		if(sym->flags.type == RECORD && subref)
		{
			snprintf(alt, sizeof(alt), "%s%s", opt, subref);
			sym = getLocal(alt);
		}
	}
	if(sym)
		return readSymbol(sym);

	if(!ext)
	{
		missing(opt);
		return NULL;
	}

	buf = newString(opt);
	*strrchr(buf, '.') = 0;

	sym = getLocal(buf);
	delString(buf);
	if(!sym)
	{
		missing(opt);
		return NULL;
	}

	if(++tempidx >= SCRIPT_TEMP_SPACE)
		tempidx = 0;
	buf = temps[tempidx];

	if((pos = atoi(ext + 1)) > 0)
	{
		char packtoken = getPackToken();

		dec = sym->data;
		while(--pos && dec)
		{
			dec = strchr(dec, packtoken);
			if(dec)
				++dec;
		}
		if(dec)
			setString(buf, slen, dec);
		else
			buf[0] = 0;
		dec = strchr(buf, packtoken);
		if(dec)
			*dec = 0;
	}
	else if(!strnicmp(ext, ".len", 4))
		sprintf(buf, "%d", strlen(sym->data));
	else if(!stricmp(ext, ".size"))
		sprintf(buf, "%d", sym->flags.size);
	else if(!strnicmp(ext, ".val", 4) || !strnicmp(ext, ".int", 4))
		sprintf(buf, "%d", atoi(sym->data));
	else if(!strnicmp(ext, ".dec", 4))
	{
		dec = strchr(sym->data, '.');
		if(dec)
			++dec;
		else
			dec = "0";
		sprintf(buf, "%d", atoi(dec));
	}
	else if(!stricmp(ext, ".bool"))
	{
		setString(buf, slen, "false");
		switch(*sym->data)
		{
		case 'y':
		case 'Y':
		case 't':
		case 'T':
			setString(buf, slen, "true");
			break;
		default:
			if(atoi(sym->data))
				setString(buf, slen, "true");
		}
	}
	else if(!stricmp(ext, ".used"))
		switch(sym->flags.type)
		{
		case ARRAY:
			snprintf(buf, sizeof(buf), "%d", sym->data[1]);
			break;
		case SEQUENCE:
		case STACK:
		case CACHE:
		case FIFO:
			snprintf(buf, sizeof(buf), "%d", sym->data[2]);
			break;
		default:
			return "1";
		}
	else if(!stricmp(ext, ".count"))
		switch(sym->flags.type)
		{
		case ARRAY:
		case CACHE:
		case SEQUENCE:
		case FIFO:
		case STACK:
			snprintf(buf, sizeof(buf), "%d", sym->data[4]);
			break;
		default:
			return "1";
		}
	else if(NULL != (prop = Property::find(ext + 1)))
		prop->getProperty(sym->data, buf, getSymbolSize());
	else if(!stricmp(ext, ".ref"))
	{
		switch(sym->flags.type)
		{
		case REF:
			return sym->data + sizeof(ScriptSymbol *);
		case ALIAS:
			return sym->data;
		default:
			return (char *)sym->id;
		}
	}
	else if(!stricmp(ext, ".max"))
		switch(sym->flags.type)
		{
		case ARRAY:
			sprintf(buf, "%d", sym->data[2] * 256 + sym->data[3]);
			break;
		case CACHE:
		case SEQUENCE:
		case FIFO:
		case STACK:
			sprintf(buf, "%d", sym->data[4]);
			break;
		default:
			sprintf(buf, "%d", sym->flags.size);
		}
	else if(!stricmp(ext, ".index"))
	{
		if(sym->flags.type == ARRAY)
		{
			sprintf(buf, "%d", sym->data[0] + 1);
			return buf;
		}
		buf = NULL;
	}
	else if(!stricmp(ext, ".type"))
		switch(sym->flags.type)
		{
		case ARRAY:
			return "array";
		case CACHE:
			return "cache";
		case ALIAS:
			return "alias";
		case SEQUENCE:
			return "sequence";
		case STACK:
			return "stack";
		case COUNTER:
			return "counter"; 
		case FIFO:
			return "fifo";
		default:
			return "string";
		}
	else
		buf = NULL;

	return buf;
}

char *ScriptInterp::getTempBuffer(void)
{
	char *tmp = temps[tempidx++];
	if(tempidx >= SCRIPT_TEMP_SPACE)
		tempidx = 0;

	tmp[0] = 0;
	return tmp;
}

char *ScriptInterp::getString(void)
{
	int index = script[stack].index;
	unsigned len = 0;
	unsigned max = getSymbolSize();
	char *tmp = getTempBuffer();
	char *opt;

	while(NULL != (opt = getOption(NULL)) && len < max)
	{
		script[stack].index = index;
		if(!stricmp(opt, ",") || !stricmp(opt, ")"))
			break;

		snprintf(tmp + len, max - len + 1, "%s", getValue(NULL));
		index = script[stack].index;
		len = (unsigned)strlen(tmp);
	}
	return tmp;
}

char *ScriptInterp::getValue(const char *def)
{
	char *opt = getOption(NULL);
	Attr *a;

	if(!opt)
		return (char *)def;

	if(*opt == '+')
		a = attr;
	else
		a = NULL;

	while(a)
	{
		if(!stricmp(a->id, opt + 1))
			break;
		a = a->next;
	}
	
	if(a)
		return a->meta(this, opt);

	opt = getContent(opt);
	if(!opt)
		return (char *)def;

	return opt;
}	

const char *ScriptInterp::getMember(void)
{
	char *cmd = script[stack].line->cmd;

	while(*cmd && *cmd != '.')
		++cmd;

	if(*cmd)
		return ++cmd;
	return NULL;
}

#ifdef	CCXX_NAMESPACES
}
#endif

