// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of ccscript.
//
// The exception is that, if you link the ccscript library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccscript library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccscript.  If you copy code from other releases into a copy of
// ccscript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for ccscript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.

#include <cc++/config.h>
#include <cc++/misc.h>
#include <cc++/slog.h>
#include <cc++/file.h>
#include <cc++/strchar.h>
#include <cc++/export.h>
#include <cstdlib>
#include <fstream>
#include <cstdio>
#include "bayonnescript.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

#ifdef	WIN32
#include <malloc.h>
#define	alloca	_alloca
#endif

static int keyindex(const char *keyword)
{
	unsigned key = 0;

	while(*keyword)
		key ^= key << 1 ^ (*(keyword++) & 0x1f);

	return key % KEYWORD_INDEX_SIZE;
}

ScriptCommand::ScriptCommand(ScriptCommand *ini) :
Keydata(), Mutex()
{
	memcpy(&keywords, &ini->keywords, sizeof(keywords));
	memcpy(&traps, &ini->traps, sizeof(traps));
	active = NULL;
	keyword_count = ini->keyword_count;
	trap_count = ini->trap_count;
	imask = ini->imask;
	copied = true;
	dbcount = 0;
	dbc = NULL;
	tq = NULL;
}

ScriptCommand::ScriptCommand() :
Keydata(), Mutex()
{
	ifstream cf;

	static Script::Define interp[] = {
		{"restart", &ScriptInterp::scrRestart, &ScriptCommand::chkIgnore},
		{"error", &ScriptInterp::scrError, &ScriptCommand::chkHasArgs},
		{"exit", &ScriptInterp::scrExit, &ScriptCommand::chkNoArgs},
		{"set", &ScriptInterp::scrSet, &ScriptCommand::chkIgnore},
		{"add", &ScriptInterp::scrSet, &ScriptCommand::chkHasArgs},
		{"number", &ScriptInterp::scrNumber, &ScriptCommand::chkHasArgs},
		{"clear", &ScriptInterp::scrClear, &ScriptCommand::chkHasVars},
		{"select", &ScriptInterp::scrSelect, &ScriptCommand::chkHasArgs},
		{"goto", &ScriptInterp::scrGoto, &ScriptCommand::chkHasArgs},
		{"call", &ScriptInterp::scrCall, &ScriptCommand::chkHasArgs},
		{"gosub", &ScriptInterp::scrCall, &ScriptCommand::chkHasArgs},
		{"source", &ScriptInterp::scrCall, &ScriptCommand::chkHasArgs},
		{"return", &ScriptInterp::scrReturn, &ScriptCommand::chkIgnore},
		{"size", &ScriptInterp::scrSize, &ScriptCommand::chkHasArgs},
		{"var", &ScriptInterp::scrSize, &ScriptCommand::chkIgnore},
		{"init", &ScriptInterp::scrSet, &ScriptCommand::chkIgnore},
		{"if", &ScriptInterp::scrIf, &ScriptCommand::chkHasArgs},
		{"do", &ScriptInterp::scrDo, &ScriptCommand::chkIgnore},
		{"for", &ScriptInterp::scrFor, &ScriptCommand::chkHasArgs},
		{"read", &ScriptInterp::scrRead, &ScriptCommand::chkIgnore},
		{"repeat", &ScriptInterp::scrRepeat, &ScriptCommand::chkHasArgs},
		{"dump", &ScriptInterp::scrDump, &ScriptCommand::chkHasArgs},
		{"gather", &ScriptInterp::scrGather, &ScriptCommand::chkHasArgs},
		{"loop", &ScriptInterp::scrLoop, &ScriptCommand::chkIgnore},
		{"fifo", &ScriptInterp::scrFifo, &ScriptCommand::chkHasArgs},
		{"break", &ScriptInterp::scrBreak, &ScriptCommand::chkIgnore},
		{"continue", &ScriptInterp::scrContinue, &ScriptCommand::chkIgnore},
		{"swap", &ScriptInterp::scrSwap, &ScriptCommand::chkHasArgs},
		{"once", &ScriptInterp::scrOnce, &ScriptCommand::chkHasArgs},
		{"lock", &ScriptInterp::scrLock, &ScriptCommand::chkIgnore},
		{"const", &ScriptInterp::scrConst, &ScriptCommand::chkIgnore},
		{"fconst", &ScriptInterp::scrConst, &ScriptCommand::chkIgnore},
		{"alias", &ScriptInterp::scrAlias, &ScriptCommand::chkHasArgs},
		{"data", &ScriptInterp::scrData, &ScriptCommand::chkHasArgs},
		{"dup", &ScriptInterp::scrDup, &ScriptCommand::chkHasArgs},
		{"remove", &ScriptInterp::scrRemove, &ScriptCommand::chkHasArgs},
		{"sequence", &ScriptInterp::scrSequence, &ScriptCommand::chkHasArgs},
		{"cache", &ScriptInterp::scrCache, &ScriptCommand::chkHasArgs},
		{"array", &ScriptInterp::scrArray, &ScriptCommand::chkHasArgs},
		{"index", &ScriptInterp::scrIndex, &ScriptCommand::chkHasArgs},
		{"stack", &ScriptInterp::scrStack, &ScriptCommand::chkHasArgs},
		{"lifo", &ScriptInterp::scrStack, &ScriptCommand::chkHasArgs},
		{"counter", &ScriptInterp::scrCounter, &ScriptCommand::chkHasArgs},
		{"skip", &ScriptInterp::scrSkip,  &ScriptCommand::chkHasArgs},
		{"enable", &ScriptInterp::scrEnable, &ScriptCommand::chkHasArgs},
		{"disable", &ScriptInterp::scrDisable, &ScriptCommand::chkHasArgs},
		{"then", &ScriptInterp::scrThen, &ScriptCommand::chkNoArgs},
		{"else", &ScriptInterp::scrElse, &ScriptCommand::chkNoArgs},
		{"endif", &ScriptInterp::scrEndif, &ScriptCommand::chkNoArgs},
		{"case", &ScriptInterp::scrCase, &ScriptCommand::chkHasArgs},
		{"endcase", &ScriptInterp::scrEndcase, &ScriptCommand::chkNoArgs},
		{"otherwise", &ScriptInterp::scrCase, &ScriptCommand::chkNoArgs},
		{"label", &ScriptInterp::scrLabel, &ScriptCommand::chkHasArgs},
		{"ref", &ScriptInterp::scrRef, &ScriptCommand::chkHasArgs},
		{"map", &ScriptInterp::scrMap, &ScriptCommand::chkHasArgs},
		{"begin", &ScriptInterp::scrBegin, &ScriptCommand::chkIgnore},
		{"struct", &ScriptInterp::scrStruct, &ScriptCommand::chkHasArgs},
		{"endstruct", &ScriptInterp::scrStruct, &ScriptCommand::chkNoArgs},
		{"atomic", &ScriptInterp::scrBegin, &ScriptCommand::chkIgnore},
		{"end", &ScriptInterp::scrEnd, &ScriptCommand::chkNoArgs},
		{"decimal", &ScriptInterp::scrDecimal, &ScriptCommand::chkHasArgs},
		{NULL, NULL, NULL}};

	copied = false;
	imask = 0;
	unsigned i;
	memset(&keywords, 0, sizeof(keywords));
	for(i = 0; i < TRAP_BITS; ++i)
		traps[i] = "<undefined>";

	active = NULL;
	keyword_count = 0;
	trap_count = 0;
	dbcount = 0;
	dbc = NULL;
	tq = NULL;

	// add special default exit and error traps

	trap("exit");
	trap("error", false);
	load(interp);
}

void *ScriptCommand::getDatabase(void)
{
	if(!dbc)
		return NULL;

	++dbcount;
	return dbc;
}

void *ScriptCommand::endDatabase(void)
{
	void *d;

	dblock.enterMutex();
	if(!dbcount)
	{
		dblock.leaveMutex();
		return NULL;
	}

	if(!(--dbcount))
	{
		d = dbc;
		dbc = NULL;
		dblock.leaveMutex();
		return d;
	}
	
	dblock.leaveMutex();
	return NULL;
}

void ScriptCommand::setDatabase(void *db)
{
	dbcount = 1;
	dbc = db;
	return;
}

bool ScriptCommand::setThreadQueue(ThreadQueue *q)
{
	if(tq)
		return false;

	tq = q;
	return true;
}

Script::Method ScriptCommand::getHandler(const char *keyword)
{
	Keyword *key;
	char keybuf[33];
	int len = 0;
	char *kw = keybuf;
	ScriptModule *mod;

	while(len++ < 32 && *keyword && *keyword != '.')
		*(kw++) = (*keyword++);
	*kw = 0;
	keyword = keybuf;

	key = keywords[keyindex(keyword)];

	while(key)
	{
		if(!stricmp(key->keyword, keyword))
			return key->method;

		key = key->next;
	}
	mod = ScriptModule::find(keyword);
	if(mod)
		return mod->handler;
	return (Method)NULL;
}

const char *ScriptImage::preproc(const char *token)
{
	PreParse *p = ScriptModule::preparse;

	while(p)
	{
		if(!stricmp(token, p->id))
			return p->parse(this, token);
		p = p->next;
	}
	return "unknown keyword";
}

char *ScriptCommand::check(char *keyword, Line *line, ScriptImage *img)
{
	ScriptModule *mod;
	Keyword *key;
	char keybuf[33];
	int len = 0;
	char *kw = keybuf;

	while(len++ < 32 && *keyword && *keyword != '.')
		*(kw++) = *(keyword++);

	*kw = 0;
	keyword = keybuf;
	key = keywords[keyindex(keyword)];

	while(key)
	{
		if(!stricmp(key->keyword, keyword))
			return check(key->check, line, img);

		key = key->next;
	}
	mod = ScriptModule::find(keyword);
	if(mod)
		return mod->checkScript(line, img);
	return "unknown command";
}

unsigned ScriptCommand::getTrapId(const char *trapname)
{
	unsigned i;

	for(i = 0; i < TRAP_BITS; ++i)
	{
		if(!stricmp(traps[i], trapname))
			return i;
	}
	return 0;
}

unsigned long ScriptCommand::getTrapMask(unsigned id)
{
	return 1 << id;
}

unsigned long ScriptCommand::getTrapMask(const char *trapname)
{
	unsigned long mask = 1;
	unsigned i;

	for(i = 0; i < TRAP_BITS; ++i)
	{
		if(!stricmp(traps[i], trapname))
			return mask;

		mask = mask << 1;
	}
	return 0;
}

void ScriptCommand::load(Script::Define *keydefs)
{
	size_t len;
	int key;
	Keyword *script;

	for(;;)
	{
		if(!keydefs->keyword)
			break;

		len = strlen(keydefs->keyword) + 1;
		key = keyindex(keydefs->keyword);
		script = (Keyword *)alloc(sizeof(Keyword) + len - 1);
		setString(script->keyword, len, keydefs->keyword);
		script->method = keydefs->method;
		script->check = keydefs->check;
		script->next = keywords[key];
		keywords[key] = script;
		++keydefs;
	}
}

int ScriptCommand::trap(const char *trapname, bool inherited)
{
	if(inherited)
		imask |= (1 << trap_count);
	traps[trap_count++] = alloc((char *)trapname);
	return trap_count;
}

bool ScriptCommand::isInherited(unsigned id)
{	
	if(imask & (1 << id))
		return true;

	return false;
}

char *ScriptCommand::chkIgnore(Line *line, ScriptImage *img)
{
	return NULL;
}

char *ScriptCommand::chkUse(Line *line, ScriptImage *img)
{
	int argc = line->argc;
	char **argv = line->args;

	if(!argc)
		return "missing args";

#ifdef	HAVE_MODULES

	while(argc--)
	{
		if(!Script::use(*argv))
			return "package missing";
		++argv;
	}
	return NULL;
#else
	return "dynamic loader unsupported";
#endif
}

char *ScriptCommand::chkModule(Line *line, ScriptImage *img)
{
	ScriptModule *mod = ScriptModule::first;
	char keybuf[32];

	char *kw = line->cmd;
	unsigned len = 0;
	while(*kw && len < sizeof(keybuf) - 1 && *kw != '.')
		keybuf[len++] = *kw;
	keybuf[len] = 0;

	while(mod)
	{
		if(!stricmp(kw, mod->cmd))
			break;
		return NULL;
	}
	return mod->checkScript(line, img);
}

char *ScriptCommand::chkHasModify(Line *line, ScriptImage *img)
{
	if(!line->argc)
		return "arguments missing";

	if(line->argc < 2)
		return "no values to modify";

	if(*(line->args[0]) != '%' && *(line->args[0]) != '@')
		return "invalid variable assignment";

	return NULL;
}

char *ScriptCommand::chkHasList(Line *line, ScriptImage *img)
{
	if(!line->argc)
		return "arguments missing";

	if(line->argc < 2)
		return "no values to assign";

	if(*(line->args[0]) != '%' && *(line->args[0]) !='@')
		return "invalid variable assignment";

	return NULL;
}

char *ScriptCommand::chkHasArgs(Line *line, ScriptImage *img)
{
	if(!line->argc)
		return "arguments missing";

	return NULL;
}

char *ScriptCommand::chkNoArgs(Line *line, ScriptImage *img)
{
	if(line->argc)
		return "invalid arguments";

	return NULL;
} 

char *ScriptCommand::chkHasVars(Line *line, ScriptImage *img)
{
	unsigned index = 0;
	char ch;

	if(line->argc < 1)
		return "no arguments";

	while(index < line->argc)
	{
		ch = *(line->args[index++]);
		if(ch != '%' && ch != '@')
			return "variable argument required";
	}

	return NULL;
}

ScriptSymbol::ScriptSymbol(unsigned size, size_t paging, const char *id) :
SharedMemPager(paging, setName(id))
{
	symsize = size;
	symlimit = (unsigned)(paging - 32 - sizeof(Symbol));
	memset(index, 0, sizeof(index));
	record = NULL;
}

ScriptSymbol::~ScriptSymbol()
{
	Symbol *sym = index[SYMBOL_INDEX_SIZE];
	Symbol *next;

	while(sym)
	{
		next = sym->next;
		delete[] sym;
		sym = next;
	}
}

const char *ScriptSymbol::setName(const char *id)
{
	snprintf(_idname, sizeof(_idname), "%s", id);
	return _idname;
}

void ScriptSymbol::setExclusive(bool enable)
{
	if(enable)
		enterMutex();
	else
		leaveMutex();
}

void ScriptSymbol::purge(void)
{
	MemPager::purge();
	memset(index, 0, sizeof(index));
}

unsigned ScriptSymbol::gather(Symbol **idx, unsigned max, const char *prefix, const char *suffix)
{
	char *ext;
	unsigned key = 0;
	unsigned count = 0;
	unsigned pointer, marker;
	Symbol *node;

	enterMutex();

	while(max && key <= SYMBOL_INDEX_SIZE)
	{
		node = index[key++];
		while(node && max)
		{
			if(strnicmp(node->id, prefix, strlen(prefix)))
			{
				node = node->next;
				continue;
			}

			if(suffix)
			{
				ext = strrchr(node->id, '.');
				if(!ext)
				{
					node = node->next;
					continue;
				}
				if(stricmp(++ext, suffix))
				{
					node = node->next;
					continue;
				}
			}

			pointer = 0;
			while(pointer < count)
			{
				if(stricmp(node->id, idx[pointer]->id) < 0)
					break;
				++pointer;
			}
			marker = pointer;
			pointer = count;
			while(pointer > marker)
			{
				idx[pointer] = idx[pointer - 1];
				--pointer;
			}
			idx[marker] = node;
			--max;
			++count;
			node = node->next;
		}
	}
	leaveMutex();
	return count;	
}


unsigned ScriptSymbol::getIndex(const char *symname)
{
	unsigned int key = 0;

	while(*symname)
		key ^= (key << 1) ^ (*(symname++) & 0x1f);

	return key % SYMBOL_INDEX_SIZE;
}

void ScriptSymbol::commit(Symbol *sym)
{
	return;
}

Script::Symbol *ScriptSymbol::getEntry(const char *symname, unsigned size)
{
	int key = 0;
	Symbol *sym, *node;
	char symcode[128];
	ScriptSymbol *ref;
	bool large = false;
	char *buf;
	char *ext = NULL;
	int ind;
	unsigned len;

	enterMutex();

retry:
	if(*symname == '%' || *symname == '&')
		++symname;

        if(record && size && !strchr(symname, '.'))
        {
                sym = NULL;
                goto add;
        }


	setString(symcode, sizeof(symcode), symname);
	ext = strchr(symcode, ':');
	if(ext)
	{
		*(ext++) = 0;
		symname = symcode;
	}
	key = getIndex(symname);
large:
	sym = index[key];
	while(sym)
	{
		if(!stricmp(sym->id, symname))
			break;

		sym = sym->next;
	}
	if(!sym && key < SYMBOL_INDEX_SIZE)
	{
		key = SYMBOL_INDEX_SIZE;
		goto large;
	}

	key = getIndex(symname);

	if(sym)
	{
		if(sym->flags.type == ALIAS)
		{
			symname = sym->data;
			goto retry;
		}
		else if(sym->flags.type == REF)
		{
			symname = sym->data + sizeof(ScriptSymbol *);
			ref = *((ScriptSymbol **)sym->data);
			leaveMutex();
			if(ext)
			{
				len = (unsigned)(strlen(symname) + strlen(ext) + 3);
				buf = (char *)alloca(len);
				buf[0] = 0;
				snprintf(buf, len, "%s:%s", symname, ext);
				symname = buf;
			}
			return ref->getEntry(symname, size);
		}
	}

	if(sym || !size)
	{
		if(sym && ext)
			switch(sym->flags.type)
			{
			case RECORD:
				memcpy(&sym, sym->data + 1, sizeof(void *));
				while(sym)
				{
					if(!stricmp(sym->id, ext))
						break;
					sym = sym->next;
				}
				break;		
			case ARRAY:
				ind = atoi(ext);
				if(ind < 0)
					ind += sym->data[4];
				if(ind < 0)
					ind = 0;
				if(ind >= sym->data[4])
					ind = sym->data[4] - 1;
				sym->data[0] = ind;
				break;
			default:
				break;
			}
		leaveMutex();
		return sym;
	}

add:
	if(size > symlimit)
	{
		key = SYMBOL_INDEX_SIZE;
		large = true;
		sym = (Symbol *)new char[sizeof(Symbol) + size];
	}
	else
		sym = (Symbol *)alloc(sizeof(Symbol) + size);

	sym->id = MemPager::alloc((char *)symname);

	if(record && !strchr(symname, '.'))
	{
		sym->next = NULL;
		if(*record)
		{
			node = *record;
			for(;;)
			{
				if(!node->next)
				{
					node->next = sym;
					break;
				}
				node = node->next;
			}				
		}
		else
			*record = sym;	
	}
	else
	{
		sym->next = index[key];
		index[key] = sym;
	}

	sym->flags.size = size;
	sym->flags.commit = false;
	sym->flags.initial = true;
	sym->flags.large = large;
	sym->flags.system = false;
	sym->flags.readonly = false;
	sym->flags.type = NORMAL;
	sym->data[0] = 0;
	leaveMutex();
	return sym;
}

Script::Symbol *ScriptSymbol::getAlias(const char *symname)
{
	int key;
	Symbol *sym;

	enterMutex();
	if(*symname == '%' || *symname == '&')
		++symname;
	key = getIndex(symname);
	sym = index[key];
	while(sym)
	{
		if(!stricmp(sym->id, symname))
			break;

		sym = sym->next;
	}

	if(sym)
	{
		if(sym->flags.type != ALIAS)
			sym = NULL;
	}
	leaveMutex();
	return sym;
}

char *ScriptSymbol::getSymbol(const char *id)
{
	Symbol *sym = getEntry(id);
	if(!sym)
		return NULL;

	if(sym->flags.initial)
		return NULL;

	return readSymbol(sym);
}

void *ScriptSymbol::getPointer(const char *id)
{
	Symbol *sym = getEntry(id);
	void *dp;

	if(!sym)
		return NULL;

	if(sym->flags.initial)
		return NULL;

	if(sym->flags.type != POINTER)
		return NULL;

	memcpy(&dp, sym->data + 1, sizeof(void *));
	return dp;
}

bool ScriptSymbol::setPointer(const char *id, void *data)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	sym = getEntry(id, sizeof(void *) + 1);
	if(!sym->flags.initial && sym->flags.type != POINTER)
		return false;

	enterMutex();
	sym->data[0] = 0;
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = POINTER;
	memcpy(sym->data + 1, &data, sizeof(void *));
	leaveMutex();
	return true;
}

bool ScriptSymbol::setStruct(const char *id)
{
        Symbol *sym;
	void *data = NULL;

	if(!id)
	{
		record = NULL;
		return false;
	}

        if(*id == '%' || *id == '&')
                ++id;

        sym = getEntry(id, sizeof(void *) + 1);
        if(!sym->flags.initial && sym->flags.type != RECORD)
                return false;

	if(sym->flags.initial)
		memcpy(sym->data + 1, &data, sizeof(void *));

        enterMutex();
        sym->data[0] = 0;
        sym->flags.initial = false;
        sym->flags.readonly = true;
        sym->flags.type = RECORD;
	record = (Symbol **)(sym->data + 1);
        leaveMutex();
        return true;
}


bool ScriptSymbol::swapSymbol(const char *oldname, const char *newname)
{
	unsigned ol = getIndex(oldname);
	unsigned np = getIndex(newname);
	Symbol *onode, *nnode;
	Symbol *oprior = NULL, *nprior = NULL;
	Symbol *onext, *nnext;

	enterMutex();
	onode = index[ol];
	while(onode)
	{
		if(!stricmp(onode->id, oldname))
			break;
		oprior = onode;
		onode = onode->next;
	}

	nnode = index[np];
	while(nnode)
	{
		if(!stricmp(nnode->id, newname))
			break;

		nprior = nnode;
		nnode = nnode->next;
	}

	if(!nnode || !onode)
	{
		leaveMutex();
		return false;
	}

	nnext = nnode->next;
	onext = onode->next;

	if(nprior)
		nprior->next = onode;
	else
		index[np] = onode;

	if(oprior)
		oprior->next = nnode;
	else
		index[ol] = nnode;

	nnode->next = onext;
	onode->next = nnext;
	onode->id = newname;
	nnode->id = oldname;

	leaveMutex();
	return true;
}

bool ScriptSymbol::setAlias(const char *id, const char *source)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	if(*source == '%' || *source == '&')
		++source;

	sym = getEntry(id, (unsigned)strlen(source));
	if(!sym->flags.initial)
		return false;

	enterMutex();
	setString(sym->data, sym->flags.size + 1, source);
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = ALIAS;
	leaveMutex();
	return true;
}

bool ScriptSymbol::makeCounter(const char *id)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	sym = getEntry(id, 11);
	if(!sym->flags.initial)
		return false;
	enterMutex();
	sym->flags.initial = false;
	sym->flags.type = COUNTER;
	setString(sym->data, sym->flags.size + 1, "0");
	leaveMutex();
	return true;
}

bool ScriptSymbol::makeFifo(const char *id, unsigned char size, unsigned char rec)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	sym = getEntry(id, size * (rec + 1) + 5);
	if(!sym->flags.initial)
		return false;

	enterMutex();
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = FIFO;
	sym->data[0] = 0;
	sym->data[1] = 0;	// head
	sym->data[2] = 0;	// tail
	sym->data[3] = rec;	// rec
	sym->data[4] = size;
	leaveMutex();
	return true;
}

bool ScriptSymbol::makeStack(const char *id, unsigned char size, unsigned char rec)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	sym = getEntry(id, size * (rec + 1) + 5);
	if(!sym->flags.initial)
		return false;

	enterMutex();
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = STACK;
	sym->data[0] = 0;
	sym->data[1] = 0;	// head
	sym->data[2] = 0;	// tail
	sym->data[3] = rec;	// rec
	sym->data[4] = size;
	leaveMutex();
	return true;
}

bool ScriptSymbol::makeArray(const char *id, unsigned char size, unsigned short rec)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	sym = getEntry(id, size * (rec + 1) + 5);
	if(!sym->flags.initial)
		return false;

	enterMutex();
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = ARRAY;
	sym->data[0] = 0;	// index
	sym->data[1] = 0;	// hiwater
	sym->data[2] = rec / 256;
	sym->data[3] = rec % 256;
	sym->data[4] = size;
	leaveMutex();
	return true;
}

bool ScriptSymbol::makeSequence(const char *id, unsigned char size, unsigned char rec)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	sym = getEntry(id, size * (rec + 1) + 5);
	if(!sym->flags.initial)
		return false;

	enterMutex();
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = SEQUENCE;
	sym->data[0] = 0;
	sym->data[1] = 0;	// head
	sym->data[2] = 0;	// tail
	sym->data[3] = rec;	// rec
	sym->data[4] = size;
	leaveMutex();
	return true;
}

bool ScriptSymbol::makeCache(const char *id, unsigned char size, unsigned char rec)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	sym = getEntry(id, size * (rec + 1) + 5);
	if(!sym->flags.initial)
		return false;

	enterMutex();
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = CACHE;
	sym->data[0] = 0;
	sym->data[1] = 0;	// head
	sym->data[2] = 0;	// tail
	sym->data[3] = rec;	// rec
	sym->data[4] = size;
	leaveMutex();
	return true;
}

bool ScriptSymbol::removeSymbol(Symbol *sym, const char *value)
{
	unsigned char size, rec, head, tail;
	unsigned i, j;
	unsigned len;

	if(sym->flags.type != FIFO &&
	   sym->flags.type != SEQUENCE &&
	   sym->flags.type != CACHE &&
	   sym->flags.type != STACK)
		return false;

	enterMutex();
	size = sym->data[4];
	rec = sym->data[3];
	head = sym->data[1];
	tail = sym->data[2];
	len = rec + 1;

	i = head;
	while(i != tail)
	{
		if(!strcmp(sym->data + 5 + i * len, value))
			break;
		if(++i >= size)
			i = 0;
	}

	if(i == tail)
	{
		leaveMutex();
		return true;
	}

	if(i == head && sym->flags.type == FIFO)
	{
		if(++head >= size)
			head = 0;
		sym->data[0] = head;
		leaveMutex();
		return true;
	}

	j = i + 1;
	while(j < size)
	{
		setString(sym->data + 5 + ((j - 1) * len), len, 
			sym->data + 5 + (j * len));
		++j;
	}

	if((unsigned)sym->data[1] > i)
		--sym->data[1];
	if((unsigned)sym->data[2] > i)
		--sym->data[2];
	leaveMutex();
	return true;
}

bool ScriptSymbol::postSymbol(Symbol *sym, const char *value)
{
	unsigned short arec;
	unsigned char head, tail, pos, size, rec;
	unsigned offset = 0;
	unsigned len;

	if(sym->flags.type != FIFO && sym->flags.type != SEQUENCE && sym->flags.type != STACK && sym->flags.type != CACHE && sym->flags.type != ARRAY)
		return false;

	enterMutex();
	if(sym->flags.type == ARRAY)
	{
		pos = sym->data[0];
		arec = sym->data[2] * 256 + sym->data[3];
		size = sym->data[4];
		len = arec + 1;
		if(pos >= size)
		{
			leaveMutex();
			return true;
		}
		setString(sym->data + 5 + (pos * len), len, value);
		if(++pos > size)
			pos = size;
		if(pos > sym->data[1])
			sym->data[1] = pos;
		sym->data[0] = pos;
		leaveMutex();
		return true;
	}
	else
	{
		head = sym->data[1];
		pos = tail = sym->data[2];
		size = sym->data[4];
		rec = sym->data[3];
	}

	len = rec + 1;
	if(tail >= size && sym->flags.type == CACHE)
	{
		while(offset < (unsigned)(size - 1))
		{
			setString(sym->data + 5 + (offset * len), len,
				sym->data + 5 + ((offset + 1) * len));
			++offset;
		}
		tail = size;
		pos = size - 1;
	}
	else if(++tail >= size && sym->flags.type == FIFO)
		tail = 0;

	if(pos >= size)
	{
		leaveMutex();
		return false;
	}
	if(tail == head && (sym->flags.type != CACHE && sym->flags.type  != SEQUENCE))
	{
		leaveMutex();
		return false;
	}

	len = rec + 1;
	setString(sym->data + 5 + (pos * len), len, value);
	sym->data[2] = tail;
	leaveMutex();
	return true;
}

char *ScriptSymbol::readSymbol(Symbol *sym)
{
	unsigned short arec;
	long value;
	char *data;
	unsigned char head = 0, rec, size, pos;
	switch(sym->flags.type)
	{
	case COUNTER:
		value = atoi(sym->data);
		sprintf(sym->data, "%ld", ++value);
		return sym->data;
	case ARRAY:
		pos = sym->data[0];
		arec = sym->data[2] * 256 + sym->data[3];
		size = sym->data[4];
		if(pos >= sym->data[1])
			return "";
		if(pos >= size)
			pos = size - 1;
		return sym->data + 5 + (pos * (arec + 1));
	case CACHE:
		enterMutex();
		if(!sym->data[1])
			head = sym->data[1] = sym->data[2];
		if(head)
			head = --sym->data[1];
		rec = sym->data[3];
		leaveMutex();
		return sym->data + 5 + (head * (rec + 1));
	case STACK:
		enterMutex();
		if(!sym->data[2])
		{
			leaveMutex();
			return sym->data;
		}
		head = --sym->data[2];
		rec = sym->data[3];
		leaveMutex();
		return sym->data + 5 + (head * (rec + 1));
	case SEQUENCE:
		enterMutex();
		if(sym->data[1] == sym->data[2])
			sym->data[1] = 0;
		leaveMutex();
	case FIFO:
		enterMutex();
		if(sym->data[1] == sym->data[2])
		{
			leaveMutex();
			return sym->data;
		}
		size = sym->data[4];
		rec = sym->data[3];
		head = sym->data[1];
		data =  sym->data + head * (rec + 1) + 5;
		if(++head >= size)
			head = 0;
		sym->data[1] = head;
		leaveMutex();
		return data;
	default:
		return sym->data;
	}
}

char *ScriptSymbol::setConst(const char *id, const char *value)
{
	Symbol *sym;

	if(*id == '%' || *id == '&')
		++id;

	sym = getEntry(id, (unsigned)strlen(value));
	if(!sym)
		return NULL;

	if(!sym->flags.initial)
		return sym->data;

	enterMutex();
	setString(sym->data, sym->flags.size + 1, value);
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = NORMAL;
	leaveMutex();
	return sym->data;
}

char *ScriptSymbol::setSymbol(const char *id, const char *value)
{
	long val;

	Symbol *sym = getEntry(id);
	if(!sym)
		return NULL;

	if(sym->flags.readonly)
		return NULL;

	if(!value)
		value = "";

	enterMutex();
	sym->flags.initial = false;
	switch(sym->flags.type)
	{
	case ARRAY:
	case SEQUENCE:
	case CACHE:
	case FIFO:
	case STACK:
		if(postSymbol(sym, value))
			return "";
		else
			return NULL;
	case COUNTER:
		val = atoi(value);
		snprintf(sym->data, sym->flags.size + 1, "%ld", --val);
		break;
	default:
		setString(sym->data, sym->flags.size + 1, value);
	}
	if(sym->flags.commit)
		commit(sym);

	leaveMutex();
	return sym->data;
}

void ScriptSymbol::clrSymbol(const char *id)
{
	Symbol *sym = getEntry(id);

	if(!sym)
		return;

	if(sym->flags.readonly || sym->flags.system)
		return;

	enterMutex();
	sym->flags.initial = true;
	sym->data[0] = 0;
	if(sym->flags.commit)
		commit(sym);
	leaveMutex();
	return;
}

char *ScriptSymbol::setSymbol(const char *id, int size)
{
	Symbol *sym;

	if(!size)
		size = symsize;

	sym = getEntry(id, size);
	if(!sym)
		return NULL;

	return sym->data;
}

ScriptImage::ScriptImage(ScriptCommand *cmd, const char *symset) :
Keydata(symset)
{
	static Script::Initial initial[] = {
		{"script.index", 6 * SCRIPT_STACK_SIZE, ""},
		{"script.stack", 3, "-"},
		{"script.error", 64, "none"},
		{"script.home", 64, "none"},
		{"script.token", 1, ","},
		{"script.decimal", 1, "."},
		{NULL, 0, NULL}
	};

	cmds = cmd;
	memset(index, 0, sizeof(index));
	refcount = 0;
	inccount = 0;
	scrStream = (istream *)&scrSource;
	ilist = NULL;

	load(initial);
}

void ScriptImage::initial(const char *keyword, const char *value, unsigned size)
{
	InitialList *init;

	if(!size)
		size = (unsigned)strlen(value);

	init = (InitialList *)alloc(sizeof(InitialList));
	init->name = alloc((char *)keyword);
	init->size = size;
	init->value = alloc((char *)value);
	init->next = ilist;
	ilist = init;
}

void ScriptImage::load(Initial *init)
{
	while(init->name)
	{
		initial(init->name, init->value, init->size);
		++init;
	}
}

void ScriptImage::purge(void)
{
	MemPager::purge();
	memset(index, 0, sizeof(index));
	refcount = 0;
}

void ScriptImage::commit(void)
{
	while(inccount)
		include(incfiles[--inccount]);

	cmds->enterMutex();
	if(cmds->active)
	{
		if(!cmds->active->refcount)
			delete cmds->active;
	}
	cmds->active = this;
	cmds->leaveMutex();
}

char *ScriptImage::getToken(char **pre)
{
	static char temp[513];
	char *cp = temp + 1;
	char *base = temp + 1;
	char q;
	int level;

	if(pre)
		if(*pre)
		{
			cp = *pre;
			*pre = NULL;
			return cp;
		}

	if(*bp == '=')
	{	
		++bp;
       		if(*bp == '{')
        	{
                	level = -1;
                	while(*bp)
                	{
                        	if(*bp == '}' && !level)
                                	break;
                        	if(*bp == '{')
                                	++level;
                        	else if(*bp == '}')
                                	--level;
                        	*(cp++) = *(bp++);
                	}
                	if(*bp == '}')
                        	++bp;
                	*cp = 0;
			*base = 0x01;	// special token
                	return base;
        	}

		if(*bp == ' ' || *bp == '\t' || !*bp)
			return "";
		if(*bp == '\"' || *bp == '\'')
		{
			q = *(bp++);
			while(q)
			{
				switch(*bp)
				{
				case '\\':
					++bp;
					if(!*bp)
					{
						q = 0;
						break;
					}
					switch(*bp)
					{
					case 't':
						*(cp++) = '\t';
						++bp;
						break;
					case 'b':
						*(cp++) = '\b';
						++bp;
						break;
					case 'n':
						*(cp++) = '\n';
						++bp;
						break;
					default:
						*(cp++) = *(bp++);
					}
					break;
				case 0:
					q = 0;
					break;
				default:
					if(*bp == q)
					{
						++bp;
						q = 0;
					}
					else
						*(cp++) = *(bp++);
				}
			}
			*cp = 0;
			return base;
		}
		while(*bp != ' ' && *bp != '\t' && *bp)
			*(cp++) = *(bp++);
		*cp = 0;
		return base;
	}

	if(!quote)
		while(*bp == ' ' || *bp == '\t')
			++bp;

	if(!quote && *bp == '#' && (!bp[1] || bp[1] == '!' || isspace(bp[1])))
		return NULL;

	if(!quote && bp[0] == '%' && bp[1] == '%' && (!bp[2] || isspace(bp[2])))
		return NULL;


	if(!*bp)
	{
		paren = 0;
		quote = false;
		return NULL;
	}

	if(*bp == '\"' && !quote)
	{
		++bp;
		quote = true;
	}

	if(!quote && *bp == '{')
	{
		level = -1;
		while(*bp)
		{
			if(*bp == '}' && !level)
				break;
			if(*bp == '{')
				++level;
			else if(*bp == '}')
				--level;
			*(cp++) = *(bp++);
		}
		if(*bp == '}')
			++bp;
		*cp = 0;
		return base;
	}

	if(!quote)
	{
		if(*bp == ',' && paren)
		{
			++bp;
			return ",";
		}			
retry:
		while(*bp && !isspace(*bp) && *bp != ',' && *bp != '=' && *bp != '(' && *bp != ')' )
			*(cp++) = *(bp++);

		if(*bp == '(')
			++paren;
		else if(*bp == ')')
			--paren;

		if(*bp == '=' && cp == base)
		{
			*(cp++) = *(bp++);
			goto retry;
		}

		if(*bp == '=' && cp == base + 1 && ispunct(*base))
		{
			*(cp++) = *(bp++);
			goto retry;
		}

		if((*bp == '(' || *bp == ')') && cp == base)
			*(cp++) = *(bp++);

		*cp = 0;
		if(*bp == ',' && !paren)
			++bp;
		else if(*bp == '=')
			*(--base) = *(bp);
		if(!strcmp(base, "=="))
			return ".eq.";
		if(!strcmp(base, "="))
			return "-eq";
		if(!strncmp(base, "!=", 2))
			return ".ne.";
		if(!strncmp(base, "<>", 2))
			return "-ne";
		if(!strcmp(base, "<"))
			return "-lt";
		if(!strcmp(base, "<="))
			return "-le";
		if(!strcmp(base, ">"))
			return "-gt";
		if(!strcmp(base, ">="))
			return "-ge";
		return base;
	}

	if(*bp == '\\' && (bp[1] == '%' || bp[1] == '&' || bp[1] == '.'))
	{
		++bp;
		*(cp++) = '{';
		*(cp++) = *(bp++);
	}

requote:
	if(isalnum(*bp) || strchr("~/:,. \t\'", *bp))
	{
		while(isalnum(*bp) || strchr("~=/:,. \t\'", *bp))
			*(cp++) = *(bp++);
	}
	else while(!isspace(*bp) && *bp && *bp != '\"')
		*(cp++) = *(bp++);

	if(*bp == '\n' || !*bp)
		paren = 0;

	if(*bp == '\n' || !*bp || *bp == '\"')
		quote = false;

	if(*bp == '\\' && bp[1])
	{
		++bp;
		switch(*bp)
		{
		case 0:
			break;
		case 't':
			*(cp++) = '\t';
			++bp;
			break;
		case 'b':
			*(cp++) = '\b';
			++bp;
			break;
		case 'n':
			*(cp++) = '\n';
			++bp;
			break;
		default:
			*(cp++) = *(bp++);
		}
		goto requote;
	}

	if(*bp == '\n' || *bp == '\"')
		++bp;

	*cp = 0;
	return base;
}

unsigned ScriptImage::gather(const char *suffix, Name **array, unsigned max)
{
	unsigned count = 0;
	unsigned sort = 0;
	unsigned key = 0;
	Name *scr;
	char *ext;
	int ins;

	while(count < max && key < KEYWORD_INDEX_SIZE)
	{
		scr = index[key];
		while(scr && count < max)
		{
			ext = strstr(scr->name, "::");
			if(!ext)
			{
				scr = scr->next;
				continue;
			}
			ext += 2;
			if(stricmp(ext, suffix))
			{
				scr = scr->next;
				continue;
			}
			sort = 0;
			while(sort < count)
			{
				if(stricmp(scr->name, array[sort]->name) < 0)
					break;
				++sort;
			}
			ins = count;
			while(ins > (int)sort)
			{
				array[ins] = array[ins - 1];
				--ins;
			}
			array[sort] = scr;
			++count;
			scr = scr->next;
		}
		++key;
	}
	return count;
}

Script::Name *ScriptImage::dupScript(const char *src, const char *dest)
{
	Name *scr = getScript(dest);
	Name *nscr;
	unsigned key;
	if(scr)
		return scr;

	scr = getScript(src);
	if(!scr)
		return NULL;

	if(scr->mode == Name::mCOPY)
		return NULL;

	key = keyindex(dest);
	duplock.enterMutex();
	nscr = (Name *)alloc(sizeof(Name));
	memcpy(nscr, scr, sizeof(Name));
	nscr->name = alloc((char *)dest);
	nscr->mode = Name::mCOPY;
	scr->mode = Name::mCOPIED;
	nscr->next = index[key];
	index[key] = nscr;
// COPY EVENTS??? - uses list from "copied", just like traps are reused
// since the trap pointers are initially memcpied.  access mode maybe
// should be protected if was public???
	duplock.leaveMutex();
	return nscr;
}

Script::Name *ScriptImage::getScript(const char *name)
{
	int key = keyindex(name);
	Name *scr = index[key];

	while(scr)
	{
		if(!stricmp(scr->name, name))
			break;

		scr = scr->next;
	}
	return scr;
}

int ScriptImage::compile(const char *scrname)
{
	char buffer[129];
	char *token;
	char *ext;

#ifdef	WIN32
	setString(buffer, sizeof(buffer), scrname);
	token = strrchr(buffer, '\\');
	if(!token)
			token = strrchr(buffer, '/');
#else
	setString(buffer, sizeof(buffer), scrname);
	token = strrchr(buffer, '/');
#endif

	if(!token)
		token = buffer;
	else
		++token;

	ext = strrchr(token, '.');
	if(ext)
		*ext = 0;

	return compile(scrname, token);
}

Script::Name *ScriptImage::include(const char *token)
{
	char buffer[256];
	const char *prefix = cmds->getLast("include");
	Name *inc;

	if(!prefix)
		return NULL;

	snprintf(buffer, sizeof(buffer), "%s/%s.bm", prefix, token);

	inc = getScript(token);
	if(!inc)
	{
		if(!isFile(buffer))
			return NULL;
		if(!canAccess(buffer))
			return NULL;
		compile(buffer, (char *)token);
		inc = getScript(token);
	}
	return inc;
}

int ScriptImage::compile(const char *scrname, char *name)
{
	int rtn;

	scrSource.open(scrname);
	if(!scrSource.is_open())
		return 0;

	rtn = compile((istream *)&scrSource, name, scrname);
	scrSource.close();
	scrSource.clear();
	return rtn;
}

int ScriptImage::compile(istream *str, char *name, const char *scrname)
{
	const char *errmsg = NULL;
	const char *basename = name;
	char filename[65];
	unsigned lnum = 0;
	char namebuf[256];
	char tagbuf[128];
	char rulebuf[512];
	char gvarname[128];
	char path[128];
	char csname[128];
	char *command, *token, *pretoken = NULL;
	char *args[SCRIPT_MAX_ARGS + 1];
	int maxargs = SCRIPT_MAX_ARGS;
	char *err, *cp = strrchr(scrname, '.');
	const char *var = cmds->getLast("datafiles");
	const char *project = cmds->getLast("name");
	bool trapflag, syncflag = false;
	int argc, key, tlen, initkey = 0, initcount = 0;
	unsigned i;
	unsigned short count, number, total = 0;
	size_t gvarlen;
	Name::Event *events, *esave;
	Name *script;
	Line *line, *last;
	unsigned long addmask, submask, trapmask, mask, cmask = 0;
	Method handler;
	unsigned char loopid, looplevel;
	size_t offset = 0;
	Name *base = NULL, *init = NULL;
	bool sub = false;
	streampos pos;
	bool ignore;
	Name::Access access = Name::aPUBLIC;
	char temp[32];
	char *tag;
	char *pmode = NULL;
	char *ftoken = NULL;
	bool bm = false;
	unsigned ctlen;
	bool first = true;
	unsigned long addPmask = 0, subPmask = ~0;
	unsigned long addPmask1 = 0, subPmask1 = ~0;

	buffer = (char *)malloc(512);
	bufsize = 512;

	if(project && !stricmp(project, "bayonne"))
		project = NULL;

	scrStream = str;

	enum
	{
		cSCRIPT,
		cDATA,
		cMAP
	}	cmode = cSCRIPT;

	if(cp && !stricmp(cp, ".bm"))
		bm = true;

	if(bm)
		access = Name::aPROTECTED;

	gvarname[0] = '%';
	snprintf(gvarname + 1, 48, "%s.", name);
	gvarlen = strlen(gvarname);

	cp = (char *)strrchr(scrname, '/');
	if(cp)
		++cp;
	else
		cp = (char *)scrname;
	snprintf(filename, sizeof(filename), "%s", cp);

compile:
	trapflag = false;
	count = number = 0;
	last = NULL;
	addmask = submask = trapmask = 0;
	handler = NULL;
	loopid = looplevel = 0;
	bool then = false;
	events = NULL;
	bool loaded = false;	

	setString(csname, sizeof(csname), name);
	cp = strstr(csname, "::");
	if(cp && !stricmp(cp, "::main"))
	{
		initkey = SCRIPT_INDEX_SIZE;
		*cp = 0;
	}

        key = keyindex(csname);

	if(first && bm)
		initkey = SCRIPT_INDEX_SIZE;
	else if(first)
		initkey = key;

	script = (Name *)alloc(sizeof(Name));
	memset(script, 0, sizeof(Name));
	script->name = alloc(csname);
	script->mask = 0;
	script->events = NULL;
	addPmask1 = addPmask;
	subPmask1 = subPmask;
	addPmask = 0;
	subPmask = ~0;

	if(!first)
		script->next = index[key];

	script->mode = Name::mORIGINAL;
	script->filename = alloc(filename);

	if(first)
		init = script;

	if(pmode)
	{
		if(!strnicmp(pmode, "pub", 3) || !stricmp(pmode, "program"))
			script->access = Name::aPUBLIC;
		else if(!strnicmp(pmode, "priv", 4) || !stricmp(pmode, "state"))
			script->access = Name::aPRIVATE;
		else if(!strnicmp(pmode, "prot", 4))
			script->access = Name::aPROTECTED;
		else if(!strnicmp(pmode, "fun", 3))
			script->access = Name::aFUNCTION;
		else
			script->access = access;
	}
	else
		script->access = access;

	if(!first)
		index[key] = script;

	pmode = NULL;

	if(!base)
		base = script;

	if(sub)
	{
		sub = false;
		memcpy(script->trap, base->trap, sizeof(base->trap));
	}

	for(;;)
	{
		if(ftoken)
			goto first;

		quote = false;
		paren = 0;
		if(!then)
		{
			cmask = 0;

			if(offset > bufsize - 160)
			{
				bufsize += 512;
				buffer = (char *)realloc(buffer, bufsize);
			}

			scrStream->getline(buffer + offset, bufsize - 1 - offset);
			if(scrStream->eof())
				break;

			++lnum;
			bp = strrchr(buffer, '\\');
			if(bp)
			{
				++bp;
				if(isspace(*bp) || !*bp)
				{
					offset = bp - buffer - 1;
					continue;
				}
			}

			bp = buffer;
		}
		else
			then = false;

		offset = 0;
		++number;
first:
		while(NULL != (token = getToken(&ftoken)))
		{
			if(*token == '~')
			{
		                esave = (Name::Event *)alloc(sizeof(Name::Event));
                                esave->name = alloc(token + 1);
                                esave->line = NULL;
                                esave->next = events;
                                esave->type = '~';
                                events = esave;
                                continue;
                        }
			if(*token == '@' || *token == '{')
			{
				esave = (Name::Event *)alloc(sizeof(Name::Event));
				esave->name = alloc(token + 1);
				esave->line = NULL;
				esave->next = events;
				esave->type = '@';
				events = esave;
				continue;
			}

			// compatibility mode... ::label
			if(!strnicmp(token, "::", 2))
			{
				cmode = cSCRIPT;
				if(bm)
					pmode = "protected";
				else
					pmode = "public";
				name = token + 2;
				token = "::";
				break;
			}

			if(!stricmp(token, "private") || !stricmp(token,
"protected") || !stricmp(token, "public") || !stricmp(token, "program")
|| !stricmp(token, "state") || !stricmp(token, "module"))
			{
				if(!stricmp(token, "module"))
				{
					ftoken = "fconst";
					token = "program";
				}
				cmode = cSCRIPT;
				pmode = alloc(token);
repname1:
				name = getToken();
				if(!name)
					break;
				if(*name == '+')
				{
					if(!stricmp(name, "+dtmf"))
						addPmask |= 0x08;
					else
						addPmask |= cmds->getTrapModifier(name + 1);
					goto repname1;
				}

				if(*name == '-')
				{
					subPmask &= ~cmds->getTrapModifier(name + 1);
					goto repname1;
				}

				token = "::";
				break;
			}


			if(!stricmp(token, "service"))
			{
				ftoken = "options";
				cmode = cSCRIPT;
				pmode = "public";
				goto repname1;
			}

			if(!strnicmp(token, "func", 4))
			{
				ftoken = "fconst";
				cmode = cSCRIPT;
				pmode = "function";
				goto repname1;
			}

			if(!strnicmp(token, "proc", 4))
			{
				cmode = cSCRIPT;
				pmode = "function";
				goto repname1;
			}

			if(!strncmp(token, "::", 2))
			{
				name = token + 2;
				break;
			}
			tlen = (int)strlen(token);
			if(token[tlen - 1] == ':')
			{
				token[tlen - 1] = 0;
				name = token;
				token = "::";
				break;
			}

			if(cmode != cSCRIPT)
				break;

			if(*token == '^')
			{
				if(!trapflag)
				{
					trapmask = 0;
					trapflag = true;
				}
			}

			if(!stricmp(token, "->") && !last)
			{
				token = "goto";
				break;
			}

			if(*token != '^' && *token != '+' && *token != '-' && *token != '?')
				break;

			if(*token == '^')
				mask = cmds->getTrapMask(token + 1);
			else
				mask = cmds->getTrapModifier(token + 1);

			if(!mask)
			{
				slog.error("%s: unknown trap id; %s(%d)", token + 1, filename, lnum);
				continue;
			}

			switch(*token)
			{
			case '^':
				last = NULL;
				script->mask |= mask | cmds->getTrapDefault();
				trapmask |= mask;
				break;
			case '+':
				addmask |= mask;
				break;
			case '-':
				submask |= mask;
				break;
			case '?':
				cmask |= mask;
			}
		}

		if(!token)
			continue;

		if(!stricmp(token, "::"))
			break;

		if(!stricmp(token, "datamap"))
		{
			cmode = cMAP;
			continue;
		}

		if(!stricmp(token, "table"))
		{
			cmode = cDATA;
			continue;
		}

		if(!stricmp(token, "use"))
		{
			while(NULL != (token = getToken()))
			{
				if(!Script::use(token))
					slog.warn() << token << ": package missing" << endl;
			}
			continue;
		}

		if(!stricmp(token, "config"))
		{
			if(!loaded && bm)
			{
				snprintf(path, sizeof(path), "/modules/%s", basename);
				loadPrefix(basename, path);
			}
			else if(!loaded && !project)
			{
				snprintf(path, sizeof(path), "/apps/%s", basename);
				loadPrefix(basename, path);
			}
			
			if(!loaded && project)
			{
				snprintf(path, sizeof(path), "/virtual/%s/%s", project, basename);
				loadPrefix(basename, path);
			}
			loaded = true;
			token = getToken();
			if(!token)
				continue;
			snprintf(path, sizeof(path), "%s.%s", basename, token);
			if(getLast(path))
				continue;

			token = getToken();
			if(!token)
				token = "";
			setValue(path, token);
			continue;
		}

		if(!stricmp(token, "dir") || !stricmp(token, "mkdir"))
		{
			while(NULL != (token = getToken()))
			{
				if(var)
				{
					snprintf(path, sizeof(path), "%s/%s", var, token);
					token = path;
				}
				Dir::create(token, File::attrGroup);
			}
			continue;
		}

		if(!stricmp(token, "phrase"))
		{
			token = getToken();
			if(!token)
				continue;
			if(!stricmp(token, "*") || !stricmp(token, "any"))
				token = "all";

			tag = getToken();
			if(!tag)
				continue;

			snprintf(tagbuf, sizeof(tagbuf), "_%s_%s", token, tag);

			i = 0;
			while(NULL != (token = getToken()))
			{
				if(!i)
					snprintf(rulebuf, sizeof(rulebuf), "%s", token);
				else
					snprintf(rulebuf + i, sizeof(rulebuf) - i, " %s", token);
			}
			setValue(tagbuf, rulebuf);
			continue;
		}

		if((!stricmp(token, "requires") || !stricmp(token, "import")) && scrStream == (istream *)&scrSource)
		{
			token = getToken();
			if(!token)
				continue;

			if(strchr(token, '/'))
				continue;


			if(inccount > 255)
				continue;
			incfiles[inccount++] = alloc(token);
   			continue;
		}

		ignore = false;
		syncflag = false;

retoken:
		if(!token)
			continue;

		if(*token == '%')
		{
			pretoken = token;
			token = "number";
		}
		else if(*token == '&')
		{
			pretoken = token;
			token = "call";
		}
		else if(!strnicmp(token, "*::", 3))
		{
			pretoken = token + 3;
			token = "call";
		}
		else if(strstr(token, "::"))
		{
			pretoken = token;
			token = "call";
		} else
		switch(cmode)
		{
		case cSCRIPT:
			pretoken = NULL;
			break;
		case cDATA:
			pretoken = token;
			token = "data";
			break;
		case cMAP:
			ctlen = (int)strlen(token) + 6;
			pretoken = (char *)alloc(ctlen);
			setString(pretoken, ctlen, "data.");
			addString(pretoken, ctlen, token);
			token = pretoken;
			pretoken = NULL;
		}

		if(!stricmp(token, "exclusive"))
		{
			syncflag = true;
			token = getToken();
			goto retoken;
		}

		if(*token == '@')
		{
			ignore = true;
			++token;
		}

		trapflag = false;
		handler = cmds->getHandler(token);
		if(handler == (Method)NULL)
		{
			errmsg = preproc(token);
			if(!errmsg)
				continue;

			addmask = submask = 0;
			if(!ignore)
				slog.error("%s: %s; %s(%d)", token, errmsg, filename, lnum);
			continue;
		}

		command = alloc(token);
		argc = 0;
		while(argc < maxargs && NULL != (token = getToken(&pretoken)))
		{
			if(token[0] == '$' && isalpha(token[1]))
			{
				if(!stricmp(token, "$script.name"))
					token = alloc((char *)basename);
				else if(!stricmp(token, "$script.file"))
					token = alloc((char *)name);
				else if(!stricmp(token, "$script.line"))
				{
					sprintf(temp, "%d", number);
					token = alloc(temp);
				}
				else
				{
					if(!strchr(++token, '.'))
					{
						snprintf(path, sizeof(path), "%s.%s", basename, token);
						token = path;
					}
					token = (char *)getDefined(token);
				}

				if(!token)
					token = "";
			}
			else if(token[0] == 0x01)
				token = alloc(++token);
			else if(token[0] == '.' && isalpha(token[1]))
			{
				cp = token + strlen(token) - 1;
				if(*cp != '.')
				{
					gvarname[0] = '%';
					snprintf(gvarname + gvarlen, sizeof(gvarname) - gvarlen, "%s", token + 1);
					token = alloc(gvarname);
				}
				else
					token = alloc(token);
			}
			else if(!strnicmp(token, "=.", 2))
			{
				gvarname[0] = '=';
				snprintf(gvarname + gvarlen, sizeof(gvarname) - gvarlen, "%s", token + 2);
				token = alloc(gvarname);
			}
			else if(!strnicmp(token, "&.", 2) || !strnicmp(token, ">.", 2))
			{
				gvarname[0] = '&';
                                snprintf(gvarname + gvarlen,sizeof(gvarname) - gvarlen, "%s", token + 2);
                                token = alloc(gvarname);
			}
			else if(!strnicmp(token, "@.", 2))
			{
				gvarname[0] = '@';
                                snprintf(gvarname + gvarlen,sizeof(gvarname) - gvarlen, "%s", token + 2);
                                token = alloc(gvarname);
			}
			else if(!strnicmp(token, "%.", 2))
			{
				gvarname[0] = '%';
				snprintf(gvarname + gvarlen, sizeof(gvarname) - gvarlen, "%s", token + 2);
				token = alloc(gvarname);
			}
			else if(*token == '>' && isalnum(token[1]))
			{
				token = alloc(token);
				*token = '&';
			}
			else
				token = alloc(token);


			args[argc++] = token;
			if(!stricmp(token, "then") && handler == &ScriptInterp::scrIf)
			{
				--bp;
				*bp = ' ';
				then = true;
				handler = &ScriptInterp::scrIfThen;
				break;
			}
		}

		args[argc++] = NULL;
		line = (Line *)alloc(sizeof(Line));
		line->line = number;
		line->lnum = lnum;
		line->cmask = cmask;
		line->mask = ((~0 & ~trapmask) | addmask) & ~submask;
		if(script->mask)
			line->mask &= cmds->getTrapHandler(script);
		if(!trapmask)
		{
			line->mask |= addPmask1;
			line->mask &= subPmask1;
		}
		line->next = NULL;
		line->args = (char **)alloc(sizeof(char *) * argc);
		line->argc = --argc;
		line->sync = syncflag;
		line->method = handler;
		line->cmd = command;
		line->loop = 0;

		addmask = submask = 0;

		if(!stricmp(command, "repeat") || !stricmp(command, "for") || !stricmp(command, "do") || !strnicmp(command, "do.", 3) || !strnicmp(command, "for.", 4))
		{
			if(!looplevel)
				++loopid;
			++looplevel;
			line->loop = loopid * 128 + looplevel;
		}

		if(!stricmp(command, "loop") || !strnicmp(command, "loop.", 5))
		{
			line->loop = loopid * 128 + looplevel;
			if(!looplevel)
			{
				slog.error("loop: nesting error %s(%d)", filename, lnum);
				continue;
			}
			else
				--looplevel;
		}
		memcpy(line->args, &args, sizeof(char *) * argc);

		err = cmds->check(command, line, this);
		if(err)
		{
			slog.error("%s: %s; %s(%d)", command, err, filename, lnum);
			continue;
		}

		++count;
		script->mask |= trapmask;
		if(!script->first)
			script->first = line;

		while(events)
		{
			esave = events->next;
			events->line = line;
			events->next = script->events;
			script->events = events;
			events = esave;
		}

		if(trapmask && !last)
		{
			for(i = 0; i < TRAP_BITS; ++i)
			{
				if((1l << i) & trapmask)
				{
					if(!script->trap[i])
						script->trap[i] = line;
				}
			}
		}

		if(last)
			last->next = line;

		last = line;
	}
	line = script->first;
	if(!script->mask)
		script->mask = cmds->getTrapDefault();
	script->mask |= addPmask1;
	script->mask &= subPmask1;
	addPmask1 = 0;
	subPmask1 = ~0;
	while(line)
	{
		line->mask &= script->mask;
		line->mask |= ((~script->mask) & cmds->imask);
		line = line->next;
	}
	total += count;

	if(first && count)
		initcount = count;
	else if(count)
		slog.info("%s: %d steps compiled", script->name, count);
	first = false;

	if(!scrStream->eof())
	{
		if(strstr(name, "::"))
			snprintf(namebuf, sizeof(namebuf), "%s", name);
		else
			snprintf(namebuf, sizeof(namebuf), "%s::%s", basename, name);
		name = namebuf;
		goto compile;
	}
	scrSource.close();
	scrSource.clear();
	if(init)
	{
		if(initkey == SCRIPT_INDEX_SIZE && initcount)
			slog.info("%s: %d steps initial", init->name, initcount);
		else if(initcount)
			slog.info("%s: %d steps compiled", init->name, initcount);
		init->next = index[initkey];
		index[initkey] = init;
	}
	free(buffer);
	return total;
}

#ifdef	CCXX_NAMESPACES
}
#endif
