// Copyright (C) 2001 Kai Germaschewski
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
using namespace std;
namespace ost {
#endif

static unsigned char ulaw_to_alaw[] =
{
	0xab, 0x55, 0xd5, 0x15, 0x95, 0x75, 0xf5, 0x35,
	0xb5, 0x45, 0xc5, 0x05, 0x85, 0x65, 0xe5, 0x25,
	0xa5, 0x5d, 0xdd, 0x1d, 0x9d, 0x7d, 0xfd, 0x3d,
	0xbd, 0x4d, 0xcd, 0x0d, 0x8d, 0x6d, 0xed, 0x2d,
	0xad, 0x51, 0xd1, 0x11, 0x91, 0x71, 0xf1, 0x31,
	0xb1, 0x41, 0xc1, 0x01, 0x81, 0x61, 0xe1, 0x21,
	0x59, 0xd9, 0x19, 0x99, 0x79, 0xf9, 0x39, 0xb9,
	0x49, 0xc9, 0x09, 0x89, 0x69, 0xe9, 0x29, 0xa9,
	0xd7, 0x17, 0x97, 0x77, 0xf7, 0x37, 0xb7, 0x47,
	0xc7, 0x07, 0x87, 0x67, 0xe7, 0x27, 0xa7, 0xdf,
	0x9f, 0x7f, 0xff, 0x3f, 0xbf, 0x4f, 0xcf, 0x0f,
	0x8f, 0x6f, 0xef, 0x2f, 0x53, 0x13, 0x73, 0x33,
	0xb3, 0x43, 0xc3, 0x03, 0x83, 0x63, 0xe3, 0x23,
	0xa3, 0x5b, 0xdb, 0x1b, 0x9b, 0x7b, 0xfb, 0x3b,
	0xbb, 0xbb, 0x4b, 0x4b, 0xcb, 0xcb, 0x0b, 0x0b,
	0x8b, 0x8b, 0x6b, 0x6b, 0xeb, 0xeb, 0x2b, 0x2b,
	0xab, 0x54, 0xd4, 0x14, 0x94, 0x74, 0xf4, 0x34,
	0xb4, 0x44, 0xc4, 0x04, 0x84, 0x64, 0xe4, 0x24,
	0xa4, 0x5c, 0xdc, 0x1c, 0x9c, 0x7c, 0xfc, 0x3c,
	0xbc, 0x4c, 0xcc, 0x0c, 0x8c, 0x6c, 0xec, 0x2c,
	0xac, 0x50, 0xd0, 0x10, 0x90, 0x70, 0xf0, 0x30,
	0xb0, 0x40, 0xc0, 0x00, 0x80, 0x60, 0xe0, 0x20,
	0x58, 0xd8, 0x18, 0x98, 0x78, 0xf8, 0x38, 0xb8,
	0x48, 0xc8, 0x08, 0x88, 0x68, 0xe8, 0x28, 0xa8,
	0xd6, 0x16, 0x96, 0x76, 0xf6, 0x36, 0xb6, 0x46,
	0xc6, 0x06, 0x86, 0x66, 0xe6, 0x26, 0xa6, 0xde,
	0x9e, 0x7e, 0xfe, 0x3e, 0xbe, 0x4e, 0xce, 0x0e,
	0x8e, 0x6e, 0xee, 0x2e, 0x52, 0x12, 0x72, 0x32,
	0xb2, 0x42, 0xc2, 0x02, 0x82, 0x62, 0xe2, 0x22,
	0xa2, 0x5a, 0xda, 0x1a, 0x9a, 0x7a, 0xfa, 0x3a,
	0xba, 0xba, 0x4a, 0x4a, 0xca, 0xca, 0x0a, 0x0a,
	0x8a, 0x8a, 0x6a, 0x6a, 0xea, 0xea, 0x2a, 0x2a
};

CapiTone::CapiTone(CapiTrunk *trunk)
	: Service((Trunk *)trunk, keythreads.priAudio())
{
	slog(Slog::levelDebug) << "CAPI: " << __FUNCTION__ << endl;

	num_frames = 2;
	frame_size = capiivr.getBufferSize();
}

CapiTone::~CapiTone()
{
	slog(Slog::levelDebug) << "CAPI: " << __FUNCTION__ << endl;

	clrAudio();
	terminate();
}

void CapiTone::initial()
{
	slog(Slog::levelDebug) << "CAPI: " << __FUNCTION__ << endl;

	CapiTrunk *trk = (CapiTrunk *) trunk;
	char name[32];

	trk->getName(name);
	trk->getTone();
	buffer = trk->data.tone.tone->getSamples();
	len = trk->data.tone.tone->getDuration() * 8000/1000;
	trk->play_frames = 0;
}

void CapiTone::run()
{
	slog(Slog::levelDebug) << "CAPI: " << __FUNCTION__ << endl;

	char name[33];
	CapiTrunk *trk = (CapiTrunk *) trunk;
	CapiDriver *drv = (CapiDriver *) trunk->getDriver();
	unsigned count;
	CapiBuffer buf(frame_size);
	unsigned int err;
	_cmsg cmsg;

	trunk->getName(name);

	setAudio();
	while (!stopped) {
		if (trk->play_frames >= (int)num_frames) {
			Semaphore::wait();
			continue;
		}
		count = len;
		if (count > frame_size)
			count = frame_size;

		for (unsigned i = 0; i < count; i++)
			buf.data[i] = ulaw_to_alaw[buffer[i]];

		buffer += count;
		len -= count;
		buf.bitReverse();
		err = DATA_B3_REQ(&cmsg, drv->appl_id, drv->msg_id++,
				  trk->ncci,
				  buf.data,
				  count,
				  0,
				  0);
		if (err != CapiNoError) {
				slog(Slog::levelError) << "CAPI: DATA_REQ err = "
						 << err << endl;
				break;
		}
		++trk->play_frames;
		if (len == 0)
			break;
	}
	clrAudio();
	while (trk->play_frames > 0)
		Semaphore::wait();

	Service::success();
}

timeout_t CapiTone::stop(void)
{
	slog(Slog::levelDebug) << "CAPI: " << __FUNCTION__ << endl;

	clrAudio();
	stopped = true;

	return ~0;
}

bool CapiTrunk::toneHandler(TrunkEvent *event)
{
	slog(Slog::levelDebug) << "CAPI: " << __FUNCTION__ << " " 
			 << event->id << endl;

	switch(event->id) {
	case TRUNK_ENTER_STATE:
		endTimer();
		if (!flags.offhook) {
			trunkError("tone-on-hook");
			handler = &CapiTrunk::stepHandler;
			return true;
		}
		enterState("tone");
		status[tsid] = 't';
		Trunk::setDTMFDetect();
		thread = new CapiTone(this);
		thread->start();
		return true;
	case TRUNK_SERVICE_SUCCESS:
		trunkSignal(TRUNK_SIGNAL_STEP);
		goto stop;
	case TRUNK_STOP_STATE:
	stop:
		handler = &CapiTrunk::stepHandler;
		return true;
	}
	return false;
}

#ifdef	CCXX_NAMESPACES
};
#endif
