// Copyright (C) 2000 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

LTAPIConfig::LTAPIConfig() :
Keydata("/drivers/ltapi")
{
	static Keydata::Define defkeys[] = {
	{"devices", "16"},
	{"wink", "100"},
	{"buffers", "1"},
	{"echo", "low"},
	{NULL, NULL}};

	load(defkeys);
}

size_t LTAPIConfig::getStack(void)
{
	const char *cp = getLast("stack");

	if(!cp)
		return keythreads.getStack();
		
	return atoi(cp) * 1024;
}

LTAPIDriver::LTAPIDriver() :
Driver()
{
	port_count = getDevices();
	ports = new LTAPITrunk *[port_count];
	memset(ports, 0, sizeof(LTAPITrunk *) * port_count);

	thread_count = thread_select = 0;
	
	slog(Slog::levelInfo) << "Generic Phone Device (ltapi) driver loaded; capacity=" << port_count << endl;
}

LTAPIDriver::~LTAPIDriver()
{
	stop();
	if(ports)
		delete[] ports;
}

int LTAPIDriver::start(void)
{
	char path[32];
	int fd, ts;
	int count = 0;
	int scount = keythreads.getServices();

	if(active)
	{
		slog(Slog::levelError) << "driver already started" << endl;
		return 0;
	}

	slog(Slog::levelInfo) << "driver starting " << scount << " service thread(s)" << endl;
	threads = new LTAPIService[scount];

	for(ts = 0; ts < port_count; ++ts)
	{
		sprintf(path, "/dev/phone%d", ts);
		fd = open(path, O_RDWR);
		if(fd < 0)
			continue;
		if(!getuid())
			fchown(fd, getuid(), getgid());
		ports[ts] = new LTAPITrunk(fd, ts);
		++count;
	}

	active = true;
	return count;
}

void LTAPIDriver::stop(void)
{
	int id;

	if(!active)
		return;

	if(ports)
	{
		for(id = 0; id < port_count; ++id)
		{
			if(ports[id])
				delete ports[id];
		}
	}
	memset(ports, 0, sizeof(LTAPITrunk *) * port_count);
	delete[] threads;
	
	slog(Slog::levelInfo) << "driver stopping service thread(s)" << endl;
	active = false;
}

Trunk *LTAPIDriver::getTrunkPort(int id)
{
	if(id < 0 || id >= port_count)
		return NULL;

	return (Trunk *)ports[id];
}

LTAPIService *LTAPIDriver::getService(void)
{
	LTAPIService *svc = &threads[thread_select++];
	if(thread_select >= thread_count)
		thread_select = 0;
	return svc;
}

void LTAPIDriver::notify(unsigned char id)
{
	TrunkEvent event;
	LTAPITrunk *trunk;

	if(id < 1 || id > port_count)
		return;

	trunk = ports[--id];
	if(!trunk)
		return;

	event.id = TRUNK_NOTIFICATION;
	trunk->postEvent(&event);
}

LTAPIDriver phivr;

#ifdef	CCXX_NAMESPACES
};
#endif
