// Copyright (C) 2000-2003 Open Source Telecom Corporation.
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne
// with the OpenH323 and Pwlib libraries, and distribute the combination, 
// without applying the requirements of the GNU GPL to the OpenH323
// and pwd libraries as long as you do follow the requirements of the 
// GNU GPL for all the rest of the software thus combined.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include "driver.h"

#ifdef	CCXX_NAMESPACES
using namespace std;
namespace ost {
#endif

OH323Channel::OH323Channel(OH323Trunk *parent) : 
	trunk(parent)
{
	receive = NULL;
	transmit = NULL;

	//dtmf = new DTMFDetect();
	dtmf = NULL;

	if(oh323ivr.getBool("inbanddtmf"))
		dtmf = new DTMFDetect();
}

OH323Channel::~OH323Channel()
{
	if(receive)
		delete receive;
	if(transmit)
		delete transmit;
	if(dtmf)
		delete dtmf;
}

BOOL OH323Channel::IsOpen()
{
	return TRUE;
}

void OH323Channel::stop(int channelType)
{
	TrunkEvent event;
	bool stopped = false;

	enterMutex();
	if(channelType == AUDIO_TRANSMIT && transmit)
	{
		stopped = true;
		delete transmit;
		transmit = NULL;
	}
	else if(channelType == AUDIO_RECEIVE && receive)
	{
		stopped = true;
		delete receive;
		receive = NULL;
	}
	else
	{
		if(receive)
		{
			stopped = true;
			delete receive;
			receive = NULL;
		}
		if(transmit)
		{
			stopped = true;
			delete transmit;
			transmit = NULL;
		}
	}

	if(stopped)
	{
		event.id = TRUNK_AUDIO_IDLE;
		trunk->postEvent(&event);
	}
	leaveMutex();
}

BOOL OH323Channel::Write(const void *buf, PINDEX len)
{
	TrunkEvent event;
	static unsigned lastDtmf = 0;
	int dtmfFound = 0;
	unsigned i;
	char digits[128] = {0};
	enterMutex();

	/*if(dtmf)
	{
		lastDtmf += len;
		dtmfFound = dtmf->putSamples((int16_t*)buf, len/2);
		if(dtmfFound && lastDtmf > 1000)
		{
			dtmf->getResult(digits, 64);
			if(digits)
			{
				for(i = 0; i < strlen(digits); i++)
				{
					event.id = TRUNK_DTMF_KEYUP;
					event.parm.dtmf.digit = (int)(digits[i] - '0');
					event.parm.dtmf.duration = 60;
					trunk->postEvent(&event);
				}
				lastDtmf = 0;
			}
		}
	}*/

	if(receive)
		receive->putBuffer(buf, len);
	leaveMutex();

	lastWriteCount = len;
	writeDelay.Delay(lastWriteCount/16);

	return TRUE;
}

BOOL OH323Channel::Read(void *buf, PINDEX len)
{
	int count = 0;

	enterMutex();
	if(transmit)
	{
		count = transmit->getBuffer(buf, len);
		if(count < 1)
			stop(AUDIO_TRANSMIT);
	}
	leaveMutex();

	if(count < len)
	{
		memset((char*)buf+count, 0, len - count);
	}

	lastReadCount = len;
	readDelay.Delay(lastReadCount/16);

	return TRUE;
}

BOOL OH323Channel::Close()
{
	closed = true;
	//if(trunk->isOpen())
	//	trunk->close();
	PChannel::Close();
	return true;
}

AudioFileChannel::AudioFileChannel(AudioFile *audio)
{
	this->audio = audio;
}

int AudioFileChannel::getBuffer(void *buf, unsigned len)
{
	return audio->getLinear((short *)buf, len/2) * 2;
}

int AudioFileChannel::putBuffer(const void *buf, unsigned len)
{
	return audio->putBuffer((void *)buf, len);
}

ToneChannel::ToneChannel(phTone *t)
{
	int i;

	this->tone = t;
	buffer = tone->getSamples();
	samples = tone->getDuration() * 8;

	for(i = 0; i < samples; i++)
		buffer[i] = phTone::ulaw2linear(buffer[i]);
}

int ToneChannel::getBuffer(void *buf, unsigned len)
{
	unsigned read = len;
	if(samples < read)
		read = samples;

	if(read > 0)
	{
		memcpy(buf, buffer, read);
		samples -= read;
		buffer += read;
	}
	return read;
}

JoinChannel::JoinChannel(Trunk *srctrk, Trunk *dsttrk)
{
	src = srctrk;
	dst = dsttrk;
}

int JoinChannel::getBuffer(void *buf, unsigned len)
{
	unsigned amount = len;
	//dst->softJoin->getBuffer((char *)buf, len);
	return amount;
}

int JoinChannel::putBuffer(const void *buf, unsigned len)
{
	unsigned amount = len;
	//src->softJoin->putBuffer((char *)buf, len);
	return amount;
}

#ifdef	CCXX_NAMESPACES
}
#endif     
