// Copyright (C) 2000-2003 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne
// with the OpenH323 and Pwlib libraries, and distribute the combination, 
// without applying the requirements of the GNU GPL to the OpenH323
// and pwd libraries as long as you do follow the requirements of the 
// GNU GPL for all the rest of the software thus combined.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include "driver.h"
#include <cc++/process.h>

#ifdef	CCXX_NAMESPACES
using namespace std;
namespace ost {
#endif

OH323Config::OH323Config() :
Keydata("/drivers/h323")
{
	static Keydata::Define defkeys[] = {
	{"first", "3128"},
	{"count", "32"},
	{"inc", "4"},
	{"interface", "*"},
	{"interval", "35"},
	{"port","1720"},
	{"usegk","0"},
	{"gatekeeperaddress",""},
	{"gatekeeperid",""},
	{"gatekeeperpassword",""},
	{"gatewayprefixes",""},
	{"username","Bayonne"},
	{"uimode","rfc2833"},
	{"faststart","1"},
	{"h245tunneling","1"},
	{"h245insetup","1"},
	{"forwarding","1"},
	{"inbanddtmf","1"},
	{"rtpmin","0"},
	{"rtpmax","0"},
	{"tcpmin","0"},
	{"tcpmax","0"},
	{"udpmin","0"},
	{"udpmax","0"},
	{NULL, NULL}};

	load(defkeys);
}

InetAddress OH323Config::getBindAddress(void)
{
	const char *cp = getLast("bind");
	if(!cp)
		cp = getLast("address");
	if(!cp)
		cp = getLast("interface");

	return InetAddress(cp);
}

bool OH323Config::getBool(const char *name)
{
	const char *cp = getLast(name);
	if(atoi(cp) == 1 || !stricmp(cp, "on"))
		return true;
	else
		return false;
}

OH323Driver::OH323Driver() :
Driver(), Thread(keythreads.priService(), keythreads.getDriverStack())
{
	port_count = oh323ivr.getRTPCount();

	ports = NULL;


//	endpoint = new OH323EndPoint;

	ports = new OH323Trunk *[port_count];

	if(ports)
		memset(ports, 0, sizeof(OH323Trunk *) * port_count);

#ifdef PTRACING
	//PTrace::Initialise(1, NULL, PTrace::Blocks | PTrace::Timestamp | PTrace::Thread |
	//			PTrace::FileAndLine);
	//slog(Slog::levelInfo) << "H323 tracing enabled" << endl;
#endif

	slog(Slog::levelInfo) << "H323 trunk driver loaded; capacity=" << port_count << endl;
}

OH323Driver::~OH323Driver()
{
	stop();
	if(ports)
		delete[] ports;

}

int OH323Driver::start(void)
{
	PIPSocket::Address interfaceAddress(getLast("interface"));
	WORD port = getSignalPort();
	unsigned idx = 0;
	const char *env = Process::getEnv("H323_PORT");
	
	if(env)
		port = atoi(env);

	int count = oh323ivr.getRTPCount();

	if(active)
	{
		slog(Slog::levelError) << "oh323 driver already started" << endl;
		return 0;
	}

	slog(Slog::levelInfo) << "oh323 driver starting..." << endl;


	if (!endpoint.Initialise())
	{
		slog(Slog::levelError) << "oh323: the endpoint failed to initialise" << endl;
		return 0;
	}

	listener = new H323ListenerTCP(endpoint, interfaceAddress, port);
	if(!endpoint.StartListener(listener))
	{
		slog.error() << "count not open h.323 listener port" << endl;
		delete listener;
		listener = NULL;
		return 0;
	}

        slog.notice() << "h323: listening on port " << port << endl;
	active = true;

	Thread::start();
	return count;
}

void OH323Driver::stop(void)
{

	if(!active)
		return;

	if(ports)
		memset(ports, 0, sizeof(OH323Trunk *) * port_count);

	active = false;
	slog(Slog::levelInfo) << "oh323 driver stopping..." << endl;


	if(listener)
	{
		endpoint.ClearAllCalls(H323Connection::EndedByLocalUser, true);
//		endpoint.RemoveListener(listener);
//		delete listener;
		listener = NULL;
	}
}

void OH323Driver::run(void)
{
	int i;
	int timer;
	TrunkEvent event;
	timeout_t timeout = atoi(getLast("interval"));
	OH323Trunk *next, *prior, *current;
	char buffer[32];

	if(!active)
		return;

	setCancel(cancelImmediate);

	while(active)
	{
		Thread::sleep(timeout);
		for(i = 0; i < port_count; i++)
		{
			if(ports[i] == NULL)
				continue;

			if(!ports[i]->getTimer())
			{
				event.id = TRUNK_TIMER_EXPIRED;
				ports[i]->postEvent(&event);
			}
		}
		prior = NULL;
		OH323Trunk::freLock.enterMutex();
		current = OH323Trunk::freFirst;
		OH323Trunk::freLock.leaveMutex();
		while(current && active)
		{
			next = current->freNext;
			if(current->handler == &OH323Trunk::idleHandler)
			{
				current->getName(buffer);
				slog.debug() << buffer << ": recycled" << endl;
				if(!prior)
				{
					OH323Trunk::freLock.enterMutex();
					OH323Trunk::freFirst = next;
					OH323Trunk::freLock.leaveMutex();
				}
				else
					prior->freNext = next;
				delete current;
				current = next;
				continue;
			}
			if(!current->getTimer())
			{
				event.id = TRUNK_TIMER_EXPIRED;
				current->postEvent(&event);
			}
			prior = current;
			current = next;
		}
	}
	Thread::sync();
}

bool OH323Driver::isTrunkClass(int id, const char *name)
{
	if(!stricmp(name, "h323"))
		return true;

	if(!stricmp(name, "callto"))
		return true;

	if(!stricmp(name, getName()))
		return true;

	return false;
}

Trunk *OH323Driver::getTrunkPort(int id)
{
	if(id < 0 || id >= port_count)
		return NULL;

	if(!ports)
		return NULL;

	if(ports[id] == NULL)
	{
		// make new trunk
		return NULL;
	}

	return (Trunk *)ports[id];
}

Trunk *OH323Driver::getOutboundTrunk(int id)
{
	if(id < 0 || id >= port_count)
		return NULL;

	if(!ports)
		return NULL;

	if(ports[id] != NULL)
		return NULL;

	ports[id] = new OH323Trunk(NULL, id);

	return (Trunk *)ports[id];
}

OH323Driver oh323ivr;
OH323EndPoint endpoint;

#ifdef	CCXX_NAMESPACES
}
#endif
