// Copyright (C) 2000 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"
#include <fcntl.h>

#ifdef  CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

OSSConfig::OSSConfig() :
Keydata("/drivers/oss")
{
	static Keydata::Define defkeys[] = {
	{"device", "/dev/dsp"},
	{"mixer", "/dev/mixer"},
	{"count", "1"},
	{"buffers", "8"},
	{"stack", "64"},
	{NULL, NULL}};

	load(defkeys);
}

size_t OSSConfig::getStack(void)
{
        const char *cp = getLast("stack");

        if(!cp)
                return keythreads.getDriverStack();

        return atoi(cp) * 1024;
}

size_t OSSConfig::getAudioStack(void)
{
	const char *cp = getLast("audiostack");

	if(!cp)
		return keythreads.getAudioStack();

	return atoi(cp) * 1024;
}

const char *OSSConfig::getDevice(int ts)
{
	char devname[16];
	const char *cp;

	snprintf(devname, 16, "device%d", ts);
	cp = getLast(devname);
	if(!cp)
		cp = getLast("device");
	return cp;
}

const char *OSSConfig::getMixer(int ts)
{
	char devname[16];
	const char *cp;

	snprintf(devname, 16, "mixer%d", ts);
	cp = getLast(devname);
	if(!cp)
		cp = getLast("mixer");
	return cp;
}

OSSDriver::OSSDriver() :
Driver(), OSSConfig(), Thread(keythreads.priService(), ossivr.getStack())
{
	long opts;
	port_count = getDeviceCount();

	ports = new OSSTrunk *[port_count];

	::pipe(iobuf);
	opts = fcntl(iobuf[0], F_GETFL);
	fcntl(iobuf[0], F_SETFL, opts | O_NDELAY);

	if(ports)
		memset(ports, 0, sizeof(OSSTrunk *) * port_count);

	slog(Slog::levelInfo) << "Generic Test (oss) driver loaded; capacity=" << port_count << endl;
}

OSSDriver::~OSSDriver()
{
	::close(iobuf[0]);
	::close(iobuf[1]);
	if(active)
		stop();

	if(ports)
		delete ports;

}

void OSSDriver::update(void)
{
	char buf;
	::write(iobuf[1], &buf, 1);
}

int OSSDriver::start(void)
{
	int count = 0;

	if(active)
	{
		slog(Slog::levelError) << "driver already started" << endl;
		return 0;
	}


	for(count = 0; count < getDeviceCount(); ++count)
        	ports[count] = new OSSTrunk(count);

	tcgetattr(0, &t);
	slog(Slog::levelInfo) << "driver starting..." << endl;
	Thread::start();

	active = true;
	return count;
}

void OSSDriver::stop(void)
{
	unsigned count;
	OSSTrunk *trk;
	if(!active)
		return;

	terminate();
        tcsetattr(0, TCSANOW, &t);

	if(ports)
	{
		for(count = 0; count < (unsigned)getDeviceCount(); ++count)
		{
			trk = ports[count];
			if(trk)
				delete trk;
		}
		memset(ports, 0, sizeof(OSSTrunk *) * port_count);
	}

	active = false;
	slog(Slog::levelInfo) << "driver stopping..." << endl;
}

void OSSDriver::run(void)
{
	TrunkEvent event;
	timeout_t timeout;
#ifdef  USE_POLL
	struct pollfd pfd[2];
	int maxfd;
#else
	struct timeval tv;
	fd_set sfd, efd;
#endif
	char ch;
	struct termios n;
	OSSTrunk *trunk;
	int port = 0;
	char name[16];
	int rtn;

	setCancel(cancelImmediate);
	slog(Slog::levelNotice) << "oss: start thread" << endl;\

#ifdef  USE_POLL
	pfd[0].fd = iobuf[0];
	pfd[1].fd = 0;
#endif

        tcgetattr(0, &t);
        tcgetattr(0, &n);
        n.c_lflag &= ~(ECHO|ICANON);
        rtn = tcsetattr(0, TCSANOW, &n);

	for(;;)
	{
		Thread::yield();
		trunk = ports[port];
		if(!trunk)
		{
			slog(Slog::levelError) << "oss: missing trunk" << endl;
			sleep(1000);
			continue;
		}

		snprintf(name, sizeof(name), "sound/%d", port);

		trunk->enterMutex();
		timeout = trunk->getTimer();
		if(!timeout)
		{
			trunk->endTimer();
			event.id = TRUNK_TIMER_EXPIRED;
			trunk->postEvent(&event);
		}
		trunk->leaveMutex();

		if(timeout > 500)
			timeout = 500;

		// if daemonfied server, then we just timeout...


		if(getppid() < 2)
		{
			
			Thread::sleep(timeout);
			continue;
		}

#ifdef  USE_POLL
                pfd[0].events = POLLIN | POLLRDNORM;
                pfd[0].revents = 0;
		pfd[1].events = POLLIN | POLLRDNORM;
		pfd[1].revents = 0;
		if(getppid() < 2)
			maxfd = 1;
		else
			maxfd = 2;
                if(poll(pfd, maxfd, timeout) > 0)
		{
			while(pfd[0].revents)
			{
				if(::read(iobuf[0], &ch, 1) < 1)
					break;
			}
			if(pfd[0].revents)
				continue;
#else
		FD_ZERO(&efd);
		FD_ZERO(&sfd);
		if(getppid() > 1)
			FD_SET(0, &sfd);
		FD_SET(iobuf[0], &sfd);
		tv.tv_sec = timeout / 1000;
		tv.tv_usec = (timeout % 1000) * 1000;
		select(iobuf[0] + 1, &sfd, NULL, &efd, &tv);
		while((FD_ISSET(iobuf[0], &sfd))
		{
			if(::read(iobuf[0], &ch, 1) < 1)
				break;
		}

		if(FD_ISSET(0, &sfd))
		{
#endif
			::read(0, &ch, 1);
                        switch(ch)
                        {
                        case '#':
                                slog(Slog::levelDebug) << name
                                        << ": digit #..." << endl;
                                event.id = TRUNK_DTMF_KEYUP;
                                event.parm.dtmf.digit = 11;
                                event.parm.dtmf.duration = 60;
                                trunk->postEvent(&event);
                                sleep(10);
                                break;
                        case '*':
                                slog(Slog::levelDebug) << name
                                        << ": digit *..." << endl;
                                event.id = TRUNK_DTMF_KEYUP;
                                event.parm.dtmf.digit = 10;
                                event.parm.dtmf.duration = 60;
                                trunk->postEvent(&event);
                                sleep(10);
                                break;
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                                slog(Slog::levelDebug) << name
                                        << ": digit " << ch << "..." << endl;
                                event.id = TRUNK_DTMF_KEYUP;
                                event.parm.dtmf.digit = ch - '0';
                                event.parm.dtmf.duration = 60;
                                trunk->postEvent(&event);
                                sleep(10);
                                break;
                        case 'r':
                        case 'R':
                                slog(Slog::levelDebug) << name
                                        << ": ringing..." << endl;
                                event.id = TRUNK_RINGING_ON;
                                event.parm.ring.digit = 0;
                                event.parm.ring.duration = 50;
                                trunk->postEvent(&event);
                                sleep(100);
                                event.id = TRUNK_RINGING_OFF;
                                trunk->postEvent(&event);
                                break;
			case 'n':
			case 'N':
				if(++port >= getTrunkCount())
					port = 0;
				break;
                        case 'D':
                        case 'd':
                                slog(Slog::levelDebug) << name
                                        << ": dialtone..." << endl;
                                event.id = TRUNK_CPA_DIALTONE;
                                trunk->postEvent(&event);
                                break;
                        case 'B':
                        case 'b':
                                slog(Slog::levelDebug) << name
                                        << ": busytone..." << endl;
                                event.id = TRUNK_CPA_BUSYTONE;
                                trunk->postEvent(&event);
                                break;
                        case ' ':
                                slog(Slog::levelDebug) << name
                                        << ": step/start..." << endl;
                                event.id = TRUNK_NULL_EVENT;
                                trunk->postEvent(&event);
                                break;
                        case 'H':
                        case 'h':
                                slog(Slog::levelDebug) << name
                                        << ": hangup..." << endl;
                                event.id = TRUNK_STOP_DISCONNECT;
                                trunk->postEvent(&event);
                                break;
                        case 'i':
                        case 'I':
                                slog(Slog::levelDebug) << name
                                        << ": idle..." << endl;
                                event.id = TRUNK_MAKE_IDLE;
                                trunk->postEvent(&event);
                                break;
			case 't':
			case 'T':
				if(trunk->trace)
				{
					slog.debug() << name << ": trace off" << endl;
					trunk->trace = false;
				}
				else
				{
					slog.debug() << name << ": trace on" << endl;
					trunk->trace = true;
				}
				break;
                        case 'X':
                        case 'x':
                        case 3:
                                slog(Slog::levelDebug) << name
                                        << ": exiting..." << endl;
                                raise(SIGINT);
			}
		}
	}
}

bool OSSDriver::isTrunkClass(int id, const char *name)
{
	if(!stricmp(name, "sound"))
		return true;

	return false;
}

Trunk *OSSDriver::getTrunkPort(int id)
{
	if(id < 0 || id >= port_count)
		return NULL;

	if(!ports)
		return NULL;

	return (Trunk *)ports[id];
}

unsigned OSSDriver::getCaps(void)
{
	return capSpeed | capDynamic;	// prevents outbound
}

OSSDriver ossivr;

#ifdef	CCXX_NAMESPACES
}
#endif
