// Copyright (C) 2000 Open Source Telecom Corporation.
// Copyright (C) 2004 Free Software Foundation
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

SIPConfig::SIPConfig() :
Keydata("/drivers/sip")
{
	static Keydata::Define defkeys[] = {
	{"count", "16"},
	{"stack", "32"},
	{"port", "5060"},
	{"rtpmin", "20000"},
	{"rtpmax", "21000"},
	{"rtpframesize", "65536"},
	{"interval", "35"},

	/*
	  These parameters are optional
	  and default to NULL
	  {"outbound_proxy", ""},
	  {"local_address", ""},
	  {"firewall_addess", ""},
	  {"auth_user", ""},
	  {"auth_passwd", ""},
	  {"auth_realm", ""},
	  {"register_uri", ""},
	  {"register_proxy", ""},
	  {"register_contact", ""},
	*/
	{"register_expiry", "3600"},

	{NULL, NULL}};

	load(defkeys);
}

size_t SIPConfig::getStack(void)
{
	const char *cp = getLast("stack");

	if(!cp)
		return keythreads.getStack();

	return atoi(cp) * 1024;
}

size_t SIPConfig::getAudioStack(void)
{
    const char *cp = getLast("audiostack");
    if(!cp)
	return keythreads.getAudioStack();
    return atoi(cp) * 1024;
}

SIPDriver::SIPDriver() :
    Driver(),
    SIPConfig(), 
    Thread(keythreads.priService() + 1, sipivr.getStack()), 
    SIPUserAgent()
{
    port_count = sipivr.getTrunkCount();
    ports = new SIPTrunk *[port_count];

    if(ports)
	memset(ports, 0, sizeof(SIPTrunk *) * port_count);

    slog(Slog::levelInfo) << "sip device driver loaded" << endl;
}

SIPDriver::~SIPDriver()
{
    stop();
    if(ports)
	delete[] ports;
}

void SIPDriver::run(void)
{
    TrunkEvent event;
    int i;
    timeout_t timeout = atoi(getLast("interval"));
    unsigned port;

    setCancel(cancelDeferred);
    while(active) {
	Thread::sleep(timeout);
	//
	// cause the processing of any
	// necessary trunk events if the
	// appropriate timer has expired
	//
	for(i=0; i < port_count; i++) {
	    if(!ports[i])
		continue;
	    ports[i]->enterMutex();
	    if(!ports[i]->getTimer()) {
		event.id = TRUNK_TIMER_EXPIRED;
		ports[i]->postEvent(&event);
	    }
	    ports[i]->leaveMutex();
	}

	//
	// send any register requests that 
	// may be required
	//
	uaRegister();

	//
	// process any pending eXosip events
	//
	uaProcess();
    }
    Thread::sync();
}

int SIPDriver::start(void)
{
    int count = sipivr.getTrunkCount();
    const char *cfgParam;

    if(active) {
	slog(Slog::levelError) << "driver already started" << endl;
	return 0;
    }
    
    slog(Slog::levelInfo) << "sip driver starting " << count << " ports" << endl;

    //
    // start the eXosip user agent
    //
    cfgParam = getLast("sipport");
    int sipport = 5060;
    if (cfgParam)
      sipport = atoi(cfgParam);
    if (sipport == 0 || sipport>65535)
	{
	    slog(Slog::levelWarning) << "sip: invalid port in config file: " 
				     << sipport << endl;
	    slog(Slog::levelWarning) << "sip: reverting to default port: 5060" << endl;
	    sipport = 5060;
	}
    uaStart(sipport);
    setUserAgent("GNU Bayonne");

    //
    // configure local ip addresses
    //
    cfgParam = getLast("local_address");
    if(cfgParam)
	setLocalAddress(cfgParam);
    cfgParam = getLast("firewall_address");
    if(cfgParam)
	setFirewallAddress(cfgParam);

    //
    // add credentials
    //
    unsigned nUsers = Keydata::getCount("auth_user");
    unsigned nPass = Keydata::getCount("auth_passwd");
    unsigned nRealm = Keydata::getCount("auth_realm");
    const char * const *authUsers = getList("auth_user");
    const char * const *authPass = getList("auth_passwd");
    const char * const *authRealm = getList("auth_realm");

    if(nPass != nUsers)
	slog(Slog::levelWarning) << "Configuration error: number of passwords does not match number of users" << endl;
    if((nRealm < (nUsers-1)) || nRealm > nUsers)
	slog(Slog::levelWarning) << "Configuration error: number of realms does not match number of users" << endl;

    for(int i = 0; i < nUsers; i++)
	{
	    const char *realm;
	    if(i >= nPass)
		break;
	    if(i >= nRealm)
		realm = NULL;
	    else
		realm = authRealm[i];

	    slog(Slog::levelInfo) << "sip: authentication" << endl;
	    slog(Slog::levelInfo) << "\tuser: " << authUsers[i] 
				  << (realm ? "@" : "")
				  << (realm ? realm : "") << endl;

	    addAuthenticationInfo(authUsers[i], authPass[i], realm);
	}

    //
    // start SIP registrations
    //
    unsigned nRegister = Keydata::getCount("register_uri");
    unsigned nRegProxy = Keydata::getCount("register_proxy");
    unsigned nRegContact = Keydata::getCount("register_contact");

    const char * const *regUri = getList("register_uri");
    const char * const *regProxy = getList("register_proxy");
    const char * const *regContact = getList("register_contact");

    if(nRegister != nRegProxy)
	slog(Slog::levelWarning) << "Configuration error: number of registrations does not match number of proxies" << endl;

    if(nRegister)
	registryInit(nRegister);

    for(int i = 0; i < nRegister; i++)
	{
	    const char *contact;
	    if(i >= nRegProxy)
		break;
	    if(i >= nRegContact)
		contact = NULL;
	    else
		contact = regContact[i];

	    int expiry = atoi(getLast("register_expiry"));

	    slog(Slog::levelInfo) << "sip: registration" << endl;
	    slog(Slog::levelInfo) << "\turi:     " << regUri[i] << endl;
	    slog(Slog::levelInfo) << "\tproxy:   " << regProxy[i] << endl;
	    slog(Slog::levelInfo) << "\tcontact: " << (contact ? contact : "") << endl;
	    slog(Slog::levelInfo) << "\texpiry:  " << expiry << endl;

	    if(!registryAdd(regUri[i], regProxy[i], contact, expiry))
		slog(Slog::levelWarning) << "sip: registration " << regUri[i] 
					 << " to proxy " << regProxy[i] << " failed"
					 << endl;
	}
    
    active = true;
    Thread::start();
    return count;
}

void SIPDriver::stop(void)
{
    if(!active)
	return;

    if(ports)
	memset(ports, 0, sizeof(SIPTrunk *) * port_count);

    slog(Slog::levelInfo) << "driver stopping service thread(s)" << endl;
    if(running) {
	active = false;
	terminate();
    }

    //
    // stop the sip user agent
    //
    uaStop();
    active = false;
}

Trunk *SIPDriver::getTrunkPort(int id)
{
    if(id < 0 || id >= port_count)
	return NULL;
    if(!ports)
	return NULL;

    return (Trunk *)ports[id];
}

Trunk *SIPDriver::getTrunkCallId(int callid)
{
    for(int i = 0; i < port_count; i++)
	if(ports[i] && ports[i]->callId == callid)
	    return ports[i];
    return NULL;
}

Trunk *SIPDriver::getTrunkUnused(int callid)
{
    char buf[33];

    portMutex.enterMutex();
    for(int i = 0; i < port_count; i++)
	if(!ports[i]) {
	    ports[i] = new SIPTrunk(i);
	    ports[i]->setCallId(callid);
	    portMutex.leaveMutex();
	    ports[i]->getName(buf);
	    slog(Slog::levelInfo) << buf << ": created" << endl;
	    return ports[i];
	}

    portMutex.leaveMutex();
    return NULL;
}

void SIPDriver::deleteTrunk(Trunk *trunk) 
{
    SIPTrunk *_trunk;
    char buf[33];

    if(!trunk)
	return;

    _trunk = (SIPTrunk*)trunk;
    _trunk->getName(buf);
    int tid = _trunk->getId();

    // we can only delete trunks that 
    // are in the idle state.
    if(!trunk->isReady()) {
	slog(Slog::levelError) << buf << ": cannot delete: busy" << endl;
	return;
    }

    portMutex.enterMutex();
    delete ports[tid];
    ports[tid] = NULL;
    portMutex.leaveMutex();

    slog(Slog::levelInfo) << buf << ": deleted" << endl;
}

bool SIPDriver::isTrunkClass(int id, const char *name)
{
    if(!stricmp(name, "sip"))
	return true;
    if(!stricmp(name, "rtp"))
	return true;
    return false;
}

/*void SIPDriver::notify(unsigned char id)
{
    return;
}
*/

SIPDriver sipivr;

#ifdef	CCXX_NAMESPACES
};
#endif
