// Copyright (C) 2000 Open Source Telecom Corporation.
// Copyright (C) 2004 Free Software Foundation
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
using namespace std;
namespace ost {
#endif

bool RTPTrunk::recordToFile(void)
{
    char buffer[32];
    Audio::Info info;
    AudioStreamFile *af;
    const char *ext;
    const char *fmt = data.record.encoding;

    getName(buffer);

    memset(&info, 0, sizeof(info));
    info.format = Audio::raw;
    info.encoding = Audio::pcm16Mono;
    info.order = 0;
    info.annotation = (char *)data.record.annotation;
    info.rate = 8000;

    ext = strrchr(data.record.name, '/');
    if(!ext)
	ext = data.record.name;
    ext = strrchr(ext, '.');
    if(!ext) 
	{
	    ext = data.record.extension;
	    strcat(data.record.name, ext);
	}

    if(!fmt)
	fmt = "pcm";
    if(!stricmp(ext, ".al"))
	fmt = "alaw";
    else if(!stricmp(ext, ".ul"))
	fmt = "mulaw";
    else if(!stricmp(ext, ".au") || !stricmp(ext, ".snd")) 
	{
	    if(!stricmp(fmt, "pcm"))
		fmt = "mulaw";
	    info.format = Audio::snd;
	    info.order = __BIG_ENDIAN;
	} 
    else if(!stricmp(ext, ".wav")) 
	{
	    info.format = Audio::riff;
	    info.order = __LITTLE_ENDIAN;
	}

    if(!stricmp(fmt, "pcm") || !stricmp(fmt, "l16") || !stricmp(fmt, "linear"))
	info.encoding = Audio::pcm16Mono;
    else if(!stricmp(fmt, "alaw"))
	info.encoding = Audio::alawAudio;
    else if(!stricmp(fmt, "g.711") || !stricmp(fmt, "g711")) 
	{
	    if(info.encoding != Audio::alawAudio)
		info.encoding = Audio::mulawAudio;
	} 
    else if(!stricmp(fmt, "ulaw") || !stricmp(fmt, "mulaw"))
	info.encoding = Audio::mulawAudio;
    else if(stricmp(fmt, "raw")) 
	{
	    slog(Slog::levelError) << buffer << ": unsupportd codec requested" << endl;
	    return false;
	}

    af = new AudioStreamFile(&info);

    if(data.record.offset != (unsigned long)-1) 
	{
	    af->open(data.record.name, Audio::modeWrite);
	    af->setPosition(data.record.offset);
	} 
    else if(data.record.append)
	af->open(data.record.name, Audio::modeWrite);
    else 
	{
	    remove(data.record.name);
	    af->create(data.record.name, &info);
	}

    if(!af->isOpen()) 
	{
	    slog(Slog::levelError) << data.record.name << ": cannot open" << endl;
	    delete af;
	    return false;
	}
    
    if(data.record.append)
	af->setPosition();
    
    enterMutex();
    recordFile = af;
    rtpStream->setSink(recordFile);
    leaveMutex();

    setTimer(data.record.timeout);

    return true;
}

    /*
     * When entering the record state, we set up the
     * destination file. Upon reception of a stop signal
     * or a DTMF digit, stop recording and clean up.
     * 
     * @param event Trunk Event that we are to handle
     */
bool RTPTrunk::recordHandler(TrunkEvent *event)
{
    char buffer[12];
    struct stat ino;
    int trim;
    AudioStreamFile *af;

    switch(event->id) 
	{
	case TRUNK_DTMF_KEYUP:
	    if(!getExitkey(data.record.term, event->parm.dtmf.digit))
		return false;
	    
	    // fall through
	case TRUNK_STOP_STATE:
	    handler = &RTPTrunk::stepHandler;
	    
	    enterMutex();
	    if(!recordFile) {
		setSymbol(SYM_RECORDED, "0");
		leaveMutex();
		trunkSignal(TRUNK_SIGNAL_STEP);
		return true;
	    }
	    
	    //
	    // stop recording, but keep a 
	    // copy of a pointer to recordFile
	    // around so we can clean up
	    //
	    rtpStream->setSink(NULL);
	    af = recordFile;
	    recordFile = NULL;
	    leaveMutex();
	    
	    //
	    // Save information about what we have recorded
	    // in trunk symbols, truncate if necessary, and
	    // clean up record state.
	    //
	    sprintf(buffer, "%ld", af->getPosition());
	    setSymbol(SYM_OFFSET, buffer);

	    if(data.record.minsize &&
	       (af->getPosition() < data.record.minsize)) 
		{
		    setSymbol(SYM_RECORDED, "0");
		    remove(data.record.name);
		    af->close();
		    delete af;
		    trunkSignal(TRUNK_SIGNAL_STEP);
		    return true;
		}
	    
	    trim = af->toBytes(af->getEncoding(), data.record.trim);
	    stat(data.record.name, &ino);
	    if(ino.st_size <= trim || data.record.frames) 
		{
		    remove(data.record.name);
		    setSymbol(SYM_RECORDED, "0");
		}
	    else 
		{
		    sprintf(buffer, "%ld",
			    af->getPosition() - data.record.trim);
		    trunk->setSymbol(SYM_RECORDED, buffer);
		    truncate(data.record.name, ino.st_size - trim);
		    if(data.record.save)
			rename(data.record.name, data.record.save);
		}

	    af->close();
	    delete af;

	    trunkSignal(TRUNK_SIGNAL_STEP);
	    return true;

	case TRUNK_ENTER_STATE:
	    enterState("record");
	    flags.dsp = DSP_MODE_VOICE;
	    status[tsid] = 'r';
	    if(data.record.text)
		cout << "*** " << data.record.text << endl;
	    if(data.record.term)
		setExitkey();
	    else
		setDTMFDetect();
	    
	    if(!rtpActive()) 
		{
		    trunkError("record-no-rtp");
		    trunkSignal(TRUNK_SIGNAL_STEP);
		    handler = &RTPTrunk::stepHandler;
		    return true;
		}

	    // The recordToFile method sets up the
	    // recording state -- prepares and opens
	    // the file, etc. If it fails, we return
	    // to the step handler -- it will have
	    // already called trunkError() with an
	    // appropriate message.
	    if(!recordToFile()) 
		{
		    trunkSignal(TRUNK_SIGNAL_STEP);
		    handler = &RTPTrunk::stepHandler;
		}

	    return true;
	}
    return false;
}

#ifdef	CCXX_NAMESPACES
};
#endif
