// Copyright (C) 2004 Free Software Foundation
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

//
// RTPStream::run(void)
// this is the main RTP engine, it exchanges
// data between the RTP peer and a file if
// the 
void RTPStream::run(void)
{
    size_t r;
    char buf[rtpBufferSize];

    // the timer has a resolution of miliseconds,
    // packetRate is in units of packets per second.
    // by default we are using 10pps, so frameTime
    // will be 100ms
    unsigned frameTime = 1000 / getPacketRate();
    setTimer(0);

    for(;;) {
	incTimer(frameTime);

	//
	// first send any packets that are
	// due to be sent
	//
	setCancel(cancelDeferred);
	enterMutex();
	if(source) {
	    source->getBuffer(buf, rtpBufferSize);
	    putBuffer(buf, rtpBufferSize);
	    r = dispatchDataPacket();
	    oBytes += r;
	}
	leaveMutex();
	setCancel(cancelImmediate);

	//
	// process any inbound packets
	// that may be waiting
	//
	// XXX possible DOS here? we could
	// get stuck here and never deliver
	// any packets...
	//
	while(isPendingData(0)) {
	    setCancel(cancelDeferred);
	    r = takeInDataPacket();
	    if(r < 0)
		return;
	    enterMutex();
	    int n = getBuffer(buf, r);
	    if(sink) {
		sink->putBuffer(buf, n);
	    }
	    leaveMutex();
	    setCancel(cancelImmediate);
	    iBytes += r;
	}
	// 
	// wait for the next cycle
	//
	Thread::sleep(getTimer());
    }
};

int RTPStream::putBuffer(void *buf, size_t len)
{
    static uint32 timestamp = 0;
    // do transcoding here!

    // This only work for G711
    int32 result = 20; /* 20ms of data */
    result *= (getCurrentRTPClockRate()/1000);
    if (timestamp == 0)
      timestamp = getCurrentTimestamp();
    timestamp += result;

    putData(timestamp, (const unsigned char *)buf, len);
    return len;
}


int RTPStream::getBuffer(void *buf, size_t len)
{
    const AppDataUnit *adu;
    size_t n;

    // do transcoding here!

    adu = getData(getFirstTimestamp());
    if(!adu)
	return 0;
    n = adu->getSize();
    if(n <= 0)
      {
        delete adu;
	return 0;
      }

    ::memcpy(buf, adu->getData(), n);
    delete adu;
    return n;
}

int RTPDTMFDetector::putBuffer(void *buf, size_t len) {
    bool dtmfFound = 0;
    static unsigned lastDtmf = 0;
    char digits[128] = {0};

    lastDtmf += len;
    dtmfFound = DTMFDetect::putSamples((int16_t *)buf, len/2);
    if(dtmfFound && lastDtmf > 1000) {
	getResult(digits, 64);
	slog << "read some dtmf digits: " << digits << endl;
	lastDtmf = 0;
    }
    if(sink)
	return sink->putBuffer(buf, len);
    return len;
};  

//
// RTPConfig class -- holds the configuration data
// for RTPTrunk
//
RTPConfig::RTPConfig() :
    Keydata("/drivers/rtp")
{
    static Keydata::Define defkeys [] = {
	{"rtpmin", "20000"},
	{"rtpmax", "21000"},
	{"rtpframesize", "1024"},
	{NULL, NULL}
    };
    load(defkeys);
};

//
// RTPTrunk class 
//
void RTPTrunk::initSyms(void)
{
    setConst(SYM_NETWORK, "none");
    setConst(SYM_INTERFACE, "rtp");
}

void RTPTrunk::getName(char *buffer)
{
    sprintf(buffer, "rtp/%d", id);
}

bool RTPTrunk::rtpStart(void)
{
    char buf[32];
    
    if(rtpStream) {
	trunkError("rtp-already-active");
	return false;
    }

    if(!localAddress || !remoteAddress || !remotePort) {
	trunkError("rtp-configure-me");
	return false;
    }

    //
    // XXX kludge -- this should be set by
    // the signalling driver that creates
    // the trunk
    //
    Audio::Info info;
    info.encoding = Audio::mulawAudio;
    info.rate = Audio::rate8khz;

    //
    // find an unused local port to use
    //
    localPort = getRTPMin();
    while(!(rtpStream = new RTPStream(this, &info, localAddress, localPort)) &&
	  localPort < getRTPMax())
	localPort++;
    if(!rtpStream) {
	trunkError("rtp-port-unavail");
	return false;
    }

    //
    // tell the rtp socket where to send the data
    //
    if(!rtpStream->addDestination(remoteAddress, remotePort)) {
	trunkError("rtp-dest-unavail");
	return false;
    }
    
    getName(buf);
    slog(Slog::levelDebug) << buf << ": " << localAddress<< ":" << localPort
			   << " --> " << remoteAddress << ":" << remotePort << endl;
    
    //
    // XXX TODO: payload format must be configurable
    // based on SDP and the defaults should live in the
    // config file
    //
    rtpStream->setPayloadFormat(StaticPayloadFormat(sptPCMU));

    // 
    // XXX should only install a DTMF detector if we 
    // are doing uncompressed audio
    //
    //dtmfDetector = new RTPDTMFDetector(this);
    //rtpStream->setSink(dtmfDetector);

    rtpStream->setMaxSendSegmentSize(getRTPFrameSize());
    rtpStream->rtpStart();

    return true;
}

void RTPTrunk::rtpStop(void)
{
    if(rtpStream) {
	char buf[33];

	//
	// if we are attached to files,
	// make sure to free them
	//
	if(playFile) {
	    rtpStream->setSource(NULL);
	    delete playFile;
	}
	if(dtmfDetector) {
	    rtpStream->setSink(recordFile);
	    delete dtmfDetector;
	}
	if(recordFile) {
	    rtpStream->setSink(NULL);
	    delete recordFile;
	}
	    
	//
	// perhaps there should be hooks here using the
	// thread on terminate/cleanup routine for more
	// detailed accounting, or else this information
	// should perhaps be made available to the CDR
	// code, but for now, just make sure we have something
	// in the log
	//
	getName(buf);
	slog(Slog::levelInfo) << buf << ": received " << rtpStream->iBytes << " octets" << endl;
	slog(Slog::levelInfo) << buf << ": sent " << rtpStream->oBytes << " octets" << endl;

	delete rtpStream;
	rtpStream = NULL;
    }
}

#ifdef	CCXX_NAMESPACES
};
#endif
