// Copyright (C) 2004 Free Software Foundation
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <ccrtp/rtp.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
#endif

class AudioStream;   
class AudioStreamFile;
class RTPStream;
class RTPDTMFDetector;
class RTPConfig;
class RTPTrunk;


/*
 * AudioStream integrates functionality from AudioBuffer
 * as well as AudioFile functionality so that it can
 * encode and decode the audio stream. The main reason
 * for this class existing is that the getBuffer and putBuffer
 * methods are not declared virtual in those two classes.
 *
 * @class AudioStream
 */

class AudioStream : public Audio
{
 protected:
    Info info;
 public:
    AudioStream(void) {};
    AudioStream(Info *inf)
	{ memcpy(&info, inf, sizeof(info)); };

    inline Encoding getEncoding(void)
	{return info.encoding;};
    
    inline unsigned getSampleRate(void)
	{return info.rate;};

    virtual int putBuffer(void *, size_t) = 0;
    virtual int getBuffer(void *, size_t) = 0;
};

/*
 * AudioStreamFile pulls in functionality from AudioFile
 * into an AudioStream derived class. This is so that we
 * can use instances of this interchangeably with RTPStream
 * defined below as sources and sinks of an RTP data stream.
 *
 * @class AudioStreamFile
 */
class AudioStreamFile : public AudioFile, public AudioStream
{
 public:
    //
    // this method is a kludge because AudioStream and
    // AudioFile both maintain Info data structures;
    // really AudioFile should be a subclass of AudioStream
    //
    AudioStreamFile(Info *inf) :
	AudioStream(inf),
	AudioFile(NULL, inf) {};
    AudioStreamFile(void) :
	AudioStream(),
	AudioFile() {};

    int putBuffer(void *buf, size_t len)
	{ return AudioFile::putBuffer(buf, len); };
    int getBuffer(void *buf, size_t len)
	{ return AudioFile::getBuffer(buf, len); };
    inline Encoding getEncoding(void)
	{ return AudioFile::getEncoding(); };
    inline unsigned getSampleRate(void)
	{ return AudioFile::getSampleRate(); };
};

/*
 * RTPStream class -- a subclass of RTPSession without
 * using RTCP, and integrating the putBuffer / getBuffer
 * functionality that comes from AudioStream.
 *
 * @class RTPStream
 */

class RTPStream : public RTPSession, public AudioStream, private TimerPort, private Mutex
{
 private:
    time_t startTime;
    RTPTrunk *trunk;
    unsigned packetRate;
    size_t rtpBufferSize;
    bool dropInbound;

    //
    // The reason that we make
    // source and sink of class AudioFile
    // rather than, say AudioBuffer is so
    // that we can connect files as well as
    // rtp streams
    //
    AudioStream *source;
    AudioStream *sink;

 public:
    uint32 iBytes;
    uint32 oBytes;
    uint32 oTimeStamp;

    RTPStream(RTPTrunk *trk, Info *inf, InetHostAddress addr, tpport_t lport) : 
	RTPSession(addr, lport),
	AudioStream(inf),
	TimerPort(),
	Mutex(),
	packetRate(50),
	dropInbound(false),
	trunk(trk),
	source(NULL),
	sink(NULL),
	iBytes(0),
	oBytes(0),
	oTimeStamp(0) {};

    void rtpStart(void) {
	//
	// Calculate the size of the buffer that
	// we should use for outbound packets
	//
	unsigned samples = getSampleRate() / getPacketRate();
	rtpBufferSize = toBytes(getEncoding(), samples);    

	//
	// Save the start time and start the
	// session
	//
	startTime = ::time(NULL); 
	RTPSession::startRunning();
    };

    //
    // These methods are overridden from the
    // AudioBuffer parent class
    // are used for doing soft joins between
    // trunks, as well as connecting files
    // to audio streams. They implement the
    // encoding and decoding of the streams.
    //
    int putBuffer(void *buf, size_t len);
    int getBuffer(void *buf, size_t len);

    // 
    // We also override this method since rtp
    // streams do not upport lseek(2) behaviour
    //
    bool hasPositioning(void)
	{ return false; };

    //
    // The main function of the thread that
    // reads and writes audio from/to the RTP
    // socket and takes care of the timing
    //
    void run(void);

    //
    // For an RTP audio stream, we have a configurable
    // packet rate, in packets per second. the default
    // is 10pps
    //
    int getPacketRate(void)
	{ return packetRate; };

    //
    // These two methods set the source and sink
    // respectively from and to which the RTP stream
    // will read and write data. An object used as
    // a source must implement something useful for
    // its getBuffer() method, and a sink must implement
    // putBuffer()
    //
    void setSource(AudioStream *src) {
	enterMutex();
	source = src;
	leaveMutex();
    };
    void setSink(AudioStream *dst) {
	enterMutex();
	sink = dst;
	leaveMutex();
    };

    // 
    // If we do not have a sink connected,
    // there is no sense processing incomming
    // packets, and this causes them to be 
    // dropped in that case.
    //
    bool onRTPPacketRecv(IncomingRTPPkt *pkt)
	{ return (sink ? true : false); }

};

/*
 * The RTPDTMFDetector daisy chains
 * to an AudioFile, it is useful with
 * RTPSession, where a typical chain might
 * look like:
 *
 *  rtpSession->dtmfDetector->audioFile
 *
 * so that dtmf events can be detected
 * while recording to an audio file
 *
 * This is currently not used.
 *
 * @class RTPDTMFDetector
 */
class RTPDTMFDetector : public DTMFDetect, public AudioStream, private Mutex
{
 private:
    AudioStream *sink;
 public:
    RTPDTMFDetector(Trunk *trk) :
	DTMFDetect(),
	AudioStream(),
	Mutex(),
	sink(NULL) {};

    int putBuffer(void *, size_t);
    int getBuffer(void *buf, size_t len)
	{ return 0; };

    void setSink(AudioStream *dst)
	{ enterMutex(); sink = dst; leaveMutex(); };

};

/*
 * The RTPConfig class holds configuration
 * data for the RTP driver.
 *
 * @class RTPConfig
 */
class RTPConfig : public Keydata
{
 public:
    RTPConfig(void);

 protected:
    tpport_t getRTPMin(void)
	{ return atoi(getLast("rtpmin")); };
    tpport_t getRTPMax(void)
	{ return atoi(getLast("rtpmax")); };
    size_t getRTPFrameSize(void)
	{ return atoi(getLast("rtpframesize")); };
};

/*
 * The RTPTrunk class is a mixin class intended to
 * be used primarily by bayonne device drivers. it
 * provides RTP audio facilities and should be useable
 * by SIP and MGCP drivers.
 *
 * @class RTPTrunk
 */
typedef bool (RTPTrunk::*trunkhandler_t)(TrunkEvent *event);

class RTPTrunk : protected RTPConfig, protected Trunk, protected AudioService, protected TimerPort
{
 private:
    InetHostAddress localAddress;
    InetHostAddress remoteAddress;
    tpport_t localPort;
    tpport_t remotePort;

    void initSyms(void);

 protected:
    trunkhandler_t handler;
    RTPStream *rtpStream;
    AudioStreamFile *playFile;
    AudioStreamFile *recordFile;
    RTPDTMFDetector *dtmfDetector;

 public:
    RTPTrunk(int ts, Driver *ivr) : 
	RTPConfig(), 
	Trunk(ts, ivr),
	TimerPort(),
	rtpStream(NULL),
	playFile(NULL),
	recordFile(NULL),
	dtmfDetector(NULL) {};

    ~RTPTrunk(void) {
	rtpStop();
    };

    /*
     * These are standard functions that
     * are expected to be implemented by
     * subclasses of Trunk.
     */

    /*
     * Should getName be implemented here? Or
     * in SIPTrunk? Should the trunk name be
     * rtp/0 or sip/0?
     */
    void getName(char *buffer);
    long unsigned int getIdleTime(void);
    void exit(void);
    void trunkStep(trunkstep_t step);
    bool postEvent(TrunkEvent *event);

    /* 
     * Before this class can be used,
     * it necessary to set the local
     * and remote port/address pairs.
     * the local port is not specified since
     * it is allocated dynamically within the
     * configured [rtpmin, rtpmax] range
     * @method setLocalAddress
     * @param ha -- the local address
     */
    void setLocalAddress(InetHostAddress ha)
	{ localAddress = ha; };
    /*
     * @method setRemoteAddress
     * @param ha -- the remote address
     * @param port -- the remote port
     */
    void setRemoteAddress(InetHostAddress ha, tpport_t port)
	{ remoteAddress = ha; remotePort = port; };

    /*
     * Since we determine what local port to
     * use based on our configured range and
     * the current open ports, we make available
     * the actual port that we are using through
     * this method
     *
     * @method getLocalPort
     */

    tpport_t getLocalPort(void)
	{ return localPort; };

    /*
     * This method creates the
     * appropriate RTP data structures
     * and threads for handling the audio
     * data. 
     *
     * @method rtpStart
     * @return true on success, false on failure
     */
    bool rtpStart(void);

    /*
     * This method does the inverse of rtpStart
     * cleaning up any RTP data structures
     * and threads that may be still active
     *
     * @method rtpStop
     */
    void rtpStop(void);

    /*
     * Determine whether or not the RTP 
     * socket is active
     *
     * @method rtpActive
     */
    bool rtpActive(void)
	{ return (rtpStream ? true : false); };

    /*
     * This method creates and initializes an AudioStreamFile
     * object according to the parameters in data.play, and
     * attaches it as a source to our rtpStream object. It
     * then sets the trunk timer to the expected length of 
     * time it will take the file to play.
     *
     * Upon error, this function calls trunkError with an
     * appropriate message and returns false.
     *
     * @method playFromFile
     */
    bool playFromFile(void);

    /*
     * This method is mostly to avoid clutter in the
     * recordHandler method. It creates and initializes
     * an AudioStreamFile instance according to the
     * parameters in data.record, and attaches it as
     * a sink to our rtpStream instance. It then sets
     * the trunk timer according to the timeout parameter
     * in data.record.
     *
     * In case of error, it will call trunkError() with 
     * an appropriate message, and return false.
     *
     * @method recordToFile
     */
    bool recordToFile(void);

    bool Join(Trunk *trk);
    void Part(void);

    //
    // These are the Bayonne event handlers
    // defined by the base RTPTrunk class
    //
    bool collectHandler(TrunkEvent *event);
    bool idleHandler(TrunkEvent *event);
    bool joinHandler(TrunkEvent *event);
    bool playHandler(TrunkEvent *event);
    bool playwaitHandler(TrunkEvent *event);
    bool recordHandler(TrunkEvent *event);
    bool ringHandler(TrunkEvent *event);
    bool sleepHandler(TrunkEvent *event);
    bool threadHandler(TrunkEvent *event);

    //
    // The step handler, the main workhorse
    // of the driver, must be implemented by
    // a subclass since the RTP driver doesn't
    // know anything about signalling -- and that
    // stuff is necessary in order to answer
    // and hang up calls for example. So we define
    // it as virtual here so that it can be 
    // referenced by the handlers that are implemented
    // directly by RTPTrunk
    //
    virtual bool stepHandler(TrunkEvent *event) = 0;
    virtual bool hangupHandler(TrunkEvent *event) = 0;
};

#ifdef	CCXX_NAMESPACES
};
#endif
