// Copyright (C) 2004 Free Software Foundation
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

    SIPUserAgent::SIPUserAgent(void) {
	registry = NULL;
	return;
    }

    SIPUserAgent::~SIPUserAgent(void) {
	uaStop();
	if(registry)
	    {
		for(int i = 0; i < registrySize; i++)
		    if(registry[i])
			delete registry[i];
		free(registry);
	    }
    };

    int SIPUserAgent::uaStart(tpport_t port) {
	bindPort = port;
	if(eXosip_init(NULL, NULL, bindPort)) 
	    return 1;

	/*
	osip_trace_initialize_syslog(TRACE_LEVEL0, "bayonne");
	osip_trace_enable_level(TRACE_LEVEL1);
	osip_trace_enable_level(TRACE_LEVEL2);
	osip_trace_enable_level(TRACE_LEVEL3);
	osip_trace_enable_level(TRACE_LEVEL4);
	osip_trace_enable_level(TRACE_LEVEL5);
	osip_trace_enable_level(TRACE_LEVEL6);
	osip_trace_enable_level(TRACE_LEVEL7);
	*/

	// reset all payload to fit application capabilities
	eXosip_sdp_negotiation_remove_audio_payloads();
	eXosip_sdp_negotiation_add_codec(::strdup("0"),
					 NULL,
					 ::strdup("RTP/AVP"),
					 NULL, NULL, NULL,
					 NULL, NULL,
					 ::strdup("0 PCMU/8000"));

	slog(Slog::levelInfo) << "sip: user agent bound to port " << bindPort << endl;
	return 0;
    };

    int SIPUserAgent::uaStop(void) {
        static int alreadyStopped = 0;	
	if (alreadyStopped==0)
	  {
	    alreadyStopped++;
	    eXosip_quit();
	    slog(Slog::levelInfo) << "sip: user agent unbound from port " << bindPort << endl;
	  }
	return 0;
    };

    void SIPUserAgent::uaRegister(void) {
	if(!registry)
	    return;

	for(int i = 0; i < registrySize; i++)
	    {
		if(!registry[i])
		    continue;
		if(registry[i]->getTimer())
		    continue;

		int expiry = registry[i]->getExpiry();
		eXosip_lock();
		eXosip_register(registry[i]->getId(), expiry);
		eXosip_unlock();
		
		// we reregister at half of the expiry interval
		// so that we always remain registered
		registry[i]->setTimer(expiry * 1000 / 2);
	    }
    }

    void SIPUserAgent::uaProcess(void) {
	eXosip_event_t *sevent;
	TrunkEvent event;

	while((sevent = eXosip_event_wait(0, 0))) {
	    Trunk *trunk = NULL;
	    SIPRegistry *r;
	    char *argv[16];
	    char buf[256];
	    char buf2[256];
	    char *cp, *bp;

	    slog(Slog::levelInfo) << "sip: got event (type, cid, did) = (" 
				  << sevent->type << ", " << sevent->cid << ", "
				  << sevent->did << ")" << endl;
	    switch(sevent->type) {
	    case EXOSIP_REGISTRATION_NEW:
		slog(Slog::levelInfo) << "sip: new registration" << endl;
		break;
	    case EXOSIP_REGISTRATION_SUCCESS:
		r = registryGet(sevent->rid);
		if(!r)
		    {
			slog(Slog::levelWarning) << "sip: registration success unknown id "
						 << sevent->rid << endl;
			break;
		    }
		// reset the authenticate state for our
		// next register attempt at the end of 
		// the expiry interval
		r->clrAuth();
		slog(Slog::levelInfo) << "sip: registration successful" << endl;
		break;
	    case EXOSIP_REGISTRATION_FAILURE:
		r = registryGet(sevent->rid);
		if(!r)
		    {
			slog(Slog::levelWarning) << "sip: registration failure unknown id " 
						 << sevent->rid << endl;
			break;
		    }
		// if we have tried an authenticated register,
		// fail until the next expiry interval
		if(r->getAuth())
		    {
			slog(Slog::levelInfo) << "sip: registration failed" << endl;
			r->clrAuth();
			break;
		    }
		// otherwise try to register again with authentication
		r->setAuth();
		r->setTimer(1);
		break;
	    case EXOSIP_REGISTRATION_REFRESHED:
		slog(Slog::levelInfo) << "sip: registration refreshed" << endl;
		break;
	    case EXOSIP_REGISTRATION_TERMINATED:
		slog(Slog::levelInfo) << "sip: registration terminated" << endl;
		break;
	    case EXOSIP_CALL_NEW:
		trunk = getTrunkUnused(sevent->cid);
		if(!trunk) {
		    slog(Slog::levelInfo) << "cannot get trunk!" << endl;
		    eXosip_lock();
		    eXosip_terminate_call(sevent->cid, sevent->did);
		    eXosip_unlock();
		    break;
		}

		// save the current dialog id
		// for use when we answer
		setTrunkDialogId(trunk, sevent->did);
		
		// set the local ip address to use
		eXosip_get_localip(buf);
		setTrunkLocalAddress(trunk, (InetHostAddress)buf);
		// set the remote rtp ip address and port
		setTrunkDestination(trunk,
				    (InetHostAddress)sevent->remote_sdp_audio_ip,
				    (tpport_t)sevent->remote_sdp_audio_port);

		// we have just allocated a new
		// trunk, so send it into the idle
		// state
		event.id = TRUNK_ENTER_STATE;
		trunk->postEvent(&event);

		trunk->getName(buf);
		slog(Slog::levelDebug) << buf << ": call to " << sevent->req_uri 
				       << " from " << sevent->remote_uri << endl;

		// extract the user part from 
		// sip:user@host:port
		memset(buf, 0, sizeof(buf));
                osip_uri_t *uri;
                osip_uri_init(&uri);
                if (osip_uri_parse(uri, sevent->req_uri)!=0)
                  {
                    strcpy(buf, "unknown404");
                    argv[0] = buf;
                    argv[1] = NULL;
                  }
                else
                  {
                    if (uri->username==NULL)
                      {
                        strcpy(buf, "unknown404");
                        argv[0] = buf;
                        argv[1] = NULL;
                      }
		    else
                      {
                        snprintf(buf, 255, "%s", uri->username);
                        argv[0] = buf;
                        argv[1] = NULL;
                      }
                  }
                osip_uri_free(uri);

                // Look For the Caller Information
		memset(buf2, 0, sizeof(buf2));
                osip_from_t *from;
                osip_from_init(&from);
                if (osip_from_parse(from, sevent->remote_uri)!=0)
                  {
                    strcpy(buf2, "unknown");
                    argv[1] = buf2;
                    argv[2] = NULL;
                  }
                else
                  {
                    if (from->url->username==NULL)
                      {
                        strcpy(buf2, "unknown");
                        argv[1] = buf2;
                        argv[2] = NULL;
                      }
                    else
                      {
                        snprintf(buf2, 255, "%s", from->url->username);
                        argv[1] = buf2;
                        argv[2] = NULL;
                      }
                  }
                osip_from_free(from);

		// start the trunk ringing...
		event.id = TRUNK_RING_START;
		event.parm.argv = argv;
		if(!trunk->postEvent(&event)) 
		{
		    // this means the attach() has 
		    // failed. send the calling party
		    // a 404 error. eXosip will give us
		    // a RELEASE event later where we will
		    // tear down the trunk.
		    eXosip_lock();
		    eXosip_answer_call(sevent->did, 404, NULL);
		    eXosip_unlock();
	            break;
		}
		eXosip_lock();
		eXosip_answer_call(sevent->did, 180, NULL);
		eXosip_unlock();
		break;
	    case EXOSIP_CALL_ACK:
		// the ack we have recieved
		// from our OK response to the
		// invite
		trunk = getTrunkCallId(sevent->cid);
		if(!trunk) {
		    slog(Slog::levelWarning) << "something's very wrong. cannot find the trunk for this call" << endl;
		    eXosip_lock();
		    eXosip_terminate_call(sevent->cid, sevent->did);
		    eXosip_unlock();
		    break;
		}
		break;
	    case EXOSIP_INFO_NEW:
		trunk = getTrunkCallId(sevent->cid);
		if(!trunk) {
		    slog(Slog::levelWarning) << "something's terribly wrong. recieved an INFO event for a nonexistant call" << endl;
		    eXosip_lock();
		    eXosip_terminate_call(sevent->cid, sevent->did);
		    eXosip_unlock();
		    break;
		}

		//
		// we are looking for DTMF digits sent as
		// INFO messages
		//
		//if(!sevent->payload)
		//  break;
		if(!strcasecmp(sevent->i_ctt->type, "application") &&
		   !strcasecmp(sevent->i_ctt->subtype, "dtmf-relay") &&
		   !osip_list_eol(sevent->i_bodies, 0)) {
		    char digit[10];
		    char duration[10];
		    memset(&digit, 0, 10);
		    memset(&duration, 0, 10);
		    char *k, *v, *l, *line;

		    //
		    // retrieve the message body from the event
		    // structure, and make a local copy that we
		    // can munge
		    //
		    osip_body_t *mbody = (osip_body_t *)osip_list_get(sevent->i_bodies, 0);
		    osip_body_t *_mbody;

		    if(!mbody)
			break;

		    if(0 != osip_body_clone(mbody, &_mbody))
			break;

		    char tmpbuf[strlen(mbody->body)];

		    //
		    // SIP INFO message bodies look like:
		    // Signal=N
		    // Duration=S
		    // Where N is the number that the user has
		    // pressed. So we parse out the N, and create
		    // a DTMF event for the trunk;
		    line = _mbody->body;
		    while(line && *line) {
			k = v = line;
			// search for the value, after the equal
			// sign
			while(*v && (*v != '=') && (*v != '\r') && (*v != '\n'))
			    v++;
			// if we are at the end of the buffer or
			// the line, break -- parse error
			if(!*v || (*v == '\n') || (*v == '\r'))
			    break;

			// bypass spaces
			v++;
			while(*v && (*v == ' ') && (*v != '\r') && (*v != '\n'))
			    v++;

			// if we are at the end of the buffer or
			// the line, break -- parse error
			if(!*v || (*v == '\n') || (*v == '\r'))
			    break;

			// WE MUST NOT CHANGE THE body: // *v++ = 0;

			// now we have the left hand side in k
			// and the right hand side in v, find
			// th end of the line;
			l = v;
			while(*l && *l != ' ' && *l != '\n' && *l != '\r')
			    l++;
			if(!*l)
			    break;

			if(!strncasecmp(k, "signal", 6))
			    strncpy(digit, v, l-v);
			else if(!strncasecmp(k, "duration",8))
			    strncpy(duration, v, l-v);


			// WE MUST NOT CHANGE THE body: // *l++ = 0;
			l++;
			while(*l && (*l == ' ' || *l == '\n' || *l == '\r'))
			    l++;

			line = l;
                        if(!*line)
                            break;
		    };
		    if(!digit || !duration) {
			slog(Slog::levelWarning) << "sip: got an unparseable INFO message" << endl;
			osip_body_free(_mbody);
			break;
		    }

	    
		    event.id = TRUNK_DTMF_KEYUP;
		    switch(*digit) {
		    case '0':
		    case '1':
		    case '2':
		    case '3':
		    case '4':
		    case '5':
		    case '6':
		    case '7':
		    case '8':
		    case '9':
			event.parm.dtmf.digit = *digit - '0';
			event.parm.dtmf.duration = atoi(duration);
			slog(Slog::levelInfo) << "sip got digit from INFO: " << event.parm.dtmf.digit << " duration: " << duration << endl;
			trunk->postEvent(&event);
			break;
		    case '*':
			event.parm.dtmf.digit = 10;
			event.parm.dtmf.duration = atoi(duration);
			slog(Slog::levelInfo) << "sip got digit from INFO: " << event.parm.dtmf.digit << " duration: " << duration << endl;
			trunk->postEvent(&event);
			break;
		    case '#':
			event.parm.dtmf.digit = 11;
			event.parm.dtmf.duration = atoi(duration);
			slog(Slog::levelInfo) << "sip got digit from INFO: " << event.parm.dtmf.digit << " duration: " << duration << endl;
			trunk->postEvent(&event);
			break;
		    default:
		      slog(Slog::levelWarning) << "sip got unknown digit in INFO: " << "xxx" << digit << "xxx" << endl;
			break;
		    }
		}
		break;
	    case EXOSIP_CALL_CLOSED: /* BYE */
		trunk = getTrunkCallId(sevent->cid);
		if(!trunk) {
		    slog(Slog::levelWarning) << "sip: got BYE for non-existant call" << endl;
		    break;
		}
		// set the dialogId to 0 so that
		// the hangup handler knows that
		// we did not cause the hangup
		setTrunkDialogId(trunk, 0);
		trunk->trunkSignal(TRUNK_SIGNAL_HANGUP);
		break;
	    case EXOSIP_CALL_RELEASED:
		trunk = getTrunkCallId(sevent->cid);
		if(!trunk) {
		    slog(Slog::levelWarning) << "SIPClient got release event fr non-existant call" << endl;
		    break;
		}
		deleteTrunk(trunk);
		break;
	    default:
		slog(Slog::levelInfo) << "unhandled eXosip event" << endl;
	    }
	    eXosip_event_free(sevent);
	}
    };

    bool SIPUserAgent::registryInit(int nitems)
	{ 
	    registry = (SIPRegistry **)malloc(nitems * sizeof(SIPRegistry *));
	    if(!registry)
		return false;
	    memset(registry, 0, nitems * sizeof(SIPRegistry *));
	    registrySize = nitems;
	    return true;
	}

    bool SIPUserAgent::registryAdd(const char *from, 
				   const char *proxy,
				   const char *contact,
				   int expiry)
	{
	    bool ret;
	    int regid;
	    char *f;
	    char *p;
	    char *c = NULL;
	    int i;

	    // sanity check our arguments
	    if(!from || !proxy)
		return false;

	    // if we don't have a registry, don't continue
	    if(!registry)
		return false;

	    // check that there is room in the registry
	    // and find an empty slot
	    for(i = 0; i < registrySize; i++)
		if(!registry[i])
		    break;
	    if(i == registrySize)
		return false;

	    // the eXosip routine wants a char * and
	    // we have been passed a const so we have
	    // to make copies of our arguments
	    f = strdup(from);
	    if(!f)
		return false;

	    p = strdup(proxy);
	    if(!p)
		{
		    free(f);
		    return false;
		}

	    if(contact)
		{
		    c = strdup(contact);
		    if(!c)
			{
			    free(p);
			    free(f);
			    return false;
			}
		}

	    // tell eXosip about this registration,
	    // send the initial registration and 
	    // set up our timer
	    eXosip_lock();
	    regid = eXosip_register_init(f, p, c);
	    if(regid < 0)
		{
		    ret = false;
		}
	    else
		{
		    ret = (eXosip_register(regid, expiry) < 0 ? false : true);
		    registry[i] = new SIPRegistry(regid, expiry);
		}
	    eXosip_unlock();
			
	    /*
	      if(c)
		free(c);
	      free(p);
	      free(f);
	    */

	    return ret;
	};

    void SIPUserAgent::answer(int did, int code, tpport_t port) {
	char buf[16];
	memset(buf, 0, sizeof(buf));
	snprintf(buf, sizeof(buf), "%d", port);
	eXosip_lock();
	eXosip_answer_call(did, code, buf);
	eXosip_unlock();
    }

    void SIPUserAgent::hangup(int cid, int did) {
	eXosip_lock();
	eXosip_terminate_call(cid, did);
	eXosip_unlock();
    }

#ifdef	CCXX_NAMESPACES
};
#endif
