// Copyright (C) 2004 Free Software Foundation
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <eXosip/eXosip.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
#endif

class SIPUserAgent;

/*
 * The SIPUserAgent class is a class providing
 * access to the eXosip User Agent API.
 *
 * @class SIPUserAgent
 */ 
class SIPUserAgent
{
 private:
    InetAddress bindAddr;
    tpport_t bindPort;

    /* 
     * SIPRegistry is a helper class that
     * manages the registration expiry and
     * reregistration. Not intended for use
     * outside of the SIPUserAgent class
     */
    class SIPRegistry : public TimerPort
	{
	private:
	    int regId;
	    int regExpiry;
	    bool authFlag;
	public:
	    SIPRegistry(int id, int expiry) :
		TimerPort(),
		regId(id),
		regExpiry(expiry),
		authFlag(false)
		{ 
		    setTimer(expiry * 1000 / 2);
		};
	    int getId(void)
		{ return regId; };
	    int getExpiry(void)
		{ return regExpiry; };
	    bool getAuth(void)
		{ return authFlag; };
	    void setAuth(void)
		{ authFlag = true; };
	    void clrAuth(void)
		{ authFlag = false; };
	};
    SIPRegistry **registry;
    int registrySize;
    SIPRegistry *registryGet(int id)
	{
	    if(!registry)
		return NULL;
	    for(int i=0; i < registrySize; i++)
		if(registry[i] && (registry[i]->getId() == id))
		    return registry[i];
	    return NULL;
	}
 public:
    SIPUserAgent(void);
    ~SIPUserAgent(void);

    /*
     * The uaStart method starts the eXosip process, 
     * which runs in a thread of its own
     * 
     * @method uaStart
     * @param port The UDP Port to listen on
     * @return 0 on success, non-zero on error
     */
    int uaStart(tpport_t port);
    /*
     * The uaStop method stops the eXosip process
     *
     * @method uaStop
     * @return 0 on success, non-zero on error.
     */
    int uaStop(void);

    /*
     * The uaRegister method sends any registration
     * requests whose timers have expired
     */
    void uaRegister(void);

    /*
     * The uaProcess method processes any events 
     * that are currently pending eXosip
     *
     * @method uaProcess
     */
    void uaProcess(void);

 protected:
    /*
     * The following functions are pure virtual and so
     * they must be implemented by our subclasses.
     */

    /*
     * getTrunkUnused returns an new trunk, and assign 
     * it the given call identification. this is the 
     * callid used by eXosip and bears no relation to
     * the trunk id used by Bayonne.
     *
     * @method getTrunkUnused
     * @param callid
     * @return a newly allocated trunk
     */
    virtual Trunk *getTrunkUnused(int callid) = 0;
    /*
     * getTrunkCallId eturns a trunk indexed by id as 
     * previously allocated by getTrunkUnused()
     * 
     * @method getTrunkCallId
     * @param callid
     * @return trunk
     */
    virtual Trunk *getTrunkCallId(int callid) = 0;
    /*
     * A trunk is associated with a dialogid whicn may
     * or may not change over the course of a call.
     * a trunk implementation may chose to interpret
     * a dialog id of 0 to mean that the remote end
     * has hung up on us, and should not call our 
     * hangup() method.
     *
     * @method setTrunkDialogId
     * @param trunk -- the trunk in question
     * @param did -- the dialog id
     */
    virtual bool setTrunkDialogId(Trunk *trunk, int did) = 0;
    /*
     * setTrunkLocalAddress provides us the ability to 
     * set the local ip address that the trunk should use
     * for our end of the RTPStream. This is typically
     * implemented by the RTPStream class from which
     * our child class SIPTrunk inherits 
     * 
     * @method setTrunkLocalAddress
     * @param trunk -- the trunk in question
     * @param ha -- the local addres
     */
    virtual bool setTrunkLocalAddress(Trunk *trunk, 
				      InetHostAddress ha) = 0;
    /*
     * setTrunkDestination provides us with the ability to
     * set the destination of the RTP stream. This is, as with
     * setTrunkLocalAddress, typically implemented by the
     * RTPStream class.
     *
     * @method setTrunkDestination
     * @param trunk -- the trunk in question
     * @param da -- the destination addres
     * @param dp -- the destination port
     */
    virtual bool setTrunkDestination(Trunk *trunk, 
				     InetHostAddress da, 
				     tpport_t dp) = 0;

    /*
     * deleteTrunk should be used to indicate that we are not
     * longer in need of the trunk and it and any resources
     * associated with it should be freed by the low level
     * driver.
     *
     * @method deleteTrunk
     * @param trunk -- the trunk to delete
     */
    virtual void deleteTrunk(Trunk *trunk) = 0;


    /*
     * These functions are provided by the SIPUserAgent
     * class in order to allow configuration of our 
     * behaviour.
     */

    /*
     * Set the user agent string that we are to 
     * present ourselves as
     *
     * @method setUserAgent
     * @param uaString
     */
    void setUserAgent(const char *uaString)
	{ eXosip_set_user_agent(uaString); };

    /*
     * Set the local address that we are to use for
     * signalling and RTP data. Otherwise it will
     * be guessed by the eXosip library
     *
     * @method setLocalAddress
     * @param ha
     */
    void setLocalAddress(InetHostAddress ha)
	{ eXosip_force_localip(ha.getHostname()); };

    /*
     * Set the outside address of the firewall that
     * we are using to facilitate NAT traversal.
     *
     * @method setFirewallAddress(InetHostAddress ha)
     * @param ha
     */
    void setFirewallAddress(InetHostAddress ha)
	{ eXosip_set_firewallip(ha.getHostname()); };

    /*
     * Add authentication information
     *
     * @method addAuthenticationInfo
     * @param username
     * @param passwd
     * @param realm
     */
    int addAuthenticationInfo(const char *username,
			       const char *passwd,
			       const char *realm)
	{ return eXosip_add_authentication_info(username, 
						username, 
						passwd, 
						NULL, 
						realm); };

    /*
     * Initialize our SIP registration list
     *
     * @returns true on success, false on failure
     */
    bool registryInit(int nitems);

    /*
     * Add an entry to our SIP registration list
     *
     * @method registryAdd
     * @param from
     * @param proxy
     * @param contact
     * @param expires
     */
    bool registryAdd(const char *from, const char *proxy, 
		     const char *contact, int expiry);

 public:
    /*
     * The following methods are implemented as calls to
     * the eXosip library and are exposd for our sub-classes
     * or any other classes as necessary.
     */

    /*
     * Answer a call with the given response code.
     * The optional argument port is used if we are
     * answering successfully (200 response code),
     * and it gets filled in in the SDP body to tell
     * the remote end where it should send the data
     * stream.
     *
     * @method answer
     * @param did -- dialog id
     * @param code -- SIP response code
     * @param port -- optional local RTP port
     */
    void answer(int did, int code, tpport_t port=0);

    /*
     * hang up on a call (eXosip_terminate_call)
     *
     * @method hangup
     * param cid -- call id
     * param did -- dialog id
     */
    void hangup(int cid, int did);
};

#ifdef	CCXX_NAMESPACES
};
#endif
