// Copyright (C) 2000 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

VirtualConfig::VirtualConfig() :
Keydata("/drivers/virtual")
{
	static Keydata::Define defkeys[] = {
	{"count", "16"},
	{"stack", "32"},
	{NULL, NULL}};

	load(defkeys);
}

size_t VirtualConfig::getStack(void)
{
	const char *cp = getLast("stack");

	if(!cp)
		return keythreads.getStack();

	return atoi(cp) * 1024;
}

VirtualDriver::VirtualDriver() :
Driver(), Thread(keythreads.priService() + 1, virtivr.getStack()), PortNotify()
{
	const char *cp = getLast("steptimer");

	if(cp)
		step_timer = atol(cp);
	else
		step_timer = keythreads.getStepDelay();

	port_count = atoi(getLast("count"));
	if(port_count > 240)
		port_count = 240;
	ports = new VirtualTrunk *[port_count];
	memset(ports, 0, sizeof(VirtualTrunk *) * port_count);
	slog(Slog::levelInfo) << "Virtual device driver loaded; capacity=" << port_count << endl;
}

VirtualDriver::~VirtualDriver()
{
	if(active)
		stop();

	if(ports)
		delete[] ports;
}

void VirtualDriver::run(void)
{
	VirtualTrunk *trunk;
	TrunkEvent event;
	timeout_t timer, expires;
	unsigned port;
	active = true;
	setCancel(cancelDeferred);

	for(;;)
	{
		timer = ~0;
		for(port = 0; port < port_count; ++port)
		{
			// in case we later make dynamic
			if(ports[port] == NULL)
				continue;

			trunk = ports[port];
			expires = trunk->getTimer();
			if(expires > 0)
				if(expires < timer)
					timer = expires;

			if(!expires)
			{
				event.id = TRUNK_TIMER_EXPIRED;
				trunk->endTimer();
				trunk->postEvent(&event);
			}
		}
		port = fetch(timer);
		switch(port)
		{
		case 0xff:
			break;
		case 0x00:
			Thread::sync();
		default:
			notify(port);
		}
		Thread::yield();
	}
}

int VirtualDriver::start(void)
{
//	unsigned ts;

	if(active)
	{
		slog(Slog::levelError) << "driver already started" << endl;
		return 0;
	}

	slog(Slog::levelInfo) << "virtual driver starting..." << endl;

//	for(ts = 0; ts < port_count; ++ts)
//		ports[ts] = new VirtualTrunk(ts);

	Thread::start();
	active = true;
	return port_count;
}

void VirtualDriver::stop(void)
{
	unsigned id;

	if(!active)
		return;

	active = false;

	if(running)
	{
		update(0);
		terminate();
	}

	if(ports && port_count)
	{
		for(id = 0; id < port_count; ++id)
		{
			if(ports[id])
				delete ports[id];
		}
	}
	memset(ports, 0, sizeof(VirtualTrunk *) * port_count);
	slog(Slog::levelInfo) << "driver stopping service thread(s)" << endl;
}

Trunk *VirtualDriver::getTrunkPort(int id)
{
	if(id < 0 || (unsigned)id >= port_count)
		return NULL;

	return (Trunk *)ports[id];
}

Trunk *VirtualDriver::getOutboundTrunk(int id)
{
	if(id < 0 || (unsigned)id >= port_count)
		return NULL;

	if(!ports || ports[id])
		return NULL;

	ports[id] = new VirtualTrunk(id);

	return (Trunk *)ports[id];
}

bool VirtualDriver::isTrunkClass(int id, const char *name)
{
	if(!stricmp(name, "virtual"))
		return true;

	return false;
}

void VirtualDriver::notify(unsigned char id)
{
	TrunkEvent event;
	VirtualTrunk *trunk;

	if(id < 1 || id > port_count)
		return;

	trunk = ports[--id];
	if(!trunk)
		return;

	event.id = TRUNK_NOTIFICATION;
	trunk->postEvent(&event);
}

VirtualDriver virtivr;

#ifdef	CCXX_NAMESPACES
}
#endif
