// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

VirtualTrunk::VirtualTrunk(int ts) :
TimerPort(), Trunk(ts, &virtivr)
{
	const char *cp = virtivr.getLast("steps");
	TrunkEvent event;
	handler = &VirtualTrunk::idleHandler;
	event.id = TRUNK_ENTER_STATE;

	if(cp)
		steps = atoi(cp);

	slog(Slog::levelInfo) << "virtual/" << id << ": generic driver started" << endl;
	(this->*handler)(&event);
	if(usage)
		setUsage(&usage->virtuse);
}

VirtualTrunk::~VirtualTrunk()
{
	handler = NULL;

	endTimer();
	
	setHookState(false);
	slog(Slog::levelInfo) << "virtual/" << id << ": device stopped" << endl;
}

void VirtualTrunk::getName(char *buffer)
{
	sprintf(buffer, "virtual/%d", id);
}

void VirtualTrunk::exit(void)
{
	if(!flags.onexit)
		if(redirect("::exit"))
		{
			flags.onexit = true;
			return;
		}

	handler = &VirtualTrunk::hangupHandler;
}

bool VirtualTrunk::postEvent(TrunkEvent *event)
{
	bool rtn = true;
	trunkhandler_t prior;
	char evt[65];

	enterMutex();
	switch(event->id)
	{
	case TRUNK_TIMER_EXIT:
		if(!exittimer)
			rtn = false;
		exittimer = 0;
		break;
	case TRUNK_TIMER_SYNC:
		if(!synctimer)
			rtn = false;
		synctimer = 0;
		break;
	case TRUNK_TIMER_EXPIRED:
		if(!getTimer())
			rtn = false;
		break;
	case TRUNK_DTMF_KEYUP:
		if(flags.offhook)
			time(&idletime);
		if(!flags.dtmf)
			rtn = false;
		break;
	default:
		break;
	}	
	if(!rtn)
	{
		leaveMutex();
		return false;
	}

	if(handler == (trunkhandler_t)NULL)
	{
		slog(Slog::levelWarning) << "virtual/" << id;
		slog() << ": no handler active; event=" << event->id << endl;
		leaveMutex();
		return false;
	}

retry:
	debug->debugEvent(this, event);
	prior = handler;
	rtn = (this->*handler)(event);
	if(rtn)
	{
		if(handler != prior)
		{
			if(prior == &VirtualTrunk::idleHandler)
				setIdle(false);
			event->id = TRUNK_ENTER_STATE;
			goto retry;
		}
		leaveMutex();
		return true;
	}

	// default handler

	rtn = true;
	switch(event->id)
	{
	case TRUNK_RINGING_ON:
		++rings;
		snprintf(evt, sizeof(evt), "ring:%d", rings);
		if(Trunk::event(evt))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		break;
	case TRUNK_ENTER_STATE:
		if(flags.offhook)
			setDTMFDetect();
		else
			setDTMFDetect(false);
		endTimer();
		break;
	case TRUNK_LINE_WINK:
		if(Trunk::event("line:wink"))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		if(!flags.offhook)
			break;
		goto drop;
	case TRUNK_CPA_DIALTONE:
	case TRUNK_STOP_DISCONNECT:
drop:
		if(flags.onexit)
			break;
		if(trunkSignal(TRUNK_SIGNAL_HANGUP))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		break;
	case TRUNK_SEND_MESSAGE:
		if(recvEvent(event))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		break;
	case TRUNK_TIMER_EXPIRED:
		trunkSignal(TRUNK_SIGNAL_TIMEOUT);
		event->id = TRUNK_STOP_STATE;
		goto retry;
		break;
        case TRUNK_TIMER_SYNC:
                if(trunkSignal(TRUNK_SIGNAL_TIME))
                {
                        event->id = TRUNK_STOP_STATE;
                        goto retry;
                }
                break;
        case TRUNK_TIMER_EXIT:
                if(trunkSignal(TRUNK_SIGNAL_TIME))
                        event->id = TRUNK_STOP_STATE;
                else
                        event->id = TRUNK_STOP_DISCONNECT;
                goto retry;

	case TRUNK_SYNC_PARENT:
		if(Trunk::event(event->parm.sync.msg))
		{
			setSymbol(SYM_STARTID, event->parm.sync.id);
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		rtn = false;
		break;
       case TRUNK_SYNC_NOTIFY:
                if(isActive())
                        ++counts;
                break;

	case TRUNK_CHILD_EXIT:
		if(!isActive())
			break;
		if(trunkSignal(TRUNK_SIGNAL_CHILD))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		break;
	case TRUNK_DTMF_KEYUP:
		if(digits < MAX_DIGITS)
			dtmf.bin.data[digits++] = digit[event->parm.dtmf.digit];
		dtmf.bin.data[digits] = 0;
		snprintf(evt, sizeof(evt), "digits:%s", dtmf.bin.data);
		if(Trunk::event(evt))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		if(trunkSignal((trunksignal_t)(event->parm.dtmf.digit + TRUNK_SIGNAL_0)))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		break;
	case TRUNK_EXIT_SHELL:
                if(event->parm.exitpid.seq != tgi.seq)
                        break;
                tgi.pid = 0;
		break;
	case TRUNK_STOP_STATE:
		endTimer();
		handler = &VirtualTrunk::stepHandler;
		break;
	case TRUNK_EXIT_STATE:
		break;
	case TRUNK_MAKE_IDLE:
		handler = &VirtualTrunk::hangupHandler;
		break;
	default:
		rtn = false;
	}
	if(handler != prior)
	{
		event->id = TRUNK_ENTER_STATE;
		goto retry;
	}
	leaveMutex();
	return rtn;
}

void VirtualTrunk::setTimer(timeout_t ptimer)
{
	TimerPort::setTimer(ptimer);
	virtivr.update();
}

void VirtualTrunk::incTimer(timeout_t ptimer)
{
	TimerPort::incTimer(ptimer);
	virtivr.update();
}

unsigned long VirtualTrunk::getIdleTime(void)
{
	time_t now;

	time(&now);
	if(handler == &VirtualTrunk::idleHandler)
		return (unsigned long)(now - idletime);

	return 0;
}

void VirtualTrunk::setHookState(bool offhook)
{
	flags.offhook = offhook;
}

#ifdef	CCXX_NAMESPACES
}
#endif
