// Copyright (C) 2000 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"
#include <fcntl.h>

#ifdef  CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

WSSConfig::WSSConfig() :
Keydata("/drivers/wss")
{
	static Keydata::Define defkeys[] = {
	{"count", "1"},
	{"buffers", "8"},
	{"stack", "64"},
	{NULL, NULL}};

	load(defkeys);
}

size_t WSSConfig::getStack(void)
{
        const char *cp = getLast("stack");

        if(!cp)
                return keythreads.getDriverStack();

        return atoi(cp) * 1024;
}

size_t WSSConfig::getAudioStack(void)
{
	const char *cp = getLast("audiostack");

	if(!cp)
		return keythreads.getAudioStack();

	return atoi(cp) * 1024;
}

WSSDriver::WSSDriver() :
Driver(), WSSConfig(), Thread(keythreads.priService(), wssivr.getStack())
{
	port_count = getDeviceCount();

	ports = new WSSTrunk *[port_count];

	if(ports)
		memset(ports, 0, sizeof(WSSTrunk *) * port_count);

	slog(Slog::levelInfo) << "Generic Test (wss) driver loaded; capacity=" << port_count << endl;
}

WSSDriver::~WSSDriver()
{
	if(active)
		stop();

	if(ports)
	{
		delete ports;
		ports = NULL;
	}
}

int WSSDriver::start(void)
{
	int count = 0;

	if(active)
	{
		slog(Slog::levelError) << "driver already started" << endl;
		return 0;
	}


	for(count = 0; count < getDeviceCount(); ++count)
        	ports[count] = new WSSTrunk(count);

	slog(Slog::levelInfo) << "driver starting..." << endl;
	Thread::start();

	active = true;
	return count;
}

void WSSDriver::stop(void)
{
	unsigned count;
	WSSTrunk *trk;

	if(!active)
		return;

	active = false;

	if(running)
	{
		update(0);
		terminate();
	}

	if(ports)
	{
		for(count = 0; count < (unsigned)getDeviceCount(); ++count)
		{
			trk = ports[count];
			if(trk)
				delete trk;
		}
		memset(ports, 0, sizeof(WSSTrunk *) * port_count);
	}

	slog(Slog::levelInfo) << "driver stopping..." << endl;
}

void WSSDriver::run(void)
{
	TrunkEvent event;
	timeout_t timeout;
	char ch;
	WSSTrunk *trunk;
	int port = 0;
	char name[16];
	HANDLE hInput;
	DWORD count;

	setCancel(cancelImmediate);
	slog(Slog::levelNotice) << "wss: start thread" << endl;

	hInput = GetStdHandle(STD_INPUT_HANDLE);

	for(;;)
	{
		Thread::yield();
		trunk = ports[port];
		if(!trunk)
		{
			slog(Slog::levelError) << "wss: missing trunk" << endl;
			sleep(1000);
			continue;
		}

		snprintf(name, sizeof(name), "sound/%d", port);

		trunk->enterMutex();
		timeout = trunk->getTimer();
		if(!timeout)
		{
			trunk->endTimer();
			event.id = TRUNK_TIMER_EXPIRED;
			trunk->postEvent(&event);
		}
		trunk->leaveMutex();

		if(timeout > 500)
			timeout = 500;

		count = WaitForSingleObject(hInput, timeout); 
		switch(count)
		{
		case WAIT_OBJECT_0:
			count = 1;
			ReadConsole(hInput, &ch, 1, &count, NULL);
			break;
		default:
			count = 0;
		}
		if(!count)
			continue;
		switch(ch)
		{
		case '#':
			slog(Slog::levelDebug) << name << ": digit #..." << endl;
			event.id = TRUNK_DTMF_KEYUP;
			event.parm.dtmf.digit = 11;
			event.parm.dtmf.duration = 60;
			trunk->postEvent(&event);
			sleep(10);
			break;
		case '*':
			slog(Slog::levelDebug) << name << ": digit *..." << endl;
			event.id = TRUNK_DTMF_KEYUP;
			event.parm.dtmf.digit = 10;
			event.parm.dtmf.duration = 60;
			trunk->postEvent(&event);
			sleep(10);
			break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			slog(Slog::levelDebug) << name << ": digit " << ch << "..." << endl;
			event.id = TRUNK_DTMF_KEYUP;
			event.parm.dtmf.digit = ch - '0';
			event.parm.dtmf.duration = 60;
			trunk->postEvent(&event);
			sleep(10);
			break;
		case 'r':
		case 'R':
			slog(Slog::levelDebug) << name << ": ringing..." << endl;
			event.id = TRUNK_RINGING_ON;
			event.parm.ring.digit = 0;
			event.parm.ring.duration = 50;
			trunk->postEvent(&event);
			sleep(100);
			event.id = TRUNK_RINGING_OFF;
			trunk->postEvent(&event);
			break;
		case 'n':
		case 'N':
			if(++port >= getTrunkCount())
				port = 0;
			break;
		case 'D':
		case 'd':
			slog(Slog::levelDebug) << name << ": dialtone..." << endl;
			event.id = TRUNK_CPA_DIALTONE;
			trunk->postEvent(&event);
			break;
		case 'B':
		case 'b':
			slog(Slog::levelDebug) << name << ": busytone..." << endl;
			event.id = TRUNK_CPA_BUSYTONE;
			trunk->postEvent(&event);
			break;
		case ' ':
			slog(Slog::levelDebug) << name << ": step/start..." << endl;
			event.id = TRUNK_NULL_EVENT;
			trunk->postEvent(&event);
			break;
		case 'H':
		case 'h':
			slog(Slog::levelDebug) << name << ": hangup..." << endl;
			event.id = TRUNK_STOP_DISCONNECT;
			trunk->postEvent(&event);
			break;
		case 'i':
		case 'I':
			slog(Slog::levelDebug) << name << ": idle..." << endl;
			event.id = TRUNK_MAKE_IDLE;
			trunk->postEvent(&event);
			break;
		case 't':
		case 'T':
			if(trunk->trace)
			{
				slog.debug() << name << ": trace off" << endl;
				trunk->trace = false;
			}
			else
			{
				slog.debug() << name << ": trace on" << endl;
				trunk->trace = true;
			}
			break;
		}
	}
}

bool WSSDriver::isTrunkClass(int id, const char *name)
{
	if(!stricmp(name, "sound"))
		return true;

	return false;
}

Trunk *WSSDriver::getTrunkPort(int id)
{
	if(id < 0 || id >= port_count)
		return NULL;

	if(!ports)
		return NULL;

	return (Trunk *)ports[id];
}

unsigned WSSDriver::getCaps(void)
{
	return capSpeed | capDynamic;	// prevents outbound
}

WSSDriver wssivr;

#ifdef	CCXX_NAMESPACES
};
#endif
