// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifndef	CCXX_BAYONNE_H_
#define	CCXX_BAYONNE_H_

#ifndef	CCXX_BAYONNESCRIPT_H_
#include <cc++/bayonnescript.h>
#endif

#ifndef	CCXX_BAYONNESYMBOLS_H_
#include <cc++/bayonnesymbols.h>
#endif

#ifdef	COMMON_XML_PARSING
#ifndef	CCXX_XML_H_
#include <cc++/xml.h>
#endif
#endif

#ifndef	CCXX_URL_H_
#include <cc++/url.h>
#endif

#ifndef	CCXX_SLOG_H_
#include <cc++/slog.h>
#endif

#ifndef	CCXX_DSO_H_
#include <cc++/file.h>
#endif

#ifndef	CCXX_SOCKET_H_
#include <cc++/socket.h>
#endif

#ifndef	CCXX_BUFFER_H_
#include <cc++/buffer.h>
#endif

#ifndef	CCXX_BAYONNEAUDIO_H_
#include <cc++/bayonneaudio.h>
#endif

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <climits>

// PIPE_BUF is the atomic i/o size for the pipe.  This ultimately is used to represent
// the maximum size of a tgi command buffer.

#ifndef	PIPE_BUF
#define	PIPE_BUF	512
#endif

#ifdef	__FreeBSD__
#undef	read
#undef	write
#undef	readv
#undef	writev
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
#endif

/* Bayonne uses a universal event record for all driver plugins and
   state handlers.  Not all drivers impliment the entire set of events
   and some drivers have specific events associated with their unique
   characteristcs.  However, most events are considered "universal".
*/

class __EXPORT Trunk;
class __EXPORT TrunkImage;
class __EXPORT Service;
class __EXPORT phTone;
class __EXPORT TrunkGroup;
class __EXPORT AudioBuffer;
class __EXPORT Conference;

#define	MAX_DIGITS	48	// maximum dtmf receive digit buffer size
#define MAX_NAME_LEN	64

struct callrec_t;

// This is used to represent the trunk select policy, if trunks should be selected
// by first to last, or last to first, as per bayonne.conf.

typedef enum
{
	SELECT_FIRST,
	SELECT_LAST
} seltype_t;


// This specifies the mode of a channel join, whether it is full duplex, or in a specific
// half duplex mode.

typedef enum
{
	JOIN_RECV,
	JOIN_XMIT,
	JOIN_FULL
} joinmode_t;

// This is used to specify the current dialing mode being used, particularly for drivers
// that use softdial support.

typedef enum
{
	PULSE_DIALER,	// pulse dial mode
	DTMF_DIALER,	// dtmf tone dialing
	MF_DIALER	// mf tone dialing
} dialmode_t;

// When a script command requires the driver to be in a specific processing state to
// perform a blocking operation, it sends a trunkStep() request to the driver.  Different
// trunkstep requests put the driver into different processing states, which then fall
// back to scripting when the state completes.

typedef enum {
	// step change requests
	TRUNK_STEP_HANGUP = 0,
	TRUNK_STEP_SLEEP,
	TRUNK_STEP_ACCEPT,
	TRUNK_STEP_REJECT,
	TRUNK_STEP_ANSWER,
	TRUNK_STEP_COLLECT,
	TRUNK_STEP_PLAY,
	TRUNK_STEP_PLAYWAIT,
	TRUNK_STEP_RECORD,
	TRUNK_STEP_TONE,
	TRUNK_STEP_DIALXFER,
	TRUNK_STEP_SOFTDIAL,
	TRUNK_STEP_FLASH,
	TRUNK_STEP_JOIN,
	TRUNK_STEP_LISTEN,
	TRUNK_STEP_RTP,
	TRUNK_STEP_DUPLEX,
	TRUNK_STEP_DETECT,
	TRUNK_STEP_REQUIRES,
	TRUNK_STEP_THREAD,
	TRUNK_STEP_SENDFAX,
	TRUNK_STEP_RECVFAX,
	TRUNK_STEP_ENTER,	// enter conference resource
	TRUNK_STEP_SOFTJOIN,
	TRUNK_STEP_DRIVER,
	TRUNK_STEP_EXIT = TRUNK_STEP_HANGUP
} trunkstep_t;

// Trunk signals are used by drivers to notify scripting that a specific ^xxx handler
// is needed based on a driver state event.  This is used to define the specific
// requests and event branch points.  The "TRUNK_SIGNAL_STEP" request is special, in that
// it simply requests processing to resume at the next script statement.

typedef	enum {
	// notify script from state handler
	TRUNK_SIGNAL_STEP = 0,				// branch to next script step

	TRUNK_SIGNAL_EXIT,				// ask script to exit
	TRUNK_SIGNAL_HANGUP=TRUNK_SIGNAL_EXIT,
	TRUNK_SIGNAL_ERROR,				// ^error handler
	TRUNK_SIGNAL_TIMEOUT,				// ^timeout handler
	TRUNK_SIGNAL_DTMF,

	TRUNK_SIGNAL_0,
	TRUNK_SIGNAL_1,
	TRUNK_SIGNAL_2,
	TRUNK_SIGNAL_3,

	TRUNK_SIGNAL_4,
	TRUNK_SIGNAL_5,
	TRUNK_SIGNAL_6,
	TRUNK_SIGNAL_7,

	TRUNK_SIGNAL_8,
	TRUNK_SIGNAL_9,
	TRUNK_SIGNAL_STAR,
	TRUNK_SIGNAL_POUND,

	TRUNK_SIGNAL_A,
	TRUNK_SIGNAL_OVERRIDE = TRUNK_SIGNAL_A,
	TRUNK_SIGNAL_B,
	TRUNK_SIGNAL_FLASH = TRUNK_SIGNAL_B,
	TRUNK_SIGNAL_C,
	TRUNK_SIGNAL_IMMEDIATE = TRUNK_SIGNAL_C,
	TRUNK_SIGNAL_D,
	TRUNK_SIGNAL_PRIORITY = TRUNK_SIGNAL_D,

	TRUNK_SIGNAL_SILENCE,
	TRUNK_SIGNAL_BUSY,
	TRUNK_SIGNAL_CANCEL,
	TRUNK_SIGNAL_FAIL = TRUNK_SIGNAL_CANCEL,
	TRUNK_SIGNAL_INVALID = TRUNK_SIGNAL_CANCEL,
	TRUNK_SIGNAL_NOTIFY,

	TRUNK_SIGNAL_NOANSWER,
	TRUNK_SIGNAL_RING,
	TRUNK_SIGNAL_ANSWER = TRUNK_SIGNAL_RING,
	TRUNK_SIGNAL_PICKUP = TRUNK_SIGNAL_RING,
	TRUNK_SIGNAL_TONE,
	TRUNK_SIGNAL_EVENT,

	TRUNK_SIGNAL_TIME,
	TRUNK_SIGNAL_MAXTIME = TRUNK_SIGNAL_TIME,
	TRUNK_SIGNAL_CHILD,
	TRUNK_SIGNAL_DRIVER,

	TRUNK_SIGNAL_GOTO = 65			// special goto completion request,
						// used by states that process labels.
}	trunksignal_t;

// Each channel receives a generic event with an id as defined from this type whenever
// the low level api or device driver needs to notify the Bayonne driver of some
// functional change in line state, or whenever the application wishes to generate
// events to feed into the Bayonne driver state machines.

typedef enum {
	// primary state handlers

	TRUNK_ENTER_STATE = 100,// newly entered handler state
	TRUNK_EXIT_STATE,	// exiting prior state (unused)
	TRUNK_STOP_STATE,	// request state termination
	TRUNK_NOTIFICATION,	// death notify event
	TRUNK_SERVICE_SUCCESS,	// service completion successful
	TRUNK_SERVICE_FAILURE,	// service completion failed
	TRUNK_SERVICE_LOGIN,	// login transaction
	TRUNK_SIGNAL_TEXT,	// unsolicited text event message
	TRUNK_SEND_MESSAGE,	// event message ipc
	TRUNK_JOIN_TRUNKS,	// join two trunks together
	TRUNK_PART_TRUNKS,	// split two trunks that were joined
	TRUNK_NULL_EVENT,	// used to push pipe driven systems
	TRUNK_CHILD_START,	// notify parent of child startup
	TRUNK_CHILD_FAIL,	// notify parent child start failed
	TRUNK_CHILD_EXIT,	// notify child channel died
	TRUNK_SIGNAL_JOIN,	// send join notification (^event)
	TRUNK_FAX_EVENT,	// fax event
	// tgi/integration control state handlers


	TRUNK_EXIT_SHELL = 200,	// tgi completion event
	TRUNK_START_DIAL,	// start with a dial command
	TRUNK_START_SCRIPT,	// start of script
	TRUNK_RING_START,	// smdi/integrated answer
	TRUNK_RING_REDIRECT,	// smdi/integrated answer options
	TRUNK_STOP_DISCONNECT,	// integrated hangup notification
	TRUNK_SHELL_START,	// fifo shell request
	TRUNK_WAIT_SHELL,	// report wait
	TRUNK_ASR_START,	// asr startup notification
	TRUNK_ASR_TEXT,		// asr text message
	TRUNK_ASR_PARTIAL,	// asr partial text message
	TRUNK_ASR_VOICE,	// asr voice detect
	TRUNK_SYNC_PARENT,	// parent notification
	TRUNK_SYNC_NOTIFY,	// semaphore count

	// in the future these will be used

	TRUNK_START_INCOMING = TRUNK_RING_START,
	TRUNK_START_OUTGOING = TRUNK_START_SCRIPT,

	// primary "mode" selection controls

	TRUNK_MAKE_TEST =  300,	// request driver perform line test
	TRUNK_MAKE_BUSY,	// request driver lockout line
	TRUNK_MAKE_IDLE,	// request driver reset line
	TRUNK_MAKE_STEP,	// pass step event internally
	TRUNK_MAKE_STANDBY,	// standby mode for carrier grade events

	// basic trunk events

	TRUNK_LINE_WINK = 400,	// used for line disconnect notification
	TRUNK_TIMER_EXPIRED,	// driver specific port timer expired
	TRUNK_TIMER_EXIT,
	TRUNK_TIMER_SYNC,
	TRUNK_RINGING_ON,	// some drivers distinguish start/stop
	TRUNK_RINGING_OFF,	// default ring event
	TRUNK_TEST_IDLE,	// some drivers have line test completion
	TRUNK_TEST_FAILURE,	// some drivers notify errors
	TRUNK_ON_HOOK,		// some drivers notify on hook
	TRUNK_OFF_HOOK,		// some drivers notify off hook
	TRUNK_CALLER_ID,	// caller id parse request
	TRUNK_RINGING_DID,	// did digit ring signal
	TRUNK_CALL_DETECT,	// ISDN call detected notification
	TRUNK_CALL_CONNECT,	// ISDN call connection notification
	TRUNK_CALL_RELEASE,	// ISDN call release notification
	TRUNK_CALL_ACCEPT,	// ISDN incoming call accepted
	TRUNK_CALL_ANSWERED,	// ISDN connect sent to the network
	TRUNK_CALL_HOLD,	// ISDN call placed on hold
	TRUNK_CALL_NOHOLD,	// ISDN call hold was rejected
	TRUNK_CALL_DIGITS,	// requested digits received
	TRUNK_CALL_OFFER,	// ISDN call offered
	TRUNK_CALL_ANI,		// ANI received
	TRUNK_CALL_ACTIVE,	// ISDN call taken off hold
	TRUNK_CALL_NOACTIVE,	// ISDN call hold retrieve failed
	TRUNK_CALL_BILLING,	// ISDN call billing acknowledge
	TRUNK_CALL_RESTART,	// ISDN call restart, success or failure
	TRUNK_CALL_SETSTATE,	// ISDN acknowledge state change
	TRUNK_CALL_FAILURE,	// ISDN midcall failure
	TRUNK_CALL_ALERTING,	// ISDN call alerting
	TRUNK_CALL_INFO,	// ISDN call info message
	TRUNK_CALL_BUSY,	// ISDN conjestion message
	TRUNK_CALL_DIVERT,	// ISDN call diversion notification
	TRUNK_CALL_FACILITY,	// ISDN call facility
	TRUNK_CALL_FRAME,	// ISDN call frame
	TRUNK_CALL_NOTIFY,	// ISDN call notify
	TRUNK_CALL_NSI,		// ISDN call nsi message
	TRUNK_CALL_RINGING,	// digital T1 incoming call
	TRUNK_CALL_DISCONNECT,	// digital T1 circuit break
	TRUNK_DEVICE_OPEN,	// device open
	TRUNK_DEVICE_CLOSE,	// device close
	TRUNK_DEVICE_BLOCKED,	// channel blocked
	TRUNK_DEVICE_UNBLOCKED,	// channel unblocked

	// basic audio processing events

	TRUNK_AUDIO_IDLE = 500,	// audio reset or completion event
	TRUNK_INPUT_PENDING,	// some drivers monitor audio i/o status
	TRUNK_OUTPUT_PENDING,	// some drivers monitor audio i/p status
	TRUNK_AUDIO_BUFFER,	// some drivers return audio buffers
	TRUNK_TONE_IDLE,	// tone generator completion event
	TRUNK_DTMF_KEYDOWN,	// some drivers distinguish tone down
	TRUNK_DTMF_KEYUP,	// default dtmf event
	TRUNK_TONE_START,	// tone detected
	TRUNK_TONE_STOP,	// some drivers have tone completion event
	TRUNK_VOX_DETECT,	// speaker detected
	TRUNK_VOX_SILENCE,	// silence detected
	TRUNK_AUDIO_START,	// some drivers may "vox" compress
	TRUNK_AUDIO_STOP,	// some drivers may "vox" compress
	TRUNK_CPA_DIALTONE,	// dialtone heard on the line
	TRUNK_CPA_BUSYTONE,
	TRUNK_CPA_RINGING,
	TRUNK_CPA_RINGBACK = TRUNK_CPA_RINGING,
	TRUNK_CPA_INTERCEPT,
	TRUNK_CPA_NODIALTONE,
	TRUNK_CPA_NORINGBACK,
	TRUNK_CPA_NOANSWER,
	TRUNK_CPA_CONNECT,
	TRUNK_CPA_FAILURE,
	TRUNK_CPA_GRUNT,
	TRUNK_CPA_REORDER,
	TRUNK_DSP_READY,	// dsp resource became available
	TRUNK_CPA_STOPPED,

	// basic station processing event extensions

	TRUNK_START_RINGING = 600,	// ring for incoming lines
	TRUNK_START_TRANSFER,		// transfer mode ring
	TRUNK_START_INTERCOM,		// intercom mode ring
	TRUNK_START_RECALL,		// transfer recall mode ring
	TRUNK_START_DIALING,		// initiate outgoing call
	TRUNK_STOP_RINGING,		// cancel a ring request
	TRUNK_STATION_OFFHOOK,
	TRUNK_STATION_ONHOOK,
	TRUNK_STATION_FLASH,
	TRUNK_STATION_ANSWER,		// answering ringing port
	TRUNK_STATION_PICKUP,		// pickup foreign port
	TRUNK_STATION_CONNECT,		// auto-answer connect

	// driver specific events and anomolies

	TRUNK_DRIVER_SPECIFIC=8000	// very oddball events
} trunkevent_t;

// This is most often associated with drivers that dynamically allocate dsp resources as
// needed on demand.  This is used to indicate which dsp resource has currently been
// allocated for the port.

typedef enum
{
	DSP_MODE_INACTIVE = 0,	// dsp is idle
	DSP_MODE_VOICE,		// standard voice processing
	DSP_MODE_CALLERID,	// caller id support
	DSP_MODE_DATA,		// fsk modem mode
	DSP_MODE_FAX,		// fax support
	DSP_MODE_TDM,		// TDM bus with echo cancellation
	DSP_MODE_RTP,		// VoIP full duplex
	DSP_MODE_DUPLEX,	// mixed play/record
	DSP_MODE_JOIN,		// support of joined channels
	DSP_MODE_CONF,		// in conference
	DSP_MODE_TONE		// tone processing
} dspmode_t;

// This specifies the mode of the trunk port, particularly in regard to cdr logging,
// whether it is being used for an incoming or outgoing call, or is available, or has
// been taken out of service.

typedef enum
{
	TRUNK_MODE_INCOMING = 0,	// this channel is receiving a call
	TRUNK_MODE_OUTGOING,		// this channel is placing a call
	TRUNK_MODE_INACTIVE,		// this channel is idle
	TRUNK_MODE_UNAVAILABLE		// this channel is not in service
} trunkmode_t;

// This is used to represent statistical data that is collected by Bayonne.  Each
// trunk group can collect it's own statistical data, which can be used to compute
// traffic engineering, busy hour info for acd agents, etc.

typedef enum
{
	STAT_MAX_INCOMING,
	STAT_MAX_OUTGOING,
	STAT_TOT_INCOMING,
	STAT_TOT_OUTGOING,
	STAT_ACTIVE_CALLS,
	STAT_NOW_INCOMING,
	STAT_NOW_OUTGOING,
	STAT_SYS_INCOMING,
	STAT_SYS_OUTGOING,
	STAT_SYS_UPTIME,
	STAT_SYS_ACTIVITY,
	STAT_CURRENT_CALLS,
	STAT_CMAX_INCOMING,
	STAT_CMAX_OUTGOING,
	STAT_AVAIL_CALLS
} statitem_t;

// This is used to determine how each driver's play command will work.

typedef	enum
{
	PLAY_MODE_NORMAL,	// play audio normally
	PLAY_MODE_ONE,		// play only first audio file found
	PLAY_MODE_ANY,		// play any (all) audio files found
	PLAY_MODE_TEMP,		// play and delete temporary audio file
	PLAY_MODE_TEXT,		// perform text to speech processing for this play
	PLAY_MODE_FILE,		// perform text to speech from an input file
	PLAY_MODE_MOH,		// play in music on hold mode
	PLAY_MODE_NONE		// cache synth only
} playmode_t;

// This is actually not used, but may be used later to determine the file write mode
// for a record operation

typedef enum
{
	WRITE_MODE_NONE,	// no write
	WRITE_MODE_REPLACE,	// replace original
	WRITE_MODE_APPEND,	// append to original
	WRITE_MODE_INITIAL	// write if intial only
} writemode_t;

// This is used for tgi based text to speech support

typedef enum
{
	TTS_GATEWAY_TEXT,
	TTS_GATEWAY_FILE
} ttsmode_t;

// Each trunk port has a capability flag which indicates what features that port
// supports.

#define TRUNK_CAP_VOICE		0x00000001	// supports voice telephone calls
#define	TRUNK_CAP_DIAL		0x00000002	// supports dialing
#define TRUNK_CAP_SENDFAX	0x00000004	// supports sending of faxes
#define	TRUNK_CAP_RECVFAX	0x00000008	// supports receipt of faxes
#define	TRUNK_CAP_DATA		0x00000010	// supports fsk modem
#define	TRUNK_CAP_TTS		0x00000020	// supports text to speech directly
#define	TRUNK_CAP_ASR		0x00000040	// supports asr (listen command)
#define	TRUNK_CAP_STATION	0x00000080	// is pots to plugin telephone
#define TRUNK_CAP_TIE		0x00000160	// is a tie line
#define TRUNK_CAP_TRUNK		0x00002000	// is a trunk port

// This structure holds the executive data for managing a tgi session for each channel

typedef	struct
{
	bool dtmf;		// whether dtmf is enabled for this tgi
	int pid;		// process id of tgi process
	unsigned short seq;	// sequence check value to make sure not stale
	void *data;		// unused?
}	execdata_t;

// Used to indicate audio playback speed.

typedef enum
{
	SPEED_FAST,
	SPEED_SLOW,
	SPEED_NORMAL
}	trunkspeed_t;

// This is a special "data" block that is embedded in each channel.  The content of this
// data block depends on which state the driver is currently processing (stepped) into.
// Hence, different driver states each use different representions of this state config
// data.

typedef	union
{
	// when in driver answer state

	struct
	{
		unsigned rings;		// rings to answer on
		timeout_t timeout;	// maximum wait time
		const char *transfer;	// id to transfer to if transfering answer
		Trunk *intercom;	// intercom for transfering answer
		const char *station;	// if answer to fax tone, station id
		const char *fax;	// fax branch script vector
	}	answer;

	// to be used in future fax state

	struct
	{
		char pathname[256];
		const char *station;	// fax station id
	}	fax;

	// used when in playing state

	struct
	{
		char list[256];		// a buffer to store comma seperated prompt list
		char *name;		// a pointer into the list
		const char *extension;	// file extension if passed in play command
		unsigned long offset;	// offset from start of file
		unsigned long limit;	// maximum bytes into file
		unsigned short term;	// keys that can directly terminate play
		playmode_t mode;	// mode of play
		writemode_t write;		// write mode for any output
		timeout_t timeout, maxtime;	// maximum play time
		timeout_t position;		// audio position timemark
		unsigned repeat;		// number of repeats
		unsigned volume;		// play volume
		float gain, pitch;		// pitch and gain control
		trunkspeed_t speed;		// speed of playback
		const char *voice;		// voice library override
		const char *text;		// text alt tag for driver
		const char *cache;		// cache to use
		bool lock;			// lock flag
		unsigned channel;		// channel id where supported
	}	play;

	// processing data block for record command

	struct
	{
		char *name, *save;		// filename to record, and optionally
						// save under if successful
		const char *encoding;		// audio encoding mode
		const char *annotation;		// file annotation to embed
		const char *extension;		// file extension to use
		const char *text;		// alt tag text for driver
		timeout_t timeout;		// max record time
		timeout_t position;		// new time mark
		unsigned long offset;		// starting offset in file
		unsigned short term;		// dtmf keys to exit record
		unsigned long silence;
		unsigned long trim;		// samples to trim from end
		unsigned long minsize;		// minimum valid file size
		unsigned volume;		// record volume and gain
		float gain;
		short frames;			// frames to buffer if rotating
		bool append;			// flag to append to existing file
		bool info;
		char filepath[65];		// buffer for full filename
		char savepath[65];		// buffer for save name if used
		char altinfo[128];
	}	record;

	// dial command processing

	struct
	{
		char digits[65];		// number to dial
		char *digit;			// a pointer into digits
             	char *callingdigit;
		bool exit;			// hangup on completion flag
		dialmode_t dialer;		// dialing mode
		timeout_t interdigit;		// delay between digits (tone)
		timeout_t digittimer;		// timer for pulse dialing
		timeout_t timeout;		// wait time for call progress
		timeout_t offhook;		// hook time for flash
		timeout_t onhook;		// hook time for flase
		unsigned pulsecount;		// counter for pulse dialing
		bool sync;			// syncflag
	}	dialxfer;

	// collect handling

	struct
	{
		timeout_t timeout, first;	// interdigit and first digit timeout
		unsigned count;			// total digits to collect
		unsigned short term;		// digits which are terminating
		unsigned short ignore;		// digits to ignore
		void *map;			// map table, unused
		void *var;			// var to save results into if given
	}	collect;

	// asr listen state handling

	struct
	{
		timeout_t first;		// delay for first word
		timeout_t next;			// interword delay
		unsigned short term, count;	// word count
		float gain, pitch;		// gain control
		unsigned volume;		// volume control
		void *save;			// var to save results into
		char *wordlist[32];		// optional terminating wordlist
		char buffer[256];		// buffer for wordlist
	}	listen;

	// sleep state handling

	struct
	{
		timeout_t wakeup;		// wakeup timer
		unsigned rings;			// wakeup on ring n
		unsigned loops;			// loop
		const char *save;		// var to save tgi results
		unsigned short term;		// termination mask
		unsigned count;			// semaphore style wait
	}	sleep;

	// tone state handling

	struct
	{
		timeout_t wakeup, duration;	// tone and silence duration
		unsigned loops;			// number of times to repeat
		phTone *tone;			// tone entry if from config
		unsigned freq1, freq2;		// tone definition if manually set
		int ampl1, ampl2;		// tone definition if manually set
		Trunk *dialing;			// used when intercom dialing
		bool recall;			// used when intercom dialing
	}	tone;
	struct
	{
		Trunk *src;
		const char *msg;
		unsigned seq;
	}	send;

	// join state handling to join calls

	struct
	{
		timeout_t wakeup, maxwait;	// join wait times
		bool hangup;			// hangup if join is done?
		joinmode_t direction;		// full or half duplex?
		Trunk *trunk, *waiting;		// trunk to wait for
		phTone *tone;			// tone while waiting
		float inpgain, outgain;		// gain control pads
		char *recfn;			// record file if logged
		const char *encoding;		// encoding if logged
		const char *annotation;		// annotation if logged
		const char *extension;		// extension if logged
		unsigned count;			// retry count to sync
		unsigned seq;			// sequence check
		trunkevent_t reason;
		bool local;
		time_t	start;			// start of join
	}	join;

	// enter state handling for conference resource

	struct
	{
		Conference *resource;		// resource entered
		timeout_t maxtime;		// max time in conference
		unsigned volume;		// volume control
		float input, output;		// gain controls
	}	enter;

	// xml load state handler

	struct
	{
		TrunkImage *image;		// image buffer to transcode into
		const char *url;		// url being requested
		const char *section;		// section to start at
		const char *parent;		// parent that requested
		const char *fail;		// fail label to branch to
		const char *database;		// used by sql driver
		char **vars;			// vars to pass
		bool post, attach, gosub;
		timeout_t timeout;		// maximum runtime before giving up
		char userid[64];		// user logged in as
		char filepath[256];		// work buffer
	}	load;

	// intercom state handling

	struct
	{
		Trunk *answer;			// answering for
		const char *transfer;		// transfering to
	}	intercom;
}	trunkdata_t;

// When posting an event to a trunk port, each event id also has optional data that
// may be passed based on the event id type.  The different trunk event data items
// are shown here:

typedef struct
{
	// event id

	trunkevent_t id;	// event id

	// data block based on event id

	union
	{
		// used for dtmf events from low level device

		struct
		{
			unsigned digit: 4;		// dtmf digit recieved
			unsigned duration: 12;		// duration of digit
			unsigned e1: 8;			// energy level of tones
			unsigned e2: 8;
		} dtmf;

		// used for tone detect from low level driver

		struct
		{
			unsigned tone: 8;
			unsigned energy: 8;		// energy level of tone
			unsigned duration: 16;		// duration of tone
			char *name;			// name of tone
		} tone;
	
		// used by events that pass intertrunk messages which must be
		// sequence checked to confirm.  Usually send command.

		struct
		{
			unsigned seq;		// sequence id to confirm
			Trunk *src;		// source making request
			const char *msg;	// text message
		} send;

		// used by low level driver to notify inbound ring

		struct
		{
			unsigned digit:  4;	// optional id if ring cadence supported
			unsigned duration: 24;	// duration of ring
		} ring;

		// used to process results of lookup requests

		struct
		{
			unsigned seq;
			bool result;
			char *data;
		} lookup;

		// used to perform intercom transfer

		struct
		{
			unsigned tid;		// original port 
			const char *transfer;	// session id being transfered
		} intercom;

		// used by some fax code

		struct
                {
                        int type;    
                        int param0;
                        int param1;
                        int channel;
                } fax;

		// used by tgi when exiting

		struct
		{
			unsigned status;	// process status result
			unsigned short seq;	// sequence check to validate
		}	exitpid;

		// used by tgi to notify started

		struct
		{
			int pid;		// process id of new tgi process
			unsigned short seq;	// sequence value
		}	waitpid;

		struct
		{
			const char *msg;
			const char *id;
		}	sync;

		trunkevent_t reason;		// generic reason code
		unsigned span;			// span related event, span id
		unsigned card;			// card related event, card id
		unsigned tid;			// port related event, port id
		bool ok;			
		int status, fd, pid;
		Trunk *trunk;			// inter-trunk related, ref trunk
		void *data;
		char **argv;			// argument list
		char *error;	
		timeout_t duration;		// time related
		trunkstep_t step;		// step request
		char dn[8];
		dspmode_t dsp;
	} parm;
} TrunkEvent;

#pragma pack(1)

// The status node is used to report and share Bayonne line states when Bayonne servers
// form a global call state map.  It is also reported to Bayonne site monitoring tools.

typedef	struct {
	time_t	update;			// update time (live) of last message
	char name[16];			// name of node
	struct in_addr addr;		// ip address of node
	unsigned long uptime;		// total uptime of node
	unsigned long calls;		// total calls processed for node
	unsigned char version;		// protocol version id
	unsigned char buddies;		// number of failover buddies elected
	unsigned char spansize;		// number of spans
	unsigned short ports;		// total port count
	unsigned char service;		// service level flag
	unsigned char dialing;	
	char schedule[16];		// current scheduled mode
	char stat[960];			// port statistics
}	statnode_t;

#pragma pack()

/* This is used to bind user defined "functions" which may be loaded
   in a DSO module.
*/

__EXPORT bool getLogical(const char *string);
__EXPORT timeout_t getSecTimeout(const char *string);
__EXPORT timeout_t getMSTimeout(const char *string);
__EXPORT timeout_t getTimePosition(const char *string);
class __EXPORT UsageStat;

/**
 * A call statistic class is used to manipulate call stats with a mutex
 * lock to prevent errors during "adjustment" periods.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short call statistic collection object.
 */
class __EXPORT CallStat : public Mutex
{
protected:
	static unsigned long upincoming;
	static unsigned long upoutgoing;
	static time_t uptime;
	volatile bool dirty;

	time_t updated, prior, idled;

	int capacity;
	struct
	{
		int incoming;
		int outgoing;
	}	active, max, lastmax;

	struct
	{
		long incoming;
		long outgoing;
	}	total, lasttotal;

public:
	CallStat();

	/**
	 * Get current call capacity.
	 *
	 * @return capacity of this stat item.
	 */
	inline int getCapacity(void)
		{return capacity;};

	/**
	 * get a stat item.
	 *
	 * @param statitem to request.
	 * @return item value.
	 */
	long getStat(statitem_t item);

	/**
	 * Get a stat record at once.
	 *
	 * @param pointer to stats to copy.
	 */
	void getStat(unsigned long *save);

	/**
	 * inc active incoming call count.
	 */
	void incIncoming(void);

	/**
	 * dec active incoming call count.
	 */
	void decIncoming(void);

	/**
	 * inc active outging call count.
	 */
	void incOutgoing(void);

	/**
	 * dec active outgoing call count.
	 */
	void decOutgoing(void);

	/**
	 * Get idle time in seconds.
	 */
	long getIdleTime();

	/**
	 * Update stats, active to last.
	 */
	void update(void);
};

/**
 * Phrasebook modules are used to convert things like numbers into
 * sets of prompts that form spoken words.  Translations are dso based
 * and are represented by the "languages" plugin.  A translator for
 * spanish may handle not just numbers, dates, etc, but also the
 * selection of masculine and feminine forms of numbers based on
 * usage, etc.  The translator classes provide some basic mechanics
 * for phrasebook tts in Bayonne.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short phrase translations dso base for tts.
 */
class __EXPORT Translator : protected Keydata
{
private:
	friend __EXPORT Translator *getTranslator(const char *name);
	static Translator *first;
	Translator *next;

protected:
	/**
	 * Return the language this translator supports.
	 *
	 * @return language supported.
	 */
	virtual char *getName(void) = 0;

	/**
	 * get play buffer object.
	 *
	 * @return put buffer
	 * @param trunk object
	 */
	char *getPlayBuffer(Trunk *trunk);

	Translator(const char *conf);

public:
	/**
	 * Perform a phrasebook translation of the current script
	 * statement and issue play request.
	 *
	 * @return ccscript error message or NULL.
	 * @param trunk object for phrasebook.
	 */
	virtual char *speak(Trunk *trunk) = 0;
};

/* Bayonne config file istanciation classes.  In Bayonne these are
   created as keydata objects out of bayonne.conf during process
   startup automatically.  This is Bayonne runtime configuration magic.
*/

/**
 * Load /etc/bayonne [tones] key values for user defined tone sets.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load and build user defined tones.
 */
class __EXPORT KeyTones : protected Keydata
{
public:
	/**
	 * Initialize tone data.
	 */
	KeyTones();
};

/**
 * Load localization rules from [localize].
 * May have defaults appropriate to US.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load localization rules.
 */
class __EXPORT KeyLocal : public Keydata
{
public:
	/**
	 * Load local rule set.
	 */
	KeyLocal();
};

/**
 * Load /etc/bayonne [voices] to select and map voice libraries to
 * different translators.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load interpreter imports to use.
 */
class __EXPORT KeyVoices : public Keydata
{
public:
	/**
	 * Initialize keyimports data.
	 */
	KeyVoices();
};


/**
 * Load /etc/bayoone [paths] key value pairs.  Has internal defaults
 * if section or file is missing.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load keypaths path location configuration data.
 */
class __EXPORT KeyPaths : public Keydata
{
public:
	/**
	 * Initialize keypath data.
	 */
	KeyPaths();

	/**
	 * Get the driver and tgi base directories.
	 */
	inline const char *getLibexec(void)
		{return getLast("libexec");};

	/**
	 * Get library tgi exec search path.
	 */
	inline const char *getTgipath(void)
		{return getLast("tgipath");};

	/**
	 * Get prefix for DSO modules at install.
	 */
	inline const char *getLibpath(void)
		{return getLast("libpath");};

	/**
	 * Get the primary working storage for writable messages.
	 */
	inline const char *getDatafiles(void)
		{return getLast("datafiles");};

	/**
	 * Get the wrappers working space for inter-system bridging.
	 */
	inline const char *getWrappers(void)
		{return getLast("wrappers");};

	/**
	 * Get the runfile directory.
	 */
	inline const char *getRunfiles(void)
		{return getLast("runfiles");};

	/**
	 * Get the spool directory.
	 */
	inline const char *getSpool(void)
		{return getLast("spool");};

	/**
	 * Get the log directory.
	 */
	inline const char *getLogpath(void)
		{return getLast("logpath");};

	/**
	 * Get the primary script prefix directory.
	 */
	inline const char *getScriptFiles(void)
		{return getLast("scripts");};

	/**
	 * Get the prompt directory.
	 */
	inline const char *getPromptFiles(void)
		{return getLast("prompts");};

	/**
	 * Get the pre-cache directory.
	 */
	inline const char *getCache(void)
		{return getLast("cache");};

#ifndef	WIN32
	/**
	 * Set auto-location prefix.
	 */
	void setPrefix(char *path);
#endif
};

/**
 * Load /etc/bayonne [network] key value pairs.  These are used to
 * specify dilu access methods and bindings.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load database access bindings.
 */
class __EXPORT KeyNetwork : public Keydata
{
public:
	/**
	 * Initialize keydatabase data.
	 */
	KeyNetwork();

	/**
	 * Get refresh timer in seconds for inter-node activity.
	 *
	 * @return refresh in seconds.
	 */
	unsigned getRefresh(void)
		{return atoi(getLast("refresh"));};

	/**
	 * Get time delay for declairing a node "dead".
	 *
	 * @return time delay in seconds.
	 */
	unsigned getTimeToLive(void)
		{return atoi(getLast("live"));};

	/**
	 * Get time to elect a new buddy.
	 *
	 * @return time to elect in seconds.
	 */
	unsigned getTimeToElect(void)
		{return atoi(getLast("elect"));};

	/**
	 * Get time to expire a buddy.
	 *
	 * @return time to expire a buddy.
	 */
	unsigned getTimeToExpire(void)
		{return atoi(getLast("expire"));};

	/**
	 * Get broadcast address to use.
	 *
	 * @return broadcast address.
	 */
	InetHostAddress getBroadcast(void);

	/**
	 * Get bind address to use.
	 *
	 * @return binding address.
	 */
	InetAddress getAddress(void);

	/**
	 * Get bind address for monitoring module.
	 *
	 * @return monitor address.
	 */
	InetAddress getMonitorAddress(void);

	/**
	 * Get port for binding.
	 *
	 * @return port.
	 */
	tpport_t getPort(void);

	/**
	 * Get port for monitoring module.
	 *
	 * @return monitor port.
	 */
	tpport_t getMonitorPort(void);
};

/**
 * Load proxy settings and provide access to proxy info.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load proxy info
 */
class __EXPORT KeyProxy : public Keydata
{
public:
	/**
	 * Initialize proxy settings.
	 */
	KeyProxy();

	/**
	 * Get proxy host for http.
	 *
         * @return address of proxy or NULL
	 */
	const char *getHTTPServer(void);

	/**
	 * Get proxy port number.
	 *
	 * @return port number or 0 if not set.
	 */
	tpport_t getHTTPPort(void);

	/**
	 * Get server connect timeout value.
	 *
	 * @return timeout interval.
	 *
	 */
	timeout_t getTimeout(void);
};	

/**
 * Load /etc/bayonne [memory] key value pairs.  This is used to
 * configurate space management properties of the runtime environment
 * including audio buffering, page size allocations, etc.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load memory related options.
 */
class __EXPORT KeyMemory : public Keydata
{
public:
	/**
	 * Initialize keymemory data.
	 */
	KeyMemory();

	/**
	 * Get default symbol space size for variables.
	 *
	 * @return number of bytes of storage for new symbols.
	 */
	inline int getSymbolSize(void)
		{return atoi(getLast("symbols"));};

	/**
	 * Get default page allocation size to use for "paged" memory
	 * allocations.
	 *
	 * @return page size for default paging.
	 */
	inline int getPageSize(void)
		{return atoi(getLast("page"));};

	/**
	 * Get maximum users.
	 *
	 * @return maximum users.
	 */
	inline size_t getUserCount(void)
		{return atoi(getLast("users"));};

	/**
	 * Get maximum preferences per user.
	 *
	 * @return maximum preferences.
	 */
	inline size_t getPrefCount(void)
		{return atoi(getLast("prefs"));};
};

/**
 * Load /etc/bayonne [thread] key value pairs.  Has internal defaults
 * if section or file is missing.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load keythreads priority and session count configuration.
 */
class __EXPORT KeyThreads : public Keydata
{
public:
	/**
	 * Initialize keythread data.
	 */
	KeyThreads();

	/**
	 * Get relative priority to run service threads at.
	 *
	 * @return service thread priority (relative).
	 */
	inline int priService(void)
		{return atoi(getLast("services"));};

	/**
	 * Get the relative priority to run gui at.
	 *
	 * @return gui priority (relative).
	 */
	inline int priGUI(void)
		{return atoi(getLast("gui"));};

	/**
	 * Get the relative priority to run rtp at.
	 *
	 * @return rtp priority (relative).
	 */
	inline int priRTP(void)
		{return atoi(getLast("rtp"));};

	/**
	 * Get number of milliseconds to delay each script step.
	 *
	 * @return millisecond delay interval.
	 */
	inline int getStepDelay(void)
		{return atoi(getLast("stepdelay"));};

	/**
	 * Get the minimal step interval.
	 *
	 * @return millisecond minimal interval.
	 */
	inline int getStepInterval(void)
		{return atoi(getLast("stepinterval"));};

	/**
	 * Get the reset delay required for settling the DSP.
	 *
	 * @return millisecond delay from dsp reset.
	 */
	inline int getResetDelay(void)
		{return atoi(getLast("resetdelay"));};

	/**
	 * Get default stack size for threads.
	 *
	 * @return stack size in "k" increments.
	 */
	size_t getStack(void);


	/**
	 * Get default stack size for audio threads.
	 *
	 * @return stack size in "k" increments.
	 */
	size_t getAudioStack(void);

	/**
	 * Get default stack size for driver threads.
	 *
	 * @return stack size in "k" increments.
	 */
	size_t getDriverStack(void);

	/**
	 * Get count of service pool threads to use.
	 *
	 * @return thread count for service pool.
	 */
	int getServices(void);

	/**
	 * Get relative priority to run audio streams at.
	 *
	 * @return audio thread priority (relative).
	 */
	inline int priAudio(void)
		{return atoi(getLast("audio"));};

	/**
	 * Get the auditing flag.
	 *
	 * @return thread auditing flag.
	 */
	inline bool getAudit(void)
		{return getLogical(getLast("audit"));};
	/**
	 * Get relative priority to run gateway (TGI) sessions at.
	 *
	 * @return tgi gateway process priority (niceness).
	 */
	inline int priGateway(void)
		{return atoi(getLast("gateways"));};

	/**
	 * Get the relative priority to run network management sessions.
	 *
	 * @return priority for management threads.
	 */
	inline int priManager(void)
		{return atoi(getLast("managers"));};

	/**
	 * Get the relative priority for network service thread.
	 *
	 * @return priority for lookup thread.
	 */
	inline int priNetwork(void)
		{return atoi(getLast("network"));};

	/**
	 * Get the relative priority for switch integration module.
	 *
	 * @return priority for switch integration.
	 */
	inline int priSwitch(void)
		{return atoi(getLast("switch"));};


	/**
	 * Scheduler network node rebroadcast frequency.
	 *
	 * @return node broadcast update interval in seconds.
	 */
	inline int getRefresh(void)
		{return atoi(getLast("refresh"));};

	/**
	 * Get number of tgi gateway proccesses to make available.
	 *
	 * @return number of tgi gateway processes.
	 */
	int getGateways(void);

	/**
	 * Get default Bayonne system priority (niceness) to start
	 * under before we adjust any relative priorities.
	 *
	 * @return primary "nice" process priority.
	 */
	inline int getPriority(void)
		{return atoi(getLast("priority"));};

	/**
	 * Get scheduling policy to use.
	 *
	 * @return policy id, or "0" for default.
	 */
	int getPolicy(void);

	/**
	 * Get memory locking and local pages.
	 *
	 * @return number of k of stack to pre-allocate.
	 */
	inline int getPages(void)
		{return atoi(getLast("pages"));};
};

extern __EXPORT KeyThreads keythreads;

/**
 * A base class new for driver conference resource objects.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Conference object resource.
 *
 */
class __EXPORT Conference
{
private:
	friend class Trunk;
	friend class Driver;

	Conference *next, *prev;
	Driver *driver;
	char name[16];
	unsigned size, used;
	bool reuse;	// set when released
	bool ready;	// ready for use

protected:
	Conference(Driver *driver, const char *id, unsigned size);

	virtual bool enter(Trunk *trunk) = 0;
	virtual void leave(Trunk *trunk) = 0;
	void unlink(void);
	void drop(void);		// dec used
	void enterMutex(void);
	void leaveMutex(void);

public:
	virtual ~Conference()
		{unlink();};

	void release(void);
};

/**
 * The Keyroute class is a container of routing information that 
 * associates a route key, such as a telephone number, with a data 
 * element such as a scriptname or start path.  These lists are ordered
 * and searched for patterns in digit length order for matching entries.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Route key data object.
 */
class __EXPORT Keyroute : public String
{
protected:
	friend class KeyServer;

	class key
	{
	public:
		key *next;
		char id[33];
		String value;
	};	
	
	key *idx[32];
	key *fre;		// free list

public:
	/**
	 * Construct a route table.
	 */
	Keyroute();

	/**
	 * Add or change a route entry.
	 *
	 * @param id of route to add.
	 * @param data to associate.
	 */
	void set(const char *id, const char *data);

	/**
	 * Remove a route entry.
	 *
	 * @param id to remove.
	 */
	void clear(const char *id);

	/**
	 * Match a dialing string to a route by increase size.
	 *
	 * @return route data.
	 * @param data to search with.
	 */
	const char *first(const char *id);

	/**
	 * Match a dialihng string to a route by decrease size.
	 *
	 * @return route data.
	 * @param data to search with.
	 */
	const char *last(const char *id);

	/**
	 * Dump an index table of routes.
	 *
	 * @return number of entries dumped.
	 * @param table index.
	 * @param size of table index.
	 */ 
	unsigned getIndex(char **list, unsigned max);
};

/**
 * Trunk "groups" provide keydata configuration information that apply
 * to a group of trunk ports represented under a common "group" identity.
 * These are initially given a [trunks] section for default group values
 * followed by a specific entry.  The [server] groups key lists the active
 * trunk groups.  A special default group is also created.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Trunk group configuration.
 */
class __EXPORT TrunkGroup : public Keydata, public CallStat
{
private:
	friend class KeyServer;
	friend class Request;
	friend class Policy;
	friend class Trunk;
	friend class MappedStats;
	static TrunkGroup *first;
	TrunkGroup *next;
	unsigned trump;
	Request *reqfirst, *reqlast;
	Policy *polFirst;
	unsigned members;

	friend inline const char *getGroups(void)
		{return TrunkGroup::first->getLast("groups");};

	friend __EXPORT void cancel(TrunkGroup *group, const char *tag);
	friend __EXPORT Request *locate(TrunkGroup *group, const char *tag, int *pos = NULL);

public:
	/**
	 * Create a trunk group data key from the bayonne.conf file.
	 *
	 * @param name of trunk group to load.
	 */
	TrunkGroup(char *name = NULL);

	/**
	 * Get the name of this trunk group.
	 *
	 * @return name of trunk group.
	 */
	inline const char *getName(void)
		{return getLast("name");};

	/**
	 * Interface to force logging of call statistics.
	 */
	static void logStats(void);

	/**
	 * Get the number of rings before answering.
	 *
	 * @return number of rings to answer.
	 */
	inline unsigned getAnswer(void)
		{return atoi(getLast("answer"));};

	/**
	 * Get the accepting state option for this trunk.
	 *
	 * @return true if should accept.
	 */
	bool getAccept(void);

	/**
	 * Get dialtone detect state for the trunk.
	 *
	 * @return true if dialtone detect should be enabled.
	 */
	bool getDetect(void);

	/**
	 * Get the number of milliseconds for active caller id.
	 *
	 * @return number of milliseconds of caller id.
	 */
	inline timeout_t getCallerid(void)
		{return getMSTimeout(getLast("callerid"));};

	/**
	 * Get pickup gaurd time for this trunk group.
	 *
	 * @return number of milliseconds for call pickup.
	 */
	inline timeout_t getPickup(void)
		{return getMSTimeout(getLast("pickup"));};

	/**
	 * Get handling for trunk group "pending requests".
	 *
	 * @return symbol or timer value.
	 */
	inline const char *chkRequest(void)
		{return getLast("requests");};

	/**
	 * Get a trunk group port selection method.
	 *
	 * @return select method.
	 */
	seltype_t getSelect(void);

	/**
	 * Get trunk threashold for requests to be posted.
	 *
	 * @return threashold.
	 */
	inline unsigned getThreashold(void)
		{return atoi(getLast("threashold"));};

	/**
	 * Get call progress analysis timeout for dialing.
	 *
	 * @return timeout in seconds.
	 */
	inline unsigned getAnalysis(void)
		{return atoi(getLast("analysis"));};

	/**
	 * Get ready timer for trunk before handling requests when
	 * idle.
	 *
	 * @return ready timer in milliseconds.
	 */
	inline timeout_t getReady(void)
		{return getMSTimeout(getLast("ready"));};
	
	/**
	 * Get the initial idle timer for this group.
	 *
	 * @return idle time initial.
	 */
	inline timeout_t getIdleTime(void)
		{return getMSTimeout(getLast("idletime"));};

	/**
	 * Get the trunk seizure timer for dialtone detect.
	 *
	 * @return siezure time for dialtone.
	 */
	inline unsigned getSiezeTime(void)
		{return atoi(getLast("siezetime"));};

	/**
	 * Get the time delay of each ring.
	 *
	 * @return ring time between rings.
	 */
	inline unsigned getRingTime(void)
		{return atoi(getLast("ringtime"));};

	/**
	 * Get call progress minimum digits
	 *
	 * @return number of digits (only for dialogic drivers).
	 */
	inline unsigned getMinDigits(void)
		{return atoi(getLast("mindigits"));};

	/**
	 * Get call progress time out to get more digits
	 *
	 * @return Timeout in seconds
	 */
	inline unsigned getMDigTimeOut(void)
		{return atoi(getLast("mdigtimeout"));};

	/**
	 * Get disconnect gaurd time before answering.
	 *
	 * @return gaurd time in milliseconds.
	 */
	inline timeout_t getHangup(void)
		{return getMSTimeout(getLast("hangup"));};

	/**
	 * Get default hook flash time for this trunk group.
	 *
	 * @return hookflash time in milliseconds.
	 */
	inline timeout_t getFlash(void)
		{return getMSTimeout(getLast("flash"));};

	/**
	 * Get dialtone wait time for this trunk group.
	 *
	 * @return dialtone wait time in milliseconds.
	 */
	inline timeout_t getDialtone(void)
		{return getMSTimeout(getLast("dialtone"));};

	/**
	 * Get dialing speed in milliseconds.
	 *
	 * @return dialspeed in milliseconds.
	 */
	inline timeout_t getDialspeed(void)
		{return getMSTimeout(getLast("dialspeed"));};

	/**
	 * Get the telephone number associated with this trunk group
	 * if one is associated with it.
	 *
	 * @return telephone number if known.
	 */
	const char *getNumber(void);

	/**
	 * Used when mapping trunk groups to activated trunks.
	 */
	inline void incCapacity(void)
		{++capacity;};

	/**
	 * Get the next active request pending for this group.
	 *
	 * @return next request queued or NULL.
	 */
	Request *getRequest(void);

	/**
	 * Find a named trunk group.
	 *
	 * @return trunk group object if found.
	 */
	static TrunkGroup *getGroup(const char *name = NULL);

	/**
	 * Assign a trunk to it's appropriate trunk group.
	 *
	 * @param trunk to look for.
	 * @return trunk group assigned.
	 */
	static TrunkGroup *assign(Trunk *trunk);

	/**
	 * Get next group record.
	 *
	 * @return next group link.
	 */
	inline TrunkGroup *getNext(void)
		{return next;};
};

/**
 * This class is a cache for server specific configuration information
 * which may be configured from /etc/bayonne.conf [server].
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short load server configuration data.
 */
class __EXPORT KeyServer : public Keydata, public ThreadLock
{
private:
	friend class Fifo;

	Keyroute incoming, outgoing;

public:
	/**
	 * Load active server configuration.
	 */
	KeyServer();

	/**
	 * Get the name of the node identifier.
	 *
	 * @return node id.
	 */
	inline const char *getNode(void)
		{return getLast("node");};

	/**
	 * Return default script attach login state.
	 *
	 * @return login id.
	 */
	inline const char *getLogin(void)
		{return getLast("login");};

	/**
	 * Return policy search order.
	 *
	 * @return policy search order.
	 */
	inline const char *getPolicyOrder(void)
		{return getLast("policy");};

	/**
	 * Get the remote access password.
	 *
	 * @return remote access password.
	 */
	inline const char *getPassword(void)
		{return getLast("password");};

	/**
	 * Get tgi token seperator.
	 *
	 * @return token seperator.
	 */
	inline const char *getToken(void)
		{return getLast("token");};

	/**
	 * Load all active trunk group records.
	 */
	void loadGroups(bool test);

	/**
	 * Get number of nodes.
	 *
	 * @return node count.
	 */
	inline int getNodeCount(void)
		{return atoi(getLast("nodes"));};

	const char *getIncoming(Trunk *trunk);

	/**
	 * Print updated route map to runfile.
	 */
	void printRoutes(void);
};

/**
 * This class is used to load and manage "plugin" modules as called for
 * in /etc/bayonne.conf [plugins].
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Load and manage plugins support.
 */
class __EXPORT Plugins : public Keydata
{
private:
	unsigned pidcount;
	int pids[32];

public:
	/**
	 * Load plugins key data and initialize.
	 */
	Plugins();

	/**
	 * Unload all active DSO modules for plugins.
	 */
	~Plugins();

	/**
	 * Get the name of the driver API being used.
	 *
	 * @return driver api name.
	 */
	char *getDriverName(void);

	/**
	 * Load a debug module or stub interface.
	 */
	void loadDebug(void);

	/**
	 * Load a database module.
	 */
	void loadDatabase(void);

	/**
	 * Attempt to load a DSO IVR API driver subsystem on top of the
	 * Bayonne server.  On failure a DSO exception is thrown.
	 */
	void loadDriver(void);

#ifndef	WIN32
	/**
	 * Attempt to load a sql query module for Bayonne.
	 */
	void loadSQL(void);
#endif

	/**
	 * Pre-load ccaudio codec modules that may be needed.
	 */
	void loadCodecs(void);

	/**
	 * Attempt to load DSO based script modules into the server.
	 */
	void loadModules(void);

	/**
         * Attempt to load DSO based tts modules into the server.
         */
        void loadTTS(void);

	/**
	 * Attempt to load generic server extenions.
	 */
	void loadExtensions(void);

	/**
	 * Attempt to load DSO based TTS translation modules into server.
	 */
	void loadTranslators(const char *lcp = NULL);

	/**
	 * Attempt to load TGI modules, have to be done before other
	 * modules...
	 */
	void loadTGI(void);

};

/**
 * We derive a Bayonne server version of ScriptCommand, aaScript,
 * which holds most common elements of the script engine for Bayonne
 * use.  Individual drivers may further derive sub-dialects.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Bayonne script dialect.
 */
class __EXPORT aaScript : public ScriptCommand
{
protected:
	/**
	 * New GetTrapMask() used to set DTMF bit when dtmf digits
	 * are also requested in ^traps.
	 *
	 * @return trap mask to apply.
	 * @param trap name being evaluated.
	 */
	unsigned long getTrapMask(const char *trapname);

	/**
	 * Linked list of boxes.
	 */
	aaScript *next;

public:
	/**
	 * Default scripting environment.
	 */
	aaScript();

	/**
	 * Create and bind additional script boxes.
	 */
	aaScript(aaScript *ini);

	/**
	 * Find a script box by name.
	 */
	static aaScript *get(const char *id, aaScript *base = NULL);

	/**
	 * Add unused commands to aaScript for compiler clean-ness.
	 *
	 * @param names of commands to add.
	 */
	void addDummy(const char *names);
};

/**
 * We derive a Bayonne compiler and script image container, aaImage,
 * to hold active script sets for the trunk class script engine.  This
 * class is almost never further derived in drivers, though the driver
 * "getScript" method is used to load it.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Bayonne script image.
 */
class __EXPORT aaImage : public ScriptImage
{
protected:
	/**
	 * Used to parse and determine if a given directory file
	 * should be "compiled".  Usually tests for *.scr.
	 *
	 * @return true if this file should be compiled.
	 * @param file name to test.
	 */
	virtual bool isScript(const char *scriptname, const char *ext);

	/**
	 * Used to scan and compile a script collection from a specified
	 * directory.
	 *
	 * @param directory to scan.
	 */
	void scanDir(char *path, const char *ext, size_t size);


public:
	/**
	 * Default image compiler.
	 */
	aaImage(aaScript *script);
};

/**
 * The PortNotify class is used to buffer and pass port notify
 * events, which, in the past, when used in some drivers, were
 * typically done with a "pipe" and a select or poll call.  The
 * new Common C++ 1.1 allows us to do this far more efficiently
 * through a Buffer object which can be attached to a driver or
 * service as needed.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Bayonne driver port notification events.
 */
class __EXPORT PortNotify : private Buffer
{
private:
	unsigned head, tail;
	unsigned char buffer[256];	// 256 notify events deep...

	int onPeek(void *buffer);
	int onWait(void *buffer);
	int onPost(void *buffer);

public:
	/**
	 * Post a port id to the buffer.  The special id 0xff is
	 * used to resequence timer queues, and 0x00 is often used
	 * to stop a driver.
	 *
	 * @param port id to pass.
	 */
	void update(unsigned char port=0xff);

protected:
	/**
	 * Waits for a specified timeout for a port id to be
	 * available.  If the id is available, it is returned.
	 *
	 * @return port id or 0xff if timeout/reschedule event.
	 * @param time to wait in milliseconds.
	 */
	unsigned char fetch(timeout_t timeout);

	/**
	 * This class is often constructed as part of a Driver or
	 * service thread class.  It is never used stand-alone.
	 */
	PortNotify();
};

/**
 * The EventBuffer class is used to buffer and pass port trunks
 * events, which, in the past, when used in some drivers, were
 * typically done with a "pipe" and a select or poll call.  The
 * new Common C++ 1.1 allows us to do this far more efficiently
 * through a Buffer object which can be attached to a driver or
 * service as needed.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Bayonne event record buffering.
 */
class __EXPORT EventBuffer : private Buffer
{
private:
	unsigned head, tail;
	TrunkEvent *buffer;

	int onPeek(void *buffer);
	int onWait(void *buffer);
	int onPost(void *buffer);

public:
	/**
	 * Post a TrunkEvent to the buffer.  If no event is passed,
	 * a "NULL" trunk event is used.
	 *
	 * @param port id to pass.
	 */
	void update(TrunkEvent *event = NULL);

protected:
	/**
	 * Waits for a specified timeout for an event to be
	 * available.  If the event is available, it is returned,
	 * otherwise at timeout a timer expired event is generated.
	 *
	 * @param time to wait in milliseconds.
	 * @param TrunkEvent object to fill.
	 */
	void fetch(timeout_t timeout, TrunkEvent *event);

	/**
	 * This class is often constructed as part of a Driver or
	 * service thread class.  It is never used stand-alone.
	 *
	 * @param number of events to buffer.
	 */
	EventBuffer(unsigned count);
};

/**
 * The AudioBuffer class is for mixing one-to-one
 * soft joins.
 *
 * @author Mark Lipscombe <markl@gasupnow.com>
 * @short Bayonne audio mixer
 */
class __EXPORT AudioBuffer : protected Mutex
{
public:
	AudioBuffer();
	~AudioBuffer();

	void getBuffer(char *data, unsigned amount);
	void putBuffer(const char *data, unsigned amount);
private:
	char *buf;
	unsigned len;
	unsigned start;
	unsigned size;
};

/**
 * We derive a Bayonne server version of ScriptInterp, "Trunk",
 * which holds most common elements of the script engine for Bayonne
 * use.  This is also the base of the channel port structure for
 * Bayonne.  Drivers will further derive this as "DriverTrunk".
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Bayonne channel port script engine.
 */
class __EXPORT Trunk : public ScriptInterp
{
public:
	class __EXPORT Threaded : public ScriptModule
	{
	protected:
		Threaded(const char *id);
	public:
		virtual Service *getService(Trunk *trunk, Line *line, trunkdata_t *data) = 0;
		virtual timeout_t getTimeout(Trunk *trunk, trunkdata_t *data)
			{return data->sleep.wakeup;};
	};

	class __EXPORT Database : protected Threaded, protected ScriptData, protected Keydata
	{
	public:
		Database();
		virtual bool header(ScriptInterp *interp) = 0;
		char *parseScript(ScriptInterp *interp, Line *line);
		void moduleAttach(ScriptInterp *interp);
		bool isId(const char *id);
		timeout_t getTimeout(Trunk *trunk, trunkdata_t *data);
	};

private:
	friend class Request;
	friend class Route;
	friend class ThreadModule;
	friend class MappedCalls;
	friend class aaScript;
	friend class Translator;
	friend class ScriptInterface;
	friend class Service;
	friend class AudioService;
	friend class Fifo;
	friend class Driver;
	friend class TrunkGroup;

	typedef union
	{
        	Symbol sym;
        	char data[sizeof(Symbol) + 12];
	}       Number;

	typedef union
	{
        	Symbol bin;
        	char data[sizeof(Symbol) + MAX_DIGITS];
	}       Digit;

	char _dbgname[33];
	const char *_genName(Driver *drv, int port);

protected:

        const char *volatile trunkgid;
	UsageStat *use;
	static char status[];
	phTone *tonetmp;
	int tsid;
	static struct callrec_t *callrec;
	Trunk *ctx;

	void setAudioPosition(timeout_t pos);
	void incIncoming(void);
	void decIncoming(void);
	void incOutgoing(void);
	void decOutgoing(void);
	void setUsage(UsageStat *us);

public:
	bool scrThread(void);

private:
	unsigned member;
	char apppath[64];

	bool scrSession(void);
	bool scrPolicy(void);
	bool scrConfig(void);
	bool scrSend(void);
	bool scrStart(void);
	bool scrLibexec(void);
	bool scrControl(void);
	bool scrHangup(void);
	bool scrDebug(void);
	bool scrAlog(void);
	bool scrAudit(void);
	bool scrSleep(void);

public:
	static bool hasDriver(ScriptInterp *interp, const char *v);
	static bool hasVoice(ScriptInterp *interp, const char *v);
	static bool hasAppVoice(ScriptInterp *interp, const char *v);
	static bool hasSysVoice(ScriptInterp *interp, const char *v);
	static bool hasAltVoice(ScriptInterp *interp, const char *v);
	static bool hasGroup(ScriptInterp *interp, const char *v);
	static bool hasPlugin(ScriptInterp *interp, const char *v);
	static bool hasSysPrompt(ScriptInterp *interp, const char *v);
	static bool hasVarPrompt(ScriptInterp *interp, const char *v);
	static bool isNode(ScriptInterp *interp, const char *v);
	static bool isService(ScriptInterp *interp, const char *v);
	static bool isSchedule(ScriptInterp *interp, const char *v);
	static bool ifDTMF(ScriptInterp *interp, const char *v);
	static bool ifFeature(ScriptInterp *interp, const char *v);
	static bool isExtension(ScriptInterp *interp, const char *v);
	static bool isStationPort(ScriptInterp *interp, const char *v);
	static bool isVirtual(ScriptInterp *interp, const char *v);
	static bool isHunt(ScriptInterp *interp, const char *v);
	static bool isActiveUser(ScriptInterp *interp, const char *v);
	static bool isDnd(ScriptInterp *interp, const char *v);
	static bool ifRinging(ScriptInterp *interp, const char *v);
	static bool ifRunning(ScriptInterp *interp, const char *v);
	static bool ifPort(ScriptInterp *interp, const char *v);

protected:
	bool scrCommit(void);
	bool scrChange(void);
	bool scrPassword(void);
	bool scrLogin(void);
	bool scrLogout(void);
	bool scrSync(void);
	bool scrAnswer(void);
	bool scrDial(void);
	bool scrTransfer(void);
	bool scrHold(void);
	bool scrOptions(void);
	bool scrRoute(void);

private:
	bool scrTone(void);
	bool scrAccept(void);
	bool scrReject(void);
	bool scrCollect(void);
	bool scrFlash(void);
	bool scrEcho(void);
	bool scrSlog(void);
	bool scrSay(void);
	bool scrAsr(void);
	bool scrListen(void);
	bool scrAssign(void);
	bool scrAltPlay(void);
	bool scrAltSpeak(void);
	bool scrPlay(void);
	bool scrSendFax(void);
	bool scrRecvFax(void);
	bool scrExiting(void);
	bool scrCount(void);
	bool scrScan(void);
	bool scrCopy(void);
	bool scrBuild(void);
	bool scrCreate(void);
	bool scrDelete(void);
	bool scrMove(void);
	bool scrErase(void);
	bool scrRecord(void);
	bool scrSpeak(void);
	bool scrDummy(void);
	bool scrSchedule(void);
	bool scrSignal(void);
	bool scrIdle(void);
	bool scrBusy(void);
	bool scrExamine(void);
	bool scrService(void);
	bool scrUserinfo(void);
	bool scrHuntinfo(void);
	bool scrStatinfo(void);
	bool scrConference(void);
	bool scrEnter(void);
	bool scrRelease(void);
	bool scrJoin(void);
	bool scrWait(void);
	bool scrRing(void);
	bool scrPickup(void);
	bool altDial(void);

protected:

	void setExclusive(bool enable)
		{globals.setExclusive(enable);};

	Number numbers[6];
	static ScriptSymbol globals;
	static char digit[16];
	ScriptInterface *script;
	TrunkGroup *group;
	int id;
	unsigned spanid, cardid;
	time_t starttime, idletime, synctimer, exittimer;

	int exitfd;
	char exitmsg[128];
	volatile unsigned seq;
	int idle_timer;
	unsigned rings, digits, counts;

	Driver *driver;

public:
	AudioBuffer *softJoin;
	unsigned long eventmask;	// used when "advance" is locked down

protected:
	Trunk *joined;

	Service *thread;
	trunkdata_t data;
	execdata_t tgi;
	Digit dtmf;
	char buffer[65];

	struct
	{
		bool offhook: 1;
		bool dtmf: 1;
		bool script: 1;
		bool reset: 1;
		bool timer : 1;
		bool audio: 1;
		bool once : 1;
		bool ready : 1;
		bool echo : 1;
		unsigned onexit : 1;
		trunkmode_t trunk: 2;
		dspmode_t dsp: 4;
		bool dnd : 1;
		bool cid : 1;	// captured cid state
		bool sent : 1;	// cid info sent
		bool listen : 1; // listening
		bool bgm : 1;	// bgm playing
	} flags;

	/**
	 * Set idle count of the driver.
	 */
	void setIdle(bool mode);

	/**
	 * Set symbol constants on attach.
	 */
	virtual void initSyms(void) = 0;

	/**
	 * Our default mask includes timeout.
	 *
	 * @return default mask.
	 */
	unsigned long getTrapDefault(void)
		{return 0x00000007;};

	/**
	 * Replace a symbol if a valid value is passed for a replacement.
	 *
	 * @param symbol name.
	 * @param replacement value.
	 */
	void repSymbol(const char *id, const char *data);

public:
	/**
	 * Get trunk global id.
	 *
	 * @return gid
	 */
	inline const char *getTrunkGid(void)
		{return trunkgid;};
	
	/**
	 * This provides an interface to internal symbol definitions.
	 *
	 * @return symbol entry.
	 * @param symbol name.
	 * @param allocation size if not found.
 	 */
	Symbol *getEntry(const char *symname, unsigned size);

	/**
	 * A derived Commit handler, allows "clear %digits", etc.
	 *
	 * @param symbol entry.
 	 */
	void commit(Symbol *sym);

	/**
	 * Cleardigits handler.
	 *
	 * @param whether to clear all or just lead digit.
	 */
	void cleardigits(bool all);

	/**
	 * Fetch a pending digit.
	 *
	 * @return digit that is pending.
	 */
	char getdigit(void);

	/**
	 * Check and get terminating mask.
	 *
	 * @return true if mask terminates.
	 * @param mask to check.
	 * @param input event digit.
	 */
	bool getExitkey(unsigned short mask, unsigned keycode);

	/**
	 * Set exit keytrap for a masked key.
	 */
	void setExitkey(void);

	bool isStation, isRinging;

	/**
	 * Flag if admin user.
	 */
	bool isAdmin(void);

	/**
	 * Get a fax station identifier for a script.
	 *
	 * @return pointer to station id string.
	 */
	const char *getStation(void);

	/**
	 * Get a "timeout" option.  This is like getValue, however
	 * the default timeout supplied is from the constant table,
	 * and special options for numeric times of various types can
	 * be used.
	 *
	 * @return default timeout in milli-seconds.
	 * @param optional string to parse rather than option.
	 */
	timeout_t getTimeout(const char *keyword = NULL);

	/**
	 * Get a "interdigit" timeout option.  This is like getValue,
	 * however the interdigit value supplied is from the const table.
	 *
	 * @return interdigit timeout in milli-seconds.
	 */
	timeout_t getInterdigit(const char *keyword = NULL);

	/**
 	 * Get a dtmf bit "mask".
	 *
	 * @return dtmf bit mask of digits.
	 */
	unsigned short getDigitMask(const char *keyword = NULL);

	/**
	 * Test if terminating key had been pressed.
	 *
	 * @param current exit mask to test.
	 * @return true if exit key was pressed.
	 */
	bool hasExitMask(unsigned short mask);

	/**
	 * Get a dtmf exit "mask".
	 *
	 * @return dtmf exit mask of digits.
	 */
	unsigned short getExitMask(void);

	/**
	 * Notify the script subsystem of a completion event.
	 *
	 * @param completion event signal id.
	 */
	bool trunkSignal(trunksignal_t);

	/**
	 * Notify the script subsystem of a error event.
	 *
	 * @param name of script error.
	 */
	void trunkError(const char *errmsg = NULL);

	/**
	 * Branch to a named script event handler, inhereted...
	 *
	 * @return true if handler exists
	 * @param name of handler to branch to.
	 */
	bool event(const char *event, bool inheret = true);

protected:
	/**
	 * Rewrite a telephone number based on trunk group stored
	 * digit rewrite rules, and post into dial command buffer.
	 *
	 * @param number to dial.
	 */
	void dialRewrite(const char *dialstring);

	/**
	 * This function sets dtmf detection based on the script
	 * interpreter's current trap mask.  This is often used as
	 * the initial handler for setting dtmf detection when
	 * entering each trunk driver's state.
	 */
	virtual void setDTMFDetect(void);

	/**
	 * Set actual dtmf detection in the derived trunk class driver.
	 * This typically is called by the "initial" state handler when
	 * entering a trunk driver call processing state when an implicit
	 * setting is required (such as "collect", which forces enable).
	 *
	 * @param true to enable DTMF detection.
	 */
	virtual void setDTMFDetect(bool enable)
		{flags.dtmf = enable;};

	/**
	 * This is used to reset service threads and generally cleanup
	 * the session handler.  It is a virtual since driver specific
	 * implimentations may vary.
	 */
	virtual void stopServices(void);

	/**
	 * Used to perform state transitions when trunk is in "step"
	 * state, from the Bayonne script engine.  This call is used
         * rather than "postEvent" with TRUNK_MAKE_STEP since the Bayonne
	 * engine is already in the context of the callback thread
	 * and invoked from a postEvent initiated call.  Hence, this
	 * saves the overhead rather of a recursive postEvent call.
	 *
	 * @param new state to use.
	 */
	virtual void trunkStep(trunkstep_t step) = 0;

	/**
	 * This is used to see if the total timer has expired.
	 *
	 * @return true if should hangup.
	 */
	bool idleHangup();

	/**
	 * This is used to determine if the trunk is currently "idle"
	 * and for how long.
	 *
	 * @return number of seconds idle.
 	 */
	virtual unsigned long getIdleTime(void) = 0;

public:
	/**
	 * Compute a prefix path based on prefix passed
	 *
	 * @return prefix path.
	 */
	const char *getPrefixPath(void);

protected:
	/**
	 * This is used to post a step update back to the script engine.
	 */
	inline bool scriptStep(void)
		{return ScriptInterp::step();};	

	/**
	 * We override ScriptInterp::Attach with our own that adds
	 * additional support.  This attach initializes a series of
	 * required and default "variables" for the script interpreter.
	 *
	 * @return true on success
	 * @param name of script to start.
	 */
	bool attach(const char *scrname);

	/**
	 * We override ScriptInterp::initialize with a local one that
	 * initializes key variables before init segments run.
	 */
	void initialize(void);

	/**
	 * We override ScriptInterp::Detach with our own that adds
	 * additional support to purge the variable pool.
	 */
	void detach(void);

	/**
	 * We get the scheduled or dnis or callerid map table values.
	 *
	 * @return argument list for startup.
	 * @param buffer for defaults.
	 */
	char **getInitial(char **args);

	/**
	 * Set a list of keyword values into the variable space of the
	 * script interpreter.
	 *
	 * @param list of keyword pairs.
	 */
	void setList(char **list);

	/**
	 * Accept method for accept scripts.
	 */
	virtual void accept(void)
		{return;};

	/**
	 * Get the voice library extension set to use.
	 */
	virtual const char *getLibexec(void)
		{return ".au";};

	/**
	 * Start the listen thread for the current port and return true
	 * if supported.
	 */
	virtual bool setListen(bool on)
		{return false;};

	/**
	 * Start the background audio play thread for the current port.
	 */
	virtual bool setBMG(bool on)
		{return false;};

public:
	/**
	 * Get the default audio encoding format to use.
	 */
	virtual const char *getDefaultEncoding(void)
		{return "ulaw";};

	/**
	 * Get the string name of an audio encoding type.
	 */
	static const char *getEncodingName(Audio::Encoding e);

protected:
	/**
	 * Reject method for reject scripts.  In case needed.
	 */
	virtual void reject(void)
		{return;};

	/**
	 * Enter a state and post it's identifier in local sym.
	 */
	void enterState(const char *state);

	Trunk(int port, Driver *drv, int card = 0, int span = 0);

	~Trunk();
public:
	/**
	 * Get the timeslot port number.
	 *
	 * @return driver port.
	 */
	unsigned getId(void)
		{return id;};
	/**
	 * Get the trunk sequence number.
	 *
	 * @return sequence.
	 */
	inline unsigned getSequence(void)
		{return seq;};

	inline Driver *getDriver(void)
		{return driver;};

	/**
	 * Set the sleep save marker for sleeping tasks.
	 *
	 * @param sym name.
	 */
	inline void setSave(const char *save)
		{data.sleep.save = save;};

	/**
	 * Get driver capabilities.
	 *
	 * @return capability mask.
	 */
	virtual unsigned long getCapabilities(void)
		{return TRUNK_CAP_VOICE | TRUNK_CAP_DIAL;};

	/**
	 * Get global symbol stace.
	 *
	 * @return pointer to global symbols.
	 */
	static inline ScriptSymbol *getGlobals(void)
		{return &globals;};


	/**
	 * Get the device logical name number.
	 *
	 * @param Buffer to store name.
	 */
	virtual void getName(char *buffer) = 0;

	/**
	 * Invoke a runtime state handler for a trunk driver.  This
	 * must be in an event in the derived TrunkDriver class.
	 *
	 * @return true if event claimed.
	 * @param derived method to call.
	 */
	virtual bool postEvent(TrunkEvent *event) = 0;

protected:
	/**
	 * Receive an event message from a foreign trunk and process it
	 * locally.
	 *
	 * @return true if event message valid.
	 */
	bool recvEvent(TrunkEvent *event);

	/**
	 * Post a sync event to the parent.
	 *
	 * @return true if successful.
	 */
	bool syncParent(const char *msg);

	/**
	 * Set callrec to call type specified.
	 *
	 * @param calltype.
	 */
	void setCalls(const char *mode);

	/**
	 * Set a field, typically in the callrec structure.
	 *
	 * @param pointer to field in callrec.
	 * @param content to copy.
	 * @param size of field.
	 */
	void setField(char *field, const char *str, unsigned len);

public:
	/**
	 * Compute the DTMF digit id of a character.
	 *
	 * @return dtmf digit or -1.
	 */
	int getDigit(char digit);

	/**
	 * See if trunk is idle and available.
	 *
	 * @return true if ready.
	 */
	bool isReady(void);

#ifdef	HAVE_TGI
	/**
	 * Perform a gateway TTS operation on behalf of the current
	 * trunk.
	 *
	 * @param filename or text string.
	 * @param mode argument.
	 */
	void libtts(const char *msg, ttsmode_t mode);
#endif

	/**
	 * Make getOnce() into a public.
	 */
	inline bool getOnce(void)
		{return ScriptInterp::getOnce();};

	/**
	 * Get group membership.
	 *
	 * @return member id.
	 */
	inline unsigned getMemberId(void)
		{return member;};

	/**
	 * Process a soft tone buffer.
	 *
	 * @return soft tone object.
	 */
	phTone *getTone(void);

	/**
	 * Fetch the command interpreter in public context.
	 *
	 * @return command interpreter.
	 */
	inline ScriptCommand *getCommand(void)
		{return ScriptInterp::getCommand();};

	/**
	 * Fetch the current trunk mode flag.
	 *
	 * @return trunk call mode.
	 */
	inline trunkmode_t getTrunkMode(void)
		{return flags.trunk;};

	inline Trunk *getJoined(void)
		{return joined;};

	inline void setJoined(Trunk *trk)
		{joined = trk;};
};

/**
 * The system fifo is a class that both handles a fifo "control interface"
 * and that can process string commands as events through the Bayonne
 * driver.  The Fifo is used by tgi and may be used by other system
 * services.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Bayonne fifo command executive.
 */
class __EXPORT Fifo : public Mutex, public Script
{
private:
	int fd;
	Thread *main;

protected:
	char schedule[33];

	Trunk *logResult(char **args);
	Trunk *assignRing(char **args);
	Trunk *assignDial(char **args);
	Trunk *sendEvent(char **args);
	Trunk *clearRing(char **args);
	Trunk *clearDial(char **args);
	Trunk *clearRDGroup(char **args);
	Trunk *exitPid(char **args);
	Trunk *waitPid(char **args);
	Trunk *setSymbol(char **argv);
	Trunk *putSymbol(char **argv);
	Trunk *setGlobal(char **argv);
	Trunk *addSymbol(char **argv);
	Trunk *setSize(char **argv);
	Trunk *addRoute(char **argv);
	Trunk *delRoute(char **argv);

	Trunk *login(char **argv);
	Trunk *runScript(char **argv);
	Trunk *startScript(char **argv);
	Trunk *testScript(char **argv);
	Trunk *ringScript(char **argv);
	Trunk *redirectScript(char **argv);
	Trunk *setSpan(char **argv);
	Trunk *setCard(char **argv);
	Trunk *busyLine(char **argv);
	Trunk *idleLine(char **argv);
	Trunk *hangupLine(char **argv);
	Trunk *reqScript(char **argv);
	Trunk *setSchedule(char **argv);
	Trunk *postKey(char **argv);
	Trunk *sync(char **argv);
	Trunk *setMixer(char **argv);
	Trunk *setLimit(char **argv);
	Trunk *mapFiles(char **argv);
	Trunk *submit(char **argv);
	Trunk *reload(char **argv);
	Trunk *saveId(char **argv);
	Trunk *createId(char **argv);
	Trunk *deleteId(char **argv);
	Trunk *shellStart(char **argv);
	Trunk *stop(char **argv);
	Trunk *collectCmd(char **argv);
	Trunk *optionCmd(char **argv);
	Trunk *getSymbol(char **argv);
	Trunk *run(char **argv);
	Trunk *asr(char **argv);
	Trunk *text(char **argv);
	Trunk *notify(char **argv);

public:
	/**
	 * Issue a "command" request either from the fifo port itself
	 * or from a service thread.
	 *
	 * @return NULL if failed.
	 * @param request string as would be passed by fifo.
	 * @param fd optional output redirection.
	 */
	Trunk *command(const char *cmdstring, std::ostream *fd = NULL);

	/**
	 * Get a file handle to the fifo "command" interface.
	 *
	 * @return file descriptor to use.
	 */
	void setControl(int fd);

	void control(const char *ctrl);

	Fifo();
};

#define	MAX_SPANS	64
#define	MAX_CARDS	64

/**
 * The driver class represents an abstract means of accessing the
 * internals of a Bayonne driver plug-in module.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Bayonne driver interface class.
 */
class __EXPORT Driver : public Mutex
{
protected:
	friend class TrunkGroup;
	friend class Trunk;
	friend class MappedDrivers;
	friend class Conference;

	Conference *first, *last;
	bool active;
	unsigned portCount, downCount;
	volatile unsigned idleCount;
	AtomicCounter refTimer;
        time_t nextTimer;

	unsigned tsid;
	int index;

	inline Driver *getFirst(void)
		{return drvFirst;};

	virtual Conference *getConference(const char *id, unsigned size = 0)
		{return NULL;};

public:
	friend __EXPORT Driver *getDriver(const char *name);
	static Driver *drvFirst;
	static unsigned drvIndex;
	Driver *drvNext;
	static void setNodes();

	enum {	capDaemon = 0x0001,
		capPSTN = 0x0002,
		capJoin = 0x00004,
		capSpans = 0x0008,
		capSpeed = 0x0010,	// driver can do rate adj
		capPitch = 0x0020,	// driver can pitch adj
		capGain = 0x0040,	// driver can gain adj
		capListen = 0x0080,	// driver can listen to asr
		capIP = 0x00160,	// driver is IP based
		capDynamic = 0x1000,	// driver has dynamic outbound
		capSwitch = 0x2000,
		capConference = 0x4000,
		capOffline = 0x8000};

     	/**
         * inc ref timer for sec tics.
         */
        inline void ticTimer(void)
                {++refTimer;};

	/**
	 * Second tick interval counter.
	 */
	virtual void secTick(void);

	/**
	 * Return driver capabilities.
	 */
	virtual unsigned getCaps(void)
		{return capDaemon | capPSTN;};

	/**
	 * Create an instance of the driver.
	 */
	Driver();

	/**
	 * Get the name of the driver.
	 */
	virtual char *getName(void)
		{return "Default";};

	/**
	 * Report if driver is idle.
	 */
	bool isIdle(void);

	/**
	 * Report if driver is down.
	 */
	bool isDown(void);	

	/**
	 * Start a driver; start service threads, build device nodes,
	 * etc.
	 * 
	 * @return number of active device ports.
	 */
	virtual int start(void) = 0;

	/**
	 * Shutdown the driver, service threads, etc.
	 */
	virtual void stop(void) = 0;

	/**
	 * Get a local copy of the status string.
	 *
	 * @param local status buffer.
	 */
	static void getStatus(char *buffer);

	/**
	 * Get total driver port capacity as a count.
	 *
	 * @return total port capacity count.
	 */
	static unsigned getCount(void);

	/**
	 * Load a script image.  Usually this is not driver specific,
	 * though it can be made to be so.
	 *
	 * @return ScriptImage base abstract.
	 */
	virtual aaImage *getImage(void);

	/**
	 * Get total number of port identifiers (timeslots) used by
	 * the current driver.
	 *
	 * @return trunk timeslots/ports allocated.
	 */
	virtual unsigned getTrunkCount(void) = 0;

	/**
	 * Get active number of ports actually deployed by the current
	 * driver.
	 *
	 * @return actual trunks used.
	 */
	virtual unsigned getTrunkUsed(void) 
		{return getTrunkCount();};

	/**
	 * Get the trunk group member trunk.
	 *
	 * @return trunk group port id.
	 * @param trunk group to locate.
	 * @param member id to locate.
	 */
	int getTrunkMember(TrunkGroup *grp, unsigned member);

	/**
	 * Get an individual port from the driver by timeslot/id.
	 *
	 * @return trunk port object.
	 */
	virtual Trunk *getTrunkPort(int id) = 0;

	/**
	 * Return driver classification either by supporting group
	 * name or because the id used is valid for this driver or
	 * the specified port.
	 */
	virtual bool isTrunkClass(int id, const char *name);

	enum {
		TRUNK_URI = 0x0001,
		TRUNK_EXTENSION = 0x0002,
		TRUNK_GROUP = 0x0004,
		TRUNK_TIE = 0x0008,
		TRUNK_LOCAL = 0x0032,
		TRUNK_START = 0x0160
	};

	static Trunk *getTrunk(const char *name, bool create = false,
		Driver *driver = NULL);
	static Driver *getIndexedDriver(int idx);

	/**
	 * Get an individual port from the driver by timeslot/id
	 * for the outbound (gives drivers which dynamically 
	 * allocate trunks the opportunity to create one).
	 *
	 * @return trunk port object.
	 */
	virtual Trunk *getOutboundTrunk(int id)
		{ return getTrunkPort(id); };

	/**
	 * Get a trunk by a id or global id string.
	 *
	 * @param trunk or port reference id.
	 * @return trunk port object.
	 */
	Trunk *getTrunkId(const char *id);

	/**
	 * Span event operations.
	 *
	 * @return true if success.
	 * @param span to effect.
	 * @param event to pass.
	 */
	virtual bool spanEvent(unsigned span, TrunkEvent *event);

	/**
	 * card event operations.
	 *
	 * Card event operations.
	 *
	 * @return true if successful.
	 * @param card to effect.
	 * @param event to pass.
	 */
	virtual bool cardEvent(unsigned card, TrunkEvent *event)
		{return false;};

	int getDriverIndex(void)
		{return index;};
};

/**
 * New DSO class for installing a "debugging" or "monitor" plugin.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Regression test/debug DSO interface.
 */
class __EXPORT Debug : protected Mutex, protected Script
{
public:
	/**
	 * Register DSO debug object.
	 */
	Debug();

	/**
	 * Stack trace support.
	 */
	static void stackTrace(int signo);

	/**
	 * Virtual for trace mode vs begin in script stepping.
	 *
	 * @return true to force exit after test.
	 */
	virtual bool debugTrace(void)
		{return false;};

	/**
	 * Interface for "final" handling of debug.
	 *
	 * @return true if ignore final exit.
	 */
	virtual bool debugFinal(int signo)
		{return false;};

	/**
	 * Debug interface for event processing "taps".
	 */
	virtual void debugEvent(Trunk *trunk, TrunkEvent *event)
		{return;};

	/**
	 * Debug interface for state handler entry "taps".
	 */
	virtual void debugState(Trunk *trunk, char *state)
		{return;};

	/**
	 * Debug service loop code "tap".
	 */
	virtual void debugService(Trunk *trunk, char *msg)
		{return;};

	/**
	 * Debug interface for "debug" script step.
	 */
	virtual void debugScript(Trunk *trunk, char *msg)
		{return;};

	/**
	 * Debug interface for script step "tap".
	 */
	virtual void debugStep(Trunk *trunk, Line *line)
		{debugState(trunk, "step");};

	/**
	 * Debug interface for fifo "debug" statement.
	 */
	virtual bool debugFifo(char **argv)
		{return true;};

	/**
	 * Debug interface for login/logout changes.
	 */
	virtual void debugLogin(Trunk *trunk)
		{return;};
};

/**
 * AudioService holds the logic for processing audio channels.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short audio service processing.
 */
class __EXPORT AudioService
{
private: 
	char filename[256];

protected:
	Trunk *trunk;
	/**
	 * Compute and get a prompt file name.
	 *
	 * @param partial name.
	 * @return full path.
	 */
	char *getPrompt(char *name, const char *voice = NULL);

	/**
	 * Fetch the next prompt for the play list.
	 *
	 * @return get partial name.
	 */
	char *getPlayfile(void);

	/**
	 * Construct the object.
	 */
	AudioService(void);
};

/**
 * Services are threads used to support a trunk class, such as audio
 * services used for audio, etc.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short service thread support.
 */
class __EXPORT Service : public Semaphore, public Thread, public AudioService
{
protected:
	volatile bool stopped;
	trunkdata_t *data;
	TrunkGroup *group;

	/**
	 * Signal "successful" completion result.
	 */
	void success(void);

	/**
	 * Signal "failure" completion result.
	 */
	void failure(void);

	/**
	 * Mark required dsp reset.
	 */
	inline void dspReset(void)
		{trunk->flags.reset = true;};

	/**
	 * Set audio marker.
	 */
	inline void setAudio(void)
		{trunk->flags.audio = true;};

	/**
	 * Clear audio marker.
	 */
	inline void clrAudio(void)
		{trunk->flags.audio = false;};

public:
	/**
	 * Create a service thread on an existing trunk object.
	 *
	 * @param trunk object to use.
	 * @param process priority.
	 */
	Service(Trunk *trunk, int pri, size_t stack = keythreads.getStack());

	/**
	 * request to stop a service and obtain default "delay" time
	 * to use for reset interval.
	 *
	 * @return delay in milliseconds.
	 */
	virtual timeout_t stop(void);

	/**
	 * Indicate if the service thread is exiting and/or can be
	 * deleted now.  If false, then it will be cleaned by the
	 * scheduler thread.
	 *
	 * @return false if held to scheduler.
	 */
	virtual bool isExiting(void)
		{return true;};

	/**
	 * Invoke termination.
	 */
	void endService(void)
		{terminate();};

	/**
	 * Termination of service.
	 */
	virtual ~Service()
		{terminate();}
};

/**
 * Server classes are used for threaded entities such as network
 * management interfaces, which may be started and stopped under
 * server control.  Sometimes server is mixed into other dso
 * classes to stabilize start/stop of service threads.
 *
 * @short threaded server service.
 * @author David Sugar <dyfet@ostel.com>
 */
class __EXPORT Server : public Thread
{
private:
	static Server *first;
	Server *next;
	friend __EXPORT void startServers(void);
	friend __EXPORT void stopServers(void);

protected:
	Server(int pri);

	/**
	 * Used for stopServers call interface.
	 */
	virtual void stop(void)
		{terminate();};
};

/**
 * The Sync class is used to create dso objects which have entites
 * that are repetitivly called through the scheduler thread.  These
 * are sometimes used to update disk based databases from memory, or
 * perform other interval timed operations.
 */
class __EXPORT Sync
{
private:
	static Sync *first;
	Sync *next;
	time_t runtime;

protected:
	/**
	 * Abstract class, protected constructor.
	 */
	Sync(void);

	/**
	 * Return if ready for update.  If not, the current update
	 * interval may be skipped entirely.
	 */
	virtual bool isScheduled(void)
		{return true;};

	/**
	 * Return execution interval of this sync object.  How many
	 * xx minutes before attempting a new sync.
	 */
	virtual unsigned getInterval(void)
		{return 10;};

	/**
	 * Operation to perform when scheduled.
	 */
	virtual void schedule(void) = 0;

	/**
	 * Return name used in slog event when scheduling item.
	 */
	virtual char *getSyncName(void) = 0;

public:
	static void check(void);
};

/**
 * The tone class is used to build sampled single and dual frequency
 * tones that may be fed to the telephony device.  Tones are defined
 * in the "tones" section of bayonne.conf.
 *
 * @short generated sample tone.
 * @author David Sugar.
 */
class __EXPORT phTone
{
private:
	friend class KeyTones;
	friend __EXPORT phTone *getphTone(const char *name);
	static phTone *first;
	static int ulaw[256];
	static unsigned char alaw[256];
	static unsigned char a2u[128];
	phTone *next;

protected:
	char name[33];
	unsigned char *samples;
	timeout_t duration, playtime;
	unsigned freq1, freq2;

public:
	static unsigned char linear2ulaw(int sample);
	static inline unsigned char linear2alaw(int sample)
		{return alaw[linear2ulaw(sample)];};

	static unsigned char alaw2ulaw(unsigned char al);

	static inline unsigned char ulaw2alaw(unsigned char ul)
		{return alaw[ul];};

	static short ulaw2linear(unsigned char ul);
	static short alaw2linear(unsigned char al);

	/**
	 * Create single frequency tone.
	 *
	 * @param name of tone.
	 * @param duration in milliseconds.
	 * @param frequency of tone.
	 */
	phTone(const char *name, timeout_t duration, unsigned f);

	/**
	 * Create dual frequency tone.
	 *
	 * @param name of tone.
	 * @param duration in milliseconds.
	 * @param first frequency.
	 * @param second frequency.
	 */
	phTone(const char *name, timeout_t duration, unsigned f1, unsigned f2);

	~phTone();

	/**
	 * If the telephony card is capable of generating it's own
	 * tones, then it can use the clear method to remove any
	 * allocated memory for sampled data.
	 */
	void clear(void);

	/**
	 * Fetch the sample area of the tone.
	 *
	 * @return sample area.
	 */
	inline unsigned char *getSamples(void)
		{return samples;};

	/**
	 * Get the duration of the tone.
	 *
	 * @return duration of tone.
	 */
	inline timeout_t getDuration(void)
		{return duration;};

        /**
         * Get the duration of the tone.
         *
         * @return playtime of tone.
         */
        inline timeout_t getPlaytime(void)
                {return playtime;};
};

/**
 * This class is used for interfacing to DSO loaded TGI interpreters.
 *
 * @short TGI interpreter module.
 * @author David Sugar <dyfet@ostel.com>
 */
class __EXPORT TGI
{
private:
	static TGI *first;
	TGI *next;

protected:
	TGI();

	/**
	 * Check a file extension to see if it belongs to a first
	 * stage tgi script interpreter (avoids second fork).
	 *
	 * @return true if claimed.
	 * @param extension.
	 */
	virtual bool getExtension(const char *ext)
		{return false;};

public:
	/**
	 * Check command for interpreter and, if is, execute it.  If
	 * so it doesn't return but uses "exit".  This is a "second"
	 * stage tgi mod interface and is effective for single session
	 * interpreter libraries.
	 *
	 * @param script name resolved.
	 * @param argument list.
	 */
	virtual void script(char *cmd, char **args)
		{return;};

	/**
	 * Execute a first stage interpreter, uses local functions.
	 *
	 * @return shell exit code.
	 * @param fifo file descriptor.
	 * @param port number.
	 * @param unparsed command string.
	 */
	virtual int parse(int fd, char port[64], char *cmd)
		{return -1;};	

	friend __EXPORT void getInterp(char *cmd, char **args);
	friend __EXPORT TGI *getInterp(char *cmd);
};

/**
 * The tts syth classes intercept core functions as needed to create
 * an interface to a native text to speech subsystem, usually by
 * manipulating data.play.
 */
class __EXPORT TTS
{
protected:
	friend class Plugins;

	TTS();

public:
	virtual const char *getName(void) = 0;

	virtual const char *getVoices(void)
		{return "none";};

	virtual bool synth(unsigned id, trunkdata_t *data) = 0;

	friend TTS *getTTS(const char *name);
	static TTS *ttsFirst;
	TTS *ttsNext;
};


/**
 * Sessions are used for "garbage collected" entities which may be
 * detached and removed after an expiration time rather than falling
 * out of scope immediately.  This process is managed by the scheduler
 * thread.
 *
 * @short Garbage collectable objects.
 * @author David Sugar <dyfet@ostel.com>
 */
class __EXPORT Session
{
private:
	static Mutex mutex;
	static Session *first;
	static Session *last;
	Session *next, *prev;

public:
	static void clean(void);

protected:
	Session();
	virtual ~Session()
		{unlink();};

	/**
	 * Unlink the session object.
	 */
	void unlink(void);

	/**
	 * Indicate if and when this session is to be "garbage collected"
	 * by UNIX timestamp.
	 *
	 * @return 0 if still running, else garbage collection period.
	 */
	virtual time_t getExpires(void) = 0;
};

__EXPORT void setReply(const char *sym, const char *value);
__EXPORT void errlog(const char *level, const char *fmt, ...);
__EXPORT statnode_t *getNodes(const char *name);
__EXPORT bool hasTTS(void);
__EXPORT bool permitAudioAccess(const char *path, bool write);
__EXPORT bool permitFileAccess(const char *path);
__EXPORT void initScripting(void);

extern __EXPORT unsigned numbering;
extern __EXPORT bool aliases;
extern __EXPORT bool running;
extern __EXPORT KeyServer keyserver;
extern __EXPORT KeyThreads keythreads;
extern __EXPORT KeyMemory keymemory;
extern __EXPORT KeyPaths keypaths;
extern __EXPORT KeyVoices keyvoices;
extern __EXPORT KeyLocal keylocal;
extern __EXPORT KeyNetwork keynetwork;
extern __EXPORT KeyProxy keyproxy;
extern __EXPORT Plugins plugins;
//extern Driver *driver;
extern __EXPORT Debug *debug;
extern __EXPORT Fifo fifo;
extern __EXPORT ThreadLock cachelock;
extern __EXPORT aaScript *aascript;

#ifdef	CCXX_NAMESPACES
}
#endif

#endif
