// Copyright (C) 2000 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"
#include <cc++/process.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

MysqlModule::MysqlModule() : Database()
{
}

MysqlModule::~MysqlModule()
{
	MysqlQueue *node = MysqlQueue::first;
	MysqlQueue *next;

	while(node)
	{
		next = node->next;
		delete node;
		node = next;
	}
}


size_t MysqlModule::getStack(void)
{
	const char *cp = keythreads.getLast("mysqlstack");

	if(!cp && sizeof(void *) > 4)
		cp = "112";

	if(!cp)
		cp = "64";

	return atoi(cp) * 1024;
}

int MysqlModule::getPriority(void)
{
	const char *cp = getLast("priority");
	if(!cp)
		cp = keythreads.getLast("sql");
	if(cp)
		return atoi(cp);

	return 0;
}

bool MysqlModule::header(ScriptInterp *interp)
{
	const char *sz = interp->getKeyword("size");
	MYSQL_RES *res = (MYSQL_RES *)interp->getPointer(SYM_SQLRESULTS);
	MYSQL_FIELD *field;
	unsigned cols, col = 0;
	unsigned size = interp->getSymbolSize();
	Symbol *sym;
	const char *cp;

	if(sz)
		size = atoi(sz);

	if(!res)
		return false;

	cols = mysql_num_fields(res);
	if(!cols)
		return false;

	mysql_field_seek(res, col);
	while(NULL != (field = mysql_fetch_field(res)))
	{
		sym = interp->initVariable(size);
		if(!sym)
			break;

		if(sym->flags.readonly)
			continue;

		cp = field->name;
		if(!cp)
		{
			++col;
			continue;
		}
		snprintf(sym->data, sym->flags.size + 1, "%s", cp);
		if(sym->flags.commit)
			interp->commit(sym);
		++col;
	}
	return true;
}

bool MysqlModule::fetch(ScriptInterp *interp, unsigned row)
{
	const char *sz = interp->getKeyword("size");
	const char *cp = interp->getKeyword("first");
	MYSQL_FIELD *myfield;
	MYSQL_RES *res = (MYSQL_RES *)interp->getPointer(SYM_SQLRESULTS);
	MYSQL_ROW sqlrow;
	unsigned cols, col = 0;
	unsigned size = interp->getSymbolSize();
	Symbol *sym = NULL;
	char buf[11];
	char field[65];

	if(!res)
		return false;

	if(sz)
		size = atoi(sz);

	if(row >= mysql_num_rows(res))
		return false;

	mysql_data_seek(res, row);
	sqlrow = mysql_fetch_row(res);	

	cols = mysql_num_fields(res);
	if(!cols)
		return false;

	snprintf(buf, sizeof(buf), "%d", row + 1);
	interp->setSymbol(SYM_SQLROW, buf);

	if(cp)
	{
		col = atoi(cp) - 1;
		if(col >= cols)
			return false;
	}

	while(col < cols)
	{
		sym = interp->initVariable(size);
		if(!sym)
			break;
		if(sym->flags.readonly)
			continue;
		cp = sqlrow[col];
		snprintf(sym->data, sym->flags.size + 1, "%s", cp);
		if(sym->flags.commit)
			interp->commit(sym);
		++col;
	}

	col = 0;
	field[0] = '&';
	mysql_field_seek(res, 0);
	while(NULL != (myfield = mysql_fetch_field(res)))
	{
		cp = myfield->name;		
		if(!cp || !*cp)
		{
			++col;
			continue;
		}
		snprintf(field + 1, sizeof(field) - 1, "%s", cp);
		cp = interp->getKeyword(field);
		if(!cp || *cp != '&')
		{
			++col;
			continue;
		}
		if(strchr(cp, '.'))
			sym = interp->getEntry(cp, size);
		else
			sym = interp->getLocal(cp, size);
		if(!sym || sym->flags.readonly)
		{
			++col;
			continue;
		}
		cp = sqlrow[col];
		if(!cp)
			cp = "";
		snprintf(sym->data, sym->flags.size + 1, "%s", cp);
		if(sym->flags.commit)
			interp->commit(sym);
		++col;
	}
	return true;
}

Service *MysqlModule::getService(Trunk *trunk, Line *line, trunkdata_t *data)
{
	const char *mem = trunk->getMember();
	ScriptCommand *cmd = trunk->getCommand();
	ThreadQueue *tq = cmd->getThreadQueue();
	const char *dsn = cmd->getLast("database");
	Symbol *sym = NULL;
	const char *results;
	char buffer[512];
	unsigned len = 0;
	const char *query;

	if(!dsn)
		dsn = cmd->getLast("dsn");

	if(!dsn && cmd == aascript)
		dsn = getLast("database");

	if(!dsn)
	{
		trunk->error("no-database");
		return NULL;
	}

	if(!mem)
		mem = "";

	if(!stricmp(mem, "detach") || !stricmp(mem, "post"))
	{
		if(!tq)
		{
			trunk->error("no-queue");
			return NULL;
		}
		query = trunk->getKeyword("query");
		if(query)
		{
			tq->post(query, strlen(query) + 1);
			trunk->advance();
			return NULL;
		}
		while(NULL != (query = trunk->getValue(NULL)) && len < sizeof(buffer))
		{
			snprintf(buffer + len, sizeof(buffer) - len, "%s", query);
			len += strlen(query);
		}
		if(len < sizeof(buffer))
			++len;
		tq->post(buffer, len);
		trunk->advance();
		return NULL;
	}

	if(!stricmp(mem, "fetch") || !stricmp(mem, "header"))
	{
		trunk->advance();
		return NULL;
	}

	if(!stricmp(mem, "end") || !stricmp(mem, "exit"))
	{
		trunk->advance();
		return NULL;
	}

	if(!dsn)
	{
		trunk->error("no-database");
		return NULL;
	}

	results = trunk->getKeyword("results");
	if(!results)
		results = trunk->getKeyword("save");
	if(results && *results != '&')
		results = NULL;

	if(results)
		sym = trunk->getLocal(++results, 0);
	else
		return new MysqlThread(dsn, NULL, trunk);

	if(!sym)
	{
		trunk->error("no-symbol-to-save");
		return NULL;
	}

	switch(sym->flags.type)
	{
	case ScriptModule::ARRAY:
		sym->data[0] = sym->data[1];
		break;
	case ScriptModule::FIFO:
	case ScriptModule::SEQUENCE:
	case ScriptModule::STACK:
	case ScriptModule::CACHE:
		sym->data[1] = sym->data[2] = 0;
		break;
	default:
		trunk->error("symbol-invalid-type");
		return NULL;
	}
	return new MysqlThread(dsn, results, trunk);
}

void MysqlModule::moduleAttach(ScriptInterp *interp)
{
	ScriptCommand *cmd = interp->getCommand();
	const char *dsn;
	static Mutex lock;

	Database::moduleAttach(interp);
	interp->setConst(SYM_SQLDRIVER, "mysql");
	interp->setSymbol(SYM_SQLROW, "0");

	dsn = cmd->getLast("dsn");
	if(!dsn)
		dsn = cmd->getLast("database");

	if(!dsn && cmd == aascript)
		dsn = getLast("database");

	if(!dsn)
		return;

	lock.enterMutex();
	if(cmd->getThreadQueue())
	{
		lock.leaveMutex();
		return;
	}
	cmd->setThreadQueue(new MysqlQueue(cmd));
	lock.leaveMutex();
}

void MysqlModule::moduleDetach(ScriptInterp *interp, const char *script)
{
	ScriptCommand *cmd = interp->getCommand();
	MYSQL *conn = NULL;
	MYSQL_RES *res = (MYSQL_RES *)interp->getPointer(SYM_SQLRESULTS);
        const char *cs = interp->getSymbol(SYM_SQLCONNECT);
	MysqlQueue *tq = (MysqlQueue *)cmd->getThreadQueue();

	if(tq)
		tq->cdr(interp);

	if(cmd != aascript)
		tq = (MysqlQueue *)aascript->getThreadQueue();
	else
		tq = NULL;

	if(tq)
		tq->cdr(interp);

	if(res)
		mysql_free_result(res);

        if(cs && *cs)
		conn = (MYSQL *)cmd->endDatabase();

        if(conn)
        {
                slog.debug("mysql: %s: disconnecting", cs);
		mysql_close(conn);
		delete conn;
        }
}

MysqlModule mysql;

#ifdef	CCXX_NAMESPACES
}
#endif
