// Copyright (C) 2000 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"
#include <cc++/process.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

ODBCModule::ODBCModule() : Database()
{
	long err = SQLAllocHandle(SQL_HANDLE_ENV, SQL_NULL_HANDLE, &hODBC);

	if(err != SQL_SUCCESS && err != SQL_SUCCESS_WITH_INFO)
	{
		errlog("fatal", "odbc: cannot allocate");
		return;
	}

	err = SQLSetEnvAttr(hODBC, SQL_ATTR_ODBC_VERSION, (void*)SQL_OV_ODBC3, 0);
	if(err != SQL_SUCCESS && err != SQL_SUCCESS_WITH_INFO)
	{
		errlog("fatal", "odbc: wrong version");
		SQLFreeHandle(SQL_HANDLE_ENV, hODBC);
		return;
	}
	errlog("notice", "ODBC Driver loaded");
}

ODBCModule::~ODBCModule()
{
	ODBCQueue *node = ODBCQueue::first;
	ODBCQueue *next;

	while(node)
	{
		next = node->next;
		delete node;
		node = next;
	}
}

bool ODBCModule::fetch(ScriptInterp *interp, unsigned row)
{
	const char *sz = interp->getKeyword("size");
	const char *cp = interp->getKeyword("first");
	SQLINTEGER len;
	SQLSMALLINT cols, col = 1;
	SQLSMALLINT clen;
	unsigned size = interp->getSymbolSize();
	Symbol *sym = NULL;
	unsigned current = atoi(interp->getSymbol(SYM_SQLROW));
	SQLHSTMT hstmt = interp->getPointer(SYM_SQLRESULTS);
	long err;
	char buf[11];
	char field[65];

	// if no active dataset, we skip entirely
	if(current < 0 || !hstmt)
		return false;

	if(row == 0)	// goto start
		err = SQLFetchScroll(hstmt, SQL_FETCH_FIRST, 0);
	else if(row == current)	// goto next
		err = SQLFetchScroll(hstmt, SQL_FETCH_NEXT, 0);
	else	// set position
		err = SQLFetchScroll(hstmt, SQL_FETCH_ABSOLUTE, row + 1);

	if(err != SQL_SUCCESS && err != SQL_SUCCESS_WITH_INFO)
		return false;

	if(sz)
		size = atoi(sz);

	snprintf(buf, sizeof(buf), "%d", row + 1);
	interp->setSymbol(SYM_SQLROW, buf);
	SQLNumResultCols(hstmt, &cols);
	if(cols < 1)
		return false;

	if(cp)
	{
		col = atoi(cp);
		if(!col)
			++col;
		if(col > cols)
			return false;
	}

	while(col <= cols)
	{
		sym = interp->initVariable(size);
		if(!sym)
			break;
		if(sym->flags.readonly)
			continue;
		SQLGetData(hstmt, col, SQL_C_CHAR, sym->data, sym->flags.size + 1, &len);
		if(sym->flags.commit)
			interp->commit(sym);
		++col;
	}
	col = 0;
	field[0] = '&';
	while(++col <= cols)
	{
		SQLColAttribute(hstmt, col, SQL_DESC_NAME, field + 1, sizeof(field) - 1, &clen, NULL);
		if(!field[1])
			continue;

		cp = interp->getKeyword(field);
		if(!cp)
			continue;
		if(*cp != '&')
			continue;
		++cp;
		if(strchr(cp, '.'))
			sym = interp->getEntry(cp, size);
		else
			sym = interp->getLocal(cp, size);
		if(!sym)
			continue;
		
		if(sym->flags.readonly)
			continue;
		SQLGetData(hstmt, col, SQL_C_CHAR, sym->data, sym->flags.size + 1, &len);
		if(sym->flags.commit)
			interp->commit(sym);
	}
	return true;
}

bool ODBCModule::header(ScriptInterp *interp)
{
	const char *sz = interp->getKeyword("size");
	SQLSMALLINT len;
	SQLSMALLINT cols, col = 1;
	unsigned size = interp->getSymbolSize();
	Symbol *sym = NULL;
	SQLHSTMT hstmt = interp->getPointer(SYM_SQLRESULTS);

	// if no active dataset, we skip entirely
	if(!hstmt)
		return false;

	if(sz)
		size = atoi(sz);

	SQLNumResultCols(hstmt, &cols);
	if(cols < 1)
		return false;

	while(col <= cols)
	{
		sym = interp->initVariable(size);
		if(!sym)
			break;
		if(sym->flags.readonly)
			continue;

		SQLColAttribute(hstmt, col, SQL_DESC_NAME,sym->data, sym->flags.size + 1, &len, NULL);
		if(sym->flags.commit)
			interp->commit(sym);
		++col;
	}
	return true;
}

Service *ODBCModule::getService(Trunk *trunk, Line *line, trunkdata_t *data)
{
	const char *mem = trunk->getMember();
	ScriptCommand *cmd = trunk->getCommand();
	ThreadQueue *tq = cmd->getThreadQueue();
	const char *dsn = cmd->getLast("database");
	Symbol *sym = NULL;
	const char *results;
	const char *query;
	char buffer[512];
	size_t len = 0;

	if(!dsn)
		dsn = cmd->getLast("dsn");

	if(!dsn && cmd == aascript)
		dsn = getLast("database");

	if(!dsn)
	{
		trunk->error("no-database");
		return NULL;
	}

	if(!mem)
		mem = "";

	if(!stricmp(mem, "detach") || !stricmp(mem, "post"))
	{
		if(!tq)
			goto query;
//		{
//			trunk->error("no-queue");
//			return NULL;
//		}
		query = trunk->getKeyword("query");
		if(query)
		{
			tq->post(query, (unsigned)strlen(query) + 1);
			trunk->advance();
			return NULL;
		}
		while(NULL != (query = trunk->getValue(NULL)) && len < sizeof(buffer))
		{
			snprintf(buffer + len, sizeof(buffer) - len, "%s", query);
			len += strlen(query);
		}
		if(len < sizeof(buffer))
			++len;
		tq->post(buffer, (unsigned)len);
		trunk->advance();
		return NULL;
	}

	if(!stricmp(mem, "fetch") || !stricmp(mem, "header"))
	{
		trunk->advance();
		return NULL;
	}

	if(!stricmp(mem, "end") || !stricmp(mem, "exit"))
	{
		trunk->advance();
		return NULL;
	}

query:
	if(!dsn)
	{
		trunk->error("no-database");
		return NULL;
	}

	results = trunk->getKeyword("results");
	if(!results)
		results = trunk->getKeyword("save");
	if(results && *results != '&')
		results = NULL;

	if(results)
		sym = trunk->getLocal(++results, 0);
	else
		return new ODBCThread(dsn, NULL, trunk);

	if(!sym)
	{
		trunk->error("no-symbol-to-save");
		return NULL;
	}

	switch(sym->flags.type)
	{
	case ScriptModule::ARRAY:
		sym->data[0] = sym->data[1];
		break;
	case ScriptModule::FIFO:
	case ScriptModule::SEQUENCE:
	case ScriptModule::STACK:
	case ScriptModule::CACHE:
		sym->data[1] = sym->data[2] = 0;
		break;
	default:
		trunk->error("symbol-invalid-type");
		return NULL;
	}
	return new ODBCThread(dsn, results, trunk);
}

void ODBCModule::moduleAttach(ScriptInterp *interp)
{
	ScriptCommand *cmd = interp->getCommand();
	static Mutex lock;
	const char *dsn;

	Database::moduleAttach(interp);
	interp->setConst(SYM_SQLDRIVER, "odbc");

	dsn = cmd->getLast("dsn");
	if(!dsn)
		dsn = cmd->getLast("database");

	if(!dsn && cmd == aascript)
		dsn = getLast("database");

	if(!dsn)
		return;
/*
	lock.enterMutex();
	if(cmd->getThreadQueue())
	{
		lock.leaveMutex();
		return;
	}
	cmd->setThreadQueue(new ODBCQueue(cmd));
	lock.leaveMutex();
*/
}

void ODBCModule::moduleDetach(ScriptInterp *interp, const char *script)
{
	ScriptCommand *cmd = interp->getCommand();
	SQLHDBC hdbc = NULL;
	SQLHSTMT hstmt = (SQLHSTMT)interp->getPointer(SYM_SQLRESULTS);
	const char *cs = interp->getSymbol(SYM_SQLCONNECT);

	if(hstmt)
		SQLFreeHandle(SQL_HANDLE_STMT, hstmt);

	if(cs && *cs)
		hdbc = (SQLHDBC)cmd->endDatabase();

	if(hdbc)
	{
		slog.debug("odbc: %s: disconnecting", cs);
		SQLDisconnect(hdbc);
		SQLFreeHandle(SQL_HANDLE_DBC, hdbc);
	}
}

int ODBCModule::getPriority(void)
{
	const char *cp = odbc.getLast("priority");
	if(!cp)
		cp = keythreads.getLast("sql");
	if(cp)
		return atoi(cp);

	return 0;
}

size_t ODBCModule::getStack(void)
{
	const char *cp = odbc.getLast("odbcstack");
	if(!cp)
		cp = keythreads.getLast("odbcstack");

	if(!cp && sizeof(void *) > 4)
		cp = "256";

	if(!cp)
		cp = "160";

	return atoi(cp) * 1024;
}


ODBCModule odbc;

#ifdef	CCXX_NAMESPACES
}
#endif
