# Copyright (C) 2000-2001 Open Source Telecom Corporation.
#
# This file is free software; as a special exception the author gives
# unlimited permission to copy and/or distribute it, with or without
# modifications, as long as this notice is preserved.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY, to the extent permitted by law; without even the
# implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# Bayonne perl support for controlling Bayonne 2.x server

package BAYONNE;

use KEYDATA;
use IO::Socket::UNIX;
use Attribute::Protected;
use enum qw(:Error_ Success Failed Arguments Invalid Session Connect Initial);

$SESSION{trunkid} = undef;
$VERSION = "1.0";
$ERROR = Error_Initial;
$VOICE = "sys";
$EXTENSION = ".au";
$KEYCONFIG = "/etc/bayonne";
if($ENV{'CONFIG_KEYDATA'})
{
	$KEYCONFIG=$ENV{'CONFIG_KEYDATA'};
}

KEYDATA::load "$KEYCONFIG/server.conf";

sub pathname {
	my($path) = shift;
	my($prefix) = shift;
	my($voice) = shift;
	my($extension) = shift;
	my($datafiles) = $KEYDATA::PATHS{datafiles};
	my($voices) = $KEYDATA::PATHS{voices};
	my($prompts) = $KEYDATA::PATHS{prompts};
	my($tmp) = "/tmp/.bayonne";
	my($tmpfs) = "/dev/shm/.bayonne";

	unless(-d "/dev/shm") {
		$tmpfs = $tmp;
	}

	if(defined $KEYDATA::PATHS{tmpfs}) {
		$tmpfs = "$KEYDATA::PATHS{tmpfs}/.bayonne";
	}

	if(!defined $voice) {
		$voice = "$voices/$VOICE";
	} else {
		$voice = "$voices/$voice";
	}

	if(!defined $extension) {
		$extension = $EXTENSION;
	}

	if(!defined $prefix) {
		$prefix = $datafiles;
	} else {
		$prefix = "$datafiles/$prefix";
	}
	
	if(index($path, ':') < 0) {
		if(index($path, '/') > -1) {
			if(index($path, '.', index($path, '/')) > -1) {
				return "$prefix/$path";
			}
			return "$prefix/$path$extension";
		}
		if(index($path, '.') > -1) {
			return "$prompts/$path";
		}
		return "$voice/$path$extension";	
	}

	if(substr($path, 0, 5) eq "http:") {
		return $path;
	}

	if((substr($path, 0, 4) eq "tmp:") || (substr($path, 0, 5) eq "temp:")) {
		$path = substr($path, index($path, ':') + 1);
		if(index($path, '.') > -1) {
			return "$tmp/$path";
		} else {
			return "$tmp/$path$extension";
		}
	}

        if((substr($path, 0, 4) eq "tmp:") || (substr($path, 0, 5) eq "temp:")) {
                $path = substr($path, index($path, ':') + 1);
                if(index($path, '.') > -1) {
                        return "$tmp/$path";
                } else {
                        return "$tmp/$path$extension";
                }
        }

        if((substr($path, 0, 4) eq "ram:") || (substr($path, 0, 4) eq "mem:")) {
                $path = substr($path, index($path, ':') + 1);
                if(index($path, '.') > -1) {
                        return "$tmpfs/$path";
                } else {
                        return "$tmpfs/$path$extension";
                }
        }

	if(substr($path, 0, 8) eq "prompts:") {
		$path = substr($path, 8);
		if(index($path, '.') > -1) {
			return "$prompts/$path";
		} else {
			return "$prompts/$path/$extension";
		}
	}

	my($section) = substr($path, 0, index($path, ':') - 1);
	$path = substr($path, index($path, ':') + 1);
	if(index($path, '.') > -1) {
		return "$voices/$section/$path";
	}
	return "$voices/$section/$path$extension";
}

sub control : Private {
	my($command) = shift;
	my($client) = "$KEYDATA::PATHS{runfiles}/perl-tmp-$$-";
	my($server) = "$KEYDATA::PATHS{runfiles}/bayonne.packet";
	my($data);

	my $packet = IO::Socket::UNIX->new(
		Local => $client,
		Peer => $server,
		Type => SOCK_DGRAM);

	if(!defined $packet) {
		$ERROR = Error_Connect;
		return Error_Connect;
	}

	$packet->send("X?$command\0", 0);
	$packet->recv($data, 16, 0);
	close($packet);
	unlink($client);

	if(!defined $data) {
		$ERROR = Error_Connect;
		return Error_Connect;
	}
	$data = substr($data, 1);
	if($data == "0") {
		$ERROR = Error_Failed;
		return Error_Failed;
	}

	$ERROR = Error_Success;

	if(substr($data, 0, 1) == "S")
	{
		$data;
		$_ =~ /(.*?)=(.*)/;
		my($name, $value) = ($1, $2);
		if(substr($name, 0, 9) == "Ssession.") {
			$SESSION{substr($name, 9)} = $value;
		}
	}

	$ERROR = Error_Success;
	return Error_Success;
}
	
sub compile
{
	my($section) = shift;
	if(!defined $section) {
		$section = "-bayonne";
	}
	return control("compile $section");	
}

sub start($$\%)
{
	my($class) = shift;
	my($script) = shift;
	my(%vars) = %{(shift)};
	my($command) = "start $class $script";
	my($key);
	foreach $key (keys %vars) {
		my($value) = "$vars{$key}";
		$command = "$command $key=$value";
	}
	if(control($command) == Error_Success) {
		return $SESSION{trunkid};
	}
	return undef;
} 

sub stop
{
        my($ref) = shift;
        if(!defined $ref) {
                $ERROR = Error_Arguments;
                return Error_Arguments;
        }

        if(substr($ref, 0, 1) != "-") {
                $ERROR = Error_Session;
                return Error_Session;
        }

        return control("stop $ref");
}

sub drop
{
	my($ref) = shift;
	if(!defined $ref) {
		$ERROR = Error_Arguments;
		return Error_Arguments;
	}

	if(substr($ref, 0, 1) != "-") {
		$ERROR = Error_Session;
		return Error_Session;
	}

	return control("hangup $ref");
}

sub port
{
	my($driver, $port, $mode) = @_;
	if(!defined $mode) {
		$ERROR = Error_arguments;
		return Error_Arguments;
	}

	if(($mode == "idle") || ($mode == "up")) {
		return control("idle $driver/$port");
	} elsif($mode == "busy") {
		return control("busy $driver/$port");
	}

	$ERROR = Error_Invalid;
	return Error_Invalid;
}

sub span
{
        my($driver, $span, $mode) = @_;
        if(!defined $mode) {
                $ERROR = Error_arguments;
                return Error_Arguments;
        }

        if(($mode == "idle") || ($mode == "up")) {
                return control("span $driver $span up");
        } elsif($mode == "busy") {
                return control("span $driver $span busy");
	} elsif(($mode == "down") || ($mode == "stop")) {
		return control("span $driver $span down");
	}

        $ERROR = Error_Invalid;
        return Error_Invalid;
}

sub card
{
        my($driver, $card, $mode) = @_;
        if(!defined $mode) {
                $ERROR = Error_arguments;
                return Error_Arguments;
        }

        if(($mode == "idle") || ($mode == "up")) {
                return control("card $driver $card up");
        } elsif($mode == "busy") {
                return control("card $driver $card busy");
        } elsif(($mode == "down") || ($mode == "stop")) {
                return control("card $driver $card down");
	}

        $ERROR = Error_Invalid;
        return Error_Invalid;
}

sub down
{
	my($service) = shift;
	if(!defined $service) {
		return control("down");
	}

	return control("service $service");	
}

sub up
{
	return control("up");
}

sub clearIncoming
{
	my($id) = shift;
	if(!defined $id) {
		$id = "default";
	}
	return control("route clear incoming $id");
}

sub setIncoming
{
        my($id) = shift;
        my($value) = shift;
	if(!defined $value) {
		return control("route clear incoming $id");
	} else {
        	return control("route set incoming $id $value");
	}
}

1;
