// Copyright (C) 2000 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"
#include <cc++/process.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

PostgresModule::PostgresModule() : Database()
{
	const char *cp = getLast("server");
	if(!cp)
		cp = getLast("host");

	if(cp)
		Process::setEnv("PGHOST", cp, true);

	cp = getLast("port");
	if(cp)
		Process::setEnv("PGPORT", cp, true);
}

PostgresModule::~PostgresModule()
{
	PostgresQueue *node = PostgresQueue::first;
	PostgresQueue *next;

	while(node)
	{
		next = node->next;
		delete node;
		node = next;
	}
}


size_t PostgresModule::getStack(void)
{
	const char *cp = keythreads.getLast("pgstack");

	if(!cp && sizeof(void *) > 4)
		cp = "160";

	if(!cp)
		cp = "96";

	return atoi(cp) * 1024;
}

int PostgresModule::getPriority(void)
{
	const char *cp = postgres.getLast("priority");
	if(!cp)
		cp = keythreads.getLast("sql");
	if(cp)
		return atoi(cp);

	return 0;
}

bool PostgresModule::header(ScriptInterp *interp)
{
	const char *sz = interp->getKeyword("size");
        PGresult *res = (PGresult *)interp->getPointer(SYM_SQLRESULTS);
	unsigned cols, col = 0;
	unsigned size = interp->getSymbolSize();
	Symbol *sym;
	const char *cp;

	if(sz)
		size = atoi(sz);

	if(!res)
		return false;

	cols = PQnfields(res);
	if(!cols)
		return false;

	while(col < cols)
	{
		sym = interp->initVariable(size);
		if(!sym)
			break;

		if(sym->flags.readonly)
			continue;

		cp = PQfname(res, col);
		if(!cp)
		{
			++col;
			continue;
		}
		snprintf(sym->data, sym->flags.size + 1, "%s", cp);
		if(sym->flags.commit)
			interp->commit(sym);
		++col;
	}
	return true;
}

bool PostgresModule::fetch(ScriptInterp *interp, unsigned row)
{
	const char *sz = interp->getKeyword("size");
	const char *cp = interp->getKeyword("first");
	PGresult *res = (PGresult *)interp->getPointer(SYM_SQLRESULTS);
	unsigned cols, col = 0;
	unsigned size = interp->getSymbolSize();
	Symbol *sym = NULL;
	char buf[11];
	char field[65];

	if(!res)
		return false;

	if(sz)
		size = atoi(sz);

	if((long)row >= PQntuples(res))
		return false;

	cols = PQnfields(res);
	if(!cols)
		return false;

	snprintf(buf, sizeof(buf), "%d", row + 1);
	interp->setSymbol(SYM_SQLROW, buf);

	if(cp)
	{
		col = atoi(cp) - 1;
		if(col >= cols)
			return false;
	}

	while(col < cols)
	{
		sym = interp->initVariable(size);
		if(!sym)
			break;
		if(sym->flags.readonly)
			continue;
		cp = PQgetvalue(res, row, col);
		snprintf(sym->data, sym->flags.size + 1, "%s", cp);
		if(sym->flags.commit)
			interp->commit(sym);
		++col;
	}

	col = 0;
	field[0] = '&';
	while(col < cols)
	{
		cp = PQfname(res, col);		
		if(!cp || !*cp)
		{
			++col;
			continue;
		}
		snprintf(field + 1, sizeof(field) - 1, "%s", cp);
		cp = interp->getKeyword(field);
		if(!cp || *cp != '&')
		{
			++col;
			continue;
		}
		if(strchr(cp, '.'))
			sym = interp->getEntry(cp, size);
		else
			sym = interp->getLocal(cp, size);
		if(!sym || sym->flags.readonly)
		{
			++col;
			continue;
		}
		cp = PQgetvalue(res, row, col);
		if(!cp)
			cp = "";
		snprintf(sym->data, sym->flags.size + 1, "%s", cp);
		if(sym->flags.commit)
			interp->commit(sym);
		++col;
	}
	return true;
}

Service *PostgresModule::getService(Trunk *trunk, Line *line, trunkdata_t *data)
{
	const char *mem = trunk->getMember();
	ScriptCommand *cmd = trunk->getCommand();
	ThreadQueue *tq = cmd->getThreadQueue();
	const char *dsn = cmd->getLast("database");
	Symbol *sym = NULL;
	const char *results;
	char buffer[512];
	unsigned len = 0;
	const char *query;

	if(!dsn)
		dsn = cmd->getLast("dsn");

	if(!dsn && cmd == aascript)
		dsn = getLast("database");

	if(!dsn)
	{
		trunk->error("no-database");
		return NULL;
	}

	if(!mem)
		mem = "";

	if(!stricmp(mem, "detach") || !stricmp(mem, "post"))
	{
		if(!tq)
		{
			trunk->error("no-queue");
			return NULL;
		}
		query = trunk->getKeyword("query");
		if(query)
		{
			tq->post(query, strlen(query) + 1);
			trunk->advance();
			return NULL;
		}
		while(NULL != (query = trunk->getValue(NULL)) && len < sizeof(buffer))
		{
			snprintf(buffer + len, sizeof(buffer) - len, "%s", query);
			len += strlen(query);
		}
		if(len < sizeof(buffer))
			++len;
		tq->post(buffer, len);
		trunk->advance();
		return NULL;
	}

	if(!stricmp(mem, "fetch") || !stricmp(mem, "header"))
	{
		trunk->advance();
		return NULL;
	}

	if(!stricmp(mem, "end") || !stricmp(mem, "exit"))
	{
		trunk->advance();
		return NULL;
	}

	if(!dsn)
	{
		trunk->error("no-database");
		return NULL;
	}

	results = trunk->getKeyword("results");
	if(!results)
		results = trunk->getKeyword("save");
	if(results && *results != '&')
		results = NULL;

	if(results)
		sym = trunk->getLocal(++results, 0);
	else
		return new PostgresThread(dsn, NULL, trunk);

	if(!sym)
	{
		trunk->error("no-symbol-to-save");
		return NULL;
	}

	switch(sym->flags.type)
	{
	case ScriptModule::ARRAY:
		sym->data[0] = sym->data[1];
		break;
	case ScriptModule::FIFO:
	case ScriptModule::SEQUENCE:
	case ScriptModule::STACK:
	case ScriptModule::CACHE:
		sym->data[1] = sym->data[2] = 0;
		break;
	default:
		trunk->error("symbol-invalid-type");
		return NULL;
	}
	return new PostgresThread(dsn, results, trunk);
}

void PostgresModule::moduleAttach(ScriptInterp *interp)
{
	ScriptCommand *cmd = interp->getCommand();
	const char *dsn;
	static Mutex lock;

	Database::moduleAttach(interp);
	interp->setConst(SYM_SQLDRIVER, "postgres");
	interp->setSymbol(SYM_SQLROW, "0");

	dsn = cmd->getLast("dsn");
	if(!dsn)
		dsn = cmd->getLast("database");

	if(!dsn && cmd == aascript)
		dsn = getLast("database");

	if(!dsn)
		return;

	lock.enterMutex();
	if(cmd->getThreadQueue())
	{
		lock.leaveMutex();
		return;
	}
	cmd->setThreadQueue(new PostgresQueue(cmd));
	lock.leaveMutex();
}

void PostgresModule::moduleDetach(ScriptInterp *interp, const char *script)
{
	ScriptCommand *cmd = interp->getCommand();
	PGconn *conn = NULL;
	PGresult *res = (PGresult *)interp->getPointer(SYM_SQLRESULTS);
        const char *cs = interp->getSymbol(SYM_SQLCONNECT);
	PostgresQueue *tq = (PostgresQueue *)cmd->getThreadQueue();

	if(tq)
		tq->cdr(interp);

	if(cmd != aascript)
		tq = (PostgresQueue *)aascript->getThreadQueue();
	else
		tq = NULL;

	if(tq)
		tq->cdr(interp);

	if(res)
		PQclear(res);

        if(cs && *cs)
		conn = (PGconn *)cmd->endDatabase();

        if(conn)
        {
                slog.debug("pgsql: %s: disconnecting", cs);
		PQfinish(conn);
        }
}

PostgresModule postgres;

#ifdef	CCXX_NAMESPACES
}
#endif
