// Copyright (C) 2000 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

PostgresThread::PostgresThread(const char *dsn, const char *r, Trunk *trk) :
Service(trk, postgres.getPriority(), postgres.getStack())
{
	dbname = dsn;
	results = r;
}

void PostgresThread::run(void)
{
	ScriptCommand *cmd = trunk->getCommand();
	PGresult *res = (PGresult *)trunk->getPointer(SYM_SQLRESULTS);
	PGconn *conn = (PGconn *)cmd->getDatabase();	
	char buf[256];
	const char *query = trunk->getKeyword("query");
	const char *user, *password;
	unsigned len = 0;
	const char *opt;
	const char *errmsg;
	unsigned row = 0, col = 0, rows, cols;
	const char *token = trunk->getKeyword("token");

	if(!token)
		token = ",";

	cmd->dblock.enterMutex();
	if(!conn)
	{

		user = postgres.getLast("user");
		password = postgres.getLast("password");

		if(password)
			snprintf(buf, sizeof(buf), "dbname=%s user=%s password=%s",
				dbname, user, password);
		else if(user)
			snprintf(buf, sizeof(buf), "dbname=%s user=%s",
				dbname, user);
		else
			snprintf(buf, sizeof(buf), "dbname=%s", dbname);

		conn = PQconnectdb(buf);
		if(!conn)
		{
			errlog("failed", "Database=%s, Error=unknown", dbname);
			cmd->dblock.leaveMutex();
			Service::failure();
		}
		if(PQstatus(conn) == CONNECTION_BAD)
		{
			errlog("missing", "Database=%s, Error=%s", dbname, PQerrorMessage(conn));
			PQfinish(conn);
			cmd->dblock.leaveMutex();
			Service::failure();
		}
		cmd->setDatabase(conn);
                trunk->setConst(SYM_SQLCONNECT, dbname);
                slog.debug("pgsql: %s: connected", dbname);
	}
	cmd->dblock.leaveMutex();

	trunk->setPointer(SYM_SQLRESULTS, NULL);
	if(res)
		PQclear(res);
	
	if(!query && !trunk->getMember())
	{
		query = buf;
		while(len < sizeof(buf) && NULL != (opt = trunk->getValue(NULL)))
		{
			snprintf(buf + len, sizeof(buf) - len, "%s", opt);
			len = strlen(buf);
		}
	}
	if(!query)
		Service::success();

	trunk->setSymbol(SYM_SQLROW, "0");
	errmsg = NULL;

	res = PQexec(conn, query);
        switch(PQresultStatus(res))
        {
        case PGRES_FATAL_ERROR:
                errmsg = PQresultErrorMessage(res);
                errlog("failed", "Database=%s; Error=%s", dbname, errmsg);
                break;
        case PGRES_NONFATAL_ERROR:
        case PGRES_BAD_RESPONSE:
                errmsg = PQresultErrorMessage(res);
		errlog("failed", "Database=%s; Error=%s", dbname, errmsg);
	case PGRES_COMMAND_OK:
	case PGRES_COPY_OUT:
	case PGRES_COPY_IN:
		snprintf(buf, sizeof(buf), "%d", PQcmdTuples(res));
		trunk->setSymbol(SYM_SQLCHANGES, buf);
 	}

	if(errmsg)
	{
		trunk->setSymbol(SYM_SQLERROR, errmsg);
		Service::failure();
	}
	else
	{
		snprintf(buf, sizeof(buf), "%d", PQntuples(res));
		trunk->setSymbol(SYM_ROWS, buf);
		snprintf(buf, sizeof(buf), "%d", PQnfields(res));
		trunk->setSymbol(SYM_COLS, buf);
	}

	if(PQresultStatus(res) != PGRES_TUPLES_OK)
	{
		PQclear(res);
		results = NULL;
	}
	else
		trunk->setPointer(SYM_SQLRESULTS, res);

	if(!results)
		Service::success();

	rows = PQntuples(res);
	if(rows < 1)
		Service::success();

	cols = PQnfields(res);
	while(row < rows)
	{
		col = 0;
		while(col < cols)
		{
			if(!col)
				snprintf(buf, sizeof(buf), "%s", PQgetvalue(res, row, col));
			else
				snprintf(buf + len, sizeof(buf) - len, "%c%s", *token, PQgetvalue(res, row, col));
			++col;
			len = strlen(buf);
		}
		trunk->setVariable(results, 0, buf);
		++row;
	}

	Service::success();
}

#ifdef	CCXX_NAMESPACES
}
#endif
