// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release 
// of ccscript.
// 
// The exception is that, if you link the ccscript library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccscript library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccscript.  If you copy code from other releases into a copy of
// ccscript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for ccscript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include <cc++/config.h>
#include <ctime>
#include "bayonnescript.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
#endif

static class DateProperty : public Script::Property
{
public:
	DateProperty() : Property("date") {};
	DateProperty(const char *id) : Property(id) {};

private:
	virtual void setProperty(char *dp, char *tp, unsigned size);
	virtual void getProperty(char *dp, char *tp, unsigned size);
	virtual long getValue(const char *dp);
	virtual bool isProperty(const char *dp);
	virtual void setValue(char *dp, unsigned size, long value);
	virtual long adjustValue(long value);
	unsigned getPropertySize(void)
		{return 10;};
} pDate;

static long toJulian(long year, long month, long day)
{
	long julian;

        if(year < 0)
                year--;

        julian = day - 32075l +
                1461l * (year + 4800l + ( month - 14l) / 12l) / 4l +
                367l * (month - 2l - (month - 14l) / 12l * 12l) / 12l -
                3l * ((year + 4900l + (month - 14l) / 12l) / 100l) / 4l;
	return julian;
}

static class YearProperty : public DateProperty
{
public:
	YearProperty() : DateProperty("year") {};
	
private:
	void setProperty(char *dp, char *tp, unsigned size);

	void getProperty(char *dp, char *tp, unsigned size)
		{setProperty(tp, dp, size);};

	unsigned getPropertySize(void)
		{return 4;};
} pYear;

static class WeekdayProperty : public DateProperty
{
public:
	WeekdayProperty() : DateProperty("weekday") {};

private:
	void setProperty(char *dp, char *tp, unsigned size);

	void getProperty(char *dp, char *tp, unsigned size)
		{setProperty(tp, dp, size);};

	unsigned getPropertySize(void)
		{return 10;};
} pWeekday;

static class MonthProperty : public DateProperty
{
public:
	MonthProperty() : DateProperty("month") {};

private:
	void setProperty(char *dp, char *tp, unsigned size);

	void getProperty(char *dp, char *tp, unsigned size)
		{setProperty(tp, dp, size);};

	unsigned getPropertySize(void)
		{return 10;};
} pMonthof;

bool DateProperty::isProperty(const char *dp)
{
	if(strchr(dp, '.'))
		return true;

	if(strchr(dp, '-'))
		return true;

	if(strchr(dp, '/'))
		return true;

	return false;
}

long DateProperty::adjustValue(long value)
{
	if(value < 0)
		return -value;
	return value;
}

void DateProperty::setValue(char *dp, unsigned size, long julian)
{
	// difference between dates, not really a date itself...

	if(julian < 365 * 256)
	{
		snprintf(dp, size + 1, "%ld", julian);
		return;
	}

        double i, j, k, l, n;

        l = julian + 68569.0;
        n = int( 4 * l / 146097.0);
        l = l - int( (146097.0 * n + 3)/ 4 );
        i = int( 4000.0 * (l+1)/1461001.0);
        l = l - int(1461.0*i/4.0) + 31.0;
        j = int( 80 * l/2447.0);
        k = l - int( 2447.0 * j / 80.0);
        l = int(j/11);
        j = j+2-12*l;
        i = 100*(n - 49) + i + l;

	if(size > 9)
		snprintf(dp, 11, "%04d%c%02d%c%02d", int(i), '-', int(j), '-', int(k));
	else if(size > 7)
		snprintf(dp, 9, "%02d/%02d/%02d", int(k), int(j), int(i));
	else if(size > 4)
		snprintf(dp, 6, "%02d/%02d", int(k), int(j));
	else
		*dp = 0;
}

long DateProperty::getValue(const char *dp)
{
	int year = 0, mon = 0, day = 0;
	char *ext;

	if(NULL != (ext = strchr(dp, '/')))
	{
		mon = atoi(dp);
		day = atoi(++ext);
		ext = strchr(ext, '/');
		if(ext)
			year = atoi(++ext);
	}
	else if(NULL != (ext = strchr(dp, '.')))
	{
		day = atoi(dp);
		mon = atoi(++ext);
		ext = strchr(ext, '.');
		if(ext)
			year = atoi(++ext);
	}
	else if(NULL != (ext = strchr(dp, '-')))
	{
		year = atoi(dp);
		mon = atoi(++ext);
		ext = strchr(ext, '-');
		if(ext)
			day = atoi(++ext);
	}
	else if(strlen(dp) > 7)
		sscanf(dp, "%04d%02d%02d", &year, &mon, &day);
	else if(strlen(dp) > 5)
		sscanf(dp, "%02d%02d%02d", &year, &mon, &day);
	else
		sscanf(dp, "%02d%02d", &mon, &day);

	return toJulian(year, mon, day);
}


void DateProperty::getProperty(char *dp, char *tp, unsigned size)
{
	int year = 0, mon = 0, day = 0;
	char *ext;
	time_t now;
	struct tm *dt;

	if(NULL != (ext = strchr(dp, '/')))
	{
		mon = atoi(dp);
		day = atoi(++ext);
		ext = strchr(ext, '/');
		if(ext)
			year = atoi(++ext);
	}
	else if(NULL != (ext = strchr(dp, '.')))
	{
		day = atoi(dp);
		mon = atoi(++ext);
		ext = strchr(ext, '.');
		if(ext)
			year = atoi(++ext);
	}
	else if(NULL != (ext = strchr(dp, '-')))
	{
		year = atoi(dp);
		mon = atoi(++ext);
		ext = strchr(ext, '-');
		if(ext)
			day = atoi(++ext);
	}
	else if(strlen(dp) > 7)
		sscanf(dp, "%04d%02d%02d", &year, &mon, &day);
	else if(strlen(dp) > 5)
		sscanf(dp, "%02d%02d%02d", &year, &mon, &day);
	else
		sscanf(dp, "%02d%02d", &mon, &day);

	if(!year)
	{
		time(&now);
		dt = localtime(&now);
		++dt->tm_mon;
		year = dt->tm_year + 1900;
		if(dt->tm_mday > day && dt->tm_mon == mon)
			++year;
		if(dt->tm_mon > mon)
			++year;
		if(!mon)
			mon = dt->tm_mon;
		if(!day)
			day = dt->tm_mday;
	}
	if(size < 4)
		return;
	if(size < 8)
		snprintf(tp, size, "%02d%02d", mon, day);
	else if(size < 10)
		snprintf(tp, size, "%04d%02d%02d", year, mon, day);
	else
		snprintf(tp, size, "%04d-%02d-%02d", year, mon, day);
	return;
}

void DateProperty::setProperty(char *dp, char *tp, unsigned size)
{
	time_t now;
	struct tm *dt;
	unsigned sz = 10;
	unsigned year = 0, mon = 0, day = 0;

	if(size < 10)
		sz = size;

	strncpy(dp, "0000-00-00", sz);
	if(size < 5)
		return;

	sz = (unsigned)strlen(tp);
	time(&now);
	dt = localtime(&now);
	dt->tm_year += 1900;
	++dt->tm_mon;

	if(!sz)
	{
		time(&now);
		dt = localtime(&now);
		++dt->tm_mon;
		dt->tm_year += 1900;
		if(size < 8)
			snprintf(dp, size + 1,
				"%02d/%02d", dt->tm_mon, dt->tm_mday);
		else if(size < 10)
			sprintf(dp, "%04d%02d%02d",
				dt->tm_year, dt->tm_mon, dt->tm_mday);
		else
			sprintf(dp, "%04d-%02d-%02d",
				dt->tm_year, dt->tm_mon, dt->tm_mday);
		return;
	}
	if(strchr(tp, '/'))
		sscanf(tp, "%02u/%02u/%04u", &mon, &day, &year);
	else if(strrchr(tp, '.'))
		sscanf(tp, "%02u.%02u.%04u", &day, &mon, &year);
	else if(strchr(tp, '-'))
		sscanf(tp, "%04u-%02u-%02u", &year, &mon, &day);
	else if(strlen(tp) > 7)
		sscanf(tp, "%04u%02u%02u", &year, &mon, &day);
	else if(strlen(tp) > 5)
		sscanf(tp, "%02u%02u%02u", &year, &mon, &day);
	else sscanf(tp, "%02u%02u", &mon, &day);


	if(!year)
		year = dt->tm_year;

	if(!mon)
		mon = dt->tm_mon;

	if(!day)
		day = dt->tm_mday;

	if(size < 5)
		return;
	if(size < 10)
		snprintf(dp, size + 1, "%02u/%02u", mon, day);
	else
		snprintf(dp, size + 1, "%04u-%02u-%02u", year, mon, day);
}

void YearProperty::setProperty(char *dp, char *tp, unsigned size)
{
	time_t now;
	struct tm *dt, dtm;
	unsigned sz = 10;
	unsigned year = 0, mon = 0, day = 0;

	sz = (unsigned)strlen(tp);
	time(&now);
	dt = localtime(&now);
	dt->tm_year += 1900;
	++dt->tm_mon;

	if(!sz)
	{
		time(&now);
		dt = localtime(&now);
		dt->tm_year += 1900;
		snprintf(dp, size + 1,
			"%04d", dt->tm_year);
		return;
	}
	if(sz < 4)
	{
		time(&now);
		dt = localtime(&now);
		dt->tm_year += 1900;
		snprintf(dp, size + 1,
			"%04d", dt->tm_year);
		return;
	}

	if(strchr(tp, '/'))
		sscanf(tp, "%02u/%02u/%04u", &mon, &day, &year);
	else if(strrchr(tp, '.'))
		sscanf(tp, "%02u.%02u.%04u", &day, &mon, &year);
	else if(strrchr(tp, '-'))
		sscanf(tp, "%04u-%02u-%02u", &year, &mon, &day);
	else if(strlen(tp) > 7)
		sscanf(tp, "%04u%02u%02u", &year, &mon, &day);
	else if(strlen(tp) > 5)
		sscanf(tp, "%02u%02u%02u", &year, &mon, &day);
	else sscanf(tp, "%02u%02u", &mon, &day);

	if(!year)
		year = dt->tm_year;

	if(!mon)
		mon = dt->tm_mon;

	if(!day)
		day = dt->tm_mday;

	memset(&dtm, 0, sizeof(dtm));
	dtm.tm_mday = day;
	dtm.tm_mon = mon - 1;
	if(year > 200)
		dtm.tm_year = year - 1900;
	else
		dtm.tm_year = year;
	now = mktime(&dtm);
	dt = localtime(&now);
	snprintf(dp, size + 1, "%04d", 1900 + dt->tm_year);
}

void WeekdayProperty::setProperty(char *dp, char *tp, unsigned size)
{
	static	char *days[] =
		{"sunday", "monday", "tuesday", "wednesday",
		"thursday", "friday", "saturday"};

	time_t now;
	struct tm *dt, dtm;
	unsigned sz = 10;
	unsigned year = 0, mon = 0, day = 0;

	sz = (unsigned)strlen(tp);
	time(&now);
	dt = localtime(&now);
	dt->tm_year += 1900;
	++dt->tm_mon;

	if(!sz)
	{
		time(&now);
		dt = localtime(&now);
		++dt->tm_mon;
		dt->tm_year += 1900;
		snprintf(dp, 11, "%s", days[dt->tm_wday]);
		return;
	}
	if(sz < 4)
	{
		snprintf(dp, 11, "%s", days[atoi(tp) % 7]);
		return;
	}
	if(strchr(tp, '/'))
		sscanf(tp, "%02u/%02u/%04u", &mon, &day, &year);
	else if(strrchr(tp, '.'))
		sscanf(tp, "%02u.%02u.%04u", &day, &mon, &year);
	else if(strrchr(tp, '-'))
		sscanf(tp, "%04u-%02u-%02u", &year, &mon, &day);
	else if(strlen(tp) > 7)
		sscanf(tp, "%04u%02u%02u", &year, &mon, &day);
	else if(strlen(tp) > 5)
		sscanf(tp, "%02u%02u%02u", &year, &mon, &day);
	else sscanf(tp, "%02u%02u", &mon, &day);

	if(!year)
		year = dt->tm_year;

	if(!mon)
		mon = dt->tm_mon;

	if(!day)
		day = dt->tm_mday;

	memset(&dtm, 0, sizeof(dtm));
	dtm.tm_mday = day;
	dtm.tm_mon = mon - 1;
	if(year > 200)
		dtm.tm_year = year - 1900;
	else
		dtm.tm_year = year;
	now = mktime(&dtm);
	dt = localtime(&now);
	snprintf(dp, 11, "%s", days[dt->tm_wday]);
}

void MonthProperty::setProperty(char *dp, char *tp, unsigned size)
{
	static	char *months[] =
		{"january", "february", "march", "april",
		 "may", "june", "july", "august",
		 "september", "october", "november", "december"};

	time_t now;
	struct tm *dt, dtm;
	unsigned sz = 10;
	unsigned year = 0, mon = 0, day = 0;

	sz = (unsigned)strlen(tp);
	time(&now);
	dt = localtime(&now);
	dt->tm_year += 1900;
	++dt->tm_mon;

	if(!sz)
	{
		time(&now);
		dt = localtime(&now);
		dt->tm_year += 1900;
		snprintf(dp, 11, "%s", months[dt->tm_mon]);
		return;
	}
	if(sz < 4)
	{
		snprintf(dp, 11, "%s", months[(atoi(tp) - 1) % 12]);
		return;
	}

	if(strchr(tp, '/'))
		sscanf(tp, "%02u/%02u/%04u", &mon, &day, &year);
	else if(strrchr(tp, '.'))
		sscanf(tp, "%02u.%02u.%04u", &day, &mon, &year);
	else if(strrchr(tp, '-'))
		sscanf(tp, "%04u-%02u-%02u", &year, &mon, &day);
	else if(strlen(tp) > 7)
		sscanf(tp, "%04u%02u%02u", &year, &mon, &day);
	else if(strlen(tp) > 5)
		sscanf(tp, "%02u%02u%02u", &year, &mon, &day);
	else sscanf(tp, "%02u%02u", &mon, &day);

	if(!year)
		year = dt->tm_year;

	if(!mon)
		mon = dt->tm_mon;

	if(!day)
		day = dt->tm_mday;

	memset(&dtm, 0, sizeof(dtm));
	dtm.tm_mday = day;
	dtm.tm_mon = mon - 1;
	if(year > 200)
		dtm.tm_year = year - 1900;
	else
		dtm.tm_year = year;
	now = mktime(&dtm);
	dt = localtime(&now);
	snprintf(dp, 11, "%s", months[dt->tm_mon]);
}


#ifdef	CCXX_NAMESPACES
}
#endif


