// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of ccscript.
//
// The exception is that, if you link the ccscript library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccscript library code into it.
//
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
//
// This exception applies only to the code released under the
// name ccscript.  If you copy code from other releases into a copy of
// ccscript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own for ccscript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.

#include "bayonneserver.h"
#include <sys/stat.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
#endif

static class FileEraseModule : public ScriptModule
{
private:
	char *parseScript(ScriptInterp *interp, Line *line);

public:
	FileEraseModule() : ScriptModule("erase") {};
} fileerase;

static class FileMoveModule : public ScriptModule
{
private:
	char *parseScript(ScriptInterp *interp, Line *line);

public:
	FileMoveModule() : ScriptModule("rename") {};
} filemove;

static class FileCopyModule : public Trunk::Threaded
{
private:
	char *parseScript(ScriptInterp *interp, Line *line);
	Service *getService(Trunk *trunk, Line *line, trunkdata_t *data);
	timeout_t getTimeout(Trunk *trunk, trunkdata_t *data);
public:
	FileCopyModule() : Trunk::Threaded("copy") {};
} filecopy;

static class FileAppendModule : public Trunk::Threaded
{
private:
	char *parseScript(ScriptInterp *interp, Line *line);
	Service *getService(Trunk *trunk, Line *line, trunkdata_t *data);
	timeout_t getTimeout(Trunk *trunk, trunkdata_t *data);
public:
	FileAppendModule() : Trunk::Threaded("append") {};
} fileappend;

static class FileBuildModule : public Trunk::Threaded
{
private:
	char *parseScript(ScriptInterp *interp, Line *line);
	Service *getService(Trunk *trunk, Line *line, trunkdata_t *data);
	timeout_t getTimeout(Trunk *trunk, trunkdata_t *data);
public:
	FileBuildModule() : Trunk::Threaded("build") {};
} filebuild;

static class FileInfoModule : public Trunk::Threaded
{
private:
	char *parseScript(ScriptInterp *interp, Line *line);
	Service *getService(Trunk *trunk, Line *line, trunkdata_t *data);
	timeout_t getTimeout(Trunk *trunk, trunkdata_t *data);
public:
	FileInfoModule() : Trunk::Threaded("info") {};
} fileinfo;

static class FileListModule : public Trunk::Threaded
{
private:
	char *parseScript(ScriptInterp *interp, Line *line);
	Service *getService(Trunk *trunk, Line *line, trunkdata_t *data);
	timeout_t getTimeout(Trunk *trunk, trunkdata_t *data);
public:
	FileListModule() : Trunk::Threaded("list") {};
} filelist;

class ListThread : public Service
{
private:
        Dir *dir;
	char buffer[64];
	char symid[64];
	AudioFile au;

        void run(void);
public:
        ListThread(Trunk *trk, const char *prefix, const char *sym);
        ~ListThread();
};

class InfoThread : public Service, public Audio
{
private:
        AudioFile au;
        char buffer[65];
        void run(void);

public:
        InfoThread(Trunk *trk, const char *path);
        ~InfoThread();
};

class CopyThread : public Service, public Audio
{
private:
	writemode_t mode;
	Audio::Info info;
	AudioFile inp;
	AudioFile out;
	bool done;
	char *output, *voice;
	unsigned lpos;
	char obuffer[65];

public:
	char list[256];

	CopyThread(Trunk *trk, char *path, writemode_t mode);
	~CopyThread();
	void run(void);
};

CopyThread::CopyThread(Trunk *trk, char *o, writemode_t m) :
Service(trk, 0, 20000 + (PATH_MAX * 2))
{
	const char *fmt = trk->getKeyword("encoding");
	const char *ext;

	memset(&info, 0, sizeof(info));
	snprintf(obuffer, sizeof(obuffer) - 5, "%s", o);
	o = obuffer;

	ext = strrchr(o, '/');
	if(ext)
		ext = strrchr(ext, '.');
	else
		ext = strrchr(o, '.');

	if(!ext)
	{
		ext = trk->getKeyword("extension");

		if(!ext)
			ext = trk->getSymbol(ext);

		if(!ext)
			ext = ".au";

		strcat(obuffer, ext);
	}

	if(!fmt)
	{
		fmt = trk->getDefaultEncoding();
		if(!stricmp(fmt, "pcm"))
		{
			if(!stricmp(ext, ".au") || !stricmp(ext, ".snd"))
				fmt = "mulaw";
		}
	}

	done = false;
	output = o;
	mode = m;
	lpos = 0;

	if(mode == WRITE_MODE_APPEND)
		done = true;

	if(!fmt)
		fmt = trk->getSymbol(SYM_FORMAT);

	if(!fmt)
		fmt = "raw";

	info.format = raw;
	info.encoding = mulawAudio;
	info.annotation = trk->getKeyword("annotation");
	if(!info.annotation)
		info.annotation = "copied";

	if(!stricmp(ext, ".al"))
		fmt = "alaw";
	else if(!stricmp(ext, ".ul"))
		fmt = "ulaw";
        else if(!stricmp(ext, ".au") || !stricmp(ext, ".snd"))
        {
                info.format = snd;
                info.order = __BIG_ENDIAN;
        }
        else if(!stricmp(ext, ".wav"))
        {
                info.format = riff;
                info.order = __LITTLE_ENDIAN;
        }

        if(!stricmp(fmt, "adpcm") || !stricmp(fmt, "g721") || !stricmp(fmt, "g.721"))
                info.encoding = g721ADPCM;
	else if(!stricmp(fmt, "alaw"))
		info.encoding = alawAudio;
	else if(!stricmp(fmt, "ulaw") || !stricmp(fmt, "mulaw"))
		info.encoding = mulawAudio;
        else if(!stricmp(fmt, "g723") || !stricmp(fmt, "g.723"))
                info.encoding = g723_3bit;
        else if(!stricmp(fmt, "pcm") || !stricmp(fmt, "l16") || !stricmp(fmt, "linear"))
                info.encoding = pcm16Mono;
	else if(!stricmp(fmt, "g711") || !stricmp(fmt, "g.721"))
	{
		if(info.encoding != alawAudio)
			info.encoding = mulawAudio;
	}
	else if(stricmp(fmt, "raw"))
		info.encoding = unknownEncoding;
}

CopyThread::~CopyThread()
{
	char buffer[16];

	terminate();

	inp.close();

	if(!done)
	{
		remove(output);
		trunk->setSymbol(SYM_RECORDED, "0");
	}
	else
	{
		snprintf(buffer, sizeof(buffer), "%ld", out.getPosition());
		trunk->setSymbol(SYM_RECORDED, buffer);
	}

	out.close();
}

void CopyThread::run(void)
{
	AudioCodec *ic = NULL, *oc = NULL;

	short ibuffer[1024];
	short obuffer[1024];
	short iobuffer[1024];
	char buffer[1024];
	Linear ib, ob;
	int count;
	const char *path;
	const char *voice = trunk->getKeyword("voice");
	const char *extension = trunk->getKeyword("extension");
	const char *cp;
	unsigned sc;

	if(!extension)
		extension = trunk->getSymbol(SYM_EXTENSION);

	if(!permitAudioAccess(output, true))
	{
		done = true;
		trunk->setSymbol(SYM_ERROR, "write-access-denied");
		Service::failure();
	}

	if(!voice)
		voice = trunk->getSymbol(SYM_VOICE);

	switch(mode)
	{
	case WRITE_MODE_REPLACE:
		remove(output);
		out.create(output, &info);
		break;
	case WRITE_MODE_APPEND:
		out.open(output, modeWrite);
		if(!out.isOpen())
		{
			done = false;
			out.create(output, &info);
		}
		cp = trunk->getKeyword("offset");
		if(cp)
			out.setPosition(atol(cp));
		else
			out.setPosition();
		break;
	default:
		break;
	}

	if(!out.isOpen())
	{
		trunk->setSymbol(SYM_ERROR, "write-failed");
		Service::failure();
	}

	while(list[lpos])
	{
		while(list[lpos] == ',')
			++lpos;
		path = list + lpos;
		while(list[lpos] && list[lpos] != ',')
			++lpos;
		if(list[lpos] == ',')
			list[lpos++] = 0;

		setString(buffer, sizeof(buffer), path);
		path = getPrompt(buffer, voice);
		cp = strrchr(path, '/');
		if(cp)
			cp = strrchr(cp, '.');
		else
			cp = strrchr(path, '.');
		if(!cp)
		{
			snprintf(buffer, sizeof(buffer), "%s%s", path, extension);
			path = buffer;
		}

		inp.open(path, modeRead);
		if(!inp.isOpen())
			continue;

		if(inp.getEncoding() != out.getEncoding())
		{
			if(inp.getEncoding() != pcm16Mono)
				ic = AudioCodec::getCodec(inp.getEncoding());
			if(out.getEncoding() != pcm16Mono)
				oc = AudioCodec::getCodec(out.getEncoding());
		}

		for(;;)
		{
			count = inp.getBuffer(iobuffer, Audio::toBytes(inp.getEncoding(), 1024));
			if(count < 1)
				break;

			sc = Audio::toSamples(inp.getEncoding(), count);

			if(ic)
			{
				ic->decode(ibuffer, iobuffer, sc);
				ib = ibuffer;
				count = Audio::toBytes(pcm16Mono, sc);
			}
			else
				ib = iobuffer;

			if(oc)
			{
				oc->encode(ib, obuffer, sc);
				ob = obuffer;
				count = Audio::toBytes(out.getEncoding(), sc);
			}
			else
				ob = ib;

			out.putBuffer(ob, count);
		}

		inp.close();
	}

	done = true;
	Service::success();
}


ListThread::ListThread(Trunk *trk, const char *prefix, const char *sym) :
Service(trk, 0, 4096)
{
	if(!sym)
		sym = "";
	snprintf(symid, sizeof(symid), "%s", sym);
	snprintf(buffer, sizeof(buffer), "%s", prefix);
        dir = NULL;
}

ListThread::~ListThread()
{
        terminate();
	au.close();
        if(dir)
                delete dir;
        dir = NULL;
}

void ListThread::run(void)
{
        unsigned count = 0;
        const char *match = trunk->getKeyword("match");
        const char *suffix = trunk->getKeyword("extension");
        const char *cvar = trunk->getKeyword("count");
        const char *name, *ext;
	const char *mem = trunk->getMember();
	const char *token = trunk->getKeyword("token");
        char *cp;
        unsigned mlen = 0;
	char path[256];
	struct stat ino;
	struct tm *dt;
	char date[16], time[12];

	enum
	{
		NAME,
		TRIM,
		LONG
	}	fmt = NAME;

	if(!token)
		token = trunk->getSymbol("script.token");

	if(!token)
		token = ",";

	if(mem && !stricmp(mem, "trim"))
		fmt = TRIM;
	else if(mem && !stricmp(mem, "long"))
		fmt = LONG;

        if(match)
	        mlen = (unsigned)strlen(match);

	if(!suffix)
		suffix = trunk->getKeyword("suffix");
        if(!suffix)
                suffix = trunk->getSymbol(SYM_EXTENSION);

        if(cvar)
                trunk->setVariable(cvar, 5, "0");

        dir = new Dir(buffer);
        while(NULL != (name = dir->getName()))
        {
                if(*name == '.')
                        continue;

                if(match && strnicmp(match, name, mlen))
                        continue;

                ext = strrchr(name, '.');
                if(!ext)
                        continue;

                if(stricmp(suffix, ext))
                        continue;

                ++count;
		if(!symid[0])
			continue;

		snprintf(path, sizeof(path), "%s/%s", buffer, name);
		if(stat(path, &ino))
			ino.st_mtime = 0;
		dt = localtime(&ino.st_mtime);

		snprintf(date, sizeof(date), "%04d-%02d-%02d",
			dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
		snprintf(time, sizeof(time), "%02d:%02d:%02d",
			dt->tm_hour, dt->tm_min, dt->tm_sec);

		switch(fmt)
		{
		case LONG:
	                snprintf(path, sizeof(path), "%s/%s", buffer, name);
        	        if(stat(path, &ino))
                	        ino.st_mtime = 0;
                	dt = localtime(&ino.st_mtime);

			snprintf(date, sizeof(date), "%04d-%02d-%02d",
        	                dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
                	snprintf(time, sizeof(time), "%02d:%02d:%02d",
                        	dt->tm_hour, dt->tm_min, dt->tm_sec);

			au.open(path, Audio::modeInfo);
			if(au.isOpen())
			{
				au.setPosition();
				cp = au.getAnnotation();
				if(!cp)
					cp = "";
				snprintf(path, sizeof(path), "%s%c%s%c%s%c%ld%c%s",
					date, *token, time, *token,
					name, *token, au.getPosition(), *token, cp);
				au.close();
			}
			else
				snprintf(path, sizeof(path), "%s%c%s%c%s%c%c",
					date, *token, time, *token, 
					name, *token, *token);
			trunk->setVariable(symid, 0, path);
			break;
		case TRIM:
			snprintf(buffer, sizeof(buffer), "%s", name + mlen);
			cp = strrchr(buffer, '.');
			if(cp)
				*cp = 0;
			trunk->setVariable(symid, 0, buffer);
			break;
		default:
			trunk->setVariable(symid, 0, name);
			break;
		}

        }
        snprintf(buffer, sizeof(buffer), "%d", count);
	if(cvar)
	        trunk->setVariable(cvar, 5, buffer);
        Service::success();
}


InfoThread::InfoThread(Trunk *trk, const char *path) :
Service(trk, 0, 1024)
{
        snprintf(buffer, sizeof(buffer), "%s", path);
}

InfoThread::~InfoThread()
{
        terminate();

        if(!au.isOpen())
        {
                trunk->setSymbol(SYM_OFFSET, "0");
                trunk->setSymbol(SYM_RECORDED, "0");
        }
        au.close();
}
void InfoThread::run(void)
{
        struct stat ino;
        char msgbuf[32];
        char *ann;
        char *sym;
        unsigned hour, min;
        long duration;
        const char *mem = trunk->getMember();
        const char *ext = strrchr(buffer, '/');
        const char *mimetype = "none";
	char date[16], time[12];
	struct tm *dt;

        if(ext)
                ext = strrchr(ext, '.');
        else
                ext = strrchr(buffer, '.');

        if(!ext)
                ext = "";

        if(!stricmp(ext, ".snd") || !stricmp(ext, ".au"))
        {
                mimetype = "audio/basic";
                if(!mem)
                        mem = "snd";
        }
        else if(!stricmp(ext, ".wav"))
        {
                mimetype = "audio/x-basic";
                if(!mem)
                        mem = "snd";
        }
       else if(!stricmp(ext, ".vox"))
        {
                mimetype = "audio/voxware";
                if(!mem)
                        mem = "snd";
        }
        else if(!stricmp(ext, ".ps") || !stricmp(ext, ".eps"))
                mimetype = "application/postscript";
        else if(!stricmp(ext, ".gif"))
                mimetype = "image/gif";
        else if(!stricmp(ext, ".tif") || !stricmp(ext, ".tiff"))
                mimetype = "image/tiff";
        else if(!stricmp(ext, ".txt") || !stricmp(ext, ".text"))
                mimetype = "text/plain";
        else if(!stricmp(ext, ".htm") || !stricmp(ext, ".html"))
                mimetype = "text/html";
        else if(!stricmp(ext, ".g3f"))
                mimetype = "image/g3fax";
        if(!mem)
        {
                if(!stricmp(ext, ".al") || !stricmp(ext, ".gsm"))
                        mem = "snd";
                else
                        mem = "any";
        }

        if(!stricmp(mem, "snd"))
        {
                if(!permitAudioAccess(buffer, false))
                {
                        trunk->setSymbol(SYM_ERROR, "access-denied");
                        Service::failure();
                }
        }
        else
        {
                if(!permitFileAccess(buffer))
                {
                        trunk->setSymbol(SYM_ERROR, "access-denied");
                        Service::failure();
                }
        }

        if(stat(buffer, &ino))
        {
                trunk->setSymbol(SYM_ERROR, "cannot-stat");
                Service::failure();
        }

        if(!isFile(buffer))
        {
                trunk->setSymbol(SYM_ERROR, "not-file");
                Service::failure();
        }

        sym = trunk->getKeyword("type");
        if(sym && *sym == '&')
		trunk->setVariable(sym, 16, mimetype);

        sym = trunk->getKeyword("size");
        if(sym && *sym == '&')
        {
                snprintf(msgbuf, sizeof(msgbuf), "%ld", ino.st_size);
		trunk->setVariable(sym, 11, msgbuf);
        }

        if(stricmp(mem, "snd"))
                Service::success();

        au.open(buffer, modeInfo);
        if(!au.isOpen())
        {
                trunk->setSymbol(SYM_ERROR, "cannot-open");
                Service::failure();
        }

	dt = localtime(&ino.st_mtime);

        snprintf(date, sizeof(date), "%04d-%02d-%02d",
        	dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
        snprintf(time, sizeof(time), "%02d:%02d:%02d",
        dt->tm_hour, dt->tm_min, dt->tm_sec);

	sym = trunk->getKeyword("date");
	if(sym && *sym == '&')
		trunk->setVariable(sym, 0, date);

	sym = trunk->getKeyword("time");
	if(sym && *sym == '&')
		trunk->setVariable(sym, 0, time);

        au.setPosition();
        snprintf(msgbuf, sizeof(msgbuf), "%ld", au.getPosition());
        trunk->setSymbol(SYM_OFFSET, msgbuf);
        trunk->setSymbol(SYM_RECORDED, msgbuf);
        ann = au.getAnnotation();
        if(!ann)
                ann = "";
        trunk->setSymbol(SYM_ANNOTATION, ann);

        sym = trunk->getKeyword("offset");
        if(sym && *sym == '&')
        	trunk->setVariable(sym, 11, msgbuf);

        sym = trunk->getKeyword("encoding");
        if(sym && *sym == '&')
                trunk->setVariable(sym, 8, Trunk::getEncodingName(au.getEncoding()));

        sym = trunk->getKeyword("annotation");
        if(sym  && *sym == '&')
        	trunk->setVariable(sym, 0, ann);

        sym = trunk->getKeyword("duration");
        if(sym && *sym == '&')
        {
		trunk->setVariable(sym, 9);
                switch(au.getRate(au.getEncoding()))
                {
                default:
                case rateUnknown:
                        duration = -1l;
                        trunk->setVariable(sym, 9, "unknown");
                        break;
                case rate6khz:
                        duration = au.getPosition() / 6000l;
                        break;
                case rate8khz:
                        duration = au.getPosition() / 8000l;
                        break;
                }
                if(duration > -1)
                {
                        hour = duration / 3600l;
                        duration %= 3600l;
                        min = duration / 60l;
                        duration %= 60l;
                        snprintf(msgbuf, sizeof(msgbuf), "%02d:%02d:%02ld",
                                hour, min, duration);
                        trunk->setVariable(sym, 9, msgbuf);
                }
        }
        Service::success();
}

char *FileMoveModule::parseScript(ScriptInterp *interp, Line *line)
{
	ScriptCommand *cmd = interp->getCommand();
	const char *prefix = interp->getPrefixPath();
	char *n1 = interp->getValue(NULL);
	char *n2 = interp->getValue(NULL);
	const char *var = cmd->getLast("datafiles");
	char buf1[128], buf2[128];
	char v1[128], v2[128];
	const char *ext;

	if(!n1 || !n2)
		return "move-no-files";

	if(prefix && *prefix == '/')
	{
		var = prefix;
		prefix = NULL;
	}

	if(prefix)
	{
		snprintf(buf1, sizeof(buf1) - 5, "%s/%s", prefix, n1);
		snprintf(buf2, sizeof(buf2) - 5, "%s/%s", prefix, n2);
	}
	else
	{
		snprintf(buf1, sizeof(buf1) - 5, "%s", n1);
		snprintf(buf2, sizeof(buf2) - 5, "%s", n2);
	}
	n1 = buf1;
	n2 = buf2;

	ext = strrchr(n1, '/');
	if(ext)
		ext = strchr(ext, '.');
	else
		return "move-invalid-path";

	if(!ext)
	{
		ext = interp->getKeyword("extension");
		if(!ext)
			ext = interp->getSymbol(SYM_EXTENSION);

		if(ext)
			strcat(n1, ext);
	}

	ext = strrchr(n2, '/');
	if(ext)
		ext = strchr(ext, '.');
	else
		return "move-invalid-path";

	if(!ext)
	{
		ext = interp->getKeyword("extension");
		if(!ext)
			ext = interp->getSymbol(SYM_EXTENSION);

		if(ext)
			strcat(n2, ext);
	}

	if(var)
	{
		snprintf(v1, sizeof(v1), "%s/%s", var, n1);
		snprintf(v2, sizeof(v2), "%s/%s", var, n2);
		n1 = v1;
		n2 = v2;
	}

	if(!permitAudioAccess(n1, true) || !permitAudioAccess(n2, true))
		return "access-denied";
	else if(::rename(n1, n2))
		return "move-failed";
	return NULL;
}

char *FileEraseModule::parseScript(ScriptInterp *interp, Line *line)
{
	const char *prefix = interp->getPrefixPath();
	char *name = interp->getValue(NULL);
	char buffer[128];
	char vbuf[128];
	ScriptCommand *cmd = interp->getCommand();
	const char *var = cmd->getLast("datafiles");
	const char *ext;

	if(!name)
		return "erase-no-file";

	if(!strnicmp(name, "tmp:", 4) || !strnicmp(name, "temp:", 5))
	{
		var = NULL;
		prefix = keypaths.getLast("tmp");
		name = strchr(name, ':');
		++name;
	}
	else if(!strnicmp(name, "ram:", 4))
	{
		var = NULL;
		prefix = keypaths.getLast("tmpfs");
		if(!var)
			var = keypaths.getLast("tmp");
		name += 4;
	}

	if(prefix)
		snprintf(buffer, sizeof(buffer) - 5, "%s/%s", prefix, name);
	else
		snprintf(buffer, sizeof(buffer) - 5, "%s", name);
	name = buffer;

	ext = strrchr(name, '/');
	if(ext)
		ext = strchr(ext, '.');
	else
		return "erase-invalid-path";

	if(!ext)
	{
		ext = interp->getKeyword("extension");
		if(!ext)
			ext = interp->getSymbol(SYM_EXTENSION);
		if(ext)
			strcat(name, ext);
	}

	if(var)
	{
		snprintf(vbuf, sizeof(vbuf), "%s/%s", var, name);
		name = vbuf;
	}

	if(!permitAudioAccess(name, true))
		return "access-denied";

	::remove(name);
	return NULL;
}

char *FileCopyModule::parseScript(ScriptInterp *interp, Line *line)
{
	return NULL;
}

Service *FileCopyModule::getService(Trunk *interp, Line *line, trunkdata_t *data)
{
	char buffer[65];
	char vbuf[128];
	ScriptCommand *cmd = interp->getCommand();
	const char *prefix = interp->getPrefixPath();
	char *file = interp->getKeyword("file");
	const char *opt;
	const char *ext;
	const char *var = cmd->getLast("datafiles");
	unsigned len = 0;
	CopyThread *copy;

	if(!file)
		file = interp->getValue(NULL);

	ext = strrchr(file, '/');
	if(ext)
		ext = strrchr(ext, '.');
	else
		ext = strrchr(file, '.');

	if(!ext)
	{
		ext = interp->getKeyword("extension");
		if(!ext)
			ext = interp->getSymbol(SYM_EXTENSION);
	}
	else
		ext = "";

	if(prefix && *prefix == '/')
	{
		var = prefix;
		prefix = NULL;
	}

	if(!strnicmp(file, "tmp:", 4) || !strnicmp(file, "temp:", 5))
	{
		prefix = NULL;
		var = keypaths.getLast("tmp");
		file = strchr(file, ':');
		++file;
	}
	else if(!strnicmp(file, "ram:", 4))
	{
		prefix = NULL;
		var = keypaths.getLast("tmpfs");
		if(!var)
			var = keypaths.getLast("tmp");
		file += 4;
	}

	if(prefix)
		snprintf(buffer, sizeof(buffer), "%s/%s%s", prefix, file, ext);
	else
		snprintf(buffer, sizeof(buffer), "%s%s", file, ext);

	if(var)
	{
		snprintf(vbuf, sizeof(vbuf), "%s/%s", var, buffer);
		file = vbuf;
	}
	else
		file = buffer;

	copy = new CopyThread(interp, file, WRITE_MODE_REPLACE);

	while(len < 255 && (NULL != (opt = interp->getValue(NULL))))
	{
		snprintf(copy->list + len, 256 - len, ",%s", opt);
		len = (unsigned)strlen(copy->list);
	}
	return copy;
}

timeout_t FileCopyModule::getTimeout(Trunk *trunk, trunkdata_t *data)
{
	const char *cp = trunk->getKeyword("maxTime");
	if(!cp)
		cp = "60";

	return getSecTimeout(cp);
}

char *FileAppendModule::parseScript(ScriptInterp *interp, Line *line)
{
	return NULL;
}

Service *FileAppendModule::getService(Trunk *interp, Line *line, trunkdata_t *data)
{
	char buffer[65];
	char vbuf[128];
	ScriptCommand *cmd = interp->getCommand();
	const char *prefix = interp->getPrefixPath();
	char *file = interp->getKeyword("file");
	const char *opt;
	const char *ext;
	const char *var = cmd->getLast("datafiles");
	unsigned len = 0;
	CopyThread *copy;

	if(!file)
		file = interp->getValue(NULL);

	ext = strrchr(file, '/');
	if(ext)
		ext = strrchr(ext, '.');
	else
		ext = strrchr(file, '.');

	if(!ext)
	{
		ext = interp->getKeyword("extension");
		if(!ext)
			ext = interp->getSymbol(SYM_EXTENSION);
	}
	else
		ext = "";

	if(prefix && *prefix == '/')
	{
		var = prefix;
		prefix = NULL;
	}

	if(!strnicmp(file, "tmp:", 4) || !strnicmp(file, "temp:", 5))
	{
		prefix = NULL;
		var = keypaths.getLast("tmp");
		file = strchr(file, ':');
		++file;
	}
	else if(!strnicmp(file, "ram:", 4))
	{
		prefix = NULL;
		var = keypaths.getLast("tmpfs");
		if(!var)
			var = keypaths.getLast("tmp");
		file += 4;
	}

	if(prefix)
		snprintf(buffer, sizeof(buffer), "%s/%s%s", prefix, file, ext);
	else
		snprintf(buffer, sizeof(buffer), "%s%s", file, ext);

	if(var)
	{
		snprintf(vbuf, sizeof(vbuf), "%s/%s", var, buffer);
		file = vbuf;
	}
	else
		file = buffer;

	copy = new CopyThread(interp, file, WRITE_MODE_APPEND);

	while(len < 255 && (NULL != (opt = interp->getValue(NULL))))
	{
		snprintf(copy->list + len, 256 - len, ",%s", opt);
		len = (unsigned)strlen(copy->list);
	}
	return copy;
}


timeout_t FileAppendModule::getTimeout(Trunk *trunk, trunkdata_t *data)
{
	const char *cp = trunk->getKeyword("maxTime");
	if(!cp)
		cp = "60";

	return getSecTimeout(cp);
}

char *FileBuildModule::parseScript(ScriptInterp *interp, Line *line)
{
	return NULL;
}

Service *FileBuildModule::getService(Trunk *interp, Line *line, trunkdata_t *data)
{
	Translator *tts;
	char *err;
	char vbuf[128];
	char buffer[65];
	const char *lang = interp->getKeyword("language");
	const char *voice = interp->getKeyword("voice");
	char *file = interp->getKeyword("file");
	const char *var = interp->getKeyword("datafiles");
	const char *prefix = interp->getPrefixPath();
	const char *ext;
	CopyThread *copy;

	if(!file)
		file = interp->getValue(NULL);

	if(voice && !lang)
		lang = keyvoices.getLast(voice);

	if(!lang)
		lang = interp->getSymbol(SYM_LANGUAGE);

	if(!voice)
		voice = interp->getSymbol(SYM_VOICE);

	tts = getTranslator(lang);

	if(!tts)
	{
		interp->error("language-unsupported");
		return NULL;
	}

	err = tts->speak(interp);
	if(err)
	{
		interp->error(err);
		return NULL;
	}

	ext = strrchr(file, '/');
	if(ext)
		ext = strrchr(ext, '.');
        else
                ext = strrchr(file, '.');

        if(!ext)
        {
                ext = interp->getKeyword("extension");
                if(!ext)
                        ext = interp->getSymbol(SYM_EXTENSION);
        }
        else
                ext = "";

	if(prefix && *prefix == '/')
	{
		var = prefix;
		prefix = NULL;
	}

	if(!strnicmp(file, "tmp:", 4) || !strnicmp(file, "temp:", 5))
	{
		prefix = NULL;
		var = keypaths.getLast("tmp");
		file = strchr(file, ':');
		++file;
	}
	else if(!strnicmp(file, "ram:", 4))
	{
		prefix = NULL;
		var = keypaths.getLast("tmpfs");
		if(!var)
			var = keypaths.getLast("tmp");
		file += 4;
	}

        if(prefix)
                snprintf(buffer, sizeof(buffer), "%s/%s%s", prefix, file, ext);
        else
                snprintf(buffer, sizeof(buffer), "%s%s", file, ext);

	if(var)
	{
		snprintf(vbuf, sizeof(vbuf), "%s/%s", var, buffer);
		file = vbuf;
	}
	else
		file = buffer;

	copy = new CopyThread(interp, file, WRITE_MODE_REPLACE);
	if(!copy)
		interp->error("no-thread");
	else
		strcpy(copy->list, data->play.list);
	return copy;
}

timeout_t FileBuildModule::getTimeout(Trunk *trunk, trunkdata_t *data)
{
	const char *cp = trunk->getKeyword("maxTime");
	if(!cp)
		cp = "60";

	return getSecTimeout(cp);
}

char *FileInfoModule::parseScript(ScriptInterp *interp, Line *line)
{
	return NULL;
}

Service *FileInfoModule::getService(Trunk *trunk, Line *line, trunkdata_t *data)
{
        char buffer[65];
	char vbuf[65];
	ScriptCommand *cmd = trunk->getCommand();
        const char *prefix = trunk->getPrefixPath();
        const char *file = trunk->getKeyword("file");
        const char *cp;
	const char *ext = "";
	const char *var = cmd->getLast("datafiles");

        if(!file)
                file = trunk->getValue(NULL);

        if(!file)
	{
        	trunk->error("no-file");
		return NULL;
	}

        ext = strrchr(file, '/');
        if(!ext)
                ext = file;
        ext = strrchr(ext, '.');
	if(ext)
		ext = "";
	else
	{
                ext = trunk->getKeyword("extension");
	        if(!ext)
        	        ext = trunk->getSymbol(SYM_EXTENSION);
	}

        if(prefix)
                snprintf(buffer, sizeof(buffer), "%s/%s%s", prefix, file, ext);
        else
                snprintf(buffer, sizeof(buffer), "%s%s", file, ext);

	if(var)
	{
		snprintf(vbuf, sizeof(vbuf), "%s/%s", var, buffer);
		cp = vbuf;
	}
	else
		cp = buffer;

        return new InfoThread(trunk, cp);
}

timeout_t FileInfoModule::getTimeout(Trunk *trunk, trunkdata_t *data)
{
	const char *cp = trunk->getKeyword("maxTime");
	if(!cp)
		cp = "60";

	return getSecTimeout(cp);
}

char *FileListModule::parseScript(ScriptInterp *interp, Line *line)
{
	return NULL;
}

Service *FileListModule::getService(Trunk *trunk, Line *line, trunkdata_t *data)
{
	ScriptCommand *cmd = trunk->getCommand();
	Script::Symbol *sym = NULL;
	const char *svar = trunk->getKeyword("var");
	const char *prefix = trunk->getPrefixPath();
	const char *var = cmd->getLast("datafiles");
	char buffer[65];

	if(!svar)
		svar = trunk->getKeyword("save");

	if(!svar)
		svar = trunk->getKeyword("results");

	if(svar && *svar != '&')
		svar = NULL;

        if(!prefix)
                prefix = trunk->getValue(NULL);

        if(!prefix)
        {
                trunk->error("no-prefix-dir");
                return NULL;
        }

	if(var)
	{
		snprintf(buffer, sizeof(buffer), "%s/%s", var, prefix);
		prefix = buffer;
	}

	if(svar)
	{
		sym = trunk->getLocal(++svar, 0);
		if(!sym)
		{
			trunk->error("no-symbol-to-save");
			return NULL;
		}
		switch(sym->flags.type)
		{
		case ARRAY:
			sym->data[0] = sym->data[1];
			break;
		case FIFO:
		case SEQUENCE:
		case STACK:
		case CACHE:
			sym->data[1] = sym->data[2] = 0;
			break;
		default:
			trunk->error("symbol-invalid-type");
			return NULL;
		}
	}

        if(!isDir(prefix))
        {
                trunk->error("not-directory");
		return NULL;
        }

	return new ListThread(trunk, prefix, svar);
}

timeout_t FileListModule::getTimeout(Trunk *trunk, trunkdata_t *data)
{
	const char *cp = trunk->getKeyword("maxTime");
	if(!cp)
		cp = "60";

	return getSecTimeout(cp);
}

#ifdef	CCXX_NAMESPACES
}
#endif

