// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of ccscript.
// 
// The exception is that, if you link the ccscript library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccscript library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccscript.  If you copy code from other releases into a copy of
// ccscript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for ccscript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include <cc++/url.h>
#include <cc++/strchar.h>
#include "bayonnescript.h"

#ifndef	WIN32
#include "bayonneconfig.h"
#ifdef	HAVE_REGEX_H
#include <regex.h>
#endif
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
#endif

#ifdef	HAVE_REGEX_H
static class RegexModule : public ScriptModule
{
private:
        char *parseScript(ScriptInterp *interp, Line *line);

public:
        RegexModule() : ScriptModule("regex")
                {};

}       regexmod;

char *RegexModule::parseScript(ScriptInterp *interp, Line *line)
{
	regex_t *regex;
	regmatch_t *regmatch;
	const char *exp;
	Symbol *sym;
	const char *value = "";
	char *pp;
	int from, adj;
	unsigned i, vpos, size = 0, len;
	unsigned offset = 0;
	char *dest;

	if(!size && (NULL != (pp = interp->getKeyword("size"))))
		size = atoi(pp);

	if(!size)
		size = interp->getSymbolSize();

	sym = interp->initVariable(size);
	if(!sym)
		return "symbol-not-found";

	if(sym->flags.readonly)
		return "symbol-read-only";

	size = sym->flags.size;
	exp = interp->getKeyword("exp");
	if(!exp)
		exp = interp->getValue(NULL);

	regex = new regex_t;
	memset(regex, 0, sizeof(regex_t));

	if(regcomp(regex, exp, REG_ICASE))
	{
		regfree(regex);
		delete regex;
		return "regex-invalid";
	}

	regmatch = new regmatch_t[regex->re_nsub + 1];

	dest = new char[sym->flags.size + 1];
	dest[0] = 0;
	len = 0;

	while(NULL != (value = interp->getValue()))
	{
		if(regexec(regex, sym->data + offset, regex->re_nsub + 1, regmatch, 0))
			break;

		adj = 0;
		from = 0;
		for(i = 0; i < regex->re_nsub + 1; ++i)
		{
			while(from < regmatch[i].rm_so && len < sym->flags.size)
				dest[len++] = sym->data[offset + from++];

			vpos = 0;
			while(value[vpos] && len < sym->flags.size)
				dest[len++] = value[vpos++];

			from = regmatch[i].rm_eo;
			if(from > adj)
				adj = from;
		}
		offset += adj;
	}
	while(offset < sym->flags.size && len < sym->flags.size && sym->data[offset])
		dest[len++] = sym->data[offset++];

	dest[len] = 0;
	strcpy(sym->data, dest);
	if(sym->flags.commit)
		interp->commit(sym);

	delete[] dest;

	delete[] regmatch;
	regfree(regex);
	delete regex;
	return NULL;
}

#endif

static class StringModule : public ScriptModule
{
private:
        char *parseScript(ScriptInterp *interp, Line *line);

public:
        StringModule() : ScriptModule("string")
                {};

}       stringmod;

char *StringModule::parseScript(ScriptInterp *interp, Line *line)
{
	Symbol *sym, *str;
	char fmt[5];
	const char *mem = interp->getMember();
	const char *value = "";
	char *name, *pp, *tp, *fp, *ep = NULL;
	long num;
	unsigned size = 0, len = 0, nlen;
	const char *trim = interp->getKeyword("trim");
	const char *chop = interp->getKeyword("chop");
	const char *prefix = interp->getKeyword("prefix");
	const char *suffix = interp->getKeyword("suffix");
	const char *fill = interp->getKeyword("fill");
	const char *mask = interp->getKeyword("mask");
	const char *strip = interp->getKeyword("strip");
	char packtoken = 0;
	unsigned offset = 0;

	fmt[0] = '%';

	if(!strip)
		strip = "";

	if(!trim)
		trim = "";

	if(!chop)
		chop = "";

	if(mem)
	{
		if(!stricmp(mem, "repack"))
			packtoken = ',';
		if(!stricmp(mem, "pack") || !stricmp(mem, "unpack"))
			packtoken = ',';
		if(!stricmp(mem, "chop") || !stricmp(mem, "cut"))
			packtoken = ',';
		size = atoi(mem);
	}
	else
		mem = "0";

	if(packtoken)
	{
		value = interp->getKeyword("token");
		if(value)
			packtoken = *value;
		else
			packtoken = *interp->getSymbol("script.token");
	}

	if(!size && (NULL != (pp = interp->getKeyword("size"))))
		size = atoi(pp);

	if(!size)
		size = interp->getSymbolSize();

	if(!stricmp(mem, "chop") || !stricmp(mem, "cut"))
	{
		str = interp->initVariable(0);
		if(!str)
			return "symbol-not-found";

		if(str->flags.readonly)
			return "symbol-readonly";

		while(NULL != (name = interp->getOption(NULL)))
		{
                	if(*name == '%')
                        	sym = interp->getEntry(++name, size);
                	else
                        	sym = NULL;

			if(sym)
                	{
                        	if(sym->flags.readonly)
                                	return "target-read-only";
                	}
                	else
                        	return "target-missing";

			tp = sym->data;
			value = interp->getKeyword("offset");
			if(value)
			{
				offset = atoi(value);
				pp = str->data;
				while((offset--) > 1 && *pp)
				{
					pp = strchr(pp + 1, packtoken);
					if(!pp)
					{
						pp = sym->data + strlen(sym->data);
						break;
					}
				}			
			}
			else
				pp = strrchr(str->data, packtoken);

			if(pp)
				ep = strchr(pp + 1, packtoken);
			else
				ep = NULL;

			if(!pp)
				pp = fp = str->data;
			else if(pp == str->data)
				fp = pp;
			else
				fp = pp + 1;

			if(ep)
				*(ep++) = 0;
			else
				ep = "";

			len = 0;
			while(*fp && len < sym->flags.size)
			{
				if(strchr(trim, *fp))
				{
					++fp;
					continue;
				}
				else
					trim="";

				if(strchr(strip, *fp))
				{
					++fp;
					continue;
				}

				tp[len++] = *(fp++);
			}
			tp[len] = 0;
			while(len > 0)
			{
				if(strchr(chop, sym->data[len - 1]))
					sym->data[--len] = 0;
				else
					break;
			}
                        sym->flags.initial = false;
                        if(sym->flags.commit)
                               	interp->commit(sym);

			if(*ep && pp > str->data)
				*(pp++) = packtoken;

			while(*ep)
				*(pp++) = *(ep++);
				
			*pp = 0;
			if(str->flags.commit)
				interp->commit(str);
		}
		return NULL;
	}

	if(!stricmp(mem, "unpack"))
	{
		value = interp->getKeyword("offset");
		if(value)
			offset = atoi(value);

		value = interp->getValue(NULL);
		if(!value)
			return "source-not-found";

		while(*value && (offset--) > 1)
		{
			value = strchr(value, packtoken);
			if(!value)
				value = "";
			else
				++value;
		}

		while(*value && NULL != (name = interp->getOption(NULL)))
		{

                	if(*name == '%')
                        	sym = interp->getEntry(++name, size);
                	else
                        	sym = NULL;

			if(sym)
                	{
                        	if(sym->flags.readonly)
                                	pp = NULL;
                        	else
                                	pp = sym->data;
                	}
                	else
                        	pp = NULL;


			len = 0;
			while(pp && *value && *value != packtoken)
			{
				if(strchr(trim, *value))
				{
					++value;
					continue;
				}
				else
					trim = "";

				if(strchr(strip, *value))
				{
					++value;
					continue;
				}
				if(++len > sym->flags.size)
                                	break;

				*(pp++) = *(value++);
                	}
			if(pp)
			{
                        	*pp = 0;
				while(len > 0)
				{
					if(strchr(chop, sym->data[len - 1]))
						sym->data[--len] = 0;
					else
						break;
				}
                        	sym->flags.initial = false;
                        	if(sym->flags.commit)
                                	interp->commit(sym);
                	}

                	while(*value && *value != packtoken)
                        	++value;

                	if(*value == packtoken)
                        	++value;
		}
		return NULL;
	}

	sym = interp->initVariable(size);
	if(!sym)
		return "symbol-not-found";

	size = sym->flags.size;
	if(!stricmp(mem, "repack") || !stricmp(mem, "clear") || !stricmp(mem, "new"))
		len = 0;
	else
		len = (unsigned)strlen(sym->data);

	if(sym->flags.readonly)
		return "symbol-readonly";

	if(len)
	{
		prefix = suffix = NULL;
		len = (unsigned)strlen(sym->data);
	}

	if(prefix)
	{
		while(len < size && *prefix)
			sym->data[len++] = *(prefix++);
	}

	if(mask)
	{
		while(*mask && len < size)
		switch(*mask)
		{
		case '^':
			while(*mask && len < size)
			{
				++mask;
				if(*mask == '0' || *mask == '9' || *mask == '?')
				{
					sym->data[len++] = *mask;
					continue;
				}
				break;
			}
			break;
		case '\\':
			sym->data[len++] = *(++mask);
			++mask;
			break;
		case '9':
			value = interp->getValue(NULL);
			if(!strcmp(value, "("))
				interp->getExpression(&num, 1, 0);
			else
				num = atoi(value);
			nlen = 1;
			while(*(++mask) == '9')
				++nlen;
			sprintf(fmt + 1, "%dld", nlen);
			snprintf(sym->data + len, size - len + 1, fmt, num);
			len = (unsigned)strlen(sym->data);
			value = "";
			break;
		case '0':
			value = interp->getValue(NULL);
			if(!strcmp(value, "("))
				interp->getExpression(&num, 1, 0);
			else
				num = atoi(value);
			nlen = 1;
			while(*(++mask) == '0')
				++nlen;
			sprintf(fmt + 1, "0%dld", nlen);
			snprintf(sym->data + len, size - len + 1, fmt, num);
			len = (unsigned)strlen(sym->data);
			value = "";
			break;
		case '?':
			++mask;
			if(!value)
				break;

			while(!*value)
			{
				value = interp->getValue(NULL);
				if(!value)
					break;
			}
			if(!value)
				break;

			sym->data[len++] = *(value++);
			break;
		case ',':
			if(packtoken)
				sym->data[len++] = packtoken;
			else
				sym->data[len++] = *mask;
			++mask;
			break;
		default:
			sym->data[len++] = *(mask++);
		}
	}
	else while(len < size && NULL != (value = interp->getValue(NULL)))
	{
		if(len && packtoken)
			sym->data[len++] = packtoken;

		while(*value && len < size)
		{
			if(strchr(trim, *value))
			{
				++value;
				continue;
			}
			else
				trim = "";

			if(strchr(strip, *value))
				++value;
			else
				sym->data[len++] = *(value++);
		}
	}

	if(suffix)
	{
		if(len && packtoken)
			sym->data[len++] = packtoken;

		while(len < size && *suffix)
			sym->data[len++] = *(suffix++);
	}

	while(fill && len < size)
	{
		suffix = fill;
		if(len && packtoken)
			sym->data[len++] = packtoken;

		while(len < size && *suffix)
			sym->data[len++] = *(suffix++);
	}

	while(len > 0)
	{
		if(strchr(chop, sym->data[len - 1]))
			--len;
		else
			break;
	}

	sym->data[len] = 0;
	sym->flags.initial = false;
	if(sym->flags.commit)
		interp->commit(sym);

	return NULL;
}

static class UpperProperty : public Script::Property
{
public:
	UpperProperty() : Property("upper") {};
private:
	void upper(char *dp, char *tp, unsigned size);

	void setProperty(char *dp, char *tp, unsigned size)
		{upper(dp, tp, size);};
	void getProperty(char *dp, char *tp, unsigned size)
		{upper(dp, tp, size);};

} upper;

void UpperProperty::upper(char *from, char *to, unsigned size)
{
	while(*from && size)
	{
		*to = toupper(*from);
		++to;
		++from;
	}
	*to = 0;
}

static class LowerProperty : public Script::Property
{
public:
	LowerProperty() : Property("lower") {};

private:
	void lower(char *dp, char *tp, unsigned size);

	void setProperty(char *dp, char *tp, unsigned size)
		{lower(dp, tp, size);};

	void getProperty(char *dp, char *tp, unsigned size)
		{lower(dp, tp, size);};
} lower;

void LowerProperty::lower(char *from, char *to, unsigned size)
{
	while(*from && size)
	{
		*to = tolower(*from);
		++to;
		++from;
	}
	*to = 0;
}

static class CapitalizeProperty : public Script::Property
{
public:
	CapitalizeProperty() : Property("capitalize") {};

private:
	void capitalize(char *dp, char *tp, unsigned size);

	void setProperty(char *dp, char *tp, unsigned size)
		{capitalize(dp, tp, size);};

	void getProperty(char *dp, char *tp, unsigned size)
		{capitalize(dp, tp, size);};
} capitalize;

void CapitalizeProperty::capitalize(char *from, char *to, unsigned size)
{
	if(*from && size)
	{
		*(to++) = toupper(*from);
		++from;
	}
	while(*from && size)
	{
		*to = tolower(*from);
		++to;
		++from;
	}
	*to = 0;
}

static class TrimProperty : public Script::Property
{
public:
	TrimProperty() : Property("trim") {};

private:
	void trim(char *dp, char *tp, unsigned size);

	void setProperty(char *dp, char *tp, unsigned size)
		{trim(dp, tp, size);};

	void getProperty(char *dp, char *tp, unsigned size)
		{trim(dp, tp, size);};
} trim;

void TrimProperty::trim(char *from, char *to, unsigned size)
{
	char *base = to;

	while(isspace(*from))
		++from;

	while(*from && size)
		*(to++) = *(from++);

	*to = 0;
	while(to-- > base)
	{
		if(isspace(*to))
			*to = 0;
		else
			break;
	}
}


#ifdef	CCXX_NAMESPACES
}
#endif

