/*
** Copyright (C) 1999-2000 Open Source Telecom Corporation.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <fcntl.h>
#include <time.h>
#include <syslog.h>
#include "bayonneconfig.h"
#include "bayonnepaths.h"

static char *token = "&";
static char cpath[512];
static unsigned clen = 0;
static sigset_t sigs;
static int pid = 0;

/* some commands will require the use of a url-encoded string */

static char *encode(char *source, char *dest, unsigned max)
{
        static const char *hex = "0123456789abcdef";
        unsigned len = 0;
        unsigned char ch;
        char *ret = dest;

        *dest = 0;
        if(!source)
                return dest;

        while(len < max - 4 && *source)
        {
                ch = (unsigned char)*source;
                if(*source == ' ')
                        *(dest++) = '+';
                else if(isalnum(*source) || strchr("/.-:;,", *source))
                        *(dest++) = *source;
                else
                {
                        *(dest++) = '%';
                        // char in C++ can be more than 8bit
                        *(dest++) = hex[(ch >> 4)&0xF];
                        *(dest++) = hex[ch % 16];
                }
                ++source;
        }
        *dest = 0;
        return ret;
}

static int optcmd(char *s1, char *s2)
{
	while(*s1 == '-')
		++s1;

	while(*s2 == '-')
		++s2;

	return strcmp(s1, s2);
}

static void check(void)
{
	char buffer[16];
	FILE *fp;
	char *status = getenv("BAYONNE_STATUS");
	char *p = getenv("BAYONNE_PID");
		
	token = getenv("BAYONNE_TOKEN");
	if(!token || !*token)
		token = "&";

	if(p && *p)
		pid = atoi(p);

	if(status && *status)
	{
		if(!strcmp(status, "dead"))
			exit(1);
		return;
	}

	if(!access(PATH_VARRUN_BAYONNE, R_OK))
		snprintf(cpath, sizeof(cpath) - 16, "%s/bayonne", PATH_VARRUN_BAYONNE);
	else
		snprintf(cpath, sizeof(cpath) - 16, "%s/.bayonne/", getenv("HOME"));
	clen = strlen(cpath);

	strcpy(cpath + clen, ".pid");
	fp = fopen(cpath, "r");
	if(!fp)
		exit(1);

        fgets(buffer, sizeof(buffer), fp);
        fclose(fp);
        pid = atoi(buffer);
        if(!pid)
                exit(1);
        if(kill(pid, 0))
                exit(1);
}

static int send(char *command)
{
	int fd;
	char cbuf[512];
	int sig;

	strcpy(cpath + clen, ".ctrl");
	fd = open(cpath, O_RDWR);
	if(fd < 0)
		exit(1);

	if(strchr(command, *token))
		snprintf(cbuf, sizeof(cbuf), "%d%s%s\n", getpid(), token, command);
	else
		snprintf(cbuf, sizeof(cbuf), "%d %s\n", getpid(), command);
	write(fd, cbuf, strlen(cbuf));
#ifdef	POSIX_SIGWAIT2
	sigwait(&sigs, &sig);
#else
	sig = sigwait(&sigs);
#endif
#ifdef	SIGUSR2
	if(sig == SIGUSR2)
		return 2;
#endif
	if(sig == SIGPIPE)
		return 2;
	return 0;
}

static void down(int argc, char **argv)
{
	int fd;

	if(argc > 1)
	{
		fprintf(stderr, "use: btstool --down\n");
		exit(-1);
	}
	check();
	strcpy(cpath + clen, ".ctrl");
	fd = open(cpath, O_RDWR);
	if(fd < 0)
		exit(0);
	write(fd, "down\n", 5);
	close(fd);
	while(!kill(pid, 0))
		sleep(1);
	exit(0);
} 

static void online(int argc, char **argv)
{
	if(argc > 1)
	{
		fprintf(stderr, "use: btstool --online\n");
		exit(-1);
	}
	check();
	exit(0);
}

static void compile(int argc, char **argv)
{
	char buffer[128];

	if(argc > 2)
	{
		fprintf(stderr, "use: btstool --compile [session]\n");
		exit(-1);
	}
	check();
	strcpy(buffer, "compile");
	if(argc > 1)
		snprintf(buffer, sizeof(buffer), "compile %s", argv[1]);
	exit(send(buffer));
}

static void errlog(int argc, char **argv)
{
	char buffer[512];
	unsigned len, llen;
	time_t now;
	struct tm *dt;
	static char *months[12] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	char *reason = argv[0];
	int level = LOG_ERR;

	while(*reason == '-')
		++reason;

	if(!strcmp(reason, "errlog"))
		reason = "extern";
	else if(!strcmp(reason, "filelog"))
		reason = "access";
	else if(!strncmp(reason, "fail", 4))
		reason = "failed";
	else if(!strcmp(reason, "notice") || !strcmp(reason, "notelog"))
	{
		reason = "notice";
		level = LOG_NOTICE;
	}

	time(&now);
	dt = localtime(&now);

	if(dt->tm_year < 1000)
		dt->tm_year += 1900;

	if(argc < 2)
	{
		fprintf(stderr, "use: btstool --errlog message\n");
		exit(-1);
	}
	snprintf(buffer, sizeof(buffer), "errors [%s %02d %02d:%02d:%02d %d] [%s]",
		months[--(dt->tm_mon)], dt->tm_mday,
		dt->tm_hour, dt->tm_min, dt->tm_sec, dt->tm_year, reason);
	len = strlen(buffer);
	llen = len + 1;
	while(--argc)
	{
		snprintf(buffer + len, sizeof(buffer) - len, " %s", *(++argv));
		len = strlen(buffer);
	}
	syslog(LOG_USER|level, "%s", buffer + llen);
	check();
	exit(send(buffer));
}

static void routeadd(int argc, char **argv)
{
	char buffer[160];

	if(argc != 5)
	{
		fprintf(stderr, "btstool: route: invalid arguments\n");
		exit(-1);
	}

	snprintf(buffer, sizeof(buffer), "route add %s %s %s", argv[2], argv[3], argv[4]);
	check();
	exit(send(buffer));
}

static void drop(int argc, char **argv)
{
	char buffer[128];

	if(argc != 2)
	{
		fprintf(stderr, "use: btstool --drop sessionid\n");
		exit(-1);
	}

	snprintf(buffer, sizeof(buffer), "hangup %s", argv[1]);
	check();
	exit(send(buffer));
}

static void global(int argc, char **argv)
{
        char buffer[128];
	char ebuf[256];

        if(argc != 3)
        {
                fprintf(stderr, "use: btstool --global sym value\n");
                exit(-1);
        }

	check();
	encode(argv[2], ebuf, sizeof(ebuf));
        snprintf(buffer, sizeof(buffer), "global%s%s%s%s",
		token, argv[1], token, ebuf);
        exit(send(buffer));
}


static void syncsem(int argc, char **argv)
{
        char buffer[128];

        if(argc != 2)
        {
                fprintf(stderr, "use: btstool --sync sessionid\n");
                exit(-1);
        }

        snprintf(buffer, sizeof(buffer), "sync %s", argv[1]);
        check();
        exit(send(buffer));
}

static void routedel(int argc, char **argv)
{
	char buffer[128];

	if(argc > 4)
	{
		fprintf(stderr, "btstool: route: invalid arguments\n");
		exit(-1);
	}

	snprintf(buffer, sizeof(buffer), "route del %s %s", argv[2], argv[3]);
	check();
	exit(send(buffer));
}

static void route(int argc, char **argv)
{
	int use = 0;

	if(argc > 3)
	{
		if(strcmp(argv[2], "incoming"))
		{
			fprintf(stderr, "btstool: route: %s: unknown route table\n", argv[2]);
			exit(-1);
		}
		if(!optcmd(argv[1], "--add"))
			routeadd(argc, argv);
		else if(!optcmd(argv[1], "--del"))
			routedel(argc, argv);
		fprintf(stderr, "btstool: route: %s: uknown option\n", argv[1]);
		exit(-1);
	}
	fprintf(stderr, "use: btstool route [--add|--del] routetable routeinfo...\n");
 	exit(-1);
}

void run(int argc, char **argv)
{
	char buffer[1024];
	char ebuf[256];
	char tmpname[256];
	unsigned len;
	char *key, *value, *cp;
	char *cmd = "run";
	int rts;
	FILE *fp;
	int quiet = 0;
	int session = 0;
	char *opt = argv[0];

	while(*opt == '-')
		++opt;

	if(!strcmp(opt, "quiet"))
		++quiet;
	else if(!strcmp(opt, "session"))
	{
		++session;
		cmd = "session";
	}

	if(argc < 3)
	{
		fprintf(stderr, "use: btstool --run group script [sym|tag=value...\n");
		exit(-1);
	}
	check();
	cpath[clen] = 0;
	snprintf(tmpname, sizeof(tmpname), "%s.bts-%d-", cpath, getpid());
	snprintf(buffer, sizeof(buffer), "%s%s%s%s%s%s%s",
		cmd, token, tmpname, token, argv[1], token, argv[2]);
	argc -= 3;
        argv += 3;
        len = strlen(buffer);
        while(argc--)
        {
                key = *(argv++);
                value = strchr(key, '=');
                if(value)
                        *(value++) = 0;
                else
                        value = getenv(key);
                if(!value)
                        value = "";
		encode(value, ebuf, sizeof(ebuf));
                snprintf(buffer + len, sizeof(buffer) - len, 
			"%s%s=%s", token, key, ebuf); 
                len = strlen(buffer);
        }
	mkfifo(tmpname, 0600);	
	fp = fopen(tmpname, "r+");
	rts = send(buffer);
	remove(tmpname);
	if(rts)
		exit(rts);
	for(;;)
	{
		fgets(buffer, sizeof(buffer), fp);
		if(feof(fp))
			exit(4);
		cp = strtok(buffer, " \t\r\n");
		if(!strcmp(cp, "start") && session)
		{
			cp = strtok(NULL, " \t\r\n");
			printf("%s\n", cp);
			exit(0);
		}
		if(!strcmp(cp, "fail"))
			exit(3);
		if(!strcmp(cp, "exit"))
			break;
	}
	cp = strtok(NULL, " \t\r\n");
	cp = strtok(NULL, "\r\n");
	while(isspace(*cp))
		++cp;
	if(!*cp)
		exit(0);
	if(!quiet)
		printf("%s\n", cp);		
	exit(0);
} 

void start(int argc, char **argv)
{
	char buffer[512];
	char ebuf[256];
	unsigned len;
	char *key, *value;

	if(argc < 3)
	{
		fprintf(stderr, "use: btstool --start group script [sym|tag=value] ...\n");
		exit(-1);
	}
	
	snprintf(buffer, sizeof(buffer), "start%s%s%s%s", token, argv[1], token, argv[2]);
	argc -= 3;
	argv += 3;
	len = strlen(buffer);
	while(argc--)
	{
		key = *(argv++);
		value = strchr(key, '=');
		if(value)
			*(value++) = 0;
		else
			value = getenv(key);
		if(!value)
			value = "";
		encode(value, ebuf, sizeof(ebuf));
		snprintf(buffer + len, sizeof(buffer) - len, "%s%s=%s", token, key, ebuf);
		len = strlen(buffer);
	}
	check();
	exit(send(buffer));
}

int main(int argc, char **argv)
{
	char *name = strrchr(argv[0], '/');
	char buffer[512];
	unsigned len = 0;

	sigemptyset(&sigs);
#ifdef	SIGUSR2
	sigaddset(&sigs, SIGUSR1);
	sigaddset(&sigs, SIGUSR2);
#else
	sigaddset(&sigs, SIGHUP);
	sigaddset(&sigs, SIGPIPE);
#endif
	sigprocmask(SIG_BLOCK, &sigs, NULL);

	if(name)
		++name;
	else
		name = argv[0];

	if(!strcmp("online", name))
		online(argc, argv);
	else if(!strcmp("send", name))
		goto sender;
	else if(!strcmp("compile", name))
		compile(argc, argv);
	else if(!strcmp("down", name))
		down(argc, argv);
	else if(!strcmp("drop", name) || !strcmp("stop", name))
		drop(argc, argv);
	else if(!strcmp("sync", name))
		syncsem(argc, argv);
	else if(!strcmp("global", name))
		global(argc, argv);
	else if(!strcmp("errlog", name))
		errlog(argc, argv);
	else if(!strncmp("errlog.", name, 7))
	{
		argv[0] += 7;
		errlog(argc, argv);
	} 
	else if(!strcmp("route", name))
		route(argc, argv);
	else if(!strcmp("start", name))
		start(argc, argv);
	else if(!strcmp("run", name))
		run(argc, argv);
	else if(!strcmp("run.quiet", name))
	{
		argv[0] += 4;
		run(argc, argv);
	}
	else if(!strcmp("start.session", name))
	{
		argv[0] += 6;
		run(argc, argv);
	}

	++argv;
	--argc;
	name = argv[0];
	if(!optcmd("--online", name))
		online(argc, argv);
	else if(!optcmd("--compile", name))
		compile(argc, argv);
	else if(!optcmd("--down", name))
		down(argc, argv);
	else if(!optcmd("--errlog", name))
		errlog(argc, argv);
	else if(!optcmd("--failed", name))
		errlog(argc, argv);
	else if(!optcmd("--notice", name))
		errlog(argc, argv);
	else if(!optcmd("--access", name))
		errlog(argc, argv);
	else if(!optcmd("--route", name))
		route(argc, argv);
	else if(!optcmd("--start", name))
		start(argc, argv);
	else if(!optcmd("--run", name))
		run(argc, argv);
	else if(!optcmd("--session", name))
		run(argc, argv);
	else if(!optcmd("--drop", name) || !optcmd("--stop", name))
		drop(argc, argv);
	else if(!optcmd("--sync", name))
		syncsem(argc, argv);
	else if(!optcmd("--global", name))
		global(argc, argv);

	/* a generic send facility for testing; use command specific
 	   symlinks or --commands for most uses instead of this, since 
	   those are argument and encoding aware.
	*/
	
	if(!optcmd("--send", name))
	{
sender:
		if(argc < 2)
		{
			fprintf(stderr, "use: btstool --send \"command\"\n");
			exit(-1);
		}
		check();
		while(--argc)
		{
			if(len)
				snprintf(buffer + len, sizeof(buffer) - len, "%s%s", token, *(++argv));
			else
				snprintf(buffer, sizeof(buffer), "%s", *(++argv));
			len = strlen(buffer);
		}
		exit(send(buffer));
	}

	if(argc > 1)
		fprintf(stderr, "btstool: %s: unknown option\n", argv[1]);
	else
		fprintf(stderr, "use: btstool --command [options]\n");
	exit(-1);
}
