/* Copyright (C) 2004 Open Source Telecom Corporation.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/

#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <math.h>
#include <fcntl.h>
#include <getopt.h>

static struct option long_options[] = {
	{"duration", 1, 0, 'd'},
	{"interdigit", 1, 0, 'i'},
	{"amplitude", 1, 0, 'a'},
	{"rate", 1, 0, 'r'},
	{"twist", 1, 0, 't'},
	{"format", 1, 0, 'f'},
	{0, 0, 0, 0}};

static long interdigit = 120;
static long duration = 60;
static long amplitude = 32000;
static long rate = 8000;
static long twist = 50;
static int fd = 1;
static double pos1, pos2 = 0;
static int verbose = 1;

#define	TRUE	1
#define	FALSE	0

static void gensingletone(double freq, long duration, long amp)
{
	short sample;
	double fa = ((freq * M_PI * 2) / (float)rate);
	unsigned long count = duration * rate / 1000;
	pos2 = 0;

	if(fd != 1 && verbose)
		printf("S-TONE %d %d %d\n", (long)freq, duration, (long)amp);

	while(count--)
	{
		sample = (short)(sin(pos1) * amp);	
                pos1 += fa;
		write(fd, &sample, 2);
	}	
}

static void singletone(void)
{
	long amp = amplitude;
	long duration = 1000;
	double freq = 2000.;
	char *cp = strtok(NULL, " \t\r\n");

	if(cp && *cp)
	{
		freq = (double)atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}

	if(cp && *cp)
	{
		duration = atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}

	if(cp && *cp)
		amp = atoi(cp);

	gensingletone(freq, duration, amp);
}

static void gensilence(long duration)
{
	unsigned long count = duration * rate / 1000;
	short sample = 0;
	pos1 = pos2 = 0;
	sample = 0;

	if(fd != 1 && verbose)
		printf("SILENCE %d\n", duration);

	while(count--)
		write(fd, &sample, 2);
}

static void silence(void)
{
	long duration = 1000;
	char *cp = strtok(NULL, " \t\n\r");

	if(cp && *cp)
		duration = atol(cp);

	gensilence(duration);
}

static void gendualtone(double f1, double f2, long duration, long amp, int tw)
{
	double fa1 = ((f1 * M_PI * 2) / (float)rate);
        double fa2 = ((f2 * M_PI * 2) / (float)rate);
        double a1 = amp * (100 - tw) / 100;
        double a2 = amp * tw / 100;
        unsigned long count = duration * rate / 1000;
	short sample;

	if(fd != 1 && verbose)
		printf("D-TONE %d %d %d %d %d%\n",
			(long)f1, (long)f2, duration, (long)amp, tw); 

	while(count--)
	{
		sample = (short)(sin(pos1) * a1 + sin(pos2) * a2);	
                pos1 += fa1;
                pos2 += fa2;
		write(fd, &sample, 2);
	}	
}

static void dualtone(void)
{
	int tw = twist;
	double f1 = 1000, f2 = 2000;
	long duration = 1000, amp = amplitude;
	char *cp = strtok(NULL, " \t\r\n");

	if(cp && *cp)
	{
		f1 = (double)atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}

	if(cp && *cp)
	{
		f2 = (double)atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}

	if(cp && *cp)
	{
		duration = atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}

	if(cp && *cp)
	{
		amp = atoi(cp);
		cp = strtok(NULL, " \t\r\n");
	}

	if(cp && *cp)
		tw = atoi(cp);

	gendualtone(f1, f2, duration, amp, tw);
}

static void mftones(void)
{
	const char *cp, *digits = strtok(NULL, " \t\r\n");
	long amp = amplitude;
	int tw = twist;
	long dur = 60;
	long id = 120;
	
	if(digits && *digits)
		cp = strtok(NULL, " \t\r\n");
		
	if(cp && *cp)
	{
		dur = atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}
	
	if(cp && *cp)
	{
		id = atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}
	
	if(cp && *cp)
	{
		amp = atoi(cp);
		cp = strtok(NULL, " \t\r\n");
	}
	
	if(cp && *cp)
	{
		tw = atoi(cp);
		cp = strtok(NULL, " \t\r\n");
	}	
		
	while(digits && *digits)
	{
		switch(*digits)
		{
		case '1':
			gendualtone(700, 900, dur, amp, tw);
			gensilence(id);
			break;
		case '2':
			gendualtone(700, 1100, dur, amp, twist);
			gensilence(id);
			break;
		case '3':
			gendualtone(900, 1100, dur, amp, tw);
			gensilence(id);
			break;
		case '4':
			gendualtone(700, 1300, dur, amp, tw);
			gensilence(id);
			break;
		case '5':
			gendualtone(900, 1300, dur, amp, tw);
			gensilence(id);
			break;		
		case '6':
			gendualtone(1100, 1300, dur, amp, tw);
			gensilence(id);
			break;		
		case '7':
			gendualtone(700, 1500, dur, amp, tw);
			gensilence(id);
			break;		
		case '8':
			gendualtone(900, 1500, dur, amp, tw);
			gensilence(id);
			break;		
		case '9':
			gendualtone(1100, 1500, dur, amp, tw);
			gensilence(id);
			break;		
		case '0':
			gendualtone(1300, 1500, dur, amp, tw);
			gensilence(id);
			break;				
		case 'K':
		case 'k':
		case '#':
			gendualtone(1100, 1700, 100, amp, tw);
			gensilence(200);
			break;
		case 'S':
		case 's':
		case '*':
			gendualtone(1500, 1700, dur, amp, tw);
			gensilence(id);
			break;			
		case 'b':
		case 'B':
			gensingletone(2600, 1000, amp);
			gensilence(1100);	
		}
		++digits;
	}	
}

static void abctones(void)
{
        static double sharps[] =
                {4.331, 4.861, 5.150, 5.781, 6.489, 7.293, 7.717, 0};

        static double flats[] =
                {8.176, 4.331, 4.861, 5.456, 5.781, 6.489, 7.293, 0};

        static double notes[] =
	        {8.176, 4.588, 5.150, 5.456, 6.125, 6.875, 7.717, 0};

	const char *tone = strtok(NULL, " \r\n\t");
	int note = -1;
	int octave;
	char *cp;
	int scale;
	long freq;
	long period = 1000, dur;
	long adjust = 0, adj;
	double *noteset = notes;
	long amp = amplitude;

	char *upper = "CDEFGABZ";
	char *lower = "cdefgabz";

	period = 1000 * atoi(tone);
	cp = strchr(tone, '/');
	if(cp)
		period /= atoi(++cp);

	tone = strtok(NULL, "");

	while(tone && *tone)
	{
		noteset = notes;
		adj = 0;

		if(*tone == '^')
		{
			++tone;
			noteset = sharps;
		} else if(*tone == '_')
		{
			++tone;
			noteset = flats;
		}
		cp = strchr(upper, *tone);
		if(cp)
		{
			note = cp - upper;
			octave = 4;
		}
		else
		{
			cp = strchr(lower, *tone);
			if(cp)
			{
				octave = 5;
				note = cp - lower;
			}
		}
		if(!cp)
		{
			++tone;
			continue;
		}
		++tone;
		while(*tone == '\'')
		{
			++octave;
			++tone;
		}
		while(*tone == ',')
		{
			--octave;
			++tone;
		}
		if(octave < -1 || octave > 11)
			octave = 0;

		scale = 1<<(++octave);
		freq = (long)(noteset[note] * (double)scale);
		dur = period;
		if(*tone == '/')
		{
			dur = period / atoi(++tone);
			while(isdigit(*tone))
				++tone;
		}
		else if(isdigit(*tone))
		{
			dur = period * atoi(tone);
			while(isdigit(*tone))
				++tone;
		}

		while(*tone == '>')
		{
			adj += 1000 / 16;
			++tone;
		}

		while(*tone == '<')
		{
			adj -= 1000 / 16;
			++tone;
		}

		if(adj)
		{
			dur += adj;
			adjust = -adj;
		}
		else if(adjust)
		{
			dur += adjust;
			adjust = 0;
		}

		if(freq)
			gensingletone(freq, dur, amp);
		else
			gensilence(dur);
		++tone;
	}
}

static void dtmftones(void)
{
	const char *cp, *digits = strtok(NULL, " \t\r\n");
	long amp = amplitude;
	int tw = twist;
	long dur = 60;
	long id = 120;
	
	if(digits && *digits)
		cp = strtok(NULL, " \t\r\n");
		
	if(cp && *cp)
	{
		dur = atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}
	
	if(cp && *cp)
	{
		id = atol(cp);
		cp = strtok(NULL, " \t\r\n");
	}
	
	if(cp && *cp)
	{
		amp = atoi(cp);
		cp = strtok(NULL, " \t\r\n");
	}
	
	if(cp && *cp)
	{
		tw = atoi(cp);
		cp = strtok(NULL, " \t\r\n");
	}	
		
	while(digits && *digits)
	{
		switch(*digits)
		{
		case '1':
			gendualtone(697, 1209, dur, amp, tw);
			gensilence(id);
			break;
		case '2':
			gendualtone(697, 1336, dur, amp, twist);
			gensilence(id);
			break;
		case '3':
			gendualtone(697, 1477, dur, amp, tw);
			gensilence(id);
			break;
		case 'a':
		case 'A':
			gendualtone(697, 1633, dur, amp, tw);
			gensilence(id);
			break;
		case '4':
			gendualtone(770, 1209, dur, amp, tw);
			gensilence(id);
			break;
		case '5':
			gendualtone(770, 1336, dur, amp, tw);
			gensilence(id);
			break;		
		case '6':
			gendualtone(770, 1477, dur, amp, tw);
			gensilence(id);
			break;		
		case 'b':
		case 'B':
			gendualtone(770, 1633, dur, amp, tw);
			gensilence(id);
			break;
		case '7':
			gendualtone(852, 1209, dur, amp, tw);
			gensilence(id);
			break;		
		case '8':
			gendualtone(852, 1336, dur, amp, tw);
			gensilence(id);
			break;		
		case '9':
			gendualtone(852, 1477, dur, amp, tw);
			gensilence(id);
			break;		
		case 'c':
		case 'C':
			gendualtone(852, 1633, dur, amp, tw);
			gensilence(id);
			break;
		case '*':
		case 'S':
		case 's':
			gendualtone(941, 1209, dur, amp, tw);
			gensilence(id);
			break;
		case '0':
			gendualtone(941, 1477, dur, amp, tw);
			gensilence(id);
			break;				
		case 'P':
		case 'p':
		case '#':
			gendualtone(941, 1477, dur, amp, tw);
			gensilence(id);
			break;
	
		case 'd':
		case 'D':
			gendualtone(941, 1633, dur, amp, tw);
			gensilence(id);
			break;			
		case ',':
			gensilence(1000);	
		}
		++digits;
	}	
}


int main(int argc, char **argv)
{
	FILE *input;
	char *format = NULL;
	int use = FALSE;
	int opt, opt_index, pid = 0;
	char *args[10];
	char tmp[65] = "/tmp/.tonesXXXXXX";
	char rbuf[16];
	char lbuf[132];
	char *cp;
	char *specfile = NULL;
	while(EOF != (opt = getopt_long(argc, argv, "qvs:a:r:t:f:h", 
		long_options, &opt_index)))
	{
		switch(opt)
		{
		case 'q':
			verbose = 0;
			break;
		case 'v':
			++verbose;
			break;
		case 's':
			specfile = optarg;
			break;
		case 'd':
			duration = atol(optarg);
			break;
		case 'i':
			interdigit = atol(optarg);
			break;
		case 'a':
			amplitude = atoi(optarg);
			break;
		case 'f':
			format = optarg;
			break;
		case 'r':
			rate = atoi(optarg);
			break;
		case 't':
			twist = atoi(optarg);
			break;
		default:
			use = TRUE;
		}
	}
	if(optind == argc || optind + 1 < argc)
		use = TRUE;

	if(use)
	{
		fprintf(stderr, "use: tonetool [options] [output]\n");
		exit(-1);
	}
	if(specfile)
		input = fopen(argv[optind], "r");
	else
		input = stdin;
	if(!input)
	{
		fprintf(stderr, "tonetool: %s: cannot access\n", specfile);
		exit(-1);
	}
	if(format && optind == argc)
	{
		fprintf(stderr, "tonetool: output required for format\n");
		exit(-1);
	}

	if(format)
	{
		remove(argv[optind]);		
		fd = mkstemp(tmp);
	}
		
	else if(optind < argc)
	{
		remove(argv[optind]);
		fd = creat(argv[optind], 0660);
	}

	if(fd < 0)
	{
		fprintf(stderr, "tonetool: cannot create %s", argv[optind]);
		exit(-1);
	}
	
	for(;;)
	{
		fgets(lbuf, sizeof(lbuf), input);
		if(feof(input))
			break;

		cp = strtok(lbuf, " \t\r\n");
		if(!cp)
			continue;

		if(!strncmp(cp, "sil", 3))
			silence();
		else if(!strncmp(cp, "sin", 3) || !strcmp(cp, "tone") || !strcmp(cp, "st"))
			singletone();
		else if(!strcmp(cp, "dual") || !strcmp(cp, "dt"))
			dualtone();
		else if(!strcmp(cp, "mf"))
			mftones();
		else if(!strcmp(cp, "dtmf"))
			dtmftones();
		else if(!strcmp(cp, "abc"))
			abctones();
	}

	if(format)
	{
		if(!strcmp(format, "ul"))
			format = "-U";
		else if(!strcmp(format, "al"))
			format = "-A";
		else if(!strcmp(format, "raw") || !strcmp(format, "sw"))
			format = "-sw";
		else if(!strcmp(format, "uw"))
			format = "-uw";
		else if(!strcmp(format, "pcm") || !strcmp(format, "sb"))
			format = "-sb";
		else if(!strcmp(format, "adpcm") || !strcmp(format, "726"))
			format = "-a";
		else if(!strcmp(format, "msadpcm"))
			format = "-i";
		else if(!strcmp(format, "gsm"))
			format = "-g";
		else
		{
			fprintf(stderr, "dtmfgen: unknown format %s\n", format);
			fprintf(stderr, "use ul=ulaw, al=alaw, sw=linear, pcm, adpcm, msadpcm, gsm\n");

			remove(tmp);
			exit(-1);
		}
		pid = fork();
		if(!pid)
		{
			snprintf(rbuf, sizeof(rbuf), "-r%d", rate);
			args[0] = "sox";
			args[1] = "-traw";
			args[2] = "-sw";
			args[3] = rbuf;
			args[4] = tmp;
			args[5] = format;
			args[6] = argv[optind];
			args[7] = NULL;
			execvp(*args, args);
			exit(-1);
		}			
	}
	if(pid)
		wait(&pid);

	if(format)
		remove(tmp);
	exit(0);
}

