// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "bayonnelibrary.h"

#define	BAYONNE_VERSION	VERSION
#define	BAYONNE_SERVER	PACKAGE

#ifdef CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

aaScript *aaScript::get(const char *id, aaScript *base)
{
	const char *name;
	if(!base)
		base = aascript;

	while(base)
	{
		name = base->getLast("name");
		if(name && !stricmp(name, id))
			break;
		base = base->next;
	}
	return base;
}

aaScript::aaScript(aaScript *ini) :
ScriptCommand(ini)
{
	if(ini->next)
		ini->next->next = this;
	else
		ini->next = this;

	next = NULL;
	setValue("include", ini->getLast("include"));
}

aaScript::aaScript() :
ScriptCommand()
{
	const char *cp;

	// define primary Bayonne dialect
	static Script::Define interp[] = {
		{"hangup", (Method)&Trunk::scrHangup,
			&ScriptCommand::chkIgnore},
		{"debug", (Method)&Trunk::scrDebug,
			&ScriptCommand::chkHasArgs},
		{"sleep", (Method)&Trunk::scrSleep,
			&ScriptCommand::chkHasArgs},
		{"sync", (Method)&Trunk::scrSync,
			&ScriptCommand::chkIgnore},
		{"accept", (Method)&Trunk::scrAccept,
			&ScriptCommand::chkIgnore},
		{"reject", (Method)&Trunk::scrReject,
			&ScriptCommand::chkIgnore},
		{"answer", (Method)&Trunk::scrAnswer,
			&ScriptCommand::chkIgnore},
		{"collect", (Method)&Trunk::scrCollect,
			&ScriptCommand::chkIgnore},
		{"flash", (Method)&Trunk::scrFlash,
			&ScriptCommand::chkIgnore},
		{"altplay", (Method)&Trunk::scrAltPlay,
			&ScriptCommand::chkHasArgs},
		{"play", (Method)&Trunk::scrPlay,
			&ScriptCommand::chkHasArgs},
		{"say", (Method)&Trunk::scrSay,
			&ScriptCommand::chkHasArgs},
		{"tts", (Method)&Trunk::scrSay,
			&ScriptCommand::chkHasArgs},
		{"record", (Method)&Trunk::scrRecord,
			&ScriptCommand::chkHasArgs},
		{"options", (Method)&Trunk::scrOptions,
			&ScriptCommand::chkIgnore},
		{"tone", (Method)&Trunk::scrTone,
			&ScriptCommand::chkHasArgs},
		{"dial", (Method)&Trunk::scrDial,
			&ScriptCommand::chkHasArgs},
		{"transfer", (Method)&Trunk::scrTransfer,
			&ScriptCommand::chkHasArgs},
		{"hold", (Method)&Trunk::scrHold,
			&ScriptCommand::chkNoArgs},
		{"libexec", (Method)&Trunk::scrLibexec,
			&ScriptCommand::chkHasArgs},
		{"altspeak", (Method)&Trunk::scrAltSpeak,
			&ScriptCommand::chkHasArgs},
		{"speak", (Method)&Trunk::scrSpeak,
			&ScriptCommand::chkHasArgs},
		{"start", (Method)&Trunk::scrStart,
			&ScriptCommand::chkHasArgs},
		{"busy", (Method)&Trunk::scrBusy,
			&ScriptCommand::chkIgnore},
		{"idle", (Method)&Trunk::scrIdle,
			&ScriptCommand::chkIgnore},
		{"send", (Method)&Trunk::scrSend,
			&ScriptCommand::chkHasArgs},
		{"slog", (Method)&Trunk::scrSlog,
			&ScriptCommand::chkIgnore},
                {"echo", (Method)&Trunk::scrEcho,
                        &ScriptCommand::chkIgnore},
		{"session", (Method)&Trunk::scrSession,
			&ScriptCommand::chkIgnore},
		{"statinfo", (Method)&Trunk::scrStatinfo,
			&ScriptCommand::chkIgnore},
		{"control", (Method)&Trunk::scrControl,
			&ScriptCommand::chkHasArgs},
		{"sendfax", (Method)&Trunk::scrSendFax,
			&ScriptCommand::chkHasArgs},
		{"recvfax", (Method)&Trunk::scrRecvFax,
			&ScriptCommand::chkHasArgs},
		{"exit", (Method)&Trunk::scrExiting,
			&ScriptCommand::chkIgnore},
		{"join", (Method)&Trunk::scrJoin,
			&ScriptCommand::chkHasArgs},
		{"wait", (Method)&Trunk::scrWait,
			&ScriptCommand::chkHasArgs},
		{"pickup", (Method)&Trunk::scrPickup,
			&ScriptCommand::chkHasArgs},
		{"enter", (Method)&Trunk::scrEnter,
			&ScriptCommand::chkIgnore},
		{"release", (Method)&Trunk::scrRelease,
			&ScriptCommand::chkIgnore},
		{NULL, NULL, NULL}};

	// set standard aascript trap identifiers

	trap("timeout", false);
	trap("dtmf");

	trap("0");	/* 0x10 */
	trap("1");
	trap("2");
	trap("3");

	trap("4");	/* 0x100 */
	trap("5");
	trap("6");
	trap("7");

	trap("8");	/* 0x1000 */
	trap("9");
	trap("star");
	trap("pound");

	trap("a");	/* 0x10000 */
	trap("b");
	trap("c");
	trap("d");

	trap("silence", false);
	trap("busy", false);
	trap("cancel", false);
	trap("notify", false);

	trap("noanswer");
	trap("ring");
	trap("tone");
	trap("event");

	trap("time");
	trap("child");

	load(interp);

	next = NULL;

	cp = keypaths.getLast("include");
	if(!cp)
		cp = keypaths.getLast("scripts");

	setValue("include", cp);
}

void aaScript::addDummy(const char *name)
{
	Script::Define keywords[2];

	keywords[0].keyword = name;
	keywords[0].method = (Method)&Trunk::scrDummy;
	keywords[0].check = &ScriptCommand::chkIgnore;
	keywords[1].keyword = NULL;
	keywords[1].method = (Method)NULL;
	keywords[1].check = (Check)NULL;
	load(keywords);
}

unsigned long aaScript::getTrapMask(const char *trapname)
{
	unsigned long mask;

	if(!stricmp(trapname, "hangup"))
		return 0x00000001;

	if(!strcmp(trapname, "!"))
		return 0x0000fff8;

	if(!stricmp(trapname, "override"))
		return 0x00010000;

	if(!stricmp(trapname, "flash"))
		return 0x00020000;

	if(!stricmp(trapname, "immediate"))
		return 0x00040000;

	if(!stricmp(trapname, "priority"))
		return 0x00080000;

	if(!stricmp(trapname, "part"))
		return 0x00400000;

	if(!stricmp(trapname, "fail"))
		return 0x00400000;

	if(!stricmp(trapname, "invalid"))
		return 0x00400000;

	if(!stricmp(trapname, "nodialtone"))
		return 0x01000000;

	if(!stricmp(trapname, "noringback"))
		return 0x01000000;

	if(!stricmp(trapname, "busytone"))
		return 0x00200000;

	if(!stricmp(trapname, "text"))
		return 0x00800000;

	if(!stricmp(trapname, "signal"))
		return 0x08000000;

	if(!stricmp(trapname, "maxtime"))
		return 0x10000000;

	if(!stricmp(trapname, "answer"))
		return 0x02000000;

	if(!stricmp(trapname, "pickup"))
		return 0x02000000;

	mask = ScriptCommand::getTrapMask(trapname);
	if(mask == 0x00000008)
		return 0x0000fff8;

	if(mask & 0x0000fff0)
		mask |= 0x00000008;

	return mask;
}

aaImage::aaImage(aaScript *script) :
ScriptImage((ScriptCommand *)script, "/server/script")
{
	char path[1024];
	char *syms[256];
	char **name = syms;
	const char *ext;
	const char *const *list;
	const char *ext1 = script->getLast("extensions");
	char ttsnames[128];
	unsigned ttslen = 0;

	if(!ext1)
		ext1 = ".bs";

	script->getIndex(syms, 256);

	if(script == aascript)
		errlog("notice", "Compiling default scripts...");
	else
		errlog("notice", "Compiling %s...", script->getLast("name"));

	while(*name)
	{
		ext = strrchr(*name, '.');
		if(ext && !stricmp(ext, ".scr"))
			compile(script->getLast(*name));
		else if(ext && !stricmp(ext, ext1))
			compile(script->getLast(*name));
		++name;
	}

	if(script->getLast("scripts"))
		list = script->getList("scripts");
	else
		list = NULL;

	while(list && *list)
	{
		snprintf(path, sizeof(path), "%s", *list);
		if(canAccess(path))
			scanDir(path, ext1, sizeof(path));
		++list;
	}

       static Keydata::Define keys[] = {
                {"timeout", "60"},
                {"interdigit", "6"},
                {"extdigits", "4"},
                {"localdigits", "7"},
                {"natdigits", "10"},
                {"xferextension", "FW"},
                {"dialextension", "W"},
                {"diallocal", "W9"},
                {"xferlocal", "FW9"},
                {"dialnational", "W81"},
                {"xfernational", "FW81"},
                {"dialinternational", "W8011"},
                {"xferinternational", "FW8011"},
                {"localprefix", "732"},
                {"dialoperator", "0"},
                {"fastpickup", "2"},
                {"slowpickup", "5"},
                {"country", "1"},
                {"ringtime", "6"},
                {"language", "english"},
                {"extension", ".au"},
                {"voice", "UsEngM"},
                {NULL, NULL}};

        setValue("node", keyserver.getNode());
        setValue("version", BAYONNE_VERSION);
        setValue("server", BAYONNE_SERVER);
        setValue("drivers", plugins.getLast("drivers"));

	TTS *tts = TTS::ttsFirst;

	while(tts)
	{
		if(ttslen)
			snprintf(ttsnames, sizeof(ttsnames) - ttslen, ",%s", tts->getName());
		else
			snprintf(ttsnames, sizeof(ttsnames), "%s", tts->getName());
		tts = tts->ttsNext;
		ttslen = (unsigned)strlen(ttsnames);
	}

	if(ttslen)
		setValue("tts", ttsnames);
	else
		setValue("tts", "none");

        Keydata::load(keys);

	commit();
	errlog("notice", NULL);
}

void aaImage::scanDir(char *path, const char *ext, size_t size)
{
	const char *scrname;
	char *pp = path + strlen(path);
	Dir dir(path);
	size_t ps = path + size - pp;

	*(pp++) = '/';
	--ps;

	for(;;)
	{
		scrname = dir.getName();
		if(!scrname)
			break;
		if(!isScript(scrname, ext))
			continue;

		setString(pp, ps, scrname);
		compile(path);
	}
	*(--pp) = 0;
}


bool aaImage::isScript(const char *scrname, const char *ext1)
{
	char *ext = strrchr(scrname, '.');
	if(!ext)
		return false;

	// handle in pass 2
	if(!stricmp(ext, ".scr") && !stricmp(ext1, ".bs"))
		return true;

	if(!stricmp(ext, ext1))
		return true;

	if(!stricmp(ext, ".tts") && TTS::ttsFirst && !stricmp(ext1, ".bs"))
		return true;

	return false;
}

#ifdef	CCXX_NAMESPACES
}
#endif
