// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "bayonnelibrary.h"
#include <ctime>
#include <cmath>

#ifndef	M_PI
#define	M_PI	3.14159265358979323846
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

aaScript *aascript = NULL;

void initScripting(void)
{
//	aaScript *invoke;
	aascript = new aaScript();

	// This creates the "default" script box, which is templated for other
	// script collections through a reference constructor.  A given script
	// box has the following, for which the default box has presets:
	//
	// name; name of our "box"
	// startup; default startup script (usually main)
	// voices; prefix to voice libraries
	// prompts; application specific recorded prompts
	// datafiles; where we have "var" prompts recorded if not default path
	// scripts; where we compile scripts from
	// includes; where we process includes from (if not same as scripts)
	//
	// alternately one can specify files directly, including
	// audiofile.au=filename for explicit audio audio prompts,
	// scrfile.scr=filename for explicit script files, and
	// scrfile.inc=filename for explicit include files.  These are
	// set from the argv[] and passed through idx.

	aascript->setValue("name", "-bayonne");
	aascript->setValue("scripts", keypaths.getLast("scripts"));

	ScriptInterp::addConditional("voice", &Trunk::hasVoice);
	ScriptInterp::addConditional("sys", &Trunk::hasSysVoice);
	ScriptInterp::addConditional("sysvoice", &Trunk::hasSysVoice);
	ScriptInterp::addConditional("sysprompt", &Trunk::hasSysPrompt);
	ScriptInterp::addConditional("varprompt", &Trunk::hasVarPrompt);
	ScriptInterp::addConditional("prompt", &Trunk::hasVarPrompt);
	ScriptInterp::addConditional("group", &Trunk::hasGroup);
	ScriptInterp::addConditional("policy", &Trunk::hasGroup);
	ScriptInterp::addConditional("plugin", &Trunk::hasPlugin);
	ScriptInterp::addConditional("driver", &Trunk::hasDriver);
	ScriptInterp::addConditional("node", &Trunk::isNode);
	ScriptInterp::addConditional("feature", &Trunk::ifFeature);
	ScriptInterp::addConditional("running", &Trunk::ifRunning);
	ScriptInterp::addConditional("ringing", &Trunk::ifRinging);
	ScriptInterp::addConditional("port", &Trunk::ifPort);

	new aaImage(aascript);

	if(!debug)
		new Debug();

//	invoke = new aaScript(aascript);
//	invoke->setValue("name", "-invoke");
//	invoke->setValue("scripts", keypaths.getLast("scripts"));
//	invoke->setValue("extensions", ".is");
//	new aaImage(invoke);
}


bool hasTTS(void)
{
	if(TTS::ttsFirst)
		return true;

	if(plugins.getLast("say"))
		return true;

	return false;
}

Keyroute::Keyroute() : String()
{
	memset(&idx, 0, sizeof(idx));
	fre = NULL;
}

void Keyroute::clear(const char *id)
{
	key *entry, *prior = NULL;
	size_t len;

	if(!id || !stricmp(id, "default"))
	{
		String::set("");
		return;
	}

	len = strlen(id);

	if(len < 1)
		return;

	if(len > 32)
		len = 32;

	entry = idx[len - 1];
	while(entry)
	{
		if(!strnicmp(entry->id, id, len))
			break; 
		prior = entry;
		entry = entry->next;
	}
	if(!entry)
		return;

	if(!prior)
		idx[len - 1] = entry->next;
	else
		prior->next = entry->next;

	entry->value = "";
	entry->next = fre;
	fre = entry;
}

const char *Keyroute::last(const char *id)
{
        key *entry;
        size_t len;
        size_t il = strlen(id);
	char *other = c_str();

        if(il < 1)
                return NULL;
                                                                                
        if(il > 32)
                il = 32;
        
	len = il;
                                                                        
        while(len)
        {
                entry = idx[len - 1];
                while(entry)
                {
                        if(!strnicmp(id + il - len, entry->id, len))
				return entry->value.c_str();
			entry = entry->next;
		}		                                                                        
		--len;
	}
	if(other && *other)
		return other;
	return NULL;
}

const char *Keyroute::first(const char *id)
{
	key *entry;
	size_t len;
	size_t il = strlen(id);
	char *other = c_str();

	if(il < 1)
		return NULL;

	if(il > 32)
		il = 32;

	len = il;

	while(len)
	{
		entry = idx[len - 1];
		while(entry)
		{
			if(!strnicmp(id, entry->id, len))
				return entry->value.c_str();
			entry = entry->next;
		}	
		--len;
	}
	if(other && *other)
		return other;
	return NULL;
}

void Keyroute::set(const char *id, const char *value)
{
	key *entry;
	size_t len;

	if(!id || !stricmp(id, "default"))
	{
		String::set(value);
		return;
	}

	len = strlen(id);

	if(len < 1)
		return;

	if(len > 32)
		len = 32;

	entry = idx[len - 1];
	while(entry)
	{
		if(!strnicmp(entry->id, id, len))
			break;
		entry = entry->next;
	}
	if(entry)
	{
		entry->value = value;
		return;
	}
	if(!entry && fre)
	{
		entry =  fre;
		fre = entry->next;
	}
	if(!entry)
		entry = new key;

	entry->next = idx[len - 1];
	idx[len - 1] = entry;
	strncpy(entry->id, id, len);
	entry->id[len] = 0;
	entry->value = value;
}

const char *KeyServer::getIncoming(Trunk *trunk)
{
	const char *route = getLast("testing"), *value;
	if(route)
		return route;
	readLock();
	value = trunk->getSymbol(SYM_DNID);
	if(value && *value)
		route = incoming.last(value);
	if(!route || !*route)
		route = getLast("default");
	trunk->setConst(SYM_ROUTE, route);
	route = trunk->getSymbol(SYM_ROUTE);
	unlock();
	return route;
}

static void loader(const char *name)
{
	DSO *dso = new DSO(name);
	if(!dso->isValid())
		exit(-1);
}

KeyLocal::KeyLocal() :
Keydata("/server/localize")
{
	static Keydata::Define keydefs[] = {
	{"primarycurrency", "dollar"},
	{"primarychange", "cent"},
	{"convertcurrency", "1.00"},
	{NULL, NULL}};

	load(keydefs);
}

KeyProxy::KeyProxy() :
Keydata("/server/proxy")
{
	static Keydata::Define keydefs[] = {
	{"timeout", "0"},
	{"cache", "60s"},
	{NULL, NULL}};

	load(keydefs);
}

const char *KeyProxy::getHTTPServer(void)
{
	const char *cp = getenv("BAYONNE_PROXY_SERVER");

	if(cp)
		return cp;

	cp = getLast("httpserver");
	if(!cp)
		cp = getLast("server");
	return cp;
}

tpport_t KeyProxy::getHTTPPort(void)
{
	const char *cp = getenv("BAYONNE_PROXY_PORT");

	if(cp)
		return atoi(cp);

	if(!getHTTPServer())
		return 0;

	cp = getLast("httpport");
	if(!cp)
		cp = getLast("port");
	if(cp)
		return atoi(cp);

	return 80;
}

timeout_t KeyProxy::getTimeout(void)
{
	return 1000l * atoi(getLast("timeout"));
}

KeyVoices::KeyVoices() :
Keydata("/server/voices")
{
	static Keydata::Define defimports[] = {
	{"UsEngM", "english"},
	{"UsEngF", "english"},
	{"EnglishM", "english"},
	{"EnglishF", "english"},
	{"SpanishM", "spanish"},
	{"SpanishF", "spanish"},
	{"FrenchM", "french"},
	{"FrenchF", "french"},
	{"RussianM", "russian"},
	{"RussianF", "russian"},
	{"BengaliM", "bengali"},
	{"BengaliF", "bengali"},
	{"ItalianF", "italian"},
	{"ItalianM", "italian"},
	{"GermanF", "german"},
	{"GermanM", "german"},
	{NULL, NULL}};

	load(defimports);
}

KeyPaths::KeyPaths() :
Keydata("/server/paths")
{
#ifdef	WIN32

	static Keydata::Define defpaths[] = {
	{"var", PATH_LOCALSTATE_BAYONNE},
	{"tmp", PATH_VARTMP_BAYONNE},
	{"tmpfs", PATH_VARTMP_BAYONNE},
	{"spool", PATH_VARSPOOL_BAYONNE},
	{"cache", PATH_VARCACHE_BAYONNE},
	{"runfiles", PATH_VARRUN_BAYONNE},
	{"logpath", PATH_VARLOG_BAYONNE},
	{"datafiles", PATH_VARLIB_BAYONNE},
	{"scripts", PATH_DATADIR_SCRIPTS},
	{"voices", PATH_DATADIR_VOICES},
	{"prompts", PATH_DATADIR_PROMPTS},
	{"libexec", PATH_DATADIR_LIBEXEC},
	{"includes", PATH_DATADIR_INCLUDES},
	{NULL, NULL}};

#else
	static Keydata::Define defpaths[] = {
	{"spool", PATH_VARSPOOL_BAYONNE},
	{"cache", PATH_VARCACHE_BAYONNE},
	{"runfiles", PATH_VARRUN_BAYONNE},
	{"logpath", PATH_VARLOG_BAYONNE},
	{"sox", "/usr/bin/sox"},
	{NULL, NULL}};
#endif

	load(defpaths);
}

#ifndef	WIN32

void KeyPaths::setPrefix(char *input)
{
	char tmpfs[65];
	char prefix[1024];
	const char *cp;
	char *pp;
	size_t ps;

	setString(prefix, sizeof(prefix), input);
	pp = prefix + strlen(prefix);
	ps = sizeof(prefix) - strlen(prefix);

	cp = getLast("libpath");
	if(!cp)
	{
		setString(pp, ps, "/lib/bayonne");
		setValue("libpath", prefix);
	}

	cp = getLast("libexec");
	if(!cp)
	{
		setString(pp, ps, "/libexec/bayonne");
		setValue("libexec", prefix);
	}
	cp = getLast("tgipath");
	if(!cp)
	{
		setString(pp, ps, "/libexec/bayonne:/bin:/usr/bin");
		setValue("tgipath", prefix);
	}

	cp = getLast("scripts");
	if(!cp)
	{
		setString(pp, ps, "/share/bayonne");
		setValue("scripts", prefix);
	}

	cp = getLast("prompts");
	if(!cp)
	{
		setString(pp, ps, "/share/bayonne/sys");
		setValue("prompts", prefix);
	}

	cp = getLast("voices");
	if(!cp)
	{
		setString(pp, ps, "/share/bayonne");
		setValue("voices", prefix);
	}

	cp = getLast("tmpfs");
	if(!cp)
		cp = "/dev/shm";

	if(!isDir(cp))
		cp = "/tmp";

	snprintf(tmpfs, sizeof(tmpfs), "%s/.bayonne", cp);
	setValue("tmpfs", tmpfs);
	setValue("tmp", "/tmp/.bayonne");

	cp = getenv("CONFIG_KEYDATA");
	if(!cp)
		cp = ETC_PREFIX "bayonne/";
	setValue("etc", cp);

	*pp = 0;
}

#endif

KeyServer::KeyServer() :
Keydata("/server/server")
{
	static Keydata::Define defkeys[] = {
	{"user", "bayonne"},
	{"group", "bayonne"},
	{"default", "default"},
	{"nodes", "1"},
	{"stats", "15"},
	{"token", "&"},
	{"password", "fts"},
	{"login", "none"},
	{"dialplan", "1"},
	{"policy", "dnis,card,span,port"},
	{"calls", "id,startdate,starttime,duration,source,target,route"},
	{"usage", "60"}, 
	{NULL, NULL}};

	char namebuf[128];
	char *cp;
	char *value;
	char *tok;
	FILE *fp;

	if(!getLast("node"))
	{
		gethostname(namebuf, sizeof(namebuf) - 1);
		cp = strchr(namebuf, '.');
		if(cp)
			*cp = 0;
		cp = strchr(namebuf, '-');
		if(cp)
			*cp = 0;
		setValue("node", namebuf);
	}
	load(defkeys);

	snprintf(namebuf, sizeof(namebuf), "%sroute.conf", Process::getEnv("CONFIG_KEYDATA"));
	fp = fopen(namebuf, "r");
	if(!fp)
		return;

	for(;;)
	{
		fgets(namebuf, sizeof(namebuf), fp);
		if(feof(fp))
			break;
		cp = strtok_r(namebuf, " \t\r\n", &tok);
		if(!cp || !*cp)
			continue;

		if(!stricmp(cp, "incoming"))
		{
			cp = strtok_r(NULL, " \t\r\n", &tok);
			if(!cp || !*cp)
				continue;
	
			value = strtok_r(NULL, " \t\r\n", &tok);
			if(!value || !*value)
				continue;
			incoming.set(cp, value);
		}
	}
	fclose(fp);
}

void KeyServer::printRoutes(void)
{
	Keyroute::key *key;
	int id;
	const char *other;
	FILE *tmp;
	const char *tmppath = keypaths.getLast("tmproutes");
	remove(tmppath);
	tmp = fopen(tmppath, "w");
	if(!tmp)
	{
		slog.error("route: unable to print %s", tmppath);
		return;
	}

	other = getLast("testing");
	if(other)
		fprintf(tmp, "default  %-32s %s\n", "testing", other);
	other = getLast("default");
	if(other)
		fprintf(tmp, "default  %-32s %s\n", "default", other);
	other = incoming.c_str();
	if(other && *other)
		fprintf(tmp, "incoming %-32s %s\n", "default", other);
 
	for(id = 31; id > -1; --id)
	{
		key = incoming.idx[id];
		while(key)
		{
			fprintf(tmp, "incoming %32s %s\n",
				key->id, key->value.c_str());
			key = key->next;
		}
	}

	fclose(tmp);
	rename(tmppath, keypaths.getLast("routes")); 
}

void KeyServer::loadGroups(bool test)
{
	char *group, *tok;
	char buffer[256];

	new TrunkGroup();
	slog.debug() << "loading default trunk group" << endl;

	group = (char *)TrunkGroup::first->getLast("groups");

	if(!group || test)
		return;

	setString(buffer, sizeof(buffer), group);
	group = strtok_r(buffer, " \t\n", &tok);
	while(group)
	{
		slog.debug() << "loading " << group << " trunk group" << endl;
		new TrunkGroup(group);
		group = strtok_r(NULL, " \t\n", &tok);
	}
}

KeyThreads::KeyThreads() :
Keydata("/server/threads")
{
	static Keydata::Define defpaths[] = {
	{"audit", "0"},
	{"audio", "0"},
	{"priority", "0"},
	{"gateways", "0,1"},
	{"buffers", "16"},
	{"services", "1,1"},
	{"database", "0"},
	{"network", "0"},
	{"switch", "0"},
	{"managers", "0"},
	{"gui", "0"},
	{"rtp", "0"},
	{"resetdelay", "18"},
	{"stepdelay", "36"},
	{"stepinterval", "18"},
	{"refresh", "5"},
	{"policy", "other"},
	{"pages", "0"},
	{"stack", "8"},
	{"audiostack", "16"},
	{"driverstack", "64"},
	{NULL, NULL}};

	const char *cp = getLast("pri");

	if(cp)
		setValue("priority", cp);

	load(defpaths);

	cp = getLast("autostack");
	if(cp)
		Thread::setStack(atoi(cp) * 1024);
}

int KeyThreads::getPolicy()
{
	const char *cp = getLast("policy");
#ifdef	SCHED_RR
	if(!stricmp(cp, "rr"))
		return SCHED_RR;
#endif
#ifdef	SCHED_FIFO
	if(!stricmp(cp, "fifo"))
		return SCHED_FIFO;
#endif
	return 0;
}

#ifdef	NODE_SERVICES

KeyNetwork::KeyNetwork() :
Keydata("/server/network")
{
	static Keydata::Define defkeys[] = {
	{"refresh", "5"},
	{"live", "25"},
	{"elect", "120"},
	{"expire", "300"},
	{"database", "127.0.0.1:7002"},
	{"address", "127.0.0.1:7001"},
	{"broadcast", "127.255.255.255"},
	{"monitor", "127.0.0.1:7070"},
	{NULL, NULL}};

	load(defkeys);
}

InetAddress KeyNetwork::getAddress(void)
{
	char buffer[65];
	char *cp;

	setString(buffer, sizeof(buffer), getLast("address"));
	cp = strchr(buffer, ':');
	if(*cp)
		*cp = 0;
	return InetAddress(buffer);
}

InetAddress KeyNetwork::getMonitorAddress(void)
{
	char buffer[65];
	char *cp;

	setString(buffer, sizeof(buffer), getLast("monitor"));
	cp = strchr(buffer, ':');
	if(*cp)
		*cp = 0;
	return InetAddress(buffer);
}

InetHostAddress KeyNetwork::getBroadcast(void)
{
	return InetHostAddress(getLast("broadcast"));
}

tpport_t KeyNetwork::getPort(void)
{
	char buffer[65];
	char *cp;

	setString(buffer, sizeof(buffer), getLast("address"));
	cp = strchr(buffer, ':');
	if(cp)
		return atoi(++cp);
	return 0;
}

tpport_t KeyNetwork::getMonitorPort(void)
{
	char buffer[65];
	char *cp;

	setString(buffer, sizeof(buffer), getLast("monitor"));
	cp = strchr(buffer, ':');
	if(cp)
		return atoi(++cp);
	return 0;
}

#endif

KeyMemory::KeyMemory() :
Keydata("/server/memory")
{
	static Keydata::Define defpaths[] = {
	{"symbols", "64"},
	{"page", "1024"},
	{"users", "1000"},
	{NULL, NULL}};

	load(defpaths);
}

size_t KeyThreads::getStack(void)
{
	const char *cp = getLast("stack");

	if(cp)
		return atoi(cp) * 1024 + PATH_MAX;

	return 0;
}

size_t KeyThreads::getAudioStack(void)
{
	const char *cp = getLast("audiostack");

	if(cp)
		return atoi(cp) * 1024 + (PATH_MAX * 2);

	return getStack();
}


size_t KeyThreads::getDriverStack(void)
{
	const char *cp = getLast("driverstack");

	if(cp)
		return atoi(cp) * 1024 + (PATH_MAX * 2);

	return getAudioStack();
}

int KeyThreads::getServices(void)
{
	char buf[32];
	int cnt;

	setString(buf, sizeof(buf), getLast("services"));
	char *cp = strchr(buf, ',');
	if(cp)
	{
		++cp;
		while(*cp == ' ' || *cp == '\t')
			++cp;
		cnt = atoi(cp);
		if(cnt > 0)
			return cnt;
	}
	return 1;
}

int KeyThreads::getGateways(void)
{
	char buf[32];

	setString(buf, sizeof(buf), getLast("gateways"));
	char *cp = strchr(buf, ',');
	if(cp)
	{
		++cp;
		if(*cp == ' ' || *cp == '\t')
			++cp;
		return atoi(cp);
	}
	else
		return 1;
}

Plugins::Plugins() :
Keydata("/server/plugins")
{
	if(!getLast("codecs"))
		setValue("codecs", "g.711");

	if(getLast("drivers") == NULL)
	{
#if defined(have_montecarlo_h) || defined(HAVE_MONTECARLO_H)
		setValue("drivers", "pika");
#elif defined(HAVE_DIALOGIC_SDK)
		setValue("drivers", "dialogic");
#elif defined(HAVE_VPBAPI_H)
		setValue("drivers", "vpb");
#elif defined(HAVE_CAPI20_H)
		setValue("drivers", "capi20");
#elif defined(HAVE_LINUX_TELEPHONY_H)
		setValue("drivers", "ltapi");
#elif defined(HAVE_SYS_SOUNDCARD_H)
		setValue("drivers", "oss");
#else
		setValue("drivers", "none");
#endif
	}

	if(getLast("sql") == NULL && getLast("database") == NULL)
	{
#if defined(HAVE_SQL_H) || defined(HAVE_ODBC_SQL_H)
		setValue("database", "odbc");
#elif defined(HAVE_PGSQL_POSTGRES) || defined(HAVE_POSTGRES)
		setValue("database", "postgres");
#else
		setValue("database", "none");
#endif
	}

	pidcount = 0;
}

Plugins::~Plugins()
{
//	dynunload();
#ifdef	HAVE_TGI
	while(pidcount)
		kill(pids[--pidcount], SIGTERM);
#endif
}

void Plugins::loadCodecs(void)
{
        char list[512];
        char *cp;

        cp = (char *)getLast("codecs");
        if(!cp)
                return;

        if(!stricmp(cp, "none"))
                return;

        if(!stricmp(cp, "no"))
                return;

	setString(list, sizeof(list), cp);
        cp = strtok(list, " \t\n;,");
        while(cp)
        {
                if(!*cp)
                        break;

#ifdef	AUDIO_CODEC_MODULES
                AudioCodec::load(cp);
#endif
                cp = strtok(NULL, " \t\n;,");
        }
}

void Plugins::loadModules(void)
{
	char list[512];
	char *cp;

	cp = (char *)getLast("modules");
	if(!cp)
		return;

	if(!stricmp(cp, "none"))
		return;

	if(!stricmp(cp, "no"))
		return;

	setString(list, sizeof(list), cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		Script::use(cp);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadExtensions(void)
{
	char path[256];
	char list[512];
	char *cp;
	const char *dp = Process::getEnv("BAYONNE_LIBRARY");

	cp = (char *)getLast("extensions");
	if(!cp)
		cp = (char *)getLast("server");
	if(!cp)
		cp = (char *)getLast("load");
	if(!cp)
		return;

	if(!stricmp(cp, "none"))
		return;

	if(!stricmp(cp, "no"))
		return;

	setString(list, sizeof(list), cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(strchr(cp, '/'))
			break;

#ifdef	WIN32
		if(strchr(cp, '\\'))
			break;

		if(dp && *dp)
			snprintf(path, sizeof(path), "%s/plugins/%s.rll", dp, cp);
		else
			snprintf(path, sizeof(path), "../w32/plugins/%s.rll", cp);
#else
		if(dp && *dp)
			snprintf(path, sizeof(path), "%s/plugins/%s.dso", dp, cp);
		else
			snprintf(path, sizeof(path), "../modules/plugins/%s.dso", cp);
#endif
		loader(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadTGI(void)
{
	char path[256];
	char list[512];
	char *cp;
	const char *dp = Process::getEnv("BAYONNE_LIBRARY");

	cp = (char *)getLast("tgi");
	if(!cp)
		return;

	if(!stricmp(cp, "none"))
		return;

	if(!stricmp(cp, "no"))
		return;

	setString(list, sizeof(list), cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(strchr(cp, '/'))
			break;

#ifdef	WIN32
		if(strchr(cp, '\\'))
			break;

		if(dp && *dp)
			snprintf(path, sizeof(path), "%s/tgi/%s.rll", dp, cp);
		else
			snprintf(path, sizeof(path), "../w32/tgi/%s.rll", cp);
#else
		if(dp && *dp)
			snprintf(path, sizeof(path), "%s/tgi/%s.dso", dp, cp);
		else
			snprintf(path, sizeof(path), "../modules/tgi/%s.dso", cp);
#endif
		loader(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadTranslators(const char *lcp)
{
	char path[256];
	char list[512];
	char *cp;
	const char *dp = Process::getEnv("BAYONNE_LIBRARY");

	if(!lcp)
		lcp = getLast("languages");

	if(!lcp)
		return;

	setString(list, sizeof(list), lcp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(strchr(cp, '/'))
			break;

#ifdef	WIN32
		if(strchr(cp, '\\'))
			break;

		if(!dp || !*dp)
			dp="../w32";
		snprintf(path, sizeof(path), "%s/Translators/%s.rll", dp, cp);
#else
		if(!dp || !*dp)
			dp = "../modules";
		snprintf(path, sizeof(path), "%s/translators/%s.dso", dp, cp);
#endif
		loader(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadTTS(void)
{
	char path[256];
	char list[512];
	char *cp;
	const char *dp = Process::getEnv("BAYONNE_LIBRARY");
	const char *lcp = getLast("tts");

	if(!lcp)
		return;

	setString(list, sizeof(list), lcp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(strchr(cp, '/'))
			break;

#ifdef	WIN32
		if(strchr(cp, '\\'))
			break;

		if(!dp || !*dp)
			dp="../w32";
		snprintf(path, sizeof(path), "%s/Synth/%s.rll", dp, cp);
#else
		if(!dp || !*dp)
			dp = "../modules";
		snprintf(path, sizeof(path), "%s/tts/%s.dso", dp, cp);
#endif
		loader(path);
		cp = strtok(NULL, " \t\n;,");
	}
}


void Plugins::loadDebug(void)
{
	char path[256];
	char *d = (char *)getLast("debug");
	const char *dp = Process::getEnv("BAYONNE_LIBRARY");

	if(!d || strchr(d, '/'))
	{
		new Debug();
		return;
	}

#ifdef	WIN32
	if(!dp || !*dp)
		dp = "../w32";
	snprintf(path, sizeof(path), "%s/Debug/%s.rll", dp, d);
#else
	if(!dp || !*dp)
		dp = "../modules";
	snprintf(path, sizeof(path), "%s/debug/%s.dso", dp, d);
#endif

	loader(path);
	if(!debug)
	{
		slog.notice() << "no debug handler installed" << endl;
		new Debug();
	}
}

void Plugins::loadDatabase(void)
{
	char path[256];
	const char *d = Process::getEnv("SQL_DRIVER");
	const char *dp = Process::getEnv("BAYONNE_LIBRARY");

	if(!d || !*d)
		d = getLast("database");

	if(!d)
		d = getLast("sql");

	if(!d || strchr(d, '/') || !stricmp(d, "none"))
		return;

#ifdef	WIN32
	if(!dp || !*dp)
		dp = "../w32";
	snprintf(path, sizeof(path), "%s/Databases/%s.rll", dp, d);
#else
	if(!dp || !*dp)
		snprintf(path, sizeof(path), "../modules/%s/%s.dso", d, d);
	else
		snprintf(path, sizeof(path), "%s/databases/%s.dso", dp, d);
#endif

	loader(path);
}

void Plugins::loadDriver(void)
{
	char path[256];
	char list[512];
	const char *drv = getLast("drivers");
	char *cp;
	const char *dp = Process::getEnv("BAYONNE_LIBRARY");

	snprintf(list, sizeof(list), "%s", drv);
	cp = strtok(list, " \t\n;,");

	while(cp)
	{
		if(!*cp)
			break;

		if(strchr(cp, '/'))
			break;

#ifdef	WIN32
		if(strchr(cp, '\\'))
			break;

		if(!dp || !*dp)
			dp="../w32";
		snprintf(path, sizeof(path), "%s/Drivers/%s.rll", dp, cp);
#else
		if(!dp || !*dp)
			snprintf(path, sizeof(path), "../drivers/%s/%s.dso", cp, cp);
		else
			snprintf(path, sizeof(path), "%s/drivers/%s.dso", dp, cp);
#endif

		loader(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

KeyTones::KeyTones() :
Keydata("/tones/default")
{
        static Keydata::Define keydefs[] = {
	{"ringback", "440 480 2000"},
	{"busytone", "480 620 500"},
	{"reorder", "480 620 250"},
	{"dialtone", "350 440 1000"},
	{"pbx:dialtone", "cont 350 440 30000"},
	{"intercom", "350 440 100"},
	{NULL, NULL}};

	int v1, v2, v3;
	char *tone;
	phTone *pht;
	unsigned count = getCount() + 1;
	char **tones;
	bool cont, play;
	unsigned playtime;

	if(count < 2)
	{
		load(keydefs);
		count = getCount() + 1;
	}
	tones = new char *[count];
	getIndex(tones, count);

	while(*tones)
	{
		playtime = 1000;
		cont = false;
		play = false;
		v1 = v2 = v3 = 0;
		tone = (char *)getLast(*tones);
		if(!tone || !stricmp(*tones, "tones"))
		{
			++tones;
			continue;
		}

		if(!strnicmp(*tones, "pbx:", 4))
		{
			++tones;
			continue;
		}

		while(isspace(*tone))
			++tone;

		if(!strnicmp(tone, "cont", 4))
		{
			play = true;
			cont = true;
			playtime = 0;
		}
		else if(!strnicmp(tone, "play", 4))
		{
			play = true;
			tone = strchr(tone, ' ');
			while(isspace(*tone))
				++tone;
			playtime = atoi(tone);
		}
		if(play)
			tone = strchr(tone, ' ');
		sscanf(tone, "%d %d %d", &v1, &v2, &v3);
		if(v3)
			pht = new phTone(*tones, v3, v1, v2);
		else
			pht = new phTone(*tones, v2, v1);

		if(playtime && playtime < pht->duration)
			playtime = pht->duration;
		pht->playtime = playtime;
		++tones;
	}
}

TrunkGroup *TrunkGroup::first = NULL;

TrunkGroup::TrunkGroup(char *name) :
Keydata("/server/policy"), CallStat()
{
	Keydata::Define keys[] = {
		{"answer", "1"},
		{"accept", "1"},
		{"hangup", "100"},
		{"siezetime", "12"},
		{"ringtime", "7"},
		{"flash", "200"},
		{"dialtone", "2400"},
		{"dialspeed", "160"},
		{"volume", "80"},
		{"callerid", "1500"},
		{"pickup", "500"},
		{"requests", "hangup"},
		{"select", "last"},
		{"threashold", "0"},
		{"ready", "1000"},
		{"idletime", "600"},
		{"analysis", "16"},
		{"international", "011"},
		{"national", "1"},
		{"dialmode", "dtmf"},
		{"mindigits", "0"},
		{"mdigtimeout", "3"},
		{"interval", "60"},
		{NULL, NULL}};

	char *cp;
	char namebuf[65];
	char tag[65];
	char *tok;

	if(name)
	{
		snprintf(namebuf, sizeof(namebuf), "/policy/%s", name);
		load(namebuf);
	}
	if(name && first)
	{
		cp = (char *)getLast("ports");
		if(cp)
			cp = strtok_r(cp, ",;: \t", &tok);
		while(cp)
		{
			snprintf(tag, sizeof(tag), "port.%s", cp);
			first->setValue(tag, name);
			cp = strtok_r(NULL, ",;: \t", &tok);
		}
		cp = (char *)getLast("spans");
		if(cp)
			cp = strtok_r(cp, ",;: \t", &tok);
		while(cp)
		{
			snprintf(tag, sizeof(tag), "card.%s", cp);
			first->setValue(tag, name);
			cp = strtok_r(NULL, ",;: \t", &tok);
		}
		cp = (char *)getLast("cards");
		if(cp)
			cp = strtok_r(cp, ",;: \t", &tok);
		while(cp)
		{
			snprintf(tag, sizeof(tag), "card.%s", cp);
			first->setValue(tag, name);
			cp = strtok_r(NULL, ",;: \t", &tok);
		}
	}
	else
		name = "*";

	setValue("name", name);

	load(keys);
	next = NULL;
	reqfirst = reqlast = NULL;
	polFirst = NULL;
	members = 0;

	if(!first)
		first = this;
	else
	{
		next = first;
		while(next->next)
			next = next->next;
		next->next = this;
	}
	next = NULL;
}

bool TrunkGroup::getAccept(void)
{
        const char *cp = getLast("accept");
        if(!cp)
                return false;

        switch(*cp)
        {
        case '0':
        case 'f':
        case 'F':
        case 'n':
        case 'N':
                return false;
        }
        return true;
}

bool TrunkGroup::getDetect(void)
{
	const char *cp = getLast("detect");
	if(!cp)
		return false;

	switch(*cp)
	{
	case '0':
	case 'f':
	case 'F':
	case 'n':
	case 'N':
		return false;
	}

	return true;
}

seltype_t TrunkGroup::getSelect(void)
{
	const char *cp = getLast("select");

	if(!stricmp(cp, "last"))
		return SELECT_LAST;

	return SELECT_FIRST;
}

const char *TrunkGroup::getNumber(void)
{
	const char *num = getLast("number");

	if(num)
		return num;

	return "UNKNOWN";
}

void TrunkGroup::logStats(void)
{

	char buffer[128];
	TrunkGroup *grp = first;
	const char *name;
	time_t now;
	time_t current, prior, interval;
	struct tm *dt, tbuf, ubuf;
	int hr, min;

	time(&now);
	
	while(grp)
	{
		name = grp->getName();
		if(!stricmp(name, "*"))
			name = "default";
		interval = atol(grp->getLast("interval")) * 60l;
		prior = grp->updated / interval;
		current = now / interval;
		if(prior != current)
		{
			current = now - 60l;
			dt = localtime_r(&current, &tbuf);
			hr = dt->tm_hour;
			min = dt->tm_min;
			dt = localtime_r(&grp->updated, &ubuf);
			grp->enterMutex();
			snprintf(buffer, sizeof(buffer), 
				"stats %s %0ld/%ld %d %02d/%02d %02d:%02d %02d:%02d %ld %ld %d %d\n",
				name, prior, interval, grp->capacity,
				dt->tm_mon, dt->tm_mday, 
				dt->tm_hour, dt->tm_min, hr, min,
				grp->total.incoming, grp->total.outgoing,
				grp->max.incoming, grp->max.outgoing);
			grp->update();
			grp->leaveMutex();
			fifo.command(buffer);
		}
		grp = grp->next;
	}
}

phTone *phTone::first = NULL;

unsigned char phTone::alaw[256] = {
	0xab, 0x55, 0xd5, 0x15, 0x95, 0x75, 0xf5, 0x35,
	0xb5, 0x45, 0xc5, 0x05, 0x85, 0x65, 0xe5, 0x25,
	0xa5, 0x5d, 0xdd, 0x1d, 0x9d, 0x7d, 0xfd, 0x3d,
	0xbd, 0x4d, 0xcd, 0x0d, 0x8d, 0x6d, 0xed, 0x2d,
	0xad, 0x51, 0xd1, 0x11, 0x91, 0x71, 0xf1, 0x31,
	0xb1, 0x41, 0xc1, 0x01, 0x81, 0x61, 0xe1, 0x21,
	0x59, 0xd9, 0x19, 0x99, 0x79, 0xf9, 0x39, 0xb9,
	0x49, 0xc9, 0x09, 0x89, 0x69, 0xe9, 0x29, 0xa9,
	0xd7, 0x17, 0x97, 0x77, 0xf7, 0x37, 0xb7, 0x47,
	0xc7, 0x07, 0x87, 0x67, 0xe7, 0x27, 0xa7, 0xdf,
	0x9f, 0x7f, 0xff, 0x3f, 0xbf, 0x4f, 0xcf, 0x0f,
	0x8f, 0x6f, 0xef, 0x2f, 0x53, 0x13, 0x73, 0x33,
	0xb3, 0x43, 0xc3, 0x03, 0x83, 0x63, 0xe3, 0x23,
	0xa3, 0x5b, 0xdb, 0x1b, 0x9b, 0x7b, 0xfb, 0x3b,
	0xbb, 0xbb, 0x4b, 0x4b, 0xcb, 0xcb, 0x0b, 0x0b,
	0x8b, 0x8b, 0x6b, 0x6b, 0xeb, 0xeb, 0x2b, 0x2b,
	0xab, 0x54, 0xd4, 0x14, 0x94, 0x74, 0xf4, 0x34,
	0xb4, 0x44, 0xc4, 0x04, 0x84, 0x64, 0xe4, 0x24,
	0xa4, 0x5c, 0xdc, 0x1c, 0x9c, 0x7c, 0xfc, 0x3c,
	0xbc, 0x4c, 0xcc, 0x0c, 0x8c, 0x6c, 0xec, 0x2c,
	0xac, 0x50, 0xd0, 0x10, 0x90, 0x70, 0xf0, 0x30,
	0xb0, 0x40, 0xc0, 0x00, 0x80, 0x60, 0xe0, 0x20,
	0x58, 0xd8, 0x18, 0x98, 0x78, 0xf8, 0x38, 0xb8,
	0x48, 0xc8, 0x08, 0x88, 0x68, 0xe8, 0x28, 0xa8,
	0xd6, 0x16, 0x96, 0x76, 0xf6, 0x36, 0xb6, 0x46,
	0xc6, 0x06, 0x86, 0x66, 0xe6, 0x26, 0xa6, 0xde,
	0x9e, 0x7e, 0xfe, 0x3e, 0xbe, 0x4e, 0xce, 0x0e,
	0x8e, 0x6e, 0xee, 0x2e, 0x52, 0x12, 0x72, 0x32,
	0xb2, 0x42, 0xc2, 0x02, 0x82, 0x62, 0xe2, 0x22,
	0xa2, 0x5a, 0xda, 0x1a, 0x9a, 0x7a, 0xfa, 0x3a,
	0xba, 0xba, 0x4a, 0x4a, 0xca, 0xca, 0x0a, 0x0a,
	0x8a, 0x8a, 0x6a, 0x6a, 0xea, 0xea, 0x2a, 0x2a};

int phTone::ulaw[256] = { 
	0,0,1,1,2,2,2,2,3,3,3,3,3,3,3,3,
        4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
        5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
        5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7};

unsigned char phTone::a2u[128] = {
	1, 3, 5, 7, 9, 11, 13, 15,
	16, 17, 18, 19, 20, 21, 22, 23, 
	24, 25, 26, 27, 28, 29, 30, 31, 
	32, 32, 33, 33, 34, 34, 35, 35, 
	36, 37, 38, 39, 40, 41, 42, 43, 
	44, 45, 46, 47, 48, 48, 49, 49, 
	50, 51, 52, 53, 54, 55, 56, 57, 
	58, 59, 60, 61, 62, 63, 64, 64, 
	65, 66, 67, 68, 69, 70, 71, 72, 
	73, 74, 75, 76, 77, 78, 79, 79,
	80, 81, 82, 83, 84, 85, 86, 87, 
	88, 89, 90, 91, 92, 93, 94, 95, 
	96, 97, 98, 99, 100, 101, 102, 103, 
	104, 105, 106, 107, 108, 109, 110, 111, 
	112, 113, 114, 115, 116, 117, 118, 119, 
	120, 121, 122, 123, 124, 125, 126, 127}; 

short phTone::ulaw2linear(unsigned char ul)
{
	/*short		t;

	ul = ~ul;

	t = ((ul & 0x0f) << 3) + 0x84;
	t <<= ((unsigned)ul & 0x70) >> 4;

	return ((ul & 0x80) ? (0x84 - t) : (t - 0x84));*/
	static int exp_lut[8] = {0,132,396,924,1980,4092,8316,16764};
	int sign, exponent, mantissa, sample;
	ul = ~ul;
	sign = (ul >> 4) & 0x80;
	exponent = (ul >> 4) && 0x07;
	mantissa = ul & 0x0F;
	sample = exp_lut[exponent] + (mantissa << (exponent + 3));

	return sample;
}

short phTone::alaw2linear(unsigned char al)
{
	short		t;
	short		seg;

	al ^= 0x55;

	t = (al & 0x0f) << 4;
	seg = ((unsigned)al & 0x70) >> 4;
	switch (seg) {
	case 0:
		t += 8;
		break;
	case 1:
		t += 0x108;
		break;
	default:
		t += 0x108;
		t <<= seg - 1;
	}
	return ((al & 0x80) ? t : -t);
}

unsigned char phTone::alaw2ulaw(unsigned char al)
{
	al &= 0xff; 
 	return ((al & 0x80) ? (0xFF ^ a2u[al ^ 0xD5]) : 
		(0x7F ^ a2u[al ^ 0x55])); 
} 

unsigned char phTone::linear2ulaw(int sample)
{
	int sign, exponent, mantissa, retval;

	sign = (sample >> 8) & 0x80;
	if(sign != 0) sample = -sample;
	sample += 0x84;
	exponent = ulaw[(sample >> 7) & 0xff];
	mantissa = (sample >> (exponent + 3)) & 0x0f;
	retval = ~(sign | (exponent << 4) | mantissa);
	if(!retval)
		retval = 0x02;

	return retval;
}

phTone::phTone(const char *n, timeout_t dur, unsigned f)
{
	unsigned i;
	int sample;

	if(n)
	{
		next = first;
		first = this;
	}
	else
		n = "*temp*";

        duration = playtime = dur;
        samples = new unsigned char[dur * 8];
        freq1 = f;
        freq2 = 0;
	snprintf(name, sizeof(name), "%s", n);

	double freq = (f * M_PI * 2) / 8000.;
	double pos = 0;

	for(i = 0; i < dur * 8; ++i)
	{
		sample = (int)(sin(pos) * 20000.0);
		pos += freq;
		samples[i] = linear2ulaw(sample);
	}
}

phTone::phTone(const char *n, timeout_t dur, unsigned f1, unsigned f2)
{
	unsigned i;
	int sample;

	if(n)
	{
		next = first;
		first = this;
	}
	else
		n = "*temp*";

        duration = playtime = dur;
        samples = new unsigned char[dur * 8];
        freq1 = f1;
        freq2 = f2;
	snprintf(name, sizeof(name), "%s", n);

	double fa1 = (f1 * M_PI * 2) / 8000.;
	double fa2 = (f2 * M_PI * 2) / 8000.;
	double pos1 = 0, pos2 = 0;

	for(i = 0; i < dur * 8; ++i)
	{
		sample = (int)((sin(pos1) + sin(pos2)) * 10000.0);
		pos1 += fa1;
		pos2 += fa2;
		samples[i] = linear2ulaw(sample);
	}
}

 
phTone::~phTone()
{
	if(samples)
		delete[] samples;
}


void phTone::clear(void)
{
	if(samples)
		delete[] samples;

	samples = NULL;
}

phTone *getphTone(const char *name) 
{
	phTone *tone = phTone::first;
	while(tone)
	{
		if(!stricmp(tone->name, name))
			break;
		tone = tone->next;
	}
        if(tone)
                return tone;

        name = strchr(name, ':');
        if(name)
		tone = getphTone(++name);

	return tone;
}

TrunkGroup *TrunkGroup::assign(Trunk *trunk)
{
	TrunkGroup *group = first;
	Driver *driver = trunk->driver;
	const char *dname = driver->getName();
	char buffer[65];
	const char *id;

	snprintf(buffer, sizeof(buffer), "port.%d", trunk->tsid);
	id = group->getLast(buffer);
	if(id)
		return getGroup(id);

	snprintf(buffer, sizeof(buffer), "span.%s/%d", dname, trunk->spanid);
	id = group->getLast(buffer);
	if(id)
		return getGroup(id);

	snprintf(buffer, sizeof(buffer), "card.%s/%d", dname, trunk->cardid);
	id = group->getLast(buffer);
	if(id)
		return getGroup(id);

	group = getGroup(dname);
	if(!group)
		group = first;

	return group;
}

TrunkGroup *TrunkGroup::getGroup(const char *name)
{
	TrunkGroup *group = first;

	if(!name)
		return group;

	if(!stricmp(name, "*"))
		return group;

	if(!stricmp(name, "default"))
		return group;

	while(group)
	{
		if(!stricmp(name, group->getName()))
		{
			name = group->getLast("remap");
			if(name)
				return getGroup(name);
			return group;
		}
		group = group->next;
	}
	return NULL;
}

ThreadLock cachelock;
KeyServer keyserver;
KeyPaths keypaths;
KeyLocal keylocal;
#ifdef	NODE_SERVICES
KeyNetwork keynetwork;
#endif
KeyVoices keyvoices;
KeyThreads keythreads;
KeyMemory keymemory;
KeyProxy keyproxy;
Plugins plugins;
#ifdef	NODE_SERVICES
Network *network;	// must be here for correct order of initialization
#endif
#ifdef	HAVE_TGI
int tgipipe[2];
#endif
char df_free[4] = "0";
char df_used[4] = "100";
bool running = false;
bool restart_server = false;

ScriptSymbol Trunk::globals(keymemory.getSymbolSize(), keymemory.getPageSize(), "[shared]");

#ifdef	CCXX_NAMESPACES
}
#endif
