// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "bayonnelibrary.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

Driver *Driver::drvFirst = NULL;
unsigned Driver::drvIndex = 0;
char Trunk::status[1024];

Driver::Driver() :
Mutex()
{
	active = false;

	tsid = getCount();	// prior drivers for our base

	drvNext = drvFirst;
	drvFirst = this;
	index = Driver::drvIndex++;

	portCount = downCount = idleCount = 0;
	nextTimer = 0;
	refTimer = 0;
}

bool Driver::isTrunkClass(int id, const char *name)
{
	Trunk *trunk = getTrunkPort(id);

	// many drivers do not have dynamic object creation, so we
	// just exit false if the trunk is not found...

	if(!trunk)
		return false;

	// all drivers recognize their own name

	if(!stricmp(getName(), name))
		return true;

	// many drivers recognize pstn...

	if(!stricmp(name, "pstn") && !trunk->isStation)
		return true;	

	if(!stricmp(name, "pots") && trunk->isStation)
		return true;

	if(!stricmp(name, "sta") && trunk->isStation)
		return true;

	return false;
}

EventBuffer::EventBuffer(unsigned count) :
Buffer(count)
{
	head = tail = 0;
	buffer = new TrunkEvent[count];
}

int EventBuffer::onWait(void *buf)
{
	TrunkEvent *bp = (TrunkEvent *)buf;

	*bp = buffer[head++];
	if(head >= getSize())
		head = 0;
	return 1;
}

int EventBuffer::onPost(void *buf)
{
	TrunkEvent *bp = (TrunkEvent *)buf;
	buffer[tail++] = *bp;
	if(tail >= getSize())
		tail = 0;
	return 1;
}

int EventBuffer::onPeek(void *buf)
{
	TrunkEvent *bp = (TrunkEvent *)buf;
	if(head == tail)
		return 0;

	*bp = buffer[head];
	return 1;
}

void EventBuffer::update(TrunkEvent *event)
{
	TrunkEvent ne;

	if(!event)
	{
		event = &ne;
		event->id = TRUNK_NULL_EVENT;
	}
	post(event);
}

void EventBuffer::fetch(timeout_t timeout, TrunkEvent *event)
{
	if(wait(event, timeout) < 0)
		event->id = TRUNK_TIMER_EXPIRED;
}

PortNotify::PortNotify() :
Buffer(256)
{
	head = tail = 0;
}

int PortNotify::onWait(void *buf)
{
	unsigned char *bp = (unsigned char *)buf;

	*bp = buffer[head++];
	if(head >= 256)
		head = 0;
	return 1;
}

int PortNotify::onPost(void *buf)
{
	unsigned char *bp = (unsigned char *)buf;
	buffer[tail++] = *bp;
	if(tail >= 256)
		tail = 0;
	return 1;
}

int PortNotify::onPeek(void *buf)
{
	unsigned char *bp = (unsigned char *)buf;
	if(head == tail)
		return 0;

	*bp = buffer[head];
	return 1;
}

void PortNotify::update(unsigned char data)
{
	post((void *)&data);
}

unsigned char PortNotify::fetch(timeout_t timeout)
{
	unsigned char buf;
	int rc;

	if(!timeout)
	{
		enterMutex();
		if(head == tail)
			buf = 0xff;
		else
			onWait((void *)&buf);
		leaveMutex();
		return buf;
	}

	rc = wait((void *)&buf, timeout);

	if(rc <= 0)
		return 0xff;

	return buf;
}

void Driver::setNodes(void)
{
	Driver *drv;
	char buf[256];
	char name[65];
	unsigned idx = 0;
	const char *path = keypaths.getLast("drvmap");
	int fd;
	char *cp;
	ScriptSymbol *globals = Trunk::getGlobals();

	remove(path);
	fd = creat(path, 0640);

	while(idx < drvIndex)
	{
		drv = getIndexedDriver(idx++);
		if(!drv)
			continue;
		snprintf(name, sizeof(name), "driver.%s.index", drv->getName());
		snprintf(buf, sizeof(buf), "%d", idx - 1);
		globals->setConst(name, buf);
		snprintf(name, sizeof(name), "driver.%s.ports", drv->getName());
		snprintf(buf, sizeof(buf), "%d", drv->getTrunkCount());
		globals->setConst(name, buf);
		snprintf(name, sizeof(name), "driver.%s.used", drv->getName());
		snprintf(buf, sizeof(buf), "%d", drv->getTrunkUsed());
		globals->setConst(name, buf);
		snprintf(buf, sizeof(buf), "%-12s %04d %04d\n",
			drv->getName(), drv->getTrunkCount(), drv->getTrunkUsed());
		cp = buf;
		while(*cp)
		{
			*cp = tolower(*cp);
			++cp;
		}
		write(fd, buf, IOLEN strlen(buf));
	}
	close(fd);
}


bool Driver::isIdle(void)
{
	if(portCount - idleCount)
		return false;

	return true;
}

bool Driver::isDown(void)
{
	if(portCount - downCount)
		return false;

	return true;
}

bool Driver::spanEvent(unsigned span, TrunkEvent *evt)
{
	unsigned port;
	bool rtn = false;
	Trunk *trunk;

	if(!span)
		return false;

	for(port = 0; port < (unsigned)getTrunkCount(); ++port)
	{
		TrunkEvent ev = *evt;
		trunk = getTrunkPort(port);
		if(!trunk)
			continue;

		if(trunk->spanid != span)
			continue;

		rtn = true;
		trunk->postEvent(&ev);
	}
	return rtn;
}
	
aaImage *Driver::getImage(void)
{
	//return new aaImage((aaScript *)this);
	return new aaImage(aascript);
}

Trunk *Driver::getTrunkId(const char *id)
{
	Trunk *trk, *ret;
	char *cp;

	if(!id)
		return NULL;

	cp = strchr(id, '-');
	if(!cp)
		return getTrunkPort(atoi(id));

	trk = ret = getTrunkPort(atoi(++cp));
	if(!trk)
		return NULL;

	cp = (char *)trk->trunkgid;
	if(!cp)
		cp = "";

	if(*id == '-')
		cp = strchr(cp, '-');
	if(!cp)
		cp = "";

	if(strcmp(cp, id))	
		ret = NULL;

	return ret;
}

Driver *Driver::getIndexedDriver(int idx)
{
	Driver *drv = Driver::drvFirst;
	while(drv)
	{
		if(drv->getDriverIndex() == idx)
			return drv;
		drv = drv->drvNext;
	}
	return NULL;
}

Trunk *Driver::getTrunk(const char *name, bool create, Driver *driver)
{
	Driver *drv = drvFirst;
	Trunk *trunk = NULL;
	char *cp;
	char drvid[3] = {0,0,0};
	int port;
	unsigned long caps = 0;

	if(!name)
		return NULL;

	cp = strchr(name, '/');
	if(!cp)
	{
		cp = strchr(name, '-');
		if(cp)
		{
			cp = strrchr(name, '-');
			if(cp)
			{
				strncpy(drvid, ++cp, 2);
				slog.debug() << drvid << endl;
				drv = getIndexedDriver(atoi(drvid));
			}
		}
		else if(driver)
			drv = driver;
		else
		{
			slog.warn() << "getTrunk: fall-back to default driver" << endl;
			drv = drvFirst;
		}

		if(!drv || !isdigit(*name))
			return NULL;

		if(create)
			return drv->getOutboundTrunk(atoi(name));
		else
			return drv->getTrunkId(name);
	}

	//*cp = 0;
	++cp;

	if(!strnicmp(name, "port", 4) && isdigit(*cp))
	{
		port = atoi(cp);
		while(drv)
		{
			if((unsigned)port >= drv->tsid)
				if((unsigned)port < drv->tsid + drv->getTrunkCount())
					break;
			drv = drv->drvNext;
		}
		if(!drv)
			return NULL;
		return drv->getTrunkId(cp);
	}

	if(!strnicmp(name, "pstn", 4) || !strnicmp(name, "trunk", 5) || !strnicmp(name, "trk", 3))
		caps = TRUNK_CAP_TRUNK;

	if(*cp == '*')
		port = -1;
	else
		port = atoi(cp);

	while(drv)
	{
		switch(caps)
		{
		case 0:
			if(!strnicmp(drv->getName(), name, strlen(drv->getName())))
			{
				if(port < 0)
				{
					for(port = 0; (unsigned)port < drv->getTrunkCount(); port++)
					{
						if(create)
							trunk = drv->getOutboundTrunk(atoi(cp));
						else
							trunk = drv->getTrunkId(cp);

						if(trunk)
							return trunk;
					}
				}
				else if(create)
					return drv->getOutboundTrunk(atoi(cp));
				else
					return drv->getTrunkId(cp);
			}
			break;
		}
		drv = drv->drvNext;
	}
	return NULL;
}

void Driver::secTick(void)
{
	unsigned id, max = getTrunkCount();
	Trunk *trunk;
	TrunkEvent event;
	time_t now;
	unsigned ctimer = refTimer;

	refTimer -= ctimer;
	time(&now);

	if(!ctimer)
	{
		if(!nextTimer || nextTimer > now)
			return;
	}
	
	nextTimer = 0;
	slog.debug() << getName() << ": timer tic scheduled" << endl; 
	
	for(id = 0; id < max; ++ id)
	{
		trunk = getTrunkPort(id);
		if(!trunk)
			continue;

		if(!trunk->exittimer)
			goto noexit;
				

		if(now >= trunk->exittimer)
		{
			event.id = TRUNK_TIMER_EXIT;
			trunk->postEvent(&event);
			trunk->exittimer = 0;
		}
		else if(trunk->exittimer < nextTimer || !nextTimer)
			nextTimer = trunk->exittimer;

noexit:
		if(!trunk->synctimer)
			continue;
			
		if(now >= trunk->synctimer)
		{
			event.id = TRUNK_TIMER_SYNC;
			trunk->postEvent(&event);
			trunk->synctimer = 0;
		}
		else if(trunk->synctimer < nextTimer || !nextTimer)
			nextTimer = trunk->synctimer;
	}
}

int Driver::getTrunkMember(TrunkGroup *group, unsigned member)
{
	int id, count = getTrunkCount();
	Trunk *trunk;

	for(id = 0; id < count; ++id)
	{
		trunk = getTrunkPort(id);
		if(!trunk)
			continue;

		if(trunk->group != group)
			continue;

		if(trunk->getMemberId() == member)
			return id;
	}
	return -1;
}

unsigned Driver::getCount(void)
{
	unsigned count = 0;
	Driver *drv = drvFirst;
	while(drv)
	{
		count += drv->getTrunkCount();
		drv = drv->drvNext;
	}
	return count;
}

void Driver::getStatus(char *buffer)
{
	static bool initial = false;
	if(!initial)
	{
		memset(Trunk::status, ' ', sizeof(Trunk::status));
		initial = true;
	}
	memcpy(buffer, Trunk::status, getCount());
}

Driver *getDriver(const char *name)
{
	Driver *drv = Driver::drvFirst;
	while(drv)
	{
		if(!stricmp(drv->getName(), name))
			break;
		drv = drv->drvNext;
	}
	return drv;
}

Driver *driver = NULL;

#ifdef	CCXX_NAMESPACES
}
#endif
