// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "bayonnelibrary.h"

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

Server *Server::first = NULL;
Translator *Translator::first = NULL;
TGI *TGI::first = NULL;
TTS *TTS::ttsFirst = NULL;
Sync *Sync::first = NULL;

TTS::TTS()
{
	ttsNext = ttsFirst;
	ttsFirst = this;
}

TTS *getTTS(const char *name)
{
        TTS *tts = TTS::ttsFirst;
        while(tts)
        {
                if(!stricmp(tts->getName(), name))
                        break;
                tts = tts->ttsNext;
        }
        return tts;
}

Conference::Conference(Driver *d, const char *id, unsigned sz)
{
	driver = d;
	size = sz;
	used = 0;
	reuse = false;
	ready = false;
	snprintf(name, sizeof(name), "%s", id);

	d->enterMutex();
	if(d->last)
	{
		prev = d->last;
		d->last->next = this;
		d->last = this;
		next = NULL;
	}
	else
	{
		next = prev = NULL;
		d->first = d->last = this;
	}
	d->leaveMutex();
}

void Conference::unlink(void)
{
	ready = false;
	driver->enterMutex();
	if(prev)
		prev->next = next;

	if(next)
		next->prev = prev;

	if(!next && driver->last == this)
		driver->last = prev;

	if(!prev && driver->first == this)
		driver->first = next;

	next = prev = NULL;
	driver->leaveMutex();
}

void Conference::drop(void)
{
	Driver *d = driver;

	d->enterMutex();
	if(used)
		--used;

	if(!used && reuse)
		delete this;
	d->leaveMutex();
}

void Conference::release(void)
{
	Driver *d = driver;
	d->enterMutex();
	if(!used)
		delete this;
	else
		reuse = true;
	d->leaveMutex();
}

void Conference::enterMutex(void)
{
	driver->enterMutex();
}

void Conference::leaveMutex(void)
{
	driver->leaveMutex();
}

Debug::Debug() :
Mutex()
{
	if(debug)
		THROW(this);

	debug = this;
}

Translator::Translator(const char *conf) :
Keydata(conf)
{
	char keypath[33];

	next = first;
	first = this;

	setString(keypath, sizeof(keypath), conf);
	*keypath = '~';
	load(keypath);
}

char *Translator::getPlayBuffer(Trunk *trunk)
{
	char *pbuf;
	
	pbuf = trunk->data.play.list;
	trunk->data.play.name = pbuf;
	trunk->data.play.limit = trunk->data.play.offset = 0;
	*pbuf = 0;
	return pbuf;
}

Translator *getTranslator(const char *name)
{
	Translator *trans = Translator::first;

	while(trans)
	{
		if(!stricmp(name, trans->getName()))
			return trans;
		trans = trans->next;
	}
	return NULL;
}

Sync::Sync()
{
	next = first;
	first = this;
	time(&runtime);
}

void Sync::check(void)
{
	time_t now;
	struct tm *dt, tbuf;
	Sync *sync = Sync::first;

	time(&now);
	dt = localtime_r(&now, &tbuf);

	while(sync)
	{
                if(((time_t)sync->runtime +60 * (time_t)sync->getInterval()) < now)
                {
                        time(&sync->runtime);
                        if(sync->isScheduled())
                        {	
				slog.info() << "sync: "<< sync->getSyncName() << " updated" << endl;
                                sync->schedule();
                        }
                }
                sync = sync->next;
        }
}

Server::Server(int pri) :
Thread(pri, keythreads.getStack())
{
	next = first;
	first = this;
}

void startServers(void)
{
	Server *server = Server::first;

	while(server)
	{
		server->start();
		server = server->next;
	}
}

void stopServers(void)
{
	Server *server = Server::first;

	while(server)
	{
		server->stop();
		server = server->next;
	}
}

TGI::TGI()
{
	next = first;
	first = this;
}

TGI *getInterp(char *cmd)
{
	TGI *tgi = TGI::first;
	char *ext;
	char buffer[512];
	
	setString(buffer, sizeof(buffer), cmd);
	cmd = strtok_r(buffer, " \t\n", &ext);
	ext = strrchr(cmd, '.');

	if(!ext)
		ext = cmd;

	while(tgi)
	{
		if(tgi->getExtension(ext))
			return tgi;
		tgi = tgi->next;
	}
	return NULL;
}

Trunk::Threaded::Threaded(const char *id) :
ScriptModule(id, (Method)&Trunk::scrThread)
{}

Trunk::Database::Database() :
Threaded("sql"), Keydata("/extras/sql")
{
	const char *cp = Process::getEnv("SQL_DATABASE");
	if(cp)
		setValue("database", cp);

	cp = Process::getEnv("SQL_SERVER");
	if(cp)
		setValue("server", cp);
}

void Trunk::Database::moduleAttach(ScriptInterp *interp)
{
	ScriptCommand *cmd = interp->getCommand();
	const char *db = cmd->getLast("database");

	if(!db)
		db = cmd->getLast("dsn");

	if(!db && cmd == aascript)
		db = getLast("database");

	if(!db)
		return;

	setSource(interp);
	interp->setConst(SYM_DATABASE, db);
	interp->setSymbol(SYM_SQLROW, 11);
	interp->setSymbol(SYM_ROWS, 11);
	interp->setSymbol(SYM_COLS, 11);
	interp->setSymbol(SYM_SQLCHANGES, 11);
	interp->setSymbol(SYM_SQLERROR, 64);
	interp->setPointer(SYM_SQLRESULTS, NULL);
	interp->setSymbol(SYM_SQLERROR, "none");
}

timeout_t Trunk::Database::getTimeout(Trunk *trunk, trunkdata_t *data)
{
	const char *cp = trunk->getKeyword("maxTime");
	if(!cp)
		cp = getLast("timeout");
	if(!cp)
		cp = "60s";

	return getSecTimeout(cp);
}

bool Trunk::Database::isId(const char *id)
{
	if(!stricmp(id, "sql"))
		return true;

	if(!stricmp(id, "db"))
		return true;

	if(!stricmp(id, "database"))
		return true;

	return false;
}


char *Trunk::Database::parseScript(ScriptInterp *interp, Line *line)
{
	ScriptCommand *cmd = interp->getCommand();
	const char *cp = cmd->getLast("database");
	const char *mem = interp->getMember();

	if(!cp)
		cp = cmd->getLast("dsn");

	if(!cp && cmd == aascript)
		cp = getLast("database");

	if(!cp)
		return "sql-no-database";

	if(!mem)
		mem = "";

	if(!stricmp(mem, "fetch"))
	{
		cp = interp->getKeyword("row");
		if(!cp)
			return "sql-no-row";

		if(!fetch(interp, atoi(cp) - 1))
			return "sql-no-data";
	}
	else if(!stricmp(mem, "header"))
	{
		if(!header(interp))
			return "sql-no-query";
	}
	return NULL;
}

void getInterp(char *cmd, char **args)
{
	TGI *tgi = TGI::first;

	while(tgi)
	{
		tgi->script(cmd, args);
		tgi = tgi->next;
	}
}


#ifdef  HAVE_EXECINFO_H

#include <execinfo.h>

void    Debug::stackTrace(int signo)
{
        const int maxTrace = 1000;
        void* buffer[maxTrace];
        int nTrace = backtrace ( buffer, maxTrace );
        char** trace = backtrace_symbols ( buffer, nTrace );

	slog.debug() << "trace: pid=" << pthread_self()
                << " reason=" << signo << endl;


        if ( trace ) {
                for ( int i = 0; i < nTrace; ++i ) {
                slog.debug() << "trace(" << i << "): "
                        << trace[i] << endl;
                }
        }
        // free memory
        free( trace );
}

#else
void    Debug::stackTrace(int signo) {}
#endif


Debug *debug = NULL;
TTS *tts = NULL;

#ifdef	CCXX_NAMESPACES
}
#endif
