// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "bayonnelibrary.h"
#include <cstdarg>
#include <cstdio>
#include <sys/stat.h>

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

Fifo fifo;

#define	ANY ((Trunk *)(-1))

void errlog(const char *level, const char *fmt, ...)
{
	static Keydata *missing = NULL;
	static Mutex mutex;
	char buffer[256];
	struct tm *dt, tbuf;
	time_t now;
	time(&now);
	int year;
	dt = localtime_r(&now, &tbuf);
	va_list args;
	size_t len;
	static char *months[12] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	char *m;
	const char *cp;

	if(!fmt)
	{
		mutex.enterMutex();
		if(missing)
			delete missing;
		missing = new Keydata();
		fmt = "Configuration reloaded";
		mutex.leaveMutex();
	}

	year = dt->tm_year;
	if(year < 1000)
		year += 1900;

	m = months[--dt->tm_mon];

	if(!stricmp(level, "missing"))
		level = "access";

	va_start(args, fmt);
	snprintf(buffer, sizeof(buffer),
		"errors [%s %02d %02d:%02d:%02d %d] [%s] ",
		m, dt->tm_mday, dt->tm_hour, dt->tm_min, dt->tm_sec, year, level);
	len = strlen(buffer);
	vsnprintf(buffer + len, sizeof(buffer) - len - 1, fmt, args);

	if(!stricmp(level, "access"))
	{
		slog.warn() << "missing: " << buffer + len << endl;
		mutex.enterMutex();
		if(!missing)
			missing = new Keydata();
		cp = missing->getLast(buffer + len);
		if(cp)
		{
			mutex.leaveMutex();
			return;
		}
		missing->setValue(buffer + len, "");
		mutex.leaveMutex();
	}
	else if(!stricmp(level, "debug"))
	{
		slog.debug() << "server: " << buffer + len << endl;
		return;
	}
	else if(!stricmp(level, "missing"))
		slog.warn() << "missing: " << buffer + len << endl;
	else if(!stricmp(level, "notice"))
		slog.notice() << buffer + len << endl;
	else if(!strnicmp(level, "warn", 4))
		slog.warn() << buffer + len << endl;
	else if(!stricmp(level, "failed"))
		slog.error() << buffer + len << endl;
	else if(!stricmp(level, "error"))
		slog.error() << buffer + len << endl;

	len = strlen(buffer);
	if(buffer[len - 1] != '\n')
	{
		buffer[len++] = '\n';
		buffer[len] = 0;
	}
	fifo.command(buffer);
}

Fifo::Fifo() : Mutex(), Script()
{
	fd = -1;
	main = NULL;
}

void Fifo::setControl(int cfd)
{
	main = getThread();
	fd = cfd;
}

void Fifo::control(const char *cmd)
{
        if(fd < 0)
                return;

	if(getThread() == main)
		command(cmd);
	else 	                                                                               
        	write(fd, cmd, IOLEN strlen(cmd));
}

Trunk *Fifo::addRoute(char **argv)
{
	if(!stricmp(argv[2], "incoming"))
	{
		keyserver.incoming.set(argv[3], argv[4]);
		keyserver.printRoutes();
		return ANY;
	}
	return NULL;
}

Trunk *Fifo::delRoute(char **argv)
{
        if(!stricmp(argv[2], "incoming"))
        {
                keyserver.incoming.clear(argv[3]);
		keyserver.printRoutes();
                return ANY;
        }
        return NULL;
}


Trunk *Fifo::logResult(char **argv)
{
	char path[512];
	size_t len = 0;
	unsigned argc = 1;
#ifdef	WIN32
	FILE *fp;
#else
	int fd;
#endif
	char *fmt = "%s";
	static bool out = false;

#ifdef	WIN32
	snprintf(path, sizeof(path), "%s/%s.log",
		keypaths.getLast("logpath"), argv[0]);
#else
	snprintf(path, sizeof(path), "%s/%s.%s",
		keypaths.getLast("logpath"), keyserver.getNode(), argv[0]);
#endif

	if(!stricmp(argv[0], "output") && !out)
	{
		::remove(path);
		out = true;
	}

#ifdef	WIN32
	fp = ::fopen(path, "a+");
	if(!fp)
		return false;
#else
	fd = ::open(path, O_CREAT | O_APPEND | O_RDWR, 0660);
	if(fd < 0)
		return false;
#endif

	while(len < 500 && argv[argc])
	{
		snprintf(path + len, sizeof(path) - len, fmt, argv[argc++]);
		fmt = " %s";
		len = strlen(path);
	}
	path[len++] = '\n';
	path[len] = 0;
#ifdef	WIN32
	fputs(path, fp);
	fclose(fp);
#else
	::write(fd, path, len);
	::close(fd);
#endif
	return ANY;
}
		
Trunk *Fifo::sendEvent(char **argv)
{
        TrunkEvent event;
        Trunk *trunk = Driver::getTrunk(argv[1]);
        
        if (!trunk) return(NULL);
        
        event.id = TRUNK_SEND_MESSAGE;
        event.parm.send.src = NULL;
        if (argv[2]) 
		event.parm.send.msg = argv[2];
        else 
		event.parm.send.msg = "";

        trunk->postEvent(&event);
	return trunk;
}

Trunk *Fifo::stop(char **argv)
{
	TrunkEvent event;
	Trunk *trunk = Driver::getTrunk(argv[1]);
	
	if(!trunk)
		return NULL;

	if(!trunk->isActive())
		return NULL;

	event.id = TRUNK_STOP_STATE;
	trunk->postEvent(&event);
	return trunk;
}

Trunk *Fifo::sync(char **argv)
{
        TrunkEvent event;
        Trunk *trunk = Driver::getTrunk(argv[1]);

        if(!trunk)
                return NULL;

        if(!trunk->isActive())
                return NULL;

        event.id = TRUNK_SYNC_NOTIFY;
        trunk->postEvent(&event);
        return trunk;
}

	
#ifdef	HAVE_TGI
Trunk *Fifo::shellStart(char **argv)
{
	struct stat ino;
	int fd;
	TrunkEvent event;
	Trunk *trunk = Driver::getTrunk(argv[1]);
	if(!trunk)
		return NULL;

	if(!argv[2])
		return NULL;

	stat(argv[2], &ino);
	if(!S_ISFIFO(ino.st_mode))
		return NULL;	

	fd = ::open(argv[2], O_WRONLY | O_NONBLOCK);
	remove(argv[2]);
	if( fd < 0)
		return NULL;

	event.id = TRUNK_SHELL_START;
	event.parm.fd = fd;
	if(trunk->postEvent(&event))
		return trunk;
	close(fd);
	return NULL;
}
	
Trunk *Fifo::waitPid(char **argv)
{
	int pid;
	unsigned short seq;
	TrunkEvent event;
	Trunk *trunk = Driver::getTrunk(argv[1]);
	if(!trunk)
		return NULL;

	if(!argv[2])
		return NULL;

	if(argv[3])
		seq = atoi(argv[3]);
	else
		seq = trunk->tgi.seq;

	pid = atoi(argv[2]);
	event.id = TRUNK_WAIT_SHELL;
	event.parm.waitpid.pid = pid;
	event.parm.waitpid.seq = seq;
	if(trunk->postEvent(&event))
		return trunk;
	kill(pid, SIGHUP);
	return NULL;
}
#endif

Trunk *Fifo::setGlobal(char **argv)
{
	char name[65];
	ScriptSymbol *global = Trunk::getGlobals();
	Symbol *sym;

	if(!argv[1] || !argv[2])
		return NULL;

	snprintf(name, sizeof(name), "global.%s", argv[1]);
	global->enterMutex();
	sym = global->getEntry(name, 0);
	if(!sym)
	{
		global->leaveMutex();
		return NULL;	// no such global sym predefined
	}

	snprintf(sym->data, sym->flags.size + 1, "%s", urlDecode(argv[2]));
	sym->flags.initial = false;
	global->leaveMutex();
	return ANY;
}

Trunk *Fifo::addSymbol(char **argv)
{
	unsigned len;
	Script::Symbol *sym;
	Trunk *trunk = Driver::getTrunk(argv[1]);
	if(!trunk)
		return NULL;

	if(!argv[3] || !argv[2])
		return NULL;

	trunk->enterMutex();
	sym = trunk->getLocal(argv[2], 0);
	if(sym && !sym->flags.readonly)
	{
		sym->flags.initial = false;
		len = (unsigned)strlen(sym->data);
		snprintf(sym->data + len, sym->flags.size + 1 - len, 
			"%s", argv[3]);
		if(sym->flags.commit)
			trunk->commit(sym);
	}
	trunk->leaveMutex();
	return trunk;
}

Trunk *Fifo::setSize(char **argv)
{
	Trunk *trunk = Driver::getTrunk(argv[1]);
	if(!trunk)
		return NULL;

	if(!argv[2] || !argv[3])
		return NULL;

	if(trunk->setVariable(argv[2], atoi(argv[3])))
		return trunk;

	return NULL;
}

Trunk *Fifo::setSymbol(char **argv)
{
	Trunk *trunk = Driver::getTrunk(argv[1]);
	Trunk *rts = trunk;
	if(!trunk)
		return NULL;

	if(!argv[3] || !argv[2])
		return NULL;

	trunk->enterMutex();
	if(!trunk->setVariable(argv[2], 0, argv[3]))
		rts = NULL;
	trunk->leaveMutex();
	return rts;
}


Trunk *Fifo::testScript(char **argv)
{
	char *pass[2];
	Trunk *trunk;
	TrunkEvent event;
	char name[128];

	if(!argv[1])
		return NULL;

	pass[0] = argv[1];
	pass[1] = NULL;
	argv += 2;

	while(*argv)
	{
		snprintf(name, sizeof(name), "PSTN/%s", *argv);
		trunk = Driver::getTrunk(name);
		if(!trunk)
			continue;

		event.id = TRUNK_RING_START;
		event.parm.argv = pass;
		trunk->postEvent(&event);
		++argv;
	}
	return ANY;
}

Trunk *Fifo::ringScript(char **argv)
{
	Trunk *trunk = Driver::getTrunk(argv[1]);
	TrunkEvent event;

	if(!trunk)
		return NULL;

	event.id = TRUNK_RING_START;
	event.parm.argv = &argv[2];

	if(!trunk->postEvent(&event))
		return NULL;
	return trunk;
}

Trunk *Fifo::redirectScript(char **argv)
{
	Trunk *trunk = Driver::getTrunk(argv[1]);
	TrunkEvent event;

	if(!trunk || !argv[2])
		return NULL;

	event.id = TRUNK_RING_REDIRECT;
	event.parm.argv = &argv[2];

	if(!trunk->postEvent(&event))
		return NULL;
	return trunk;
}

#ifndef	WIN32
Trunk *Fifo::runScript(char **argv)
{
	char *gid;
	char buffer[65];
	Trunk *trunk;
	int exitfd = ::open(argv[1], O_WRONLY | O_NDELAY);
	bool session = false;

	if(exitfd < 0)
		return NULL;

	if(!stricmp(argv[0], "session"))
		session = true;

	argv[1] = "start";
	trunk = startScript(++argv);
	if(trunk)
	{
		trunk->enterMutex();
		if(trunk->isActive())
		{
			snprintf(buffer, sizeof(buffer), "start %s\n", strchr(trunk->trunkgid, '-'));
			::write(exitfd, buffer, strlen(buffer));
			if(session)
				::close(exitfd);
			else
				trunk->exitfd = exitfd;
			trunk->leaveMutex();
		}
		else
		{
			trunk->leaveMutex();
			trunk = NULL;
		}
	}

	if(!trunk)
	{
		::write(exitfd, "fail - \n", 8); 		
		::close(exitfd);
	}
	return trunk;
}
#endif

Trunk *Fifo::startScript(char **argv)
{
	Trunk *trunk = Driver::getTrunk(argv[1]);
	TrunkGroup *group = TrunkGroup::getGroup(argv[1]);
	TrunkEvent event;
	Driver *drv = Driver::drvFirst;
	int port;

	if(!trunk)
		trunk = Driver::getTrunk(argv[1], true);

	if(trunk && !group)
	{
		event.id = TRUNK_START_SCRIPT;
		event.parm.argv = &argv[2];
		if(!trunk->postEvent(&event))
			return NULL;
		return trunk;
	}

	drv = Driver::drvFirst;
	while(drv)
	{
		port = drv->getTrunkCount();
		while(port--)
		{
			if(!drv->isTrunkClass(port, argv[1]))
				continue;

			trunk = drv->getTrunkPort(port);
			if(!trunk)
				trunk = drv->getOutboundTrunk(port);
			if(!trunk)
				continue;

			event.id = TRUNK_START_SCRIPT;
			event.parm.argv = &argv[2];
			if(trunk->postEvent(&event))
				return trunk;
		}
		drv = drv->drvNext;
	}

	drv = Driver::drvFirst;
	while(drv && group)
	{
		port = drv->getTrunkCount();
		
		// skip drivers with dynamic ports..
		if((drv->getCaps() & Driver::capDynamic))
		{
			drv = drv->drvNext;
			continue;
		}

		while(port--)
		{
			trunk = drv->getTrunkPort(port);

			// NEVER create for group search, must use
			// driver name to create this way, otherwise
			// all empty driver slots get filled if not
			// matching any group...

			if(!trunk)
				continue;

			if(trunk->group != group)
				continue;

			event.id = TRUNK_START_SCRIPT;
			event.parm.argv = &argv[2];
			if(trunk->postEvent(&event))
				return trunk;
		}
		drv = drv->drvNext;
	}
	return NULL;
}

Trunk *Fifo::postKey(char **argv)
{
	Trunk *trunk = Driver::getTrunk(argv[1]);
	char *digits = argv[2];
	TrunkEvent event;
	Trunk *rtn = trunk;

	if(!trunk || !argv[2])
		return NULL;

	while(digits && rtn)
	{
		event.id = TRUNK_DTMF_KEYUP;
		event.parm.dtmf.duration = 40;
		event.parm.dtmf.e1 = event.parm.dtmf.e2 = 0;
		switch(*digits)
		{
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			event.parm.dtmf.digit = *digits - '0';
			if(!trunk->postEvent(&event))
				rtn = NULL;
			break;
		case '*':
			event.parm.dtmf.digit = 10;
			if(!trunk->postEvent(&event))
				rtn = NULL;
			break;
		case '#':
			event.parm.dtmf.digit = 11;
			if(!trunk->postEvent(&event))
				rtn = NULL;
			break; 
		default:
			rtn = NULL;
		}
		if(*(++digits))
			Thread::sleep(60);
	}
	return rtn;
}

#ifdef	HAVE_TGI

Trunk *Fifo::exitPid(char **argv)
{
	Trunk *trunk = Driver::getTrunk(argv[1]);
	TrunkEvent event;

	if(!trunk)
		return NULL;

	event.id = TRUNK_EXIT_SHELL;
	if(argv[2])
		event.parm.exitpid.status = atoi(argv[2]);
	else
		event.parm.exitpid.status = 0;

	if(argv[3])
		event.parm.exitpid.seq = atoi(argv[3]);
	else
		event.parm.exitpid.seq = trunk->tgi.seq;

	if(!trunk->postEvent(&event))
		return NULL;
	return trunk;
}

#endif

Trunk *Fifo::hangupLine(char **argv)
{
	TrunkGroup *group = TrunkGroup::getGroup(argv[1]);
	Trunk *trunk = Driver::getTrunk(argv[1]);
	TrunkEvent event;
	Driver *drv;
	unsigned port;

	if(trunk && !group)
	{
		event.id = TRUNK_STOP_DISCONNECT;
		if(trunk->postEvent(&event))
			return trunk;
		else
			return NULL;
	}

	if(!group)
		return NULL;

	drv = Driver::drvFirst;
	while(drv)
	{
		for(port = 0; port < drv->getTrunkCount(); ++port)
		{
			trunk = drv->getTrunkPort(port);
			if(!trunk)
				continue;

			if(trunk->group != group)
				continue;

			event.id = TRUNK_STOP_DISCONNECT;
			trunk->postEvent(&event);
		}
		drv = drv->drvNext;
	}
	return ANY;
}

Trunk *Fifo::busyLine(char **argv)
{
	TrunkGroup *group = TrunkGroup::getGroup(argv[1]);
	Trunk *trunk = Driver::getTrunk(argv[1]);
	Driver *drv = Driver::drvFirst;
	TrunkEvent event;
	unsigned port;

	if(trunk && !group)
	{
		event.id = TRUNK_MAKE_BUSY;
		if(trunk->postEvent(&event))
			return trunk;
		return NULL;
	}
	if(!group)
		return NULL;

	while(drv)
	{
		for(port = 0; port < drv->getTrunkCount(); ++port)
		{
			trunk = drv->getTrunkPort(port);
			if(!trunk)
				continue;

			if(trunk->group != group)
				continue;

			event.id = TRUNK_MAKE_BUSY;
			trunk->postEvent(&event);
		}
		drv = drv->drvNext;
	}
	return ANY;
}

Trunk *Fifo::setSpan(char **argv)
{
	TrunkEvent event;
	unsigned span;
	Driver *drv = Driver::drvFirst;

	const char *drvname = argv[1];
	const char *cspan = argv[2];
	const char *mode = argv[3];

	if(!drvname || !cspan || !mode)
		return NULL;

	span = atoi(cspan);
	if(!stricmp(mode, "busy"))
		event.id = TRUNK_MAKE_BUSY;
	else if(!stricmp(mode, "idle") || !stricmp(mode, "up"))
		event.id = TRUNK_MAKE_IDLE;
	else if(!stricmp(mode, "stop") || !stricmp(mode, "down"))
		event.id = TRUNK_MAKE_STANDBY;
	else
		return NULL;

	while(drv)
	{
		if(!strnicmp(drv->getName(), drvname, strlen(drvname)))
		{
			if(drv->spanEvent(span, &event))
				return ANY;
			return NULL;
		}
		drv = drv->drvNext;
	}

	return NULL;
}


Trunk *Fifo::setCard(char **argv)
{
        TrunkEvent event;
        unsigned card;
	Driver *drv = Driver::drvFirst;

	const char *drvname = argv[1];
        const char *ccard = argv[2];
        const char *mode = argv[3];

        if(!drvname || !ccard || !mode)
                return NULL;

        card = atoi(ccard);
        if(!stricmp(mode, "busy"))
                event.id = TRUNK_MAKE_BUSY;
        else if(!stricmp(mode, "idle") || !stricmp(mode, "up"))
                event.id = TRUNK_MAKE_IDLE;
        else if(!stricmp(mode, "stop") || !stricmp(mode, "down"))
                event.id = TRUNK_MAKE_STANDBY;
        else
                return NULL;

	while(drv)
	{
		if(!strnicmp(drv->getName(), drvname, sizeof(drvname)))
		{
			if(drv->cardEvent(card, &event))
				return ANY;
			return NULL;
		}
		drv = drv->drvNext;
	}

	return NULL;
}

Trunk *Fifo::idleLine(char **argv)
{
	TrunkGroup *group = TrunkGroup::getGroup(argv[1]);
	Trunk *trunk = Driver::getTrunk(argv[1]);
	TrunkEvent event;
	Driver *drv = Driver::drvFirst;
	unsigned port;

	if(trunk && !group)
	{
		event.id = TRUNK_MAKE_IDLE;
		if(trunk->postEvent(&event))
			return trunk;
		return NULL;
	}

	if(!group)
		return NULL;

	while(drv)
	{
		for(port = 0; port < drv->getTrunkCount(); ++port)
		{
			trunk = drv->getTrunkPort(port);
			if(!trunk)
				continue;

			if(trunk->group != group)
				continue;

			event.id = TRUNK_MAKE_IDLE;
			trunk->postEvent(&event);
		}
		drv = drv->drvNext;
	}
	return ANY;
}

Trunk *Fifo::command(const char *cmd, ostream *fd)
{
	ScriptSymbol *globals = Trunk::getGlobals();
	char buffer[PIPE_BUF / 2];
	Trunk *rts = NULL;
	char *args[65];
	int argc = 0;
	char **argv = args;
	char *arg;
	char *sp;
	char *ptr;
	const char *token = keyserver.getToken();
	int tlen = (int)strlen(token);
	int pid = 0;
	aaScript *scr;

	if(!fd)
		fd = &slog;

	setString(buffer, sizeof(buffer), cmd);

	enterMutex();
	if(strstr(buffer, token))
	{
		ptr = buffer;
		while(isspace(*ptr))
			++ptr;

		ptr = strtok_r(ptr, "\n", &sp);
		while(NULL != (sp = strstr(ptr, token)))
		{
			argv[argc++] = ptr;
			*sp = 0;
			ptr = sp + tlen;
		}
		while(isspace(*ptr))
			++ptr;
		if(*ptr)
			argv[argc++] = ptr;
	}
	else
	{
		argv[argc++] = strtok_r(buffer, " \t\n\r", &sp);
		while(argc < 64)
		{
			arg = strtok_r(NULL, " \t\n\r", &sp);
			if(!arg)
				break;
			argv[argc++] = arg;
		}
	}
	argv[argc] = NULL;

	if(!argv[0])
	{
		leaveMutex();
		return NULL;
	}

	if(!*argv[0])
	{
		leaveMutex();
		return NULL;
	}

	if(isdigit(**argv))
	{
		pid = atoi(argv[0]);
		++argv;
		--argc;
		if(!argv[0])
		{
			++argc;
			--argv;
		}
	}

	if(!stricmp(argv[0], "service"))
	{
		if(!argv[1])
		{
			argv[0] = "-";
			rts = NULL;
		}
	}

	if(!stricmp(argv[0], "down") || !stricmp(argv[0], "service"))
	{
		if(argv[1])
			globals->setSymbol(SYM_SERVICE, argv[1]);
		else
#ifdef	WIN32
			::exit(0);
#else
			raise(SIGINT);
#endif
		rts = ANY;
	}
	else if(!stricmp(argv[0], "test"))
	{
		globals->setSymbol(SYM_SERVICE, "test");
		rts = testScript(argv);
	}
	else if(!stricmp(argv[0], "up"))
	{
		globals->setSymbol(SYM_SERVICE, "up");
		rts = ANY;
	}
#ifndef	WIN32
	else if(!stricmp(argv[0], "restart"))
	{
		restart_server = true;
		raise(SIGINT);
		rts = ANY;
	}
#endif
	else if(!stricmp(argv[0], "compile"))
	{
		scr = aascript;
		if(argv[1])
		{
			scr = aaScript::get(argv[1]);
			if(!scr)
				return false;
		}
		new aaImage(scr);
		rts = ANY;
	}
#ifdef	HAVE_TGI
	else if(!stricmp(argv[0], "wait"))
		rts = waitPid(argv);
	else if(!stricmp(argv[0], "exit"))
		rts = exitPid(argv);
#endif
	else if(!stricmp(argv[0], "set"))
		rts = setSymbol(argv);
	else if(!stricmp(argv[0], "add"))
		rts = addSymbol(argv);
	else if(!stricmp(argv[0], "size"))
		rts = setSize(argv);
	else if(!stricmp(argv[0], "global"))
		rts = setGlobal(argv);
	else if(!stricmp(argv[0], "ring"))
		rts = ringScript(argv);
	else if(!stricmp(argv[0], "redirect"))
		rts = redirectScript(argv);
	else if(!stricmp(argv[0], "busy"))
		rts = busyLine(argv);
	else if(!stricmp(argv[0], "idle"))
		rts = idleLine(argv);
	else if(!stricmp(argv[0], "span"))
		rts = setSpan(argv);
	else if(!stricmp(argv[0], "card"))
		rts = setCard(argv);
#ifndef	WIN32
	else if(!stricmp(argv[0], "run") || !stricmp(argv[0], "session"))
		rts = runScript(argv);
#endif
	else if(!strnicmp(argv[0], "start", 5))
		rts = startScript(argv);
	else if(!stricmp(argv[0], "stop"))
		rts = stop(argv);
	else if(!stricmp(argv[0], "sync"))
		rts = sync(argv);
	else if(!stricmp(argv[0], "disconnect") || !stricmp(argv[0], "hangup"))
		rts = hangupLine(argv);
	else if(!stricmp(argv[0], "post") || !stricmp(argv[0], "key"))
		rts = postKey(argv);
	else if(!stricmp(argv[0], "send"))
		rts = sendEvent(argv);
	else if(!stricmp(argv[0], "route"))
	{
		if(argv[1] && argv[2] && argv[3])
		{
			if(!stricmp(argv[1], "add") || !stricmp(argv[1], "set"))
			{
				if(argv[4])
					rts = addRoute(argv);
			}
			else if(!strnicmp(argv[1], "del", 3) || !stricmp(argv[1], "clear"))
				rts = delRoute(argv);
		}
	}
	else if(!stricmp(argv[0], "cdr") || !stricmp(argv[0], "calls"))
	{
		argv[0] = "calls";
		rts = logResult(argv);
	}
	else if(!stricmp(argv[0], "audit") || !stricmp(argv[0], "stats") || !stricmp(argv[0], "errors") || !stricmp(argv[0], "output") || !stricmp(argv[0], "usage"))
		rts = logResult(argv);
	else if(!stricmp(argv[0], "debug"))
	{
		slog.warn() << "fifo:";
		while(*(++argv))
			slog() << " " << *argv;
		slog() << endl;
		rts = ANY;
	}
	leaveMutex();
#ifndef	WIN32
#ifdef	SIGUSR2
	if(pid && rts)
		kill(pid, SIGUSR1);
	else if(pid && !rts)
		kill(pid, SIGUSR2);
#else
	if(pid & rts)
		kill(pid, SIGHUP);
	else if(pid && !rts)
		kill(pid, SIGPIPE);
#endif
#endif
	return rts;
}


#ifdef	CCXX_NAMESPACES
}
#endif
