// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "bayonnelibrary.h"

#include <cstdio>
#ifdef	HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#ifdef CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

static class DTMFProperty : public Script::Property, public Keydata
{
public:
	char dtmfmap[64];
	DTMFProperty();

private:
        void dtmf(char *dp, char *tp, unsigned size);

	void set(char dig);

        void setProperty(char *dp, char *tp, unsigned size)
                {dtmf(dp, tp, size);};
        void getProperty(char *dp, char *tp, unsigned size)
                {dtmf(dp, tp, size);};

} dtmf;

DTMFProperty::DTMFProperty() :
Script::Property("dtmf"), Keydata("/tones/dtmf")
{
	unsigned char dig = '0';

        static Keydata::Define keydefs[] = {
        {"2", "abc"},
	{"3", "def"},
	{"4", "ghi"},
	{"5", "jkl"},
	{"6", "mno"},
	{"7", "pqrs"},
	{"8", "tuv"},
	{"9", "wxyz"},
        {NULL, NULL}};

        load(keydefs);

	memset(dtmfmap, 0, sizeof(dtmfmap));

	while(dig <= '9')
		set(dig++);
}

void DTMFProperty::dtmf(char *from, char *to, unsigned size)
{
	char dig;
        while(*from && size)
        {
                dig = toupper(*(from++));
		if(dig < 32 || dig > 95)
        		continue;
		dig -= 32;
		if(!dtmfmap[(unsigned)dig])
			continue;

		*(to++) = dtmfmap[(unsigned)dig];
        }
        *to = 0;
}

void DTMFProperty::set(char key)
{
	const char *keys;
	char str[2];
	char dig;

	str[0] = key;
	str[1] = 0;

	keys = getLast(str);
	if(!keys)
		return;

	while(*keys)
	{
		dig = toupper(*(keys++));
		if(dig < 32 || dig > 95)
			continue;

		dtmfmap[dig - 32] = key;
	}
}

bool Trunk::scrRelease(void)
{
	Conference *cf;
	const char *id = getKeyword("id");
	if(!id)
		id = getValue(NULL);

	if(!id)
	{
		error("no-conference-id");
		return true;
	}

	cf = driver->getConference(id);
	if(!cf)
	{
		error("invalid-conference");
		return true;
	}
	cf->release();
	advance();
	return true;
}

bool Trunk::scrEnter(void)
{
	Conference *cf;
	const char *id = getKeyword("id");
	const char *sz = getKeyword("size");
	const char *maxtime = getKeyword("maxTime");
	const char *mem = getMember();
	const char *vol = getKeyword("volume");
	const char *igain = getKeyword("igain");
	const char *ogain = getKeyword("ogain");

	if(!id)
		id = getValue(NULL);

	if(!id)
	{
		error("no-conference-id");
		return true;
	}

	if(!vol)
		vol = "100";

	if(!sz)
		sz = "0";

	if(!maxtime)
		maxtime = "0";

	cf = driver->getConference(id, atoi(sz));
	if(!cf)
	{
		if(!trunkSignal(TRUNK_SIGNAL_INVALID))
			error("invalid-conference-id");
		return true;
	}

	data.enter.resource = cf;
	data.enter.maxtime = getSecTimeout(maxtime);
	data.enter.volume = atoi(vol);
	data.enter.input = data.enter.output = 0.0;
	
	if(igain)
		data.enter.input = (float)strtod(igain, NULL);
	if(ogain)
		data.enter.output = (float)strtod(ogain, NULL);


	if(!cf->enter(this))
	{
		if(!trunkSignal(TRUNK_SIGNAL_BUSY))
			error("cannot-enter");
		return true;
	}

	// enter.temp makes sure conference ends when all exit

	if(mem && !stricmp(mem, "temp"))
		cf->release();

	trunkStep(TRUNK_STEP_ENTER);
	return false;
}

bool Trunk::scrThread(void)
{
	Line *line = getScript();
	const char *err;
	Trunk::Threaded *mod = (Trunk::Threaded *)ScriptModule::find(line->cmd);

	if(!mod)
	{
		error("module-not-found");
		return true;
	}

	err = mod->parseScript(this, line);
	if(err)
	{
		error(err);
		return true;
	}

	thread = mod->getService(this, line, &data);
	if(!thread)
		return true;

	data.sleep.count = 0;
	data.sleep.wakeup = mod->getTimeout(this, &data);
	data.sleep.term = 0;
	if(!data.sleep.wakeup)
	{
		stopServices();
		advance();
		return true;
	}

	thread->start();
	trunkStep(TRUNK_STEP_THREAD);
	return false;
}

bool Trunk::scrExiting(void)
{
	const char *value = getValue("detach");
	const char *mem = getMember();
	unsigned len = 0;

	if(!mem)
		mem = "exit";

	while(value && len < sizeof(exitmsg) - 3)
	{
		snprintf(exitmsg + len, sizeof(exitmsg) - len, "%s", value);
		len = (unsigned)strlen(exitmsg);
		value = getValue(NULL);
	}

	exitmsg[len++] = '\n';
	exitmsg[len] = 0;
	exitmsg[--len] = 0;

	if(!stricmp(mem, "rpc"))
	{
		advance();
		return true;
	}
	return scrExit();
}

bool Trunk::scrStatinfo(void)
{
	const char *ref = getKeyword("id");
	Line *line = getScript();
	unsigned argc = 0;
	TrunkGroup *grp;
	char *opt, *var, value[12];

	if(!ref)
		ref = getKeyword("group");

	if(ref)
		grp = TrunkGroup::getGroup(ref);
	else
		grp = group;		// current trunk group

	if(!grp)
	{
		error("invalid-group-specified");
		return true;
	}

	while(argc < line->argc)
	{
		opt = line->args[argc++];
		if(*opt != '=')
			continue;

		++opt;
		var = line->args[argc++];
		if(*var == '&')
			++var;

		value[0] = 0;
		if(!stricmp(opt, "capacity") || !stricmp(opt, "size"))
			snprintf(value, sizeof(value), "%d", grp->capacity);
		else if(!stricmp(opt, "total_incoming"))
			snprintf(value, sizeof(value), "%ld", grp->total.incoming);
		else if(!stricmp(opt, "total_outgoing"))
			snprintf(value, sizeof(value), "%ld", grp->total.outgoing);
		else if(!strnicmp(opt, "avail", 5))
			snprintf(value, sizeof(value), "%d",
				grp->capacity - grp->active.incoming - grp->active.outgoing);
		else if(!stricmp(opt, "used"))
			snprintf(value, sizeof(value), "%d",
				grp->active.incoming + grp->active.outgoing);
		else if(!stricmp(opt, "incoming"))
			snprintf(value, sizeof(value), "%d", grp->active.incoming);
		else if(!stricmp(opt, "outgoing"))
			snprintf(value, sizeof(value), "%d", grp->active.outgoing);
		else if(!stricmp(opt, "max_incoming"))
			snprintf(value, sizeof(value), "%d", grp->max.incoming);
		else if(!stricmp(opt, "max_outgoing"))
			snprintf(value, sizeof(value), "%d", grp->max.outgoing);

		if(value[0])
			setVariable(var, 11, value);

	}
	advance();
	return true;		
}

bool Trunk::scrSession(void)
{
	Trunk *trk;
	const char *id = getKeyword("id");
	if(!id)
		id = getValue(NULL);

	if(!id)
	{
		ctx = this;
		advance();
		return false;	// make sure not in begin block
	}

	trk = Driver::getTrunk(id, false, getDriver());
	if(!trk)
	{
		error("invalid-session");
		return true;
	}

	if(trk == this)
	{
		ctx = this;
		return false;	// make sure not in begin block
	}

	if(*id != '-' && !isAdmin())
	{
		error("admin-required");
		return true;
	}

	ctx = trk;
	advance();
	return false;	// make sure not in begin block
}

bool Trunk::scrBusy(void)
{
	TrunkEvent event;
	Trunk *trk;
	TrunkGroup *grp = NULL;
	Driver *drv = Driver::drvFirst;
	const char *mem = getMember();
	unsigned port, tspan;

	if(!mem)
		mem = "self";
	else if(!isAdmin())
	{
		error("admin-required");
		return false;
	}

	if(!stricmp(mem, "port"))
	{
		event.id = TRUNK_MAKE_BUSY;
		mem = getValue(getKeyword("id"));
		trk = Driver::getTrunk(mem, false, getDriver());

		if(!trk)
		{
			error("busy-port-id-missing");
			return true;
		}

		if(*mem != '-' && !isAdmin())
		{
			error("busy-port-admin-required");
			return true;
		}

		if(trk == this)
		{
			error("busy-port-self-reference");
			return true;
		}

		trk->postEvent(&event);
		advance();
		return true;
	}

	if(!stricmp(mem, "span"))
	{
		if(!isAdmin())
		{
			error("busy-span-admin-required");
			return true;
		}
		mem = getValue(getKeyword("id"));
		if(!mem)
		{
			error("busy-span-id-missing");
			return true;
		}
		tspan = atoi(mem);
		if(!driver->spanEvent(tspan, &event))
			error("busy-span-id-invalid");
		else
			advance();
		return true;
	}

	if(!stricmp(mem, "card"))
	{
		if(!isAdmin())
		{
			error("busy-card-admin-required");
			return true;
		}
		mem = getValue(getKeyword("id"));
		if(!mem)
		{
			error("busy-card-id-missing");
			return true;
		}
		tspan = atoi(mem);
		if(!driver->cardEvent(tspan, &event))
			error("busy-card-id-invalid");
		else
			advance();
		return true;
	}

	if(!stricmp(mem, "group"))
	{
		if(!isAdmin())
		{
			error("busy-group-admin-required");
			return true;
		}
		mem = getValue(getKeyword("id"));
		if(mem)
			grp = TrunkGroup::getGroup(mem);
		if(!grp)
		{
			error("busy-group-id-invalid");
			return true;
		}

		while(drv)
		{
	        	for(port = 0; port < drv->getTrunkCount(); ++port)
        		{
	                	trk = drv->getTrunkPort(port);
        	        	if(!trk || trk == this)
                        		continue;

				if(trk->group != grp)
					continue;

				event.id = TRUNK_MAKE_BUSY;
				trk->postEvent(&event);
			}
			drv = drv->drvNext;
		}

		advance();
		return true;
	}
	error("busy-self-reference");
	return true;
}

bool Trunk::scrEcho(void)
{
	char buffer[256];
	unsigned len = 0;
	const char *opt;

	setString(buffer, sizeof(buffer), "output ");
	len = 7;
	while(NULL != (opt = getValue(NULL)) && len < 250)
	{
		snprintf(buffer + len, sizeof(buffer) - len - 1, "%s", opt);
		len = (unsigned)strlen(buffer);
	}
	buffer[len++] = '\n';
	buffer[len] = 0;

#ifndef	WIN32
	if(getppid() > 1)
		cout << buffer + 7 << flush;
#endif
	fifo.control(buffer);
	advance();
	return true;
}

bool Trunk::scrSlog(void)
{
	unsigned id = getId();
	const char *member = getMember();
	char *val;
	Name *obj = getObject();

	char name[32];

        if(!member)
                member = getKeyword("level");

        if(member)
        {
                if(!stricmp(member, "debug"))
                        slog(Slog::levelDebug);
                else if(!stricmp(member, "info"))
                        slog(Slog::levelInfo);
                else if(!stricmp(member, "notice"))
                        slog(Slog::levelNotice);
                else if(!strnicmp(member, "warn", 4))
                        slog(Slog::levelWarning);
                else if(!strnicmp(member, "err", 3))
                        slog(Slog::levelError);
                else if(!strnicmp(member, "crit", 4))
                        slog(Slog::levelCritical);
                else if(!stricmp(member, "alert"))
                        slog(Slog::levelAlert);
                else if(!strnicmp(member, "emerg", 5))
                        slog(Slog::levelEmergency);
                else
                        slog(Slog::levelNotice);
        }
        else
                slog(Slog::levelNotice);

	getName(name);


        slog() << name << ": " << obj->name;
        if(id)
                slog() << "(" << id << ")";

        slog() << ": ";
        while(NULL != (val = getValue(NULL)))
                slog() << val;
        slog() << endl;
        advance();
        return true;
}

bool Trunk::scrSend(void)
{
	const char *id = getKeyword("id");
	const char *pid;
	TrunkEvent event;
	Trunk *trk = NULL;
	int dig;
	const char *opt = getMember();

	if(!opt)
		opt = "message";

	if(!stricmp(opt, "signal"))
	{
		if(!id)
			id = getValue(NULL);
       		trk = Driver::getTrunk(id, false, getDriver());
        	if(!trk)
        	{
                	error("signal-no-such-trunk");
                	return true;
        	}

	        event.id = TRUNK_SIGNAL_JOIN;
        	event.parm.error = getKeyword("message");
        	if(!event.parm.error)
                	event.parm.error = getValue(NULL);
        	if(!trk->postEvent(&event))
        	{
                	error("signal-not-waiting");
                	return true;
        	}
	        advance();
        	return true;
	}

	if(!stricmp(opt, "pickup"))
		pid = getSymbol(SYM_PICKUP);
	else if(!stricmp(opt, "recall"))
		pid = getSymbol(SYM_RECALL);
	else
		pid = getKeyword("id");

	if(!pid)
		pid = getValue(NULL);

	if(!pid)
	{
		error("send-no-id");
		return true;
	}

	trk = Driver::getTrunk(pid, false, getDriver());
	if(!trk)
	{
		error("send-no-session");
		return true;
	}

	if(trk == this)
	{
		error("send-self-reference");
		return true;
	}

	if(!strnicmp(opt, "dig", 3))
	{
		opt = getKeyword("digits");
		if(!opt)
			opt = getValue(NULL);
		if(!opt)
		{
			error("no-digits");
			return true;
		}
		while(*opt)
		{
			dig = getDigit(*(opt++));
			if(dig < 0)
				continue;
			event.id = TRUNK_DTMF_KEYUP;
			event.parm.dtmf.digit = dig;
			trk->postEvent(&event);
		}
	}
	else
	{
		event.id = TRUNK_SEND_MESSAGE;
		event.parm.send.seq = seq;
		event.parm.send.src = this;
		event.parm.send.msg = getKeyword("message");
		if(!event.parm.send.msg)
			event.parm.send.msg = getValue("");
		trk->postEvent(&event);
	}

	advance();
	return true;
}

bool Trunk::scrDummy(void)
{
	error("not-supported");
	return true;
}

bool Trunk::scrIdle(void)
{
	TrunkEvent event;
	Trunk *trk;
	TrunkGroup *grp = NULL;
	Driver *drv = Driver::drvFirst;
	const char *mem = getMember();
	unsigned port, tspan;

	if(!mem)
		mem = "self";
	else if(!isAdmin())
	{
		error("admin-required");
		return true;
	}

	if(!stricmp(mem, "port"))
	{
		event.id = TRUNK_MAKE_IDLE;
		mem = getValue(getKeyword("id"));
		trk = Driver::getTrunk(mem, false, getDriver());

		if(!trk)
		{
			error("idle-port-id");
			return true;
		}

		if(trk == this)
		{
			error("idle-self-reference");
			return true;
		}

		trk->postEvent(&event);
		advance();
		return true;
	}

	if(!stricmp(mem, "span"))
	{
		mem = getValue(getKeyword("id"));
		if(!mem)
		{
			error("idle-span-id");
			return true;
		}
		tspan = atoi(mem);
		if(driver->spanEvent(tspan, &event))
			advance();
		else
			error("idle-span-invalid");
		return true;
	}

        if(!stricmp(mem, "card"))
        {
                mem = getValue(getKeyword("id"));
                if(!mem)
                {
                        error("idle-card-id");
                        return true;
                }
                tspan = atoi(mem);
                if(driver->cardEvent(tspan, &event))
                        advance();
                else
                        error("idle-card-invalid");
                return true;
        }

	if(!stricmp(mem, "group"))
	{
		mem = getValue(getKeyword("id"));
		if(mem)
			grp = TrunkGroup::getGroup(mem);
		if(!grp)
		{
			error("idle-group-id");
			return true;
		}

		while(drv)
		{	
	        	for(port = 0; port < drv->getTrunkCount(); ++port)
        		{
	                	trk = drv->getTrunkPort(port);
        	        	if(!trk || trk == this)
                        		continue;

				if(trk->group != grp)
					continue;

				event.id = TRUNK_MAKE_IDLE;
				trk->postEvent(&event);
			}
			drv = drv->drvNext;
		}

		advance();
		return true;
	}

	idle_timer = atoi(getValue("0"));
	advance();
	return true;
}

bool Trunk::scrSendFax(void)
{
	const char *prefix = getPrefixPath();
	const char *file = getValue(NULL);

	if(!(TRUNK_CAP_SENDFAX & getCapabilities()))
	{
		error("no-fax-send");
		return true;
	}

	if(!file)
	{
		error("no-file-to-send");
		return true;
	}

	if(prefix && !strchr(file, ':'))
		snprintf(data.fax.pathname, sizeof(data.fax.pathname),
			"%s/%s", prefix, file);
	else
		snprintf(data.fax.pathname, sizeof(data.fax.pathname),
			"%s", file);

	data.fax.station = getStation();
	trunkStep(TRUNK_STEP_SENDFAX);
	return false;
}
bool Trunk::scrRecvFax(void)
{
        const char *prefix = getPrefixPath();
        const char *file = getValue(NULL);

        if(!(TRUNK_CAP_RECVFAX & getCapabilities()))
        {
                error("no-fax-recv");
                return true;
        }

        if(!file)
        {
                error("no-file-to-recv");
                return true;
        }

        if(prefix && !strchr(file, ':'))
                snprintf(data.fax.pathname, sizeof(data.fax.pathname),
                        "%s/%s", prefix, file);
        else
                snprintf(data.fax.pathname, sizeof(data.fax.pathname),
                        "%s", file);

        data.fax.station = getStation();
        trunkStep(TRUNK_STEP_RECVFAX);
        return false;
}

bool Trunk::scrRecord(void)
{
	const char *name;
	const char *member = getMember();
	const char *cp;
	const char *prefix = getPrefixPath();
	char *gain = getKeyword("gain");
	char *vol = getKeyword("volume");
	ScriptCommand *cmd = getCommand();
	const char *var = cmd->getLast("datafiles");
	char buffer[256];


	if(flags.listen)
	{
		error("record-unavailable");
		return true;
	}

	cp = getKeyword("trim");
	if(!cp)
		cp = getSymbol(SYM_TRIM);
	if(!cp)
		cp = "0";
	data.record.trim = atoi(cp);

	cp = getKeyword("frames");
	if(!cp)
		cp = "0";

	data.record.frames = atoi(cp);
	cp = getKeyword("minSize");
	if(!cp)
		cp = "0";
	data.record.minsize = atoi(cp);

	if(!vol)
		vol = getSymbol(SYM_VOLUME);

	if(!vol)
		vol = "100";

	apppath[0] = 0;

	if(!member)
		member="all";

	data.record.save = getKeyword("save");
	data.record.text = getKeyword("text");

	name = getValue("");
	if(!name)
	{
		error("record-no-file");
		return true;
	}

	if(prefix && *prefix == '/')
	{
		var = prefix;
		prefix = NULL;
	}

	if(prefix && !strchr(name, ':'))
	{
		snprintf(buffer, sizeof(buffer), "%s/%s",
			prefix, name);
		name = buffer;
	}

	if(var)
		snprintf(apppath + 1, sizeof(apppath) - 6, "%s/%s", var, name);
	else
		snprintf(apppath + 1, sizeof(apppath) - 6, "%s", name);
	data.record.name = apppath + 1;

	cp = getKeyword("timeout");
	if(cp)
		data.record.timeout = getSecTimeout(cp);
	else
		data.record.timeout = getTimeout("maxTime");
	data.record.term = getDigitMask("exit");
	data.record.offset = (unsigned long)-1;
	data.record.volume = atoi(vol);
	data.record.silence = 0;
	data.record.encoding = getKeyword("encoding");
	data.record.annotation = getKeyword("annotation");
	data.record.extension = getKeyword("extension");

	if(data.record.save)
	{
		cp = strrchr(data.record.save, '.');
		if(!cp)
			cp = data.record.extension;
		if(!cp)
			cp = getSymbol(SYM_EXTENSION);
		else
			cp = "";
	}

	if(var && prefix && data.record.save)
	{
		snprintf(data.record.altinfo, sizeof(data.record.altinfo),
			"%s/%s/%s%s", var, prefix, data.record.save, cp);
		data.record.save = data.record.altinfo;
	}
	else if(prefix && data.record.save)
	{
		snprintf(data.record.altinfo, sizeof(data.record.altinfo),
			"%s/%s%s", prefix, data.record.save, cp);
		data.record.save = data.record.altinfo;
	}
	else if(var && data.record.save)
	{
		snprintf(data.record.altinfo, sizeof(data.record.altinfo),
			"%s/%s%s", var, data.record.save, cp);
		data.record.save = data.record.altinfo;
	}
	else if(data.record.save)
	{
		snprintf(data.record.altinfo, sizeof(data.record.altinfo),
			"%s%s", data.record.save, cp);
		data.record.save = data.record.altinfo;
	}

	if(!data.record.encoding)
		data.record.encoding = getDefaultEncoding();

	if(!data.record.annotation)
		data.record.annotation = "";

	if(!data.record.extension)
		data.record.extension = getSymbol(SYM_EXTENSION);

	if(gain)
		data.record.gain = (float)strtod(gain, NULL);
	else
		data.record.gain = 0.0;

	data.record.info = false;

	if(!stricmp(member, "append"))
		data.record.append = true;
	else if(!stricmp(member, "info"))
	{
		data.record.append = true;
		data.record.info = true;
	}
	else
		data.record.append = false;
	if(NULL != (cp = getKeyword("offset")))
		data.record.offset = atoi(cp);
	if(NULL != (cp = getKeyword("volume")))
		data.record.volume = atoi(cp);
	if(NULL != (cp = getKeyword("silence")))
		data.record.silence = atoi(cp);

	if(!stricmp(member, "clear") || !stricmp(member, "erase"))
	{
		cp = strrchr(data.record.name, '/');
		if(cp)
			cp = strrchr(cp, '.');
		else
			cp = strrchr(data.record.name, '.');

		if(!cp)
		{
			cp = data.record.extension;
			if(!cp)
				cp = getSymbol(SYM_EXTENSION);
			strcat(data.record.name, cp);
		}
		::remove(data.record.name);
		if(data.record.save)
			::remove(data.record.save);
		advance();
		return true;
	}

	trunkStep(TRUNK_STEP_RECORD);
	return false;
}

bool Trunk::scrTransfer(void)
{
	unsigned len;
	char *cp = (char *)group->getLast("transfer");
	if(cp)
		setString(data.dialxfer.digits, sizeof(data.dialxfer.digits), cp);
	else
		data.dialxfer.digits[0] = 0;
	len = (unsigned)strlen(data.dialxfer.digits);

	while(NULL != (cp = getValue(NULL)) && len < sizeof(data.dialxfer.digits))
	{
		setString(data.dialxfer.digits + len, sizeof(data.dialxfer.digits) - len, cp);
		len = (unsigned)strlen(data.dialxfer.digits);
	}

	data.dialxfer.digits[sizeof(data.dialxfer.digits) - 1] = 0;
	data.dialxfer.interdigit = group->getDialspeed();
	data.dialxfer.digit = data.dialxfer.digits;
	data.dialxfer.exit = true;
	data.dialxfer.timeout = 0;
	cp = getKeyword("onhook");
	if(!cp)
		cp = getKeyword("flash");
	if(!cp)
		cp = getValue(group->getLast("flash"));
	data.dialxfer.onhook = getMSTimeout(cp);
	cp = getKeyword("dialtone");
	if(!cp)
		cp = getKeyword("offhook");
	if(!cp)
		cp = getValue(group->getLast("dialtone"));
        data.dialxfer.offhook = getMSTimeout(cp);
	trunkStep(TRUNK_STEP_FLASH);
	return false;
}

bool Trunk::scrHold(void)
{
	const char *cp = group->getLast("hold");
	if(!cp)
		cp = "";

	setString(data.dialxfer.digits, sizeof(data.dialxfer.digits), cp);
	data.dialxfer.interdigit = group->getDialspeed();
        data.dialxfer.digit = data.dialxfer.digits;
        data.dialxfer.exit = true;
        data.dialxfer.timeout = 0;
	cp = getKeyword("onhook");
	if(!cp)
		cp = getKeyword("flash");
	if(!cp)
		cp = getValue(group->getLast("flash"));
        data.dialxfer.onhook = getMSTimeout(cp);
	cp = getKeyword("offhook");
	if(!cp)
		cp = getKeyword("dialtone");
	if(!cp)
		cp = getValue(group->getLast("dialtone"));
        data.dialxfer.offhook = getMSTimeout(cp);
	trunkStep(TRUNK_STEP_FLASH);
	return false;
}

bool Trunk::scrSpeak(void)
{
	Translator *tts;
	char *err;
	const char *member = getMember();
	const char *gain = getKeyword("gain");
	const char *speed = getKeyword("speed");
	const char *pitch = getKeyword("pitch");
	const char *lang = getKeyword("language");
	const char *voice = getKeyword("voice");
	const char *vol = getKeyword("volume");
	const char *zone = getKeyword("zone");

	if(!zone)
		zone = getKeyword("channel");

	if(!zone)
		zone = "0";

	if(!vol)
		vol = getSymbol(SYM_VOLUME);

	if(!vol)
		vol = "100";

	if(voice && !lang)
		lang = keyvoices.getLast(voice);

	if(!lang)
		lang = getSymbol(SYM_LANGUAGE);

	apppath[0] = 0;

	if(!member)
		member="all";

	data.play.write = WRITE_MODE_NONE;
	data.play.text = getKeyword("text");
	if(!stricmp(member, "any"))
		data.play.mode = PLAY_MODE_ANY;
	else
		data.play.mode = PLAY_MODE_NORMAL;

	data.play.term = getExitMask();
	data.play.voice = voice;

	tts = getTranslator(lang);

	if(!tts)
	{
		error("language-unsupported");
		return true;
	}
	err = tts->speak(this);
	if(err)
	{
		error(err);
		return true;
	}
	while(*data.play.name == ',')
		++data.play.name;

	if(gain)
		data.play.gain = (float)strtod(gain, NULL);
	else
		data.play.gain = 0.0;

	if(!speed)
		speed = "normal";

	if(!stricmp(speed, "fast"))
		data.play.speed = SPEED_FAST;
	else if(!stricmp(speed, "slow"))
		data.play.speed = SPEED_SLOW;
	else
		data.play.speed = SPEED_NORMAL;

	if(pitch)
		data.play.pitch = (float)strtod(pitch, NULL);
	else
		data.play.pitch = 0.0;

	data.play.extension = getKeyword("extension");
	if(!data.play.extension)
		data.play.extension = getSymbol(SYM_EXTENSION);

	data.play.offset = data.play.limit = 0;
	data.play.timeout = 0;
	data.play.maxtime = 0;
	data.play.repeat = 0;
	data.play.lock = false;
	data.play.volume = atoi(vol);
	data.play.channel = atoi(zone);
	trunkStep(TRUNK_STEP_PLAY);
	return false;
}


bool Trunk::scrControl(void)
{
	unsigned len = 0;
	char buffer[PIPE_BUF / 2];
	const char *value;
	const char *login = getSymbol(SYM_LOGIN);

#define	WORKSPC ((PIPE_BUF / 2) - 5)

	if(!login)
	{
		error("no-login");
		return true;
	}

	if(stricmp(login, "admin"))
	{
		error("admin-required");
		return true;
	}

	while(len < WORKSPC && NULL != (value = getValue(NULL)))
	{
		if(len > 0)
			buffer[len++] = ' ';
		while(*value && len < WORKSPC)
			buffer[len++] = *(value++);
	}
	if(len)
	{
		buffer[len++] = '\n';
		buffer[len] = 0;
		fifo.control(buffer);
	}
	advance();
	return true;
}

bool Trunk::altDial(void)
{
	unsigned len = 0, ilen = 0;
	const char *cp;
	const char *mem = getMember();
	char digits[64];
	char *digbuf = digits;
	char intprefix[5];
	bool intflag = false, natflag = false, soft = false;
	bool rewrite = false;

	if(!mem)
	{
		rewrite = true;
		mem = "dial";
	}

	data.dialxfer.dialer = DTMF_DIALER;
	if(!stricmp(mem, "dtmf"))
		soft = true;
	else if(!stricmp(mem, "pulse"))
	{
		data.dialxfer.dialer = PULSE_DIALER;
		soft = true;
	}
	else if(!stricmp(mem, "mf"))
	{
		data.dialxfer.dialer = MF_DIALER;
		soft = true;
	}

	cp = getKeyword("offhook");
	if(!cp)
		cp = getKeyword("flash");

	if(cp)
		data.dialxfer.offhook = getMSTimeout(cp);
	else
		data.dialxfer.offhook = 0;

	cp = getKeyword("onhook");
	if(cp)
		data.dialxfer.onhook = getMSTimeout(cp);
	else
		data.dialxfer.onhook = 0;

	while(NULL != (cp = getValue(NULL)) && len < sizeof(digits) - 1)
	{
		if(intflag)
			intflag = false;

		while(*cp && len < sizeof(digits) - 1)
		{
			if(driver->getCaps() & Driver::capPSTN)
			{
				switch(*cp)
				{
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					if(intflag && ilen < 4)
					{
						intprefix[ilen++] = *cp;
						break;
					}
				case '*':
				case '#':
				case ',':
				case '.':
				case 'a':
				case 'A':
				case 'b':
				case 'B':
				case 'c':
				case 'C':
				case 'd':
				case 'D':
					digits[len++] = *cp;
					break;
				case 's':
				case 'S':
				case 'k':
				case 'K':
					if(soft)
						digits[len++] = *cp;
			
					intflag = false;
					break;	
				case 't':
				case 'T':
				case 'p':
				case 'P':
				case '!':
				case 'f':
				case 'F':
				case 'm':
				case 'M':
					if(!stricmp(mem, "dial") || soft)
					{
						soft = true;
						digits[len++] = *cp;
					}
					intflag = false;
					break;	
				case '+':
					if(!len)
						intflag = true;
					break;
				default:
					intflag = false;
					break;
				}
			}
			else
			{
				digits[len++] = *cp;
			}
			cp++;
		}
	}
	digits[len] = 0;
	intprefix[ilen] = 0;
	digbuf = digits;

	cp = getKeyword("country");
	if(!cp)
		cp = "1";

	if(ilen)
	{
		if(!stricmp(intprefix, cp))
		{
			natflag = true;
			ilen = 0;
			digbuf += strlen(cp);
		}
	}

	cp = getKeyword("prefix");
	if(!cp)
	{
		if(!strnicmp(mem, "loc", 3))
			cp = group->getLast("local");
		else if(!strnicmp(mem, "nat", 3))
			cp = group->getLast("national");
		else if(!stricmp(mem, "cell"))
			cp = group->getLast("cell");
		else if(!strnicmp(mem, "int", 3))
			cp = group->getLast("international");
		else if(ilen)
			cp = group->getLast("international");
		else if(natflag)
			cp = group->getLast("national");
		else
			cp = group->getLast("prefix");

		if(cp)
			if(!strnicmp(digits, cp, sizeof(cp)))
				cp = "";
	}
	if(cp)
		rewrite = false;
	if(!cp)
		cp = "";

	setString(data.dialxfer.digits, sizeof(data.dialxfer.digits), cp);
	len = (unsigned)strlen(cp);

	cp = getKeyword("suffix");
	if(!len && !cp && rewrite)
	{
		dialRewrite(digbuf);
		len = (unsigned)strlen(data.dialxfer.digits);
	}
	else
	{
		setString(data.dialxfer.digits + len, sizeof(data.dialxfer.digits) - len, digbuf);
		len += (unsigned)strlen(digbuf);
	}

	if(cp)
	{
		setString(data.dialxfer.digits + len, sizeof(data.dialxfer.digits) - len, cp);
		len += (unsigned)strlen(cp);
	}

	data.dialxfer.digits[len] = 0;
	data.dialxfer.callingdigit = getKeyword("origin");
	data.dialxfer.digits[sizeof(data.dialxfer.digits) - 1] = 0;
	data.dialxfer.interdigit = group->getDialspeed();
	data.dialxfer.digittimer = data.dialxfer.interdigit / 2;
	data.dialxfer.digit = data.dialxfer.digits;
	data.dialxfer.exit = false;

	if(soft || !stricmp(mem, "digits"))
		data.dialxfer.timeout = 0;
	else
	{
		cp = getKeyword("maxTime");
		if(cp)
			data.dialxfer.timeout = getSecTimeout(cp);
		else
			data.dialxfer.timeout = group->getAnalysis() * 1000;
	}
//	advance();
//	return true;

	if(soft)
		trunkStep(TRUNK_STEP_SOFTDIAL);
	else
		trunkStep(TRUNK_STEP_DIALXFER);
	return false;
}

bool Trunk::scrDial(void)
{
	const char *mem = getMember();
	const char *ringback = getKeyword("ringback");
	const char *count = getKeyword("count");
	const char *origin = getKeyword("origin");

	if(!origin)
		origin = "";

	if(!isStation)
		return altDial();

	if(!mem)
		mem = "";

	if(!ringback)
		ringback = getKeyword("ringtone");

	if(!ringback)
		ringback = "ringback";

	if(!count)
		count = "-1";

	data.tone.freq1 = data.tone.freq2 = 0;
	data.tone.ampl1 = data.tone.ampl2 = 0;
	data.tone.wakeup = data.tone.duration = 4000;
	data.tone.loops = atoi(count);
	data.tone.tone = getphTone(ringback);
	if(!data.tone.tone)
		data.tone.tone = getphTone("ringback");

	return altDial();
}

bool Trunk::scrSay(void)
{
	char *cp;
	const char *prefix = getPrefixPath();
	const char *cache = getKeyword("cache");
	const char *gain = getKeyword("gain");
	const char *speed = getKeyword("speed");
	const char *pitch = getKeyword("pitch");
	const char *vol = getKeyword("volume");
	const char *zone = getKeyword("zone");
	unsigned len = 0, lc = 0;
	unsigned long sum = 0l, sum1 = 0l;
	const char *mem = getMember();
	bool synth = false;
	Line *line = getScript();

	if(!stricmp("tts", line->cmd))
		synth = true;

	if(!zone)
		zone = getKeyword("channel");

	if(!zone)
		zone = "0";

	if(!mem)
		mem = "text";

	if(!stricmp(mem, "nocache") && !TTS::ttsFirst)
	{
		advance();
		return true;
	}

	if((!stricmp(mem, "cache") || synth) && !TTS::ttsFirst)
	{
		advance();
		return true;
	}

	if(cache && !TTS::ttsFirst)
		cache = NULL;

	if(!hasTTS())
	{
		advance();
		return true;
	}

	if(!vol)
		vol = getSymbol(SYM_VOLUME);

	if(!vol)
		vol = "100";

	data.play.list[0] = 0;

	while(NULL != (cp = getValue(NULL)) && len < sizeof(data.play.list))
	{
		if(len)
		{
			if(TTS::ttsFirst)
				data.play.list[len++] = ' ';
			else
				data.play.list[len++] = '+';
		}
		setString(data.play.list + len, sizeof(data.play.list) - len, cp);
		len += (unsigned)strlen(cp);
	}
	if(!data.play.list[0])
	{
		error("tts-no-output");
		return true;
	}


	data.play.text = getKeyword("text");
	if(!data.play.text)
		data.play.text = data.play.list;

	data.play.voice = getKeyword("voice");
	if(!data.play.voice)
		data.play.voice = getSymbol(SYM_VOICE);
	data.play.list[sizeof(data.play.list) - 1] = 0;
#ifdef	HAVE_TGI
	if(!TTS::ttsFirst)
	{
		libtts(data.play.list, TTS_GATEWAY_TEXT);
		sprintf(data.play.list, "temp/.tts.%d.ul", id);
	}
#else
	if(!TTS::ttsFirst)
	{
		error("no-tts");
		return true;
	}
#endif
	data.play.name = data.play.list;
	data.play.repeat = 0;
	data.play.lock = false;
	data.play.maxtime = 0;
	data.play.lock = false;
	data.play.channel = atoi(zone);
	if(TTS::ttsFirst)
		data.play.timeout = 0;
	else
		data.play.timeout = getSecTimeout(getSymbol(SYM_PLAYWAIT));
	data.play.volume = atoi(vol);
	data.play.term = getExitMask();
	data.play.write = WRITE_MODE_NONE;
	data.play.mode = PLAY_MODE_TEMP;
	data.play.limit = data.play.offset = 0;
	data.play.extension = NULL;

	if(gain)
		data.play.gain = (float)strtod(gain, NULL);
	else
		data.play.gain = 0.0;

	if(!speed)
		speed = "normal";

	if(!stricmp(speed, "fast"))
		data.play.speed = SPEED_FAST;
	else if(!stricmp(speed, "slow"))
		data.play.speed = SPEED_SLOW;
	else
		data.play.speed = SPEED_NORMAL;

	if(pitch)
		data.play.pitch = (float)strtod(pitch, NULL);
	else
		data.play.pitch = 0.0;

	if(NULL != (cp = getKeyword("offset")))
		data.play.offset = atol(cp);
	if(NULL != (cp = getKeyword("volume")))
		data.play.volume = atoi(cp);
	if(NULL != (cp = getKeyword("timeout")) && !TTS::ttsFirst)
		data.play.timeout = getSecTimeout(cp);
	if(NULL != (cp = getKeyword("limit")))
		data.play.limit = atol(cp);

	if((!stricmp(mem, "cache") || synth) && !cache)
	{
		sprintf(apppath, "cache/-tts-%s-", data.play.voice);
		len = (unsigned)strlen(apppath);
		cp = data.play.list;
		while(*cp)
		{
			if(*cp != ' ')
			{
				++lc;
				sum ^= (sum << 3) ^ (*cp & 0x1f);
				sum1 ^= ((sum >> 29) ^ lc) ^ (*cp & 0x0f);
			}
			++cp;
		}
		cp = data.play.list;
		while(*cp)
		{
			sum1 = (sum1 << 3) ^ (*cp & 0x1f);
			++cp;
		}
		sprintf(apppath + len, "%08lx%08lx", sum, sum1);
		cp = apppath;
		while(*cp)
		{
			*cp = tolower(*cp);
			++cp;
		}
		data.play.cache = apppath;
	}
	else if(cache)
	{
		if(!prefix && !strchr(cache, '/'))
			prefix = "cache";
		if(prefix)
		{
			snprintf(apppath, sizeof(apppath), "%s/%s", prefix, cache);
			cache = apppath;
		}
		data.play.cache = cache;
	}
	else
		data.play.cache = NULL;

	if(TTS::ttsFirst)
	{
		if(synth)
			data.play.mode = PLAY_MODE_NONE;
		else if(!stricmp(mem, "file"))
			data.play.mode = PLAY_MODE_FILE;
		else
			data.play.mode = PLAY_MODE_TEXT;
		trunkStep(TRUNK_STEP_PLAY);
	}
	else
		trunkStep(TRUNK_STEP_PLAYWAIT);
	return false;
}

bool Trunk::scrAltPlay(void)
{
	if(hasTTS())
	{
		advance();
		return true;
	}
	return scrPlay();
}

bool Trunk::scrAltSpeak(void)
{
	if(hasTTS())
	{
		advance();
		return true;
	}
	return scrSpeak();
}

bool Trunk::scrPlay(void)
{
	Name *scr = getObject();
	char *cp;
	unsigned long mask = 0;
	const char *prefix = getPrefixPath();
	const char *gain = getKeyword("gain");
	const char *speed = getKeyword("speed");
	const char *pitch = getKeyword("pitch");
	const char *vol = getKeyword("volume");
	const char *zone = getKeyword("zone");
	bool feed = false;
	unsigned len = 0;
	const char *member = getMember();
	if(!member)
		member = "all";

	if(!stricmp(member, "feed"))
		feed = true;

	if(!zone)
		zone = getKeyword("channel");
	if(!zone)
		zone = "0";
	
	data.play.write = WRITE_MODE_NONE;

	if(!stricmp(member, "moh"))
	{
		data.play.mode = PLAY_MODE_MOH;
	}
	else if(!stricmp(member, "any"))
		data.play.mode = PLAY_MODE_ANY;
	else if(!stricmp(member, "one"))
		data.play.mode = PLAY_MODE_ONE;
	else if(!stricmp(member, "temp") || !stricmp(member, "tmp"))
		data.play.mode = PLAY_MODE_TEMP;
	/* else if(0 != (mask = getScriptMask(member)))
		data.play.mode = PLAY_MODE_ANY; */
	else
		data.play.mode = PLAY_MODE_NORMAL;

	data.play.text = getKeyword("text");
	data.play.voice = getKeyword("voice");

	if(mask)
	{
		if((mask & scr->mask) != mask)
		{
			advance();
			return true;
		}
	}

	data.play.list[0] = 0;
	while(NULL != (cp = getValue(NULL)))
	{
		if(len)
			data.play.list[len++] = ',';
		if(prefix && !strchr(cp, ':'))
			snprintf(data.play.list + len, sizeof(data.play.list) - len, "%s/%s", prefix, cp);
		else
			setString(data.play.list + len, sizeof(data.play.list) - len, cp);
		len = (unsigned)strlen(data.play.list);
		if(feed)
			break;
	}

	if(!vol)
		vol = getSymbol(SYM_VOLUME);
	if(!vol)
		vol = "100";

	data.play.list[sizeof(data.play.list) - 1] = 0;
	data.play.name = data.play.list;
	data.play.volume = atoi(vol);
	data.play.timeout = 0;
	data.play.repeat = 0;
	data.play.lock = false;
	data.play.channel = atoi(zone);
	data.play.maxtime = 0;
	data.play.term = getExitMask();
	data.play.lock = false;
	data.play.extension = getKeyword("extension");
	if(!data.play.extension)
		data.play.extension = getSymbol(SYM_EXTENSION);

	if(feed)
		data.play.maxtime = getTimeout("maxTime");

	while(*data.play.name == ',')
		++data.play.name;
	if(!*data.play.name)
	{
		error("play-no-files");
		return true;
	}

	if(gain)
		data.play.gain = (float)strtod(gain, NULL);
	else
		data.play.gain = 0.0;

	if(!speed)
		speed = "normal";

	if(!stricmp(speed, "fast"))
		data.play.speed = SPEED_FAST;
	else if(!stricmp(speed, "slow"))
		data.play.speed = SPEED_SLOW;
	else
		data.play.speed = SPEED_NORMAL;

	if(pitch)
		data.play.pitch = (float)strtod(pitch, NULL);
	else
		data.play.pitch = 0.0;

	data.play.limit = data.play.offset = 0;

	if(NULL != (cp = getKeyword("offset")))
		data.play.offset = atol(cp);

	if(NULL != (cp = getKeyword("limit")))
		data.play.limit = atol(cp);

	if(NULL != (cp = getKeyword("volume")))
		data.play.volume = atoi(cp);

	trunkStep(TRUNK_STEP_PLAY);
	return false;
}	

bool Trunk::scrFlash(void)
{
	const char *cp = getKeyword("onhook");
	if(!cp)
		cp = getKeyword("flash");
	if(!cp)
		cp = getValue(group->getLast("flash"));
	data.dialxfer.onhook = getMSTimeout(cp);

	cp = getKeyword("offhook");
	if(!cp)
		cp = getKeyword("dialtone");
	if(!cp)
		cp = getValue(group->getLast("dialtone"));

	data.dialxfer.offhook = getMSTimeout(cp);

	data.dialxfer.digit = NULL;
	trunkStep(TRUNK_STEP_FLASH);
	return false;
}

bool Trunk::scrCollect(void)
{
	unsigned copy = 0;
	Symbol *sym = NULL;
	const char *cp = getKeyword("count");
	const char *mem = getMember();
	const char *var = getKeyword("var");
	const char *ignore = getKeyword("ignore");
	const char *term = getKeyword("exit");
	unsigned digpos = 0, digscan;
	bool trim = false;

	if(!ignore)
		ignore = "";

	if(!term)
		term = "";

	if(!mem)
		mem = "all";

	if(!cp)
		cp = getKeyword("digits");

	if(!cp)
		cp = getValue("0");

	if(var)
		if(*var == '&')
			++var;

	data.collect.count = atoi(cp);
	if(data.collect.count > MAX_DIGITS)
		data.collect.count = MAX_DIGITS;

	if(var)
		sym = getLocal(var, data.collect.count);

	if(sym)
	{
		if(sym->flags.readonly)
		{
			error("collect-read-only");
			return true;
		}
	}

	if(sym)
	{
		copy = sym->flags.size;
		sym->data[0] = 0;
		if(sym->flags.commit)
			commit(sym);
	}

	if(copy > data.collect.count)
		copy = data.collect.count;		

	data.collect.var = (void *)sym;
	data.collect.map = NULL;	
	data.collect.first = data.collect.timeout = getInterdigit("timeout");
	data.collect.term = getDigitMask("exit");
	data.collect.ignore = getDigitMask("ignore");

	cp = getKeyword("nextDigit");
	if(!cp)
		cp = getKeyword("next");
	if(cp)
	{
		data.collect.timeout = getSecTimeout(cp);
		cp = getKeyword("timeout");
		if(!cp)
			data.collect.first = data.collect.timeout;
	}

	cp = getKeyword("firstDigit");
	if(!cp)
		cp = getKeyword("first");
	if(cp)
		data.collect.first = getSecTimeout(cp);

	if(!stricmp(mem, "clear"))
	{
		digits = 0;
		dtmf.bin.data[0] = 0;
	}
	else if(!stricmp(mem, "trim"))
		trim = true;
	else if(!stricmp(mem, "input"))
		trim = true;
	
	while(digpos < digits)
	{
		if(strchr(term, dtmf.bin.data[digpos]))
		{
			if(copy > digpos)
				copy = digpos;
			if(sym)
			{
				if(copy)
					strncpy(sym->data, dtmf.bin.data, copy);
				sym->data[copy] = 0;
				if(sym->flags.commit)
					commit(sym);
				digscan = ++digpos;
				while(digscan < digits)
				{
					dtmf.bin.data[digscan - digpos] = dtmf.bin.data[digscan];
					++digscan;
				}
				digits = digscan - digpos;
				dtmf.bin.data[digits] = 0;
				digits = digscan;
			}	
			else
			{
				digits = digpos;
				dtmf.bin.data[digits] = 0;
			}
			advance();
			return true;
		}

		if(strchr(ignore, dtmf.bin.data[digpos]))
		{
			digscan = digpos;
			while(digscan < digits)
			{
				dtmf.bin.data[digscan] = dtmf.bin.data[digscan + 1];
				++digscan;
			}
			continue;
		}	

		if(++digpos >= data.collect.count)
		{
			if(sym)
			{
				if(copy)
					strncpy(sym->data, dtmf.bin.data, copy);
				sym->data[copy] = 0;
				if(sym->flags.commit)
					commit(sym);
				while(digpos < digits)
				{
					dtmf.bin.data[digpos - data.collect.count] = dtmf.bin.data[digpos];
					++digpos;
				}
				digits = digpos - data.collect.count;
				dtmf.bin.data[digits] = 0;
			}
			else if(trim || *term)
			{
				digits = digpos;
				dtmf.bin.data[digits] = 0;
			}
			advance();
			return true;
		}
	}
	trunkStep(TRUNK_STEP_COLLECT);
	return false;
}

bool Trunk::scrAccept(void)
{
	if (group->getAccept())
	{
		advance();
		return true;
	}

	trunkStep(TRUNK_STEP_ACCEPT);
	return false;
}

bool Trunk::scrReject(void)
{
	trunkStep(TRUNK_STEP_REJECT);
	return false;
}

bool Trunk::scrAnswer(void)
{
	const char *mem = getMember();
	const char *kw;
	Trunk *trk = NULL;
	TrunkEvent event;
	unsigned pid;
	char *port;

	if(isStation)
	{
		if(!mem)
			mem = "";

		if(!stricmp(mem, "pickup"))
			port = getSymbol(SYM_PICKUP);
		else if(!stricmp(mem, "parent"))
			port = getSymbol(SYM_PARENT);
		else
			goto defaultanswer;

		if(!trk || trk == this)
		{
			if(!trunkSignal(TRUNK_SIGNAL_FAIL))
				error("answer-nocall");
			return true;
		}
		pid = trk->id;
		mem = trk->trunkgid;
		if(!mem)
			mem = "";
		else
			mem = strchr(mem, '-');
		setSymbol(SYM_PICKUP, mem);
		event.id = TRUNK_STATION_ANSWER;
		event.parm.trunk = this;
		if(trk->postEvent(&event))
			return scrGoto();
		if(!trunkSignal(TRUNK_SIGNAL_BUSY))
			error("answer-busy");
		return true;
	}

defaultanswer:
	/*mem = getMember();
	if(mem)
		if(!stricmp(mem, "media"))
			data.answer.media = 1;*/

	if(NULL != (kw = getKeyword("maxRing")))
		data.answer.rings = atoi(kw);
	else 	
		data.answer.rings = atoi(getValue("0"));
	if(NULL != (kw = getKeyword("maxTime")))
		data.answer.timeout = getSecTimeout(kw);
	else
		data.answer.timeout =  getSecTimeout(getValue(group->getLast("ringtime")));

	data.answer.station = getStation();
	data.answer.fax = getKeyword("fax");

	trunkStep(TRUNK_STEP_ANSWER);
	return false;
}	

bool Trunk::scrHangup(void)
{
	TrunkEvent event;
	Trunk *trk;
	TrunkGroup *grp = NULL;
	const char *mem = getMember();
	unsigned port, tspan;
	const char *id = getKeyword("id");

	if(!mem)
		mem = "self";
	else if(!isAdmin())
	{
		error("admin-required");
		return true;
	}

	if(!stricmp(mem, "port"))
	{
		event.id = TRUNK_STOP_DISCONNECT;
		mem = getValue(id);
		trk = Driver::getTrunk(mem, false, getDriver());

		if(!trk)
		{
			error("hangup-port-id-invalid");
			return true;
		}

		if(trk == this)
		{
			error("hangup-port-self-reference");
			return true;
		}

		if(*mem != ' ' && !isAdmin())
		{
			error("hangup-port-admin-required");
			return true;
		}

		trk->postEvent(&event);
		advance();
		return true;
	}

	if(!stricmp(mem, "span"))
	{
		if(!isAdmin())
		{
			error("hangup-span-admin-required");
			return true;
		}
		mem = getValue(id);
		if(!mem)
		{
			error("hangup-span-id-missing");
			return true;
		}
		tspan = atoi(mem);
		if(driver->spanEvent(tspan, &event))
			advance();
		else
			error("hangup-span-id-invalid");
		return true;			
	}

        if(!stricmp(mem, "card"))
        {
		if(!isAdmin())
		{
			error("hangup-card-admin-required");
			return true;
		}
                mem = getValue(id);
                if(!mem)
                {
                        error("hangup-card-id-missing");
                        return true;
                }
                tspan = atoi(mem);
                if(driver->cardEvent(tspan, &event))
                        advance();
                else
                        error("hangup-card-id-invalid");
                return true;
        }

	if(!stricmp(mem, "group"))
	{
		if(!isAdmin())
		{
			error("hangup-group-admin-required");
			return true;
		}
		mem = getValue(id);
		if(mem)
			grp = TrunkGroup::getGroup(mem);
		if(!grp)
		{
			error("hangup-group-id-missing");
			return true;
		}

	        for(port = 0; port < driver->getTrunkCount(); ++port)
        	{
	                trk = driver->getTrunkPort(port);
        	        if(!trk || trk == this)
                        	continue;

			if(trk->group != grp)
				continue;

			event.id = TRUNK_STOP_DISCONNECT;
			trk->postEvent(&event);
		}

		advance();
		return true;
	}
	
	if(id)
	{
		if(!strchr(id, '-') && !isAdmin())
		{
			error("admin-required");
			return true;
		}
		trk = Driver::getTrunk(id, false, getDriver());
		if(!trk)
		{
			error("hangup-id-invalid");
			return true;
		}
		event.id = TRUNK_STOP_DISCONNECT;
		trk->postEvent(&event);
		advance();
		return true;
	}

	if(!ScriptInterp::signal((unsigned int)0))
		scrExit();

	return true;
}

bool Trunk::scrDebug(void)
{
	char buf[256];
	char *value;

	buf[0] = 0;
	while(NULL != (value = getValue(NULL)))
		strcat(buf, value);
	debug->debugScript(this, buf);
	advance();
	return true;
}

bool Trunk::scrSync(void)
{
	const char *cp;
	timeout_t timer = getTimeout("time");
	time_t now;
	const char *mem = getMember();
	TrunkEvent event;
	Trunk *trunk;
	const char *id = getKeyword("id");

	if(!mem)
		mem = "none";

	time(&now);


	if(!stricmp(mem, "reset"))
	{
		counts = 0;
		advance();
		return true;
	}

	if(!stricmp(mem, "parent"))
	{
		mem = "port";
		id = getSymbol(SYM_PARENT);
	}
	
	if(!stricmp(mem, "port"))
	{
		if(!id)
			id = getKeyword("port");
		if(!id)
			id = getValue(NULL);

		if(!id)
		{
			error("missing-id");
			return true;
		}
		trunk = Driver::getTrunk(id, false, driver);
		if(!trunk)
		{
			error("invalid-port");
			return true;
		}
		event.id = TRUNK_SYNC_NOTIFY;
		trunk->postEvent(&event);
		advance();
		return true;
	}

	if(!strnicmp(mem, "max", 3) || !stricmp(mem, "exit"))
	{
		if(timer)
			exittimer = starttime + (timer / 1000);
		else
			exittimer = 0;
		++driver->refTimer;
		advance();
		return true;
	}

	if(!strnicmp(mem, "time", 4) || !stricmp(mem, "start"))
	{
		if(timer)
			synctimer = starttime + (timer / 1000);
		else
			synctimer = 0;
		++driver->refTimer;
		advance();
		return true;
	}

	if(!stricmp(mem, "current"))
	{
		if(timer)
			synctimer = now + (timer / 1000);
		else
			synctimer = 0;
		++driver->refTimer;
		advance();
		return true;
	}

	timer = (timeout_t)(timer - ((now - starttime) * 1000l));
	if(timer < 1)
	{
		advance();
		return true;
	}
	data.sleep.wakeup = timer;
	data.sleep.save = NULL;
	data.sleep.term = 0;
	data.sleep.count = 0;
	cp = getKeyword("count");
	if(cp)
		data.sleep.count = atoi(cp);
	cp = getKeyword("maxRing");
	if(!cp)
		cp = getValue("0");
	data.sleep.rings = atoi(cp);
	data.sleep.loops = 1;
	trunkStep(TRUNK_STEP_SLEEP);
	return false;
}

bool Trunk::scrStart(void)
{
	Trunk *trunk;
	TrunkGroup *grp = NULL;
	TrunkEvent event;
	Symbol *sym;
	int argc = 0;
	char *argv[32];
	char *arg = NULL, *tok;
	const char *login = getSymbol("login");
	const char *var = getKeyword("var");
	const char *submit = getKeyword("submit");
	const char *tid = NULL;
	timeout_t exp = 0;
	bool notify = false;
	bool rtn = true;
	const char *mem = getMember();
	char buffer[256];
	char args[512];
	char content[512];
	unsigned alen = 0;
	unsigned offset = 0, last = 0, span = 0;
	bool start = false;
	bool ports = false;
	Driver *startDriver = driver;
	Name *scr = getObject();
	const char *cp;
	char scrname[65];
	char subname[128];
	char *sp;
	ScriptCommand *cmd = getCommand();

	snprintf(scrname, sizeof(scrname), "%s", scr->name);
	sp = strchr(scrname, ':');
	if(sp)
		*sp = 0;

	args[0] = 0;

	if(var)
		if(*var == '&')
			++var;

	if(var)
		setVariable(var, 16);

	if(!mem)
		mem = "none";

	if(!stricmp(mem, "wait"))
	{
		exp = getTimeout("maxTime");
		notify = true;
	}
	else if(!stricmp(mem, "port"))
	{
		arg = getKeyword("first");
		if(arg)
			offset = atoi(arg);
		else
			offset = atoi(getValue("0"));
		arg = getKeyword("last");
		if(arg)
			last = atoi(arg);
		else
			last = offset;

		ports = true;
	}
	else if(!stricmp(mem, "id"))
	{
		tid = getValue("0");
		trunk = Driver::getTrunk(tid, true, driver);
		if(!trunk)
		{
			error("invalid-trunk");
			return true;
		}
		offset = last = trunk->id;
		startDriver = trunk->getDriver();
		ports = true;
	}
	else if(!stricmp(mem, "offset"))
	{
		arg = getKeyword("first");
		if(arg)
			offset = atoi(arg) + id;
		else
			offset = atoi(getValue("1")) + id;
		arg = getKeyword("last");
		if(arg)
			last = id + atoi(arg);
		else
			last = driver->getTrunkCount() - 1;
		ports = true;
	}
	else if(!stricmp(mem, "group") || !stricmp(mem, "driver"))
		start = true;
	else
		exp = getTimeout("maxTime");



	if(!ports && !span)
	{
		arg = getKeyword("group");
		if(!arg)
			arg = getKeyword("driver");

		if(!arg)
			arg = getValue("*");

		grp = TrunkGroup::getGroup(arg);
		if(!grp)
		{
			start = true;
			tid = arg;
			goto skip;
		}
		arg = getKeyword("limit");
		if(arg)
		{
			if(atoi(arg) > grp->getStat(STAT_AVAIL_CALLS))
			{
				error("call-limit-exceeded");
				return true;
			}
		}
	}

skip:

	if(start)
	{
		if(grp)
			snprintf(args + alen, sizeof(args) - alen, "start %s %s/", grp->getName(), cmd->getLast("name"));
		else if(tid)
			snprintf(args + alen, sizeof(args) - alen, "start %s %s/", tid, cmd->getLast("name"));
		else
			snprintf(args + alen, sizeof(args) - alen, "start %d %s/", offset, cmd->getLast("name"));
		alen = (unsigned)strlen(args);
	}

	arg = getKeyword("script");
	if(!arg)
		arg = getValue(NULL);
	if(!arg)
	{
		error("request-no-script");
		return true;
	}

	if(!strnicmp(arg, "::", 2))
	{
		setString(content, sizeof(content), scr->name);
		tok = strstr(content, "::");
		if(tok)
			*tok = 0;
		addString(content, sizeof(content), arg);
		arg = content;
	}
	else if(!strstr(arg, "::"))
	{
		setString(content, sizeof(content), scr->name);
		tok = strstr(content, "::");
		if(tok) 
			*tok = 0;
		addString(content, sizeof(content), "::");
		addString(content, sizeof(content), arg);
		if(image->getScript(content))
			arg = content;
	}

	setString(args + alen, sizeof(args) - alen, arg);
	alen = (unsigned)strlen(args);

	cp = trunkgid;
	if(cp)
		cp = strchr(cp, '-');
	else
		cp = "none";

	if(start || ports || span)
		snprintf(args + alen, sizeof(args) - alen, " %s=%s", SYM_PARENT, cp);
	alen = (unsigned)strlen(args);

	while(NULL != (arg = getOption(NULL)))
	{
		if(*arg != '%' && *arg != '&')
			continue;
		urlEncode(getContent(arg), content, sizeof(content));
		snprintf(args + alen, sizeof(args) - alen, " %s=%s", ++arg, content);
		alen = (unsigned)strlen(args);
	}

        if(submit)
        {
                snprintf(buffer, 255, "%s", submit);
                submit = strtok_r(buffer, ",", &tok);
	}

	while(submit)
        {
		if(*submit == '.')
		{
			snprintf(subname, sizeof(subname), "%s%s", scrname, submit);
			submit = subname;
		}
		if(strchr(submit, '.') == NULL)
		{
			sym = getLocal(submit, 0);
			if(!sym)
			{
				snprintf(subname, sizeof(subname),
					"%s.%s", scrname, subname);
				sym = getEntry(subname, 0);
			}
		}
		else		
	                sym = getEntry(submit, 0);
                submit = strtok_r(NULL, ",", &tok);
                if(!sym)
                        continue;

		urlEncode(sym->data, content, sizeof(content));
		snprintf(args + alen, sizeof(args) - alen, " %s=%s", sym->id, content);
		alen = (unsigned)strlen(args);
	}

	if(var || notify)
	{
		if(!exp)
			exp = 1000;

		data.sleep.count = 0;
		data.sleep.term = 0;
		data.sleep.save = var;
		data.sleep.wakeup = exp;
	        data.sleep.loops = 1;
	        data.sleep.rings = 0;
        	//data.sleep.save = NULL;
        	trunkStep(TRUNK_STEP_SLEEP);
		rtn = false;
	}
	else
		advance();

	if(start)
	{
		trunk = fifo.command(args);
		if(!trunk)
		{
			if(!Trunk::event("start:busy"))
				error("start-ports-busy");
		}
		cp = trunk->trunkgid;
		if(cp)
			cp = strchr(cp, '-');
		if(cp && var)
			setVariable(var, 16, cp);
		return rtn;
	}

	argv[argc++] = strtok_r(args, " ", &tok);
	while(argc < 31)
		argv[argc++] = strtok_r(NULL, " ", &tok);

	argv[argc] = NULL;

	if(offset > last)
		last = offset;

	while(offset <= last)
	{
		trunk = startDriver->getTrunkPort(offset++);
		if(!trunk)
			continue;

		if(spanid)
			if(trunk->spanid != spanid)
				continue;

		event.id = TRUNK_START_SCRIPT;
		event.parm.argv = argv;
		if(trunk->postEvent(&event))
		{
			trunk->enterMutex();
			if(var)
			{
				cp = trunk->trunkgid;
				if(cp)
					cp = strchr(cp, '-');
			}
			if(var && cp)
				setVariable(var, 16, cp);
			sym = trunk->getEntry(SYM_LOGIN, 0);
			if(sym)
				snprintf(sym->data, sym->flags.size + 1, "%s", login);			
			trunk->leaveMutex();
			break;
		}
	}

	if(offset > last)
	{
		event.id = TRUNK_CHILD_FAIL;
		postEvent(&event);
	}

	return rtn;
}

bool Trunk::scrOptions(void)
{
	const char *lang;
	const char *keys;
	const char *err = NULL;
	char voice[256];
	timeout_t timer;

	keys = getKeyword("logging");
	if(keys)
	{
		if(!stricmp(keys, "notice"))
			slog.level(Slog::levelNotice);
		else if(!stricmp(keys, "debug"))
			slog.level(Slog::levelDebug);
		else if(!stricmp(keys, "info"))
			slog.level(Slog::levelInfo);
		else if(!strnicmp(keys, "err", 3))
			slog.level(Slog::levelError);
		else if(!strnicmp(keys, "warn", 4))
			slog.level(Slog::levelWarning);
		else
			err = "options-logging-invalid";
	}
        keys = getKeyword("language");
        if(keys)
	{
		if(getTranslator(keys) != NULL)
	                setSymbol(SYM_LANGUAGE, keys);
		else
			err = "options-language-invalid";
	}

        keys = getKeyword("voice");
        if(keys)
        {
                snprintf(voice, sizeof(voice), "%s/%s", keypaths.getLast("prompts"), keys);
                if(isDir(voice))
		{
                        setSymbol(SYM_VOICE, keys);
			lang = keyvoices.getLast(keys);
			if(lang)
			{
				if(getTranslator(lang) != NULL)
					setSymbol(SYM_LANGUAGE, lang);
				else
					err = "options-language-invalid";
			}
		}
		else
			err = "options-voice-invalid";
        }

	keys = getKeyword("idlelimit");
	if(!keys)
		keys = getKeyword("idle");
	if(keys)
		idle_timer = atoi(keys);

	keys = getKeyword("timelimit");
	if(!keys)
		keys = "limit";
	if(!keys)
		keys = "time";
	if(keys)
	{
		timer = getSecTimeout(keys);
		if(timer)
			exittimer = starttime + (timer / 1000);
		else
			exittimer = 0;
		++driver->refTimer;
	}

	if(err)
		error(err);
	else
		advance();
	return true;
}

bool Trunk::scrJoin(void)
{
	Trunk *trk = NULL;
	const char *mem = getMember();
	char *port;

	data.join.hangup = false;
	data.join.waiting = NULL;
	data.join.direction = JOIN_FULL;
	data.join.recfn = NULL;
	data.join.local = true;
	data.join.wakeup = 0;

	if(!mem)
		mem = "duplex";

	if(!stricmp(mem, "pickup"))
		port = getSymbol(SYM_PICKUP);
	else if(!stricmp(mem, "parent"))
		port = getSymbol(SYM_PARENT);
	else
		port = getValue(getKeyword("id"));

	if(!stricmp(mem, "hangup"))
		data.join.hangup = true;

	trk = Driver::getTrunk(port, false, driver);
	if(!trk)
	{
		error("join-no-port");
		return true;
	}

	if(trk->getDriver() != getDriver())
		data.join.local = false;

	mem = getKeyword("count");
	if(mem)
		data.join.count = atoi(mem);
	else
		data.join.count = 0;

	mem = getKeyword("waitTime");
	if(mem)
		data.join.count = getSecTimeout(mem) /
			keythreads.getResetDelay() + 1;

	mem = getKeyword("record");
	if(mem)
	{
		data.join.recfn = (char *)mem;
		data.join.encoding = getKeyword("encoding");
		data.join.annotation = getKeyword("annotation");
		data.join.extension = getKeyword("extension");
		if(!data.join.encoding)
			data.join.encoding = getDefaultEncoding();
		if(!data.join.extension)
			data.join.extension = getSymbol(SYM_EXTENSION);
	}

	mem = getKeyword("maxTime");
	if(mem)
		data.join.wakeup = getSecTimeout(mem);

	data.join.seq = trk->getSequence();
	data.join.waiting = data.join.trunk = trk;
	data.join.wakeup = getTimeout("maxTime");

	if(data.join.local)
		trunkStep(TRUNK_STEP_JOIN);
	else
		trunkStep(TRUNK_STEP_JOIN);
	return false;
}

bool Trunk::scrWait(void)
{
	const char *mem = getMember();
	char *port = NULL;

	data.join.local = true;

	if(mem)
	{
		data.join.waiting = NULL;
		data.join.hangup = false;
		if(!stricmp(mem, "hangup"))
			data.join.hangup = true;
		else if(!stricmp(mem, "parent"))
			port = getSymbol(SYM_PARENT);
		else if(!stricmp(mem, "pickup") || !stricmp(mem, "hold"))
			port = getSymbol(SYM_PICKUP);
		else if(!stricmp(mem, "recall"))
			port = getSymbol(SYM_RECALL);

		if(port)
			data.join.waiting = Driver::getTrunk(port, false, driver);

		if(data.join.waiting && data.join.waiting->getDriver() != getDriver())
			data.join.local = false;
	}
	else
	{
		data.join.waiting = NULL;
		data.join.hangup = false;
	}

	data.join.count = 0;
	data.join.trunk = NULL;
	if(getKeyword("maxTime"))
		data.join.wakeup = getTimeout("maxTime");
	else
		data.join.wakeup = getTimeout("waitTime");

	trunkStep(TRUNK_STEP_JOIN);

	return false;
}

bool Trunk::scrPickup(void)
{
	Trunk *trk = NULL;
	TrunkEvent event;
	const char *mem = getMember();
	unsigned pid = 0, max = driver->getTrunkCount();
	char trkname[MAX_NAME_LEN];
	char appl[65];
	char digbuf[65];
	char *port = NULL;
	Name *scr = getObject();
	char *name;
	char *digits = getKeyword("digits");
	char *args[5];

	if(!digits)
		digits = getKeyword("dialing");

	if(!digits)
		digits = getKeyword("dial");

	snprintf(appl, sizeof(appl), "%s", scr->name);
	name = strstr(appl, "::");
	if(name)
		*name = 0;

	name = (char *)getKeyword("script");
	if(name)
	{
		if(!strnicmp(name, "::", 2))
		{
			snprintf(appl + strlen(appl), sizeof(appl) - strlen(appl), "%s", name);
			name = appl;
		}
	}

	if(digits && !name)
		name = (char *)keyserver.getIncoming(this);

        if(digits)
		snprintf(digbuf, sizeof(digbuf), "session.dialing=%s", digits);
        else
                snprintf(digbuf, sizeof(digbuf), "session.dialing=");
        digits = digbuf;


	if(!mem)
		mem = "incoming";

	if(!stricmp(mem, "hold"))
		port = getSymbol(SYM_PICKUP);
	else if(!stricmp(mem, "parent"))
		port = getSymbol(SYM_PARENT);
	else if(!stricmp(mem, "recall"))
		port = getSymbol(SYM_RECALL);
	else port = getKeyword("id");

	if(!port)
		port = getValue(NULL);

	if(port)
		trk = Driver::getTrunk(port, false, driver);

	if(!trk || pid >= max)
	{
		error("pickup-nocall");
		return true;
	}
	pid = trk->id;
	mem = trk->trunkgid;
	if(!mem)
	{
		snprintf(trkname, sizeof(trkname), "%s/%d", driver->getName(), trk->id);
		mem = trkname;
	}
	else
		mem = strchr(mem, '-');
	setSymbol(SYM_PICKUP, mem);
	if(name)
	{
                event.id = TRUNK_START_SCRIPT;
                event.parm.argv = args;
                args[0] = name;
                args[1] = "session.calltype=dialing";
                args[2] = digits;
        	args[3] = NULL;
	}
	else
	{
		event.id = TRUNK_STATION_PICKUP;
		event.parm.trunk = this;
	}
	if(trk->postEvent(&event))
		return scrGoto();
	if(!trunkSignal(TRUNK_SIGNAL_FAIL))
		error("pickup-failed");
	return true;
}

#ifdef	HAVE_TGI
bool Trunk::scrLibexec(void)
{
	ScriptCommand *sc = getCommand();
	tgicmd_t cmd;
	unsigned argc = 0;
	const char *gain = getKeyword("gain");
	const char *speed = getKeyword("speed");
	const char *pitch = getKeyword("pitch");
	const char *zone = getKeyword("zone");
	Line *line = getScript();
	char query[sizeof(cmd.cmd) - 160];
	char urlbuf[sizeof(cmd.cmd) - 160];
#ifdef	HAVE_SSTREAM
	ostringstream str;
	str.str() = "";
#else
	strstream str(cmd.cmd, sizeof(cmd.cmd));
#endif
	unsigned qlen = 0;
	char *qc, *tag, *opt;
	const char *member = getMember();

	if(!zone)
		zone = getKeyword("channel");
	if(!zone)
		zone = "0";

	if(!member)
		member="tgi";

	if(!strnicmp(member, "one", 3))
	{
		if(!getOnce())
		{
			advance();
			return true;
		}
	}

	if(!stricmp(member, "play"))
		data.sleep.wakeup =  getSecTimeout(getValue(getSymbol(SYM_PLAYWAIT)));
	else
		data.sleep.wakeup = getTimeout("maxTime");
	data.sleep.count = 0;
	data.sleep.term = 0;
	data.sleep.rings = 0;
	data.sleep.loops = 1;
	data.sleep.save = NULL;

	cmd.seq = ++tgi.seq;
	snprintf(cmd.port, sizeof(cmd.port), "%s/%d", driver->getName(), id);
	cmd.mode = TGI_EXEC_NORMAL;
	cmd.cmd[0] = 0;
	query[0] = 0;

	opt = getValue("--");
	str << opt;
	while(NULL != (qc = getOption(NULL)) && qlen < sizeof(query))
	{
		if(*qc != '%')
			continue;

		if(!strnicmp(qc, "%lib.", 5))
			tag = qc + 5;
		else
			tag = qc + 1;

		if(qlen)
			query[qlen++] = *keyserver.getToken();

		qc = getSymbol(qc);
		if(!qc)			// ignore non-initialized...
			continue;

		urlbuf[0] = 0;
		urlEncode(qc, urlbuf, sizeof(urlbuf));
		snprintf(query + qlen, sizeof(query) - qlen, "%s=%s", tag, urlbuf);
		qlen = strlen(query);
	}

	while(argc < line->argc && qlen < sizeof(query))
	{
		opt = line->args[argc++];
		if(*opt != '=')
			continue;

		tag = opt + 1;
		opt = line->args[argc++];

		if(qlen)
			query[qlen++] = *keyserver.getToken();

		qc = getContent(opt);
		if(!qc)
			continue;

		urlbuf[0] = 0;
		urlEncode(qc, urlbuf, sizeof(urlbuf));
		snprintf(query + qlen, sizeof(query) - qlen, "%s=%s", tag, urlbuf);
		qlen = strlen(query);
	}

	if(sc != aascript)
		str << " virtual=" << sc->getLast("name");

	str << " query=" << query;

	if(dtmf.bin.data)
		str << " digits=" << dtmf.bin.data;

	qc = getSymbol(SYM_CALLER);
	if(qc)
		str << " clid=" << qc;

	qc = getSymbol(SYM_DIALED);
	if(qc)
		str << " dnid=" << qc;

	if(!data.sleep.wakeup)
		cmd.mode = TGI_EXEC_DETACH;

	if(!stricmp(member, "play"))
	{
		data.play.write = WRITE_MODE_NONE;
		data.play.voice = NULL;
		data.play.timeout = data.sleep.wakeup;
		data.play.repeat = 0;
		data.play.lock = false;
		data.play.name = data.play.list;
		data.play.term = 0;
		data.play.mode = PLAY_MODE_TEMP;
		data.play.limit = data.play.offset = 0;
		data.play.volume = atoi(getSymbol(SYM_VOLUME));
		data.play.extension = NULL;
		data.play.channel = atoi(zone);

		if(gain)
			data.play.gain = (float)strtod(gain, NULL);
		else
			data.play.gain = 0.0;

		if(!speed)
			speed = "normal";

		if(!stricmp(speed, "fast"))
			data.play.speed = SPEED_FAST;
		else if(!stricmp(speed, "slow"))
			data.play.speed = SPEED_SLOW;
		else
			data.play.speed = SPEED_NORMAL;

		if(pitch)
			data.play.pitch = (float)strtod(pitch, NULL);
		else
			data.play.pitch = 0.0;

		sprintf(data.play.list, "temp/.tmp.%d.%s",
			id, getSymbol(SYM_EXTENSION));
		str << " audio=" << data.play.name;
	}

#ifdef	HAVE_SSTREAM
	snprintf(cmd.cmd, sizeof(cmd.cmd), "%s", str.str().c_str());
#else
	str << ends;
#endif
	::write(tgipipe[1], &cmd, sizeof(cmd));

	if(!stricmp(member, "play"))
	{
		trunkStep(TRUNK_STEP_PLAYWAIT);
		return false;
	}

	if(!data.sleep.wakeup)
	{
		advance();
		return true;
	}

	trunkStep(TRUNK_STEP_SLEEP);
	return false;
}

#else

bool Trunk::scrLibexec(void)
{
	error("libexec-unsupported");
	return true;
}

#endif

bool Trunk::scrTone(void)
{
	const char *cp;

        data.tone.recall = false;
        data.tone.dialing = NULL;
        data.tone.tone = NULL;
        data.tone.freq1 = data.tone.freq2 = 0;
        data.tone.ampl1 = data.tone.ampl2 = 0;

	cp = getValue(NULL);
	if(cp)
	{
		data.tone.tone = getphTone(cp);
		if(!data.tone.tone)
		{
			error("no-tone");
			return true;
		}
	}
	else
	{
		cp = getKeyword("frequency");
		if(cp)
		{
			data.tone.freq1 = atoi(cp);
			cp = getKeyword("amplitude");
			if(cp)
				data.tone.ampl1 = atoi(cp);
		}
		else
		{
			cp = getKeyword("freq1");
			if(cp)
				data.tone.freq1 = atoi(cp);

			cp = getKeyword("ampl1");
			if(cp)
				data.tone.ampl1 = atoi(cp);

			cp = getKeyword("freq2");
			if(cp)
				data.tone.freq2 = atoi(cp);

			cp = getKeyword("ampl2");
			if(cp)
				data.tone.ampl2 = atoi(cp);
		}
	}

	cp = getKeyword("timeout");
	if(!cp)
		cp = getValue(NULL);
	if(cp)
		data.tone.wakeup = data.tone.duration = getSecTimeout(cp);
	else
	{
		data.tone.wakeup = data.tone.tone->getPlaytime();
		data.tone.duration = data.tone.tone->getDuration();
		if(!data.tone.wakeup)
			data.tone.wakeup = 1000;
	}

	cp = getKeyword("length");
	if(cp)
		data.tone.duration = getMSTimeout(cp);

	if(data.tone.duration > data.tone.wakeup)
		data.tone.duration = data.tone.wakeup;
		
	cp = getKeyword("count");
	if(!cp)
		cp = getValue("1");
	if(!data.tone.tone->getPlaytime())
	{
		if(atoi(cp) == 1)
			data.tone.duration = data.tone.wakeup = data.tone.tone->getDuration();
		else
			data.tone.duration = data.tone.wakeup = data.tone.wakeup * atoi(cp);
		cp = "1";
	}

	data.tone.loops = atoi(cp);
	trunkStep(TRUNK_STEP_TONE);
	return false;
}

bool Trunk::scrSleep(void)
{
	timeout_t timer = getTimeout("maxTime");
	const char *cp = getKeyword("count");

	if(!timer)
	{
		advance();
		return true;
	}


	data.sleep.count = 0;
	data.sleep.term = getDigitMask("exit");
	data.sleep.wakeup = timer;
	data.sleep.loops = 1;
	data.sleep.rings = 0;
	data.sleep.save = NULL;

	if(cp)
		data.sleep.count = atoi(cp);

	trunkStep(TRUNK_STEP_SLEEP);
	return false;
}

bool Trunk::getExitkey(unsigned short term, unsigned keycode)
{
	char buf[2];

	unsigned short mask = (1 << keycode);
	if(!(mask & term))
		return false;

	buf[1] = 0;
	buf[0] = digit[keycode];
	setSymbol(SYM_EXITKEY, buf);
	return true;
}

void Trunk::setExitkey(void)
{
	setSymbol(SYM_EXITKEY, "-");
	setDTMFDetect(true);
}

#ifdef	CCXX_NAMESPACES
}
#endif
