// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifdef	__FreeBSD__
#define	getopt(a, b, c) getopt()
#include <unistd.h>
#undef getopt
#endif

#include <cc++/config.h>
#include <cc++/process.h>
#include <cc++/url.h>
#ifdef	linux
#include <sys/vfs.h>
#endif
#include <sys/wait.h>
#include <sys/utsname.h>
#include <netinet/tcp.h>
#include <sys/un.h>
#include <getopt.h>
#include <cerrno>
#include "bayonneserver.h"
#include <iomanip>

#ifdef	HAVE_URL_CURL
#include <curl/curl.h>
#endif

#ifdef	HAVE_SYS_STATVFS_H
#include <sys/statvfs.h>
#define	statfs statvfs
#endif

#ifdef	HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#ifdef	HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif

#ifdef	DLL_SERVER
#define	PLUGIN_SUBDIR	".libs/"
#else
#define	PLUGIN_SUBDIR	""
#endif

#ifdef	_POSIX_MEMLOCK
#include <sys/mman.h>
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

static int *tgipids = NULL;
static char **restart_argv;
static char **cmds;
static char *cmd0;
unsigned len0;
static char *inits[128];
static unsigned icount = 0;
static int mainpid;

static int pidfile(char *fpath)
{
	int fd;
	pid_t pid;
	int status;
	char buffer[65];

	for(;;)
	{
		fd = open(fpath, O_WRONLY | O_CREAT | O_EXCL, 0660);
		if(fd > 0)
		{
			pid = getpid();
			snprintf(buffer, sizeof(buffer), "%d\n", pid);
			write(fd, buffer, strlen(buffer));
			close(fd);
			return 0;
		}
		if(fd < 0 && errno != EEXIST)
			return -1;

		fd = open(fpath, O_RDONLY);
		if(fd < 0)
		{
			if(errno == ENOENT)
				continue;
			return -1;
		}
		sleep(2);
		status = read(fd, buffer, sizeof(buffer) - 1);
		if(status < 1)
		{
			close(fd);
			continue;
		}

		buffer[status] = 0;
		pid = atoi(buffer);
		if(pid)
		{
			if(pid == getpid())
			{
				status = -1;
				errno = 0;
			}
			else
				status = kill(pid, 0);

			if(!status || (errno == EPERM))
			{
				close(fd);
				return pid;
			}
		}
		close(fd);
		unlink(fpath);
	}
}

#ifndef	HAVE_SETENV
static void setenv(char *name, const char *value, int overwrite)
{
	char strbuf[256];

	snprintf(strbuf, sizeof(strbuf), "%s=%s", name, value);
	putenv(strbuf);
}
#endif

static void purge(const char *path)
{
	if(!isDir(path))
		return;

	char fullpath[256];
	char *ex[4];
	int pid;
	Dir dir(path);
	const char *name;

	while(NULL != (name = dir.getName()))
	{
		if(!stricmp(name, "."))
			continue;

		if(!stricmp(name, ".."))
			continue;

		pid = fork();
		if(pid < 0)
			return;

		if(pid)
		{
			waitpid(pid, NULL, 0);
			continue;
		}

		snprintf(fullpath, sizeof(fullpath), "%s/%s", path, name);
		ex[0] = "rm";
		ex[1] = "-rf";
		ex[2] = fullpath;
		ex[3] = NULL;

		exit(execvp("rm", ex));
	}
	rmdir(path);
}

static void rt(void)
{
	int min, max;
//	int pages = keythreads.getPages();
	int pri = keythreads.getPriority();
	int policy = keythreads.getPolicy();

#ifdef	_POSIX_PRIORITY_SCHEDULING
	struct sched_param p;

	sched_getparam(0, &p);
	pri += p.sched_priority;
	if(!policy)
		policy = sched_getscheduler(0);

	min = sched_get_priority_min(policy);
	max = sched_get_priority_max(policy);
	if(pri < min)
		pri = min;
	if(pri > max)
		pri = max;
	p.sched_priority = pri;
	sched_setscheduler(getpid(), policy, &p);
#else
	nice(-pri);
#endif

#ifdef	MCI_CURRENT
	if(pages)
		mlockall(MCI_CURRENT | MCI_FUTURE);
#endif
}

#ifdef	HAVE_TGI

static RETSIGTYPE tgiusr1(int signo)
{
}

static void tgi(int cfd)
{
	int argc;
	char *argv[65];
	char *user, *project;
	char buffer[PIPE_BUF / 2];
	int vpid, pool, stat;
	int count = keythreads.getGateways();
	tgipids = new int[count];
	sigset_t sigs;
	struct sigaction act;
	tgicmd_t cmd;
	TGI *tgi;
	int status;
	char *cp;
	int len;
	char **arg;
	const char *prefix;
	const char *sp;
	unsigned buffers = PIPE_BUF;
	socklen_t blen = sizeof(buffers);

	if(canModify(keypaths.getRunfiles()))
	{
		setString(buffer, sizeof(buffer), keypaths.getRunfiles());
		addString(buffer, sizeof(buffer), "/bayonne");
		setenv("SERVER_CONTROL", buffer, 1);
		setString(buffer, sizeof(buffer), keypaths.getRunfiles());
		addString(buffer, sizeof(buffer), "/bayonne.ctrl");
	}
	else
	{
		snprintf(buffer, sizeof(buffer), "%s/.bayonne/", getenv("HOME"));
		mkdir(buffer, 0700);
		setenv("SERVER_CONTROL", buffer, 1);
		snprintf(buffer, sizeof(buffer), "%s/.bayonne/.ctrl", getenv("HOME"));
	}

#ifdef	USE_SOCKETPAIR
	socketpair(AF_UNIX, SOCK_STREAM, 0, tgipipe);
	shutdown(tgipipe[0], SHUT_WR);
	shutdown(tgipipe[1], SHUT_RD);
	buffers = atoi(keythreads.getLast("buffers")) * TGI_BUF;
	setsockopt(tgipipe[1], SOL_SOCKET, SO_SNDBUF, &buffers, blen);
	setsockopt(tgipipe[0], SOL_SOCKET, SO_RCVBUF, &buffers, blen);
	getsockopt(tgipipe[0], SOL_SOCKET, SO_RCVBUF, &buffers, &blen);
#else
	pipe(tgipipe);
#endif

	for(pool = 0; pool < count; ++pool)
	{
#ifdef	__FreeBSD__
		tgipids[pool] = vfork();
#else
		tgipids[pool] = fork();
#endif
		if(!tgipids[pool])
			break;
	}

	if(pool == count)
	{
		close(tgipipe[0]);
		return;
	}

	close(tgipipe[1]);

	arg = cmds;
	while(*arg)
	{
		len = strlen(*arg);
		memset(*arg, 0, len);
		++arg;
	}
	setString(cmd0, len0, "[tgi]");
	Process::setUser(keyserver.getLast("user"));
	Process::setGroup(keyserver.getLast("group"));
	sigemptyset(&sigs);
	sigaddset(&sigs, SIGINT);
	signal(SIGINT, SIG_DFL);
	sigprocmask(SIG_UNBLOCK, &sigs, NULL);

	memset(&act, 0, sizeof(act));
	act.sa_handler = &tgiusr1;
	act.sa_flags = SA_RESTART;
	sigemptyset(&act.sa_mask);
	sigaction(SIGUSR1, &act, NULL);

	slog.notice("tgi: initialized; uid=%d, pid=%d, buffers=%d", getuid(), getpid(), buffers / TGI_BUF);
	nice(-keythreads.priGateway());

	while(cfd > -1 && cfd < 3)
		cfd = dup(cfd);
	if(cfd < 0)
		slog.error() << "tgi: fifo failed; control=" << buffer << "; failure=" << strerror(errno) << endl;
	else
		slog.debug() << "tgi: control=" << buffer << "; cfd=" << cfd << "; time=" << count * 10 << endl;

	while(read(tgipipe[0], &cmd, sizeof(cmd)) == sizeof(cmd))
	{
		argc = 0;
		slog.debug() << "tgi: cmd=" << cmd.cmd << endl;
		if(cmd.mode == TGI_EXEC_IMMEDIATE)
			tgi = getInterp(cmd.cmd);
		else
			tgi = NULL;
		if(tgi)
		{
			status = tgi->parse(cfd, cmd.port, cmd.cmd);
			continue;
		}
		vpid = vfork();
		if(vpid)
		{
			if(cmd.mode != TGI_EXEC_DETACH)
			{
				sprintf(buffer, "wait %s %d %d\n",
					cmd.port, vpid, cmd.seq);
				write(cfd, buffer, strlen(buffer));
				if(keythreads.getAudit())
					slog.debug("thread: gateway waiting on pid ", vpid);
			}
			else
			{
				if(keythreads.getAudit())
					slog.debug("thread: gateway detached pid ", vpid);
			}

			if(cmd.mode != TGI_EXEC_DETACH)
			{
#ifdef	__FreeBSD__
				wait4(vpid, &stat, 0, NULL);
#else
				waitpid(vpid, &stat, 0);
#endif
				if(keythreads.getAudit())
					slog.debug("thread: gateway exiting pid ", vpid);
				sprintf(buffer, "exit %s %d %d\n", cmd.port, WEXITSTATUS(stat), cmd.seq);
				write(cfd, buffer, strlen(buffer));
			}
			else
#ifdef	__FreeBSD__
				wait4(vpid, &stat, 0, NULL);
#else
				waitpid(vpid, &stat, 0);
#endif
			continue;
		}

		if(cmd.mode == TGI_EXEC_DETACH)
		{
                	Process::detach();
			setString(cmd0, len0, "-detach");
		}
		else
			snprintf(cmd0, len0, "tgi:%s", cmd.port);

		project = NULL;
		user = NULL;
		//sprintf(buffer, "%d", cmd.port);
		setenv("PORT_NUMBER", cmd.port, 1);
		argv[argc++] = strtok(cmd.cmd, " \t");
		while(NULL != (argv[argc] = strtok(NULL, " \t")))
		{
			/* Used for standard TGI execution */

			if(!strnicmp(argv[argc], "user=", 5))
			{
				user = argv[argc] + 5;
				setenv("PORT_USER", user, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "virtual=", 8))
			{
				project = argv[argc] + 8;
				setenv("PORT_VIRTUAL", project, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "digits=", 7))
			{
				setenv("PORT_DIGITS", argv[argc] + 7, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "dnid=", 5))
			{
				setenv("PORT_DNID", argv[argc] + 5, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "clid=", 5))
			{
				setenv("PORT_CLID", argv[argc] + 5, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "query=", 6))
			{
				setenv("PORT_QUERY", argv[argc] + 6, 1);
				continue;
			}

			// Added for TTS audio format selection

			if(!strnicmp(argv[argc], "format=", 6))
			{
				setenv("TTS_FORMAT", argv[argc] + 6, 1);
				continue;
			}

			// Added for TTS audio file type and ASR filename

			if(!strnicmp(argv[argc], "audio=", 6))
			{
				if(cmd.mode == TGI_EXEC_AUDIO)
				{
					remove(argv[argc] + 6);
					setenv("TTS_AUDIO", argv[argc] + 6, 1);
				}
				else
					setenv("ASR_AUDIO", argv[argc] + 6, 1);
				setenv("TEMP_AUDIO", argv[argc] + 6, 1);
				continue;
			}

			// Added for TTS/ASR language specification

			if(!strnicmp(argv[argc], "language=", 9))
			{
				setenv("TTS_LANGUAGE", argv[argc] + 9, 1);
				setenv("ASR_LANGUAGE", argv[argc] + 9, 1);
				continue;
			}

			// Added for TTS/ASR voice domain selection

			if(!strnicmp(argv[argc], "voice=", 6))
			{
				setenv("TTS_VOICE", argv[argc] + 6, 1);
				setenv("ASR_VOICE", argv[argc] + 6, 1);
				continue;
			}

			// Added for domain vocabulary selection

			if(!strnicmp(argv[argc], "domain=", 7))
			{
				setenv("ASR_DOMAIN", argv[argc] + 7, 1);
				continue;
			}

			// Added for ASR event trapping

			if(!strnicmp(argv[argc], "asrevt=", 7))
			{
				setenv("ASR_EVENTS", argv[argc] + 7, 1);
				continue;
			}

			// Added for passing TTS phrases

			if(!strnicmp(argv[argc], "phrase=", 7))
			{
				while(NULL != (cp = strchr(argv[argc], '+')))
					*cp = ' ';
				while(NULL != (cp = strchr(argv[argc], ',')))
					*cp = ' ';
				setenv("TTS_PHRASE", argv[argc] + 7, 1);
				continue;
			}

			// Added for passing TTS source if file based

			if(!strnicmp(argv[argc], "source=", 7))
			{
				setenv("TTS_SOURCE", argv[argc] + 7, 1);
				continue;
			} 

			// Added for url helpers and such
			if(!strnicmp(argv[argc], "href=", 5))
			{
				setenv("URL_HREF", argv[argc] + 5, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "base=", 5))
			{
				setenv("URL_BASE", argv[argc] + 5, 1);
				continue;
			}

			if(!strnicmp(argv[argc], "url=", 4))
			{
				setenv("URL_SOURCE", argv[argc] + 4, 1);
				continue;
			}

			// Added for passing TTS cache path

			if(!strnicmp(argv[argc], "cache=", 6))
			{
				setenv("TTS_CACHE", argv[argc] + 6, 1);
				continue;
			}

			++argc;
		}
		argv[argc] = NULL;
//		slog.close();
		close(0);
		close(1);
//		close(2);
		open("/dev/null", O_RDWR);
		if(cfd > 1)
		{
			dup2(cfd, 1);
			close(cfd);
		}

		prefix = getenv("SERVER_LIBEXEC");
		project = getenv("PORT_VIRTUAL");
		if(project)
		{
			snprintf(buffer, sizeof(buffer), "/virtual/%s", project);
			Keydata keyproj(buffer);
			sp = keyproj.getLast("libexec");
			if(sp)
				prefix = sp;
			sp = keyproj.getLast("path");
			if(sp)
				setenv("PATH", sp, 1);
			else
			{
				snprintf(buffer, sizeof(buffer),
					"%s:%s", getenv("PATH"), prefix);
				setenv("PATH", buffer, 1);
			}

			sp = keyproj.getLast("datafiles");
			if(sp)
				chdir(sp);
		}
		snprintf(buffer, sizeof(buffer), "%s/%s", prefix, *argv);
		getInterp(buffer, argv);

		slog.debug() << "tgi: exec " << buffer << endl;
		execvp(buffer, argv);
		slog.error() << "tgi: exec failed; " << buffer << endl;
		exit(-1);
	}
	exit(SIGINT);
}

/*
static void tgisignal(void)
{
	int count = keythreads.getGateways();
	int pid;

	if(!tgipids)
		return;

	for(pid = 0; pid < count; ++pid)
		kill(tgipids[pid], SIGUSR1);
}
*/

#endif
		
static RETSIGTYPE final(int sig)
{
	int count = keythreads.getGateways();
	int i;
	const char *cp;

        if(sig)
                errlog("failed", "Bayonne exiting; reason=%d", sig);
        else
                errlog("notice", "Bayonne exiting; normal termination");

	if(debug)
		if(debug->debugFinal(sig))
			return;

	if(getpid() != mainpid)
	{
		kill(mainpid, sig);
		Thread::sleep(~0);
	}
	signal(SIGINT, SIG_IGN);
	signal(SIGABRT, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);

	stopServers();

	if(Driver::drvFirst)
	{
		Driver *drv = Driver::drvFirst;
		while(drv)
		{
			drv->stop();
			drv = drv->drvNext;
		}
	}

//	if(sig)
//		slog.warn("exiting: reason=%d", sig);
//	else
//		slog.notice() << "normal shutdown" << endl;

#ifdef	NODE_SERVICES
	if(network)
	{
		network->stop();
		delete network;
		network = NULL;
	}
#endif

	if(tgipids)
	{
		for(i = 0; i < count; ++i)
		{
			kill(tgipids[i], SIGINT);
			waitpid(tgipids[i], NULL, 0);
		}
		tgipids = NULL;
	}

	if(restart_server)
		execvp(*restart_argv, restart_argv);

	cp = keypaths.getLast("pidfile");
	if(cp)
		remove(cp);

	cp = keypaths.getLast("ctrlfile");
	if(cp)
		remove(cp);

	cp = keypaths.getLast("pktfile");
	if(cp)
		remove(cp);

	cp = keypaths.getLast("nodefile");
	if(cp)
		remove(cp);

#ifdef	HAVE_URL_CURL
	curl_global_cleanup();
#endif

	purge(keypaths.getLast("tmpfs"));
	purge(keypaths.getLast("tmp"));
	exit(sig);
}

static void initial(int argc, char **argv, int cfd)
{
	static struct option long_options[] = {
		{"background", 0, 0, 'D'},
		{"foreground", 0, 0, 'F'},
		{"daemon", 0, 0, 'D'},
		{"help", 0, 0, 'h'},
		{"priority", 1, 0, 'p'},
		{"port", 1, 0, 'P'},
		{"driver", 1, 0, 'd'},
		{"node", 1, 0, 'n'},
		{"test", 0, 0, 't'},
		{"trace", 0, 0, 'T'},
		{"thread", 0, 0, 'A'},
		{"version", 0, 0, 'V'},
		{"voice", 1, 0, 'v'},
		{"language", 1, 0, 'l'},
		{"gui", 0, 0, 'G'},
		{"display", 1, 0, 'X'},
		{"startup", 1, 0, 'S'},
		{"debug", 0, 0, 'x'},
		{"demo", 0, 0, 'c'},
		{"init", 0, 0, 'I'},
		{"groups", 0, 0, 'g'},
		{"tts", 1, 0, 'Y'},
		{"prefix", 1, 0, 'z'},
		{"loglevel", 1, 0, 'L'},
		{0, 0, 0, 0}};

	static bool daemon = false;
	static bool usage = false;
	static bool gui = false;
	static unsigned plen = 0, ps;

	ScriptSymbol *globals = Trunk::getGlobals();
	struct utsname uts;
	TrunkGroup *policy;
	restart_argv = argv;
	char *envDriver = getenv("BAYONNE_DRIVER");
	char *envDialing = getenv("BAYONNE_DIALPLAN");
	fstream mix;
	char prefix[256], libpath[256];
	const char *cp;
	char *pp, *tok;
	sigset_t sigs;
	int opt, opt_index;
	unsigned ports;
	bool test = false;
	bool schedule = true;
	bool gdump = false;
	bool pref = false;
	char *sp = NULL;
	Driver *drv;

	cmds = argv;
	cmd0 = *cmds;
	len0 = strlen(cmd0);

	mainpid = getpid();
	sigemptyset(&sigs);
	sigaddset(&sigs, SIGTERM);
	sigaddset(&sigs, SIGQUIT);
	sigaddset(&sigs, SIGINT);
	pthread_sigmask(SIG_BLOCK, &sigs, NULL);
	slog.level(Slog::levelNotice);

	setString(prefix, sizeof(prefix), argv[0]);
	pp = strrchr(prefix, '/');
        if(pp)
        {
        	*pp = 0;
                chdir(prefix);
        }

	uname(&uts);

	getcwd(prefix, sizeof(prefix) - 1);
	pp = strrchr(prefix, '/');
	ps = prefix + sizeof(prefix) - pp;

//	keypaths.setValue("modlibpath", PATH_LIBDIR_BAYONNE);

	if(envDriver)
		plugins.setValue("drivers", envDriver);

	if(envDialing)
		keyserver.setValue("dialplan", envDialing);

	while(EOF != (opt = getopt_long(argc, argv, "L:z:cthVxP:GTDFX:d:p:d:n:AS:s:v:l:gY:I", long_options, &opt_index)))
		switch(opt)
		{
		case 'z':
			sp = optarg;
			break;
		case 'x':
			slog.level(Slog::levelDebug);
			break;
		case 'I':
			pref = true;
			break;
		case 'g':
			gdump = true;
			break;
		case 'v':
			setenv("BAYONNE_VOICE", optarg, 1);
			break;
		case 'Y':
			plugins.setValue("tts", optarg);
			break;
		case 'l':
			setenv("BAYONNE_LANGUAGE", optarg, 1);
			break;
		case 'X':
			setenv("DISPLAY", optarg, 1);
		case 'G':
			gui = true;
			plugins.setValue("debug", "gui");
			break;
		case 'P':
			plugins.setValue("debug", "tcpmon");
			setenv("TCPMON", optarg, 1);
			break;
		case 'S':
			inits[icount++] = optarg;
			break;
		case 'V':
			cout << VERSION << endl;
			exit(0);
		case 'A':
			keythreads.setValue("audit", "1");
			slog.level(Slog::levelDebug);
			break;
		case 'T':
			daemon = false;
			slog.level(Slog::levelDebug);
			plugins.setValue("debug", "trace");
			break;
		case 't':
			schedule = false;
			test = true;
			daemon = false;
			slog.level(Slog::levelDebug);
			cp = plugins.getLast("drivers");

			if(envDriver)
				plugins.setValue("drivers", envDriver);
			plugins.setValue("languages", "english");
			plugins.setValue("switch", "");

#ifdef	HAVE_THETA
			plugins.setValue("tts", "theta");
#elif	HAVE_FLITE
			plugins.setValue("tts", "flite");
#endif

#if defined(HAVE_POSTGRES) || defined(HAVE_PGSQL_POSTGRES)
			plugins.setValue("sql", "postgres");
#endif

#ifdef	COMMON_XML_PARSING
			plugins.setValue("xml", "bayonne");
#endif

//			strcpy(pp, "/modules/perl/" PLUGIN_SUBDIR "perl.tgi");
//			plugins.setValue("tgi", prefix);
			if(gui)
				plugins.setValue("debug", "gui");
			else
				plugins.setValue("debug", "trace");

			keypaths.setValue("cache", "../var/cache");
			keypaths.setValue("spool", "../var/spool");
			setString(pp, ps, "/data/script");
			keypaths.setValue("scripts", prefix);
			setString(pp, ps, "/data/sys");
			keypaths.setValue("prompts", prefix);
			setString(pp, ps, "/data");
			keypaths.setValue("voices", prefix);
			setString(pp, ps, "/data/libexec");
			keypaths.setValue("libexec", prefix);
			keypaths.setValue("datafiles", "../var");
			keypaths.setValue("logpath", "../var/log");
			break;
		case 'n':
			keyserver.setValue("node", optarg);
			break;
		case 'c':
			optarg = "oss";
			daemon = false;
		case 'd':
			if(*optarg != '/')
				envDriver = optarg;

			plugins.setValue("drivers", optarg);
			break;
		case 'p':
			keythreads.setValue("priority", optarg);
			break;
		case 'D':
			daemon = true;
			break;
		case 'F':
			daemon = false;
			break;
		case 'L':
                        if (!strcmp(optarg, "debug")) {
                          slog.level(Slog::levelDebug);
                        } else if (!strcmp(optarg, "info")) {
			  slog.level(Slog::levelInfo);
                        } else if (!strcmp(optarg, "notice")) {
			  slog.level(Slog::levelNotice);
			} else if (!strcmp(optarg, "warning")) {
			  slog.level(Slog::levelWarning);
			} else if (!strcmp(optarg, "error")) {
			  slog.level(Slog::levelError);
			} else if (!strcmp(optarg, "critical")) {
			  slog.level(Slog::levelCritical);
			} else if (!strcmp(optarg, "alert")) {
			  slog.level(Slog::levelAlert);
			} else if (!strcmp(optarg, "emergency")) {
			  slog.level(Slog::levelEmergency);
			} else {
			  cout << "Unknown log level. Must be one of debug, " <<
			    "info, notice, warning, error, critical, alert and emergency" << endl;
			  exit(-1);
			}
			break;
		default:
		case 'h':
			usage = true;
		}

	if(optind < argc)
	{
		if(test && !gui)
			plugins.setValue("debug", "trace");
		if(test)
			keyserver.setValue("testing", argv[optind++]);
		else
			keyserver.setValue("default", argv[optind++]);
		schedule = false;
	}

	if(usage || optind < argc)
	{
		clog << "use: bayonne [-options] [defscript]" << endl;
		exit(-1);
	}

	endKeydata();
	if(sp)
		keypaths.setPrefix(sp);

	if(!Process::setGroup(keyserver.getLast("group")) && !getuid())
		errlog("failed", "group=%s; cannot change", keyserver.getLast("group"));
                
	if(!getuid())
		umask(003);

	mkdir(keypaths.getDatafiles(), 0770);
        if(!isDir(keypaths.getRunfiles()))
		mkdir(keypaths.getRunfiles(), 0770);
	
	cp = keypaths.getLast("userdata");
	if(cp && !getuid())
		mkdir(cp, 0770);
	mkdir(keypaths.getLogpath(), 0770);
	chdir(keypaths.getDatafiles());
	purge("temp");
	purge(keypaths.getSpool());
	purge(keypaths.getCache());
	purge(keypaths.getLast("tmpfs"));
	purge(keypaths.getLast("tmp"));

	mkdir(keypaths.getLast("tmpfs"), 0770);
	mkdir(keypaths.getLast("tmp"), 0770);
	mkdir("temp", 0770);
	mkdir(keypaths.getCache(), 0770);
	symlink(keypaths.getCache(), "cache");
	mkdir(keypaths.getSpool(), 0770);
	symlink(keypaths.getSpool(), "spool");

        if(!getuid() && !test)
                if(!Process::setUser(keyserver.getLast("user")))
                        errlog("failed", "user=%s; cannot change", keyserver.getLast("user"));

	if(test)
		cp = NULL;
	else
	{
		cp = Process::getEnv("BAYONNE_LIBRARY");
		if(!cp || !*cp)
			cp = keypaths.getLast("libpath");
	}

	if(cp && *cp)
	{
		Process::setEnv("BAYONNE_LIBRARY", cp, true);
		snprintf(libpath, sizeof(libpath), "%s/perl5", cp);
		Process::setEnv("PERL5LIB", libpath, true);
		snprintf(libpath, sizeof(libpath), "%s/python", cp);
		Process::setEnv("PYTHONPATH", libpath, true);
	}
	else
	{
		Process::setEnv("BAYONNE_LIBRARY", "", true);
		Process::setEnv("PERL5LIB", "../modules/perl5", true);
		Process::setEnv("PYTHONPATH", "../modules/python", true);
	}

	if(daemon)
	{
		Process::detach();
		mainpid = getpid();
		slog.open("bayonne", Slog::classDaemon);
		slog.notice() << "daemon mode started" << endl;
	}
	else if(getppid() == 1)
	{
		close(0);
		close(1);
		close(2);
		open("/dev/null", O_RDWR);
		open("/dev/null", O_RDWR);
		open("/dev/null", O_RDWR);
		slog.open("bayonne", Slog::classDaemon);
		slog.notice() << "daemon init started" << endl;
	}

        if(canModify(keypaths.getRunfiles()))
                snprintf(prefix, sizeof(prefix), "%s/bayonne.routes",
                        keypaths.getRunfiles());
        else
                snprintf(prefix, sizeof(prefix), "%s/.bayonne/.routes",
                        Process::getEnv("HOME"));

	keypaths.setValue("routes", prefix);

        if(canModify(keypaths.getRunfiles()))
                snprintf(prefix, sizeof(prefix), "%s/.bayonne.routes.tmp",
                        keypaths.getRunfiles());
        else
                snprintf(prefix, sizeof(prefix), "%s/.bayonne/.routes.tmp",
                        Process::getEnv("HOME"));

        keypaths.setValue("tmproutes", prefix);


	if(canModify(keypaths.getRunfiles()))
		snprintf(prefix, sizeof(prefix), "%s/bayonne.pid",
			keypaths.getRunfiles());
	else
		snprintf(prefix, sizeof(prefix), "%s/.bayonne/.pid",
			Process::getEnv("HOME"));

	keypaths.setValue("pidfile", prefix);

	switch(pidfile(prefix))
	{
	case -1:
		slog.warn() << "server: cannot create pidfile " << prefix << endl;
	case 0:
		break;
	default:
		slog.critical() << "server: another instance running; cannot continue" << endl;
		final(-1);
	}

	setenv("SERVER_PLATFORM", plugins.getLast("drivers"), 1);
	setenv("SERVER_LIBEXEC", keypaths.getLibexec(), 1);
	setenv("SERVER_SOFTWARE", "bayonne", 1);
	setenv("SERVER_PROTOCOL", "3.0", 1);
	setenv("SERVER_VERSION", VERSION, 1);
	setenv("SERVER_TOKEN", keyserver.getToken(), 1);
	setenv("PATH", keypaths.getTgipath(), 1);

	slog.info() << "SERVER VERSION " << VERSION << "; ";
	slog() << uts.machine << " ";
	slog() << uts.sysname << " " << uts.release << endl;
	slog.info() << "TGI VERSION 3.0";
	slog() << "; driver(s)=" << plugins.getLast("drivers");
	slog() << "; etc=" << keypaths.getLast("etc") << endl;

	slog.debug() << "Loading TGI plugins..." << endl;
	plugins.loadTGI();

#ifdef	HAVE_TGI
	tgi(cfd);
	sleep(1);
#endif
	rt();

	slog.debug() << "Loading DSO plugin images..." << endl;
	plugins.loadDebug();
	plugins.loadDriver();
	plugins.loadDatabase();
	plugins.loadExtensions();
	plugins.loadModules();
	plugins.loadCodecs();
	plugins.loadTranslators();
	plugins.loadTTS();
	keyserver.loadGroups(test);

	initScripting();
	ScriptModule::init();

	sync();
	drv = Driver::drvFirst;
	ports = 0;
	new MappedUsage();
	while(drv)
	{
		slog.debug() << "Starting " << drv->getName() << " driver..." << endl;
		ports += drv->start();
		drv = drv->drvNext;
	}
	if(ports)
		slog.info("drivers started for %d port(s)", ports);
	else
	{
		slog.critical() << "no trunk ports activated" << endl;
		final(-1);
	}
	policy = TrunkGroup::getGroup(NULL);

	startServers();

#ifdef	NODE_SERVICES
	network = new Network();
	network->start();
#endif

	signal(SIGTERM, final);
	signal(SIGQUIT, final);
	signal(SIGINT, final);
	signal(SIGABRT, final);
	pthread_sigmask(SIG_UNBLOCK, &sigs, NULL);

	signal(SIGPIPE, SIG_IGN);
	sigemptyset(&sigs);
	sigaddset(&sigs, SIGPIPE);
	pthread_sigmask(SIG_BLOCK, &sigs, NULL);

	slog("bayonne", Slog::classDaemon);
	if(keyserver.getLast("config"))
		slog.notice() << "normal startup; " << keyserver.getLast("config") << endl;
	else
		slog.notice() << "normal startup" << endl;

	if(policy->getLast("groups"))
		setString(prefix, sizeof(prefix), policy->getLast("groups"));
	else
		setString(prefix, sizeof(prefix), "*");

	char pbuf[256];
	pbuf[0] = 0;
	pp = strtok_r(prefix, " ,;\t\n", &tok);
	while(pp && plen < sizeof(pbuf))
	{
		if(plen)
			pbuf[plen++] = ',';
		setString(pbuf + plen, sizeof(pbuf) - plen, pp);
		plen = strlen(pbuf);
		pp = strtok_r(NULL, " ,;\t\n", &tok);
	}

	globals->setConst(SYM_POLICIES, pbuf);

	unsigned total = 0;

	drv = Driver::drvFirst;
	while(drv)
	{
		total += drv->getTrunkCount();
		drv = drv->drvNext;
	}
	sprintf(prefix, "%d", total);
	globals->setConst(SYM_PORTS, prefix);
	globals->setConst(SYM_USER, keyserver.getLast("user"));
	globals->setConst(SYM_VERSION, VERSION);
	globals->setConst(SYM_SERVER, "bayonne");
	globals->setConst(SYM_NODE, keyserver.getNode());
	globals->setConst(SYM_SCRIPTS, keypaths.getScriptFiles());
	globals->setConst(SYM_PROMPTS, keypaths.getPromptFiles());
	globals->setConst(SYM_RELEASE, "2");
	globals->setSymbol(SYM_SERVICE, 64);
	if(test)
		globals->setSymbol(SYM_SERVICE, "test");
	else
		globals->setSymbol(SYM_SERVICE, "up");
	globals->setSymbol(SYM_SCHEDULE, 64);
	globals->setSymbol(SYM_SCHEDULE, "none");

	new KeyTones();
	new MappedCalls();
	Driver::setNodes();
	keyserver.printRoutes();

#ifdef	HAVE_URL_CURL
	curl_global_init(CURL_GLOBAL_ALL);
#endif

	running = true;
}

void check(void)
{
	Dir dir(keypaths.getCache());
	const char *entry;
	char path[128];
	time_t now;
	struct stat ino;

	slog.debug() << "server: checking cache..." << endl;

	time(&now);

	while(NULL != (entry = dir.getName()))
	{
		if(*entry == '.')
			continue;

		snprintf(path, sizeof(path), "cache/%s", entry);
		if(stat(path, &ino))
			continue;

		if(now - ino.st_mtime >= 3600)
		{
			cachelock.writeLock();
			remove(path);
			cachelock.unlock();
		}
	}
}

#define	PROTOCOL_VERSION	3

static int so = -1;
static char packet[512];

static void packetio(void)
{
	Trunk *trunk;
	char *p = packet + 2;
	struct sockaddr_un caddr;
	socklen_t clen = sizeof(caddr);
	int len = ::recvfrom(so, packet, sizeof(packet), 0, (struct sockaddr *)&caddr, &clen); 
	const char *cp;

	if(len < 0)
	{
		slog.warn() << "packet: invalid read" << endl;
		return;
	}

	while(isspace(*p))
		++p;
	packet[1] = '-';
	trunk = fifo.command(p);

	if(!packet[0])
		return;

	packet[2] = 0;

	if(trunk)
		packet[1] = '1';
	else
		packet[1] = '0';

	cp = NULL;
	if(trunk && trunk != ((Trunk *)(-1)))
		cp = trunk->getTrunkGid();

	if(cp)
		cp = strchr(cp, '-');
	if(cp && packet[1] == '1')
	{
		packet[1] = 'S';
		snprintf(packet + 2, 64, "session.trunkid=%s", cp);
	}

	clen = sizeof(caddr);
	len = ::sendto(so, packet, strlen(packet) + 1, 0, (struct sockaddr *)&caddr, clen);	
}

void setReply(const char *sym, const char *msg)
{
	if(packet[1] != '-')
		return;

	packet[1] = 'S';
	if(!msg)
		msg = "";
	snprintf(packet + 2, sizeof(packet) - 2, "%s=%s", sym, msg);
}

extern "C" int main(int argc, char **argv)
{
	static char buffer[PIPE_BUF / 2];
	static char packet[256];
	struct sockaddr_un server_addr;
	long total, used;
	struct statfs fs;
	char *p;
	unsigned bpos = 0;
#ifdef	USE_POLL
	struct pollfd pfd[2];
#else
	struct timeval tv;
	int maxfd;
	fd_set sfd, efd;
#endif
	TimerPort timer;
	timeout_t step;
	unsigned ic = 0;
	unsigned seccount = 0, mincount = 0, minhour = 0;
	ifstream init;
#ifndef	NODE_SERVICES
	statnode_t node;
	static char nodepath[256];
	int nodes;
	TrunkGroup *grp;
	const char *service;
	ScriptSymbol *globals = Trunk::getGlobals();
#endif
	Driver *drv;
	int len, fd;
	int statinterval = atoi(keyserver.getLast("stats"));

	if(canModify(keypaths.getRunfiles()))
	{
		snprintf(buffer, sizeof(buffer), "%s/bayonne.drivers",
			keypaths.getRunfiles());
		keypaths.setValue("drvmap", buffer);
		snprintf(buffer, sizeof(buffer), "%s/bayonne.asr",
			keypaths.getRunfiles());
		keypaths.setValue("asr", buffer);
		snprintf(buffer, sizeof(buffer), "%s/bayonne.stats",
			keypaths.getRunfiles());
		keypaths.setValue("stats", buffer);
		snprintf(buffer, sizeof(buffer), "%s/bayonne.calls",
			keypaths.getRunfiles());
		keypaths.setValue("calls", buffer);
		snprintf(buffer, sizeof(buffer), "%s/bayonne.usage",
			keypaths.getRunfiles());
		keypaths.setValue("usage", buffer);
		snprintf(buffer, sizeof(buffer), "%s/bayonne.ctrl",
			keypaths.getRunfiles());
		snprintf(packet, sizeof(packet), "%s/bayonne.packet",
			keypaths.getRunfiles());
#ifndef	NODE_SERVICES
		snprintf(nodepath, sizeof(nodepath), "%s/bayonne.nodes",
			keypaths.getRunfiles());
#endif
	}
	else
	{
		snprintf(buffer, sizeof(buffer), "%s/.bayonne", getenv("HOME"));
		mkdir(buffer, 0700);
		snprintf(buffer, sizeof(buffer), "%s/.bayonne/.drivers",
			getenv("HOME"));
		keypaths.setValue("drvmap", buffer);
		snprintf(buffer, sizeof(buffer), "%s/.bayonne/.asr",
			getenv("HOME"));
		keypaths.setValue("asr", buffer);
                snprintf(buffer, sizeof(buffer), "%s/.bayonne/.usage",
                        getenv("HOME"));
                keypaths.setValue("usage", buffer);
		snprintf(buffer, sizeof(buffer), "%s/.bayonne/.stats",
			getenv("HOME"));
		keypaths.setValue("stats", buffer);
		snprintf(buffer, sizeof(buffer), "%s/.bayonne/.calls",
			getenv("HOME"));
		keypaths.setValue("calls", buffer);
		snprintf(buffer, sizeof(buffer), "%s/.bayonne/.ctrl",
			getenv("HOME"));
		snprintf(packet, sizeof(packet), "%s/.bayonne/.packet",
			getenv("HOME"));

#ifndef	NODE_SERVICES
		snprintf(nodepath, sizeof(nodepath), "%s/.bayonne/.nodes",
			keypaths.getRunfiles());
#endif
	}

	keypaths.setValue("ctrlfile", buffer);
	keypaths.setValue("pktfile", packet);

	if(!getuid())
		umask(003);

	memset(&server_addr, 0, sizeof(server_addr));
	server_addr.sun_family = AF_UNIX;
	strncpy(server_addr.sun_path, packet, sizeof(server_addr.sun_path));

#ifdef  __SUN_LEN
        len = sizeof(server_addr.sun_len) + strlen(server_addr.sun_path)
			+ sizeof(addr.sun_family) + 1;

        addr.sun_len = len;
#else
        len = strlen(server_addr.sun_path) + sizeof(server_addr.sun_family) + 1;
#endif
	remove(packet);
	so = socket(AF_UNIX, SOCK_DGRAM, 0);
	if(so > -1)
	{
		if(bind(so, (struct sockaddr *)&server_addr, len))
		{
			slog.error() << "packet interface failed; path=" << packet << endl;
			so = -1;
			remove(packet);
		}
	}

	remove(buffer);
	mkfifo(buffer, 0660);
	fd = open(buffer, O_RDWR);
	fifo.setControl(fd);

	initial(argc, argv, fd);

	slog.debug() << "fifo: path=" << buffer << endl;
	if(fd < 0)
	{
		errlog("failed", "fifo access; path=%s", buffer);
		slog.warn() << "fifo: open failed" << endl;
		sleep((unsigned)(-1));
	}

#ifndef	NODE_SERVICES
	nodes = creat(nodepath, 0640);
	grp = TrunkGroup::getGroup();
	service = globals->getSymbol(SYM_SERVICE);
#endif

	while(ic < icount)
		fifo.command(inits[ic++]);

	snprintf(buffer, sizeof(buffer), "%s/startup.conf",
		keypaths.getLast("etc"));

	init.open(buffer);
	if(init.is_open())
	{
		while(!init.eof())
		{
			init.getline(buffer, sizeof(buffer));
			p = buffer + strlen(buffer) - 1;
			while(isspace(*p) && p >= buffer)
			{
				*(p--) = 0;
			}
			p = buffer;
			while(isspace(*p))
				++p;

			if(!isalpha(*p))
				continue;

			fifo.command(p);
		}
		init.close();
	}

	MappedStats stats;

	errlog("notice", "Bayonne/%s running", VERSION);
	timer.setTimer(1000);
	for(;;)
	{
		Thread::yield();
#ifdef	USE_POLL
		pfd[0].fd = fd;
		pfd[0].events = POLLIN | POLLRDNORM;
		pfd[0].revents = 0;
		if(so > -1)
		{
			pfd[1].fd = so;
			pfd[1].events = POLLIN | POLLRDNORM;
			pfd[1].revents = 0;
		}
#else
		maxfd = fd + 1;
		FD_ZERO(&efd);
		FD_ZERO(&sfd);
		FD_SET(fd, &sfd);
		if(so > -1)
			FD_SET(so, &sfd);
		if(so >= maxfd)
			maxfd = so + 1;
#endif
		step = timer.getTimer();
		if(!step)
		{
			timer.setTimer(1000);
			drv = Driver::drvFirst;
			while(drv)
			{
				drv->secTick();
				drv = drv->drvNext;
			}

#ifndef	NODE_SERVICES
			if(!(seccount % 10) && nodes > -1)
			{
				lseek(nodes, 0l, SEEK_SET);
				memset(&node, 0, sizeof(node));
	                        node.version = PROTOCOL_VERSION;
                        	node.buddies = 0;
				setString(node.name, sizeof(node.name), keyserver.getNode());
				node.ports = Driver::getCount();
	                        if(!strnicmp(service, "test::", 6))
        	                        node.service = 't';
                	        else if(service[0])
                        	        node.service = 'd';
                        	else
                                	node.service = 'u';
				node.dialing = 0;
				snprintf(node.schedule, sizeof(node.schedule), "none");
	                       	node.uptime = grp->getStat(STAT_SYS_UPTIME);
                        	node.calls = grp->getStat(STAT_SYS_ACTIVITY);
                        	Driver::getStatus(node.stat);
				time(&node.update);
				::write(nodes, &node, sizeof(node));
			}
#endif

			if(!seccount && !statfs(keypaths.getLast("datafiles"), &fs))
			{
				total = fs.f_blocks / 100;
				used = fs.f_blocks - fs.f_bfree;
				snprintf(df_used, sizeof(df_used),
					"%ld", used / total);
				snprintf(df_free, sizeof(df_free),
					"%ld", fs.f_bavail / total);
			}

			if(!(seccount % statinterval))
			{
				stats.scan();
				if(usage)
					usage->scan();
			}

			if(++seccount >= 60)
			{
				if(usage)
					usage->report();
				TrunkGroup::logStats();
				Session::clean();
				Sync::check();
				seccount = 0;
				if(++mincount >= 10)
				{
					check();
					mincount = 0;
				}
				if(++minhour >= 60)
					minhour = 0;
			}
		}

#ifdef	USE_POLL
		len = 1;
		if(so > -1)
			len = 2;
		poll(pfd, len, step);
		if(so > -1 && pfd[1].revents & POLLIN)
			packetio();
		if(pfd[0].revents & POLLIN)
                {
#else
		tv.tv_sec = step / 1000;
		tv.tv_usec = (step % 1000) * 1000;
		select(maxfd, &sfd, NULL, &efd, &tv);
		if(so > -1)
			if(FD_ISSET(so, &sfd))
				packetio();

		if(FD_ISSET(fd, &sfd))
		{
#endif
                        bpos = 0;
                        for(;;)
                        {
                                read(fd, &buffer[bpos], 1);
                                if(buffer[bpos] == '\n')
                                        break;
                                if(buffer[bpos] != '\r' && bpos < sizeof(buffer))
                                        ++bpos;
                        }
                        buffer[bpos] = 0;
			p = buffer + strlen(buffer) - 1;
			while(p > buffer)
			{
				if(*p == ' ' || *p == '\t' || *p == '\n' || *p == '\r')
					*(p--) = 0;
				else
					break;
			}
			p = buffer;
			while(*p == ' ' || *p == '\t')
				++p;
			if(!*p)
				continue;
			fifo.command(p);
		}
	}
	close(fd);
	final(0);
}

#ifdef	CCXX_NAMESPACES
}
#endif
