// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "bayonnelibrary.h"
#include <cc++/url.h>
#include <cc++/file.h>

#ifdef	HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#ifdef	HAVE_URL_CURL
#include <curl/curl.h>
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

Session *Session::first = NULL;
Session *Session::last = NULL;
Mutex Session::mutex;

Session::Session()
{
	mutex.enterMutex();
	if(last)
		last->next = this;

	prev = last;
	next = NULL;

	last = this;
	if(!first)
		first = this;
	mutex.leaveMutex();
}

void Session::unlink(void)
{
	if(next == this)
		return;

	mutex.enterMutex();
	if(prev)
		prev->next = next;
	else
		first = next;
	if(next)
		next->prev = prev;
	else
		last = prev;
	next = prev = this;
	mutex.leaveMutex();
}

void Session::clean(void)
{
	time_t now, expr;
	Session *base, *next;

	mutex.enterMutex();
	base = first;
	time(&now);

	while(base)
	{
		next = base->next;
		expr = base->getExpires();
		if(expr && expr < now)
		{
			base->unlink();
			delete base;
		}
		base = next;
	}
	mutex.leaveMutex();
}

Service::Service(Trunk *trk, int pri, size_t stack) :
Semaphore(), Thread(pri, stack), AudioService()
{
	trunk = trk;
	stopped = false;
	if(trunk->thread)
		trunk->stopServices();
	trunk->thread = this;
	data = &trunk->data;
	group = trunk->group;
}

void Service::failure(void)
{
	TrunkEvent event;

	event.id = TRUNK_SERVICE_FAILURE;
	trunk->postEvent(&event);
	Thread::sync();
}

void Service::success(void)
{
	TrunkEvent event;

	event.id = TRUNK_SERVICE_SUCCESS;
	trunk->postEvent(&event);
	Thread::sync();
}

AudioService::AudioService() {}

#ifdef	HAVE_URL_CURL
static bool urlCopy(const char *src, const char *dest)
{
	char errmsg[CURL_ERROR_SIZE + 1];
	const char *proxy = keyproxy.getHTTPServer();
	int port = keyproxy.getHTTPPort();
	FILE *fp = fopen(dest, "w");
	int rtn;	

	if(!fp)
		return false;

	CURL *curl = curl_easy_init();

	curl_easy_setopt(curl, CURLOPT_VERBOSE, 0);
	curl_easy_setopt(curl, CURLOPT_HEADER, 0);
	curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1);
	curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);
	curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, errmsg);
	curl_easy_setopt(curl, CURLOPT_URL, src);
	if(proxy && port)
	{
		curl_easy_setopt(curl, CURLOPT_PROXY, proxy);
		curl_easy_setopt(curl, CURLOPT_PROXYPORT, port);
	}
	curl_easy_setopt(curl, CURLOPT_DNS_USE_GLOBAL_CACHE, 0);
	curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
	curl_easy_setopt(curl, CURLOPT_USERAGENT, "bayonne");
	curl_easy_setopt(curl, CURLOPT_WRITEDATA, fp);
	rtn = curl_easy_perform(curl);	

	curl_easy_cleanup(curl);
	fclose(fp);
	if(rtn)
	{
		slog.error() << "curl: " << errmsg << endl;
		return false;
	}
	return true;
}
#endif

char *AudioService::getPrompt(char *name, const char *voice)
{
	Script::Name *scr = trunk->getObject();
	ScriptCommand *cmd = trunk->getCommand();
	char buffer[256];

#ifdef	HAVE_URL
	char protocol[16];
	char *bp;
#endif

	char *cp;
	const char *pname;
	const char *libname;
	const char *ext = strrchr(scr->filename, '.');
	const char *voices = cmd->getLast("voices");
	const char *prompts = cmd->getLast("prompts");
	const char *prefix = cmd->getLast("datafiles");
	const char *sys = prompts;

	if(!stricmp(ext, ".bm") || !sys)
		sys = keypaths.getLast("prompts");
	else if(!strnicmp(name, "lib:", 4))
	{
		sys = keypaths.getLast("prompts");
		name += 4;
	}

	if(!voices)
		voices = keypaths.getLast("voices");

	if(!prompts)
		prompts = keypaths.getLast("prompts");

	if(!voice)
		voice = trunk->getSymbol(SYM_VOICE);

// matched with url preproc library, works like tts through cache...

#ifdef	HAVE_URL
	if(!strnicmp(name, "http:", 5) || !strnicmp(name, "ftp:", 4) || !strnicmp(name, "https:", 6))
	{
		int cachetime = atoi(keyproxy.getLast("cache"));

		cp = strchr(name, '&');
		if(!cachetime || cp)
		{	
			if(!cp)
				cp = name + strlen(name) - 1;
			bp = cp;
			while(bp && *bp != '.' && *bp != '/' && *bp != ':')
				--bp;
			if(*bp == '/' || *bp == ':')
				bp = "";
			else
				++bp;
			snprintf(filename, sizeof(filename) - 5, "%s/url-%d.%s",
				keypaths.getLast("tmp"), trunk->tsid, bp);
			bp = strchr(filename, '&');
			if(bp)
				*bp = 0;
			return filename;
		}

		snprintf(protocol, sizeof(protocol), "%s", name);
		cp = strchr(protocol, ':');
		if(cp)
			*cp = 0;

		name = strchr(name, ':');
		while(*name == ':' || *name == '/')
			++name;
		snprintf(filename, sizeof(filename) - 5, "%s/%s-%s", 
			keypaths.getLast("cache"), protocol, name);
		name = filename + strlen(keypaths.getLast("cache"));
		while(*(++name))
		{
			if(*name == '/')
				*name = '.';
		}
		return filename;
	}
#endif

	if(!strnicmp(name, "spool:", 6))
	{
		name += 6;
		snprintf(filename, sizeof(filename) - 5, "%s/%s",
 			keypaths.getLast("spool"), name);
		return filename;
	}

	if(!strnicmp(name, "cache:", 6))
	{
		name += 6;
		snprintf(filename, sizeof(filename) - 5, "%s/%s",
			keypaths.getLast("cache"), name);
		return filename;
	} 

	if(!strnicmp(name, "tmp:", 4) || !strnicmp(name, "temp:", 5))
	{
		name = strchr(name, ':');
		snprintf(filename, sizeof(filename) - 5, "%s/%s",
			keypaths.getLast("tmp"), ++name);
		return filename;
	}

	if(!strnicmp(name, "ram:", 4) || !strnicmp(name, "mem:", 4))
	{
		name += 4;
		snprintf(filename, sizeof(filename) - 5, "%s/%s",
			keypaths.getLast("tmpfs"), name);
		return filename;
	}

	if(!strnicmp(name, "prompts:", 8))
	{
		name += 8;
		snprintf(filename, sizeof(filename) - 5, "%s/%s",
			prompts, name);
		return filename;
	}

	if(NULL != (pname = strchr(name, ':')))
	{
		ext = strrchr(++pname, '.');
		snprintf(buffer, sizeof(buffer), "%s", name);
		cp = strchr(buffer, ':');
		if(cp)
		{
			*cp = 0;
			addString(buffer, sizeof(buffer), ".lib");
		}
		else
			return name;

		libname = cmd->getLast(buffer);
		if(libname)
			snprintf(filename, sizeof(filename) - 5,
				"%s/%s", libname, pname);
		else
		{
			*cp = 0;
			snprintf(filename, sizeof(filename) - 5,
				"%s/%s/%s", voices, buffer, pname);
		}
		// if no extension, assume voice library extension...

		if(!ext)
			strcat(filename, trunk->getLibexec());

		return filename;
	}

	if(strchr(name, '/'))
	{
		if(prefix)
			snprintf(filename, sizeof(filename) - 5, "%s/%s",
				prefix, name);
		else
			snprintf(filename, sizeof(filename) - 5, "%s", name);
		return filename;
	}

	if(strchr(name, '.'))
	{
		snprintf(filename, sizeof(filename), "%s/%s",
			prompts, name);
		return filename;
	}

	snprintf(buffer, sizeof(buffer), "%s.lib", voice);
	libname = cmd->getLast(buffer);

	if(libname)
		snprintf(filename, sizeof(filename), "%s/%s%s",
			libname, name, trunk->getLibexec());
	else
		snprintf(filename, sizeof(filename), "%s/%s/%s%s",
			voices, voice, name, trunk->getLibexec());

	if(isFile(filename))
		return filename;

	snprintf(filename, sizeof(filename) - 5, "%s/%s",
			sys, name);
	return filename;
}

char *AudioService::getPlayfile(void)
{
	char buffer[128];
	char *cp = buffer;
	const char *ext;
#ifdef	HAVE_URL
	struct stat ino;
	time_t now;
#endif

	if(!trunk->data.play.name)
		return NULL;

	if(!*trunk->data.play.name)
		return NULL;

	while(*trunk->data.play.name && *trunk->data.play.name != ',')
		*(cp++) = *(trunk->data.play.name++);

	*cp = 0;
	while(*trunk->data.play.name && *trunk->data.play.name == ',')
		++trunk->data.play.name;

	cp = getPrompt(buffer, trunk->data.play.voice);

#ifdef	HAVE_URL
	if(!strnicmp(buffer, "http:", 5) || !strnicmp(buffer, "ftp:", 4) || !strnicmp(buffer, "https:", 6))
	{
		static Mutex urlmutex;
		static Semaphore urllimit(8);

		long cachetime = atoi(keyproxy.getLast("cache"));

		ext = strchr(buffer, '&');
		if(!cachetime || ext)
		{
			urllimit.wait();
			remove(cp);
			if(!urlCopy(buffer, cp))
			{
				remove(cp);
				cp = NULL;
			}
			urllimit.post();
			return cp;
		} 

		urlmutex.enterMutex();
		if(!stat(cp, &ino))
		{
			time(&now);
			if(now - ino.st_ctime < cachetime)
				goto resume;
			remove(cp);
		}
		if(urlCopy(buffer, cp))
			goto resume;
		remove(cp);
		cp = NULL;
resume:
		urlmutex.leaveMutex();
		return cp;
	}
#endif

	ext = strrchr(cp, '/');
	if(!ext)
		ext = cp;
	ext = strrchr(ext, '.');
	if(ext)
		return cp;

 	if(!ext)
		ext = trunk->data.play.extension;

	if(ext)
		strcat(cp, ext);

	return cp;
}

timeout_t Service::stop(void)
{
	stopped = true;
	return keythreads.getResetDelay();
}


#ifdef	CCXX_NAMESPACES
}
#endif
