// Copyright (C) 2000-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "bayonnelibrary.h"

#ifdef	HAVE_SSTREAM
#include <sstream>
#else
#include <strstream>
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

char Trunk::digit[16] = {
	'0', '1', '2', '3',
	'4', '5', '6', '7',
	'8', '9', '*', '#',
	'a', 'b', 'c', 'd'};

callrec_t *Trunk::callrec = NULL;

Trunk::Trunk(int port, Driver *drv, int card, int dspan) :
ScriptInterp(aascript, keymemory.getSymbolSize(), keymemory.getPageSize(), _genName(drv, port))
{
	int i;
	static char *names[] = {SYM_TIME, SYM_DATE, SYM_DURATION, SYM_COUNT, SYM_RINGS, SYM_STATE};

	use = NULL;
	driver = drv;
	ctx = this;
	trace = debug->debugTrace();

	group = TrunkGroup::assign(this);
	group->enterMutex();
	group->incCapacity();
	member = ++group->members;
	group->leaveMutex();

	++driver->portCount;

	flags.listen = flags.bgm = flags.dtmf = flags.offhook = flags.reset = false;
	flags.trunk = TRUNK_MODE_INACTIVE;
	flags.dsp = DSP_MODE_INACTIVE;
	flags.script = false;
	flags.ready = false;
	flags.onexit = false;
	flags.dnd = false;
	id = port;
	tsid = port + drv->tsid;
	spanid = dspan;
	cardid = card;
	rings = 0;
	starttime = idletime = synctimer = exittimer = 0;
	thread = NULL;
	script = NULL;
	digits = 0;
	counts = 0;
	apppath[0] = 0;
	isStation = isRinging = false;

	softJoin = NULL;
	joined = NULL;
	trunkgid = NULL;

	exitfd = -1;
	seq = 0;
	tonetmp = NULL;
	tgi.pid = 0;
	tgi.dtmf = true;

	dtmf.bin.id = SYM_DIGITS;
	dtmf.bin.flags.type = NORMAL;
	dtmf.bin.flags.readonly = false;
	dtmf.bin.flags.system = true;
	dtmf.bin.flags.initial = false;
	dtmf.bin.flags.commit = true;
	dtmf.bin.flags.size = MAX_DIGITS;
	dtmf.bin.next = NULL;

	for(i = 0; i < 6; ++i)
	{
		numbers[i].sym.id = names[i];
		numbers[i].sym.flags.type = NORMAL;
		numbers[i].sym.flags.readonly = true;
		numbers[i].sym.flags.system = true;
		numbers[i].sym.flags.initial = false;
		numbers[i].sym.flags.commit = false;
		numbers[i].sym.flags.size = 10;
		numbers[i].sym.next = NULL;
	}
	setString(numbers[5].sym.data, 11, "init");
	dtmf.bin.data[0] = 0;
	setCalls("init");
}

Trunk::~Trunk()
{
	if(use)
		use->decCapacity();
}

void Trunk::setUsage(UsageStat *us)
{
	use = us;
	if(use)
		use->incCapacity();
}

void Trunk::incIncoming(void)
{
	if(group)
		group->incIncoming();
	if(use)
		use->incIncoming();
}

void Trunk::incOutgoing(void)
{
	if(group)
		group->incOutgoing();
	if(use)
		use->incOutgoing();
}

void Trunk::decIncoming(void)
{
        if(group)
                group->decIncoming();
        if(use)
                use->decIncoming();
}

void Trunk::decOutgoing(void)
{
        if(group)
                group->decOutgoing();
        if(use)
                use->decOutgoing();
}

void Trunk::setAudioPosition(timeout_t pos)
{
	char buffer[65];
	unsigned long secs = pos / 1000;

	snprintf(buffer, sizeof(buffer), "%02d:%02d:%02d.%03d",
		secs / 3600, (secs / 60) % 60, secs % 60, pos % 1000);

	setSymbol(SYM_POSITION, buffer);
} 
	
char Trunk::getdigit(void)
{
	char digit;

	if(!digits)
		return 0;

	digit = dtmf.bin.data[0];
	cleardigits(false);
	return digit;
}

void Trunk::cleardigits(bool all)
{
	unsigned i;

	if(all || digits < 2)
	{
		dtmf.bin.data[0] = 0;
		digits = 0;
		setString(numbers[3].sym.data, 11, "0");
		return;
	}
	for(i = 1; i <= digits; ++i)
		dtmf.bin.data[i - 1] = dtmf.bin.data[i];
	--digits;
	snprintf(numbers[3].sym.data, 10, "%d", digits);
}
 
const char *Trunk::_genName(Driver *drv, int port)
{
	snprintf(_dbgname, sizeof(_dbgname), "[%s/%d]", drv->getName(), port);
	return _dbgname;
}

const char *Trunk::getEncodingName(Audio::Encoding e)
{
	switch(e)
	{
	case Audio::g721ADPCM:
		return "g721";
	case Audio::gsmVoice:
		return "gsm";
	case Audio::mulawAudio:
		return "mulaw";
	case Audio::alawAudio:
		return "alaw";
	case Audio::okiADPCM:
	case Audio::voxADPCM:
		return "vox";
	case Audio::pcm8Mono:
		return "pcm8";
	case Audio::pcm16Mono:
		return "pcm";
	default:
		return "unsupported";
	}
}

bool Trunk::isAdmin(void)
{
	const char *cp = getSymbol(SYM_LOGIN);
	if(!cp)
		return false;

	if(!stricmp(cp, "admin"))
		return true;

	return false;
}

void Trunk::dialRewrite(const char *dialstring)
{
	char rulebuf[65];
	char id[16];
	unsigned len = 0;
	char *cp;
	const char *rule = NULL;
	const char *prefix = "";
	const char *suffix = "";

	while(isdigit(dialstring[len]) && len < 15)
		++len;

	while(len && !rule)
	{
		memcpy(id, dialstring, len);
		id[len--] = 0;
		rule = group->getLast(id);
	}

	if(rule)
	{
		snprintf(rulebuf, sizeof(rulebuf), "%s", rule);
		dialstring += atoi(rule);	// prefix digits to strip
		rule = strchr(rulebuf, ',');
		if(rule)
		{
			prefix = ++rule;
			cp = strchr(rule, ',');
			if(cp)
			{
				*(cp++) = 0;
				suffix = cp;
			 }
		}
	}	
	snprintf(data.dialxfer.digits, sizeof(data.dialxfer.digits), 
		"%s%s%s", prefix, dialstring, suffix);
	data.dialxfer.digit = data.dialxfer.digits;
}	

const char *Trunk::getStation(void)
{
	const char *kw = getKeyword("station");
	if(!kw)
		kw = group->getLast("station");

	return kw;
}

void Trunk::setIdle(bool mode)
{
	if(mode)
		++driver->idleCount;
	else
		--driver->idleCount;
}

phTone *Trunk::getTone(void)
{
	if(data.tone.tone)
		return data.tone.tone;

	if(tonetmp)
		delete tonetmp;

	if(!data.tone.freq2)
		tonetmp = new phTone(NULL, data.tone.duration, data.tone.freq1);
	else
		tonetmp = new phTone(NULL, data.tone.duration, data.tone.freq1, data.tone.freq2);

	data.tone.tone = tonetmp;
	return tonetmp;
}

void Trunk::enterState(const char *state)
{
	snprintf(numbers[5].sym.data, 11, "%s", state);
	debug->debugState(this, (char *)state);
}

int Trunk::getDigit(char dig)
{
	int i;

	dig = tolower(dig);
	for(i = 0; i < 16; ++i)
	{
		if(digit[i] == dig)
			return i;
	}
	return -1;
}

#ifdef	HAVE_TGI
void Trunk::libtts(const char *msg, ttsmode_t mode)
{
	tgicmd_t cmd;
#ifdef	HAVE_SSTREAM
	ostringstream str;
	str.str() = "";
#else
	strstream str(cmd.cmd, sizeof(cmd.cmd));
#endif

	cmd.seq = ++tgi.seq;
	snprintf(cmd.port, sizeof(cmd.port), "%s/%d", driver->getName(), id);
	cmd.mode = TGI_EXEC_AUDIO;
	cmd.cmd[0] = 0;

	str << plugins.getLast("say") << ".tts";
	str << " language=" << getSymbol(SYM_LANGUAGE);
	str << " audio=" << "temp/.tts." << id << ".ul";
	str << " voice=" << getSymbol(SYM_VOICE);
	switch(mode)
	{
	case TTS_GATEWAY_TEXT:
		str << " phrase=" << msg;
		break;
	case TTS_GATEWAY_FILE:
		str << " source=" << msg;
		break;
	}
#ifdef	HAVE_SSTREAM
	snprintf(cmd.cmd, sizeof(cmd.cmd), "%s", str.str().c_str());
#else
	str << ends;
#endif
	::write(tgipipe[1], &cmd, sizeof(cmd));
}

#endif

bool Trunk::isReady(void)
{
	trunkmode_t trk = flags.trunk;
	if(trk != TRUNK_MODE_INACTIVE)
		return false;

	return flags.ready;
}

void Trunk::setDTMFDetect(void)
{
	Line *line = getScript();

	if(isActive() && line)
	{
		if(line->method == (Method)&Trunk::scrCollect)
			setDTMFDetect(true);
		else if(getMask() & 0x08)
			setDTMFDetect(true);
		else
			setDTMFDetect(false);
	}
	else
		setDTMFDetect(false);
}

void Trunk::setCalls(const char *mode)
{
	struct tm *dt, trec;
	time_t now;
	char buf[11];

	if(!callrec)
		return;

	if(!mode)
		mode = "none";

	time(&now);
	dt = localtime_r(&now, &trec);
	snprintf(buf, 6, "%02d:%02d", dt->tm_hour, dt->tm_min);
	memcpy(callrec[tsid].cr_time, buf, 5);
	setField(callrec[tsid].cr_type, mode, sizeof(callrec_t) - 7);
	callrec[tsid].cr_nl = '\n';
}

void Trunk::setField(char *field, const char *str, unsigned len)
{
	if(!callrec)
		return;

	if(!str)
		str = "";

	while(--len)
	{
		if(*str)
			*(field++) = tolower(*(str++));
		else
			*(field++) = ' ';
	}
} 

bool Trunk::syncParent(const char *msg)
{
	const char *p = getSymbol(SYM_PARENT);
	const char *g = trunkgid;
	Trunk *trunk;
	TrunkEvent event;

	if(!p)
		return false;

	if(!*p)
		return false;

	trunk = Driver::getTrunk(p);
	if(!trunk)
		return false;

	if(g)
		g = strchr(g, '-');
	else
		g = "none";

	event.id = TRUNK_SYNC_PARENT;
	event.parm.sync.msg = msg;
	event.parm.sync.id = g;
	return trunk->postEvent(&event);
}

bool Trunk::recvEvent(TrunkEvent *ev)
{
	char evt[65];
	Trunk *trk = ev->parm.send.src;
	const char *gid = trunkgid;
	if(!gid)
		return false;

	if(!trk)
	{
		snprintf(evt, sizeof(evt), "control:%s", ev->parm.send.msg);
		setSymbol(SYM_EVENTID, "server-control");
		setSymbol(SYM_EVENTMSG, ev->parm.send.msg);
		if(event(evt))
			return true;
		return trunkSignal(TRUNK_SIGNAL_EVENT);
	}

	snprintf(evt, sizeof(evt), "event:%s", ev->parm.send.msg);

	trk->enterMutex();
	if(trk->seq != ev->parm.send.seq)
	{
		trk->leaveMutex();
		return false;
	}

	gid = trk->trunkgid;
	if(!gid)
	{
		trk->leaveMutex();
		return false;
	}
	setSymbol(SYM_EVENTID, gid);
	setSymbol(SYM_EVENTMSG, ev->parm.send.msg);

	trk->leaveMutex();
	if(event(evt))
		return true;
	return trunkSignal(TRUNK_SIGNAL_EVENT);
}

char **Trunk::getInitial(char **args)
{
	args[0] = (char *)keyserver.getIncoming(this);
	args[1] = NULL;
	return args;
}

void Trunk::setList(char **list)
{
	char buffer[256];
	char *tok, *key, *value;

	while(*list)
	{
		setString(buffer, sizeof(buffer), *list);
		key = strtok_r(buffer, "=", &tok);
		value = strtok_r(NULL, "=", &tok);
		if(key && value)
		{
			if(!strnicmp(key, "server.", 7))
				key = NULL;
			else if(!stricmp(key, SYM_LOGIN))
				key = NULL;
			else if(!strnicmp(key, "driver.", 7))
				key = NULL;
			else if(!strnicmp(key, "user.", 5))
				key = NULL;  
			else if(!strnicmp(key, "line.", 5))
				key = NULL;
			else if(!strnicmp(key, "sql.", 4))
				key = NULL;
			else if(!strnicmp(key, "global.", 7))
				key = NULL;
		}
		if(key && value)
			setConst(key, urlDecode(value));
		++list;
	}
}

void Trunk::commit(Symbol *sym)
{
	if(sym == &dtmf.bin)
		digits = (unsigned)strlen(dtmf.bin.data);
	else
		ScriptSymbol::commit(sym);
}

void Trunk::repSymbol(const char *id, const char *data)
{
	if(data)
		setSymbol(id, data);
}


Script::Symbol *Trunk::getEntry(const char *name, unsigned size)
{
	int i;
	time_t now;
	struct tm *dt;
	struct tm tbuf;
	char buf[128];

	if(*name == '%')	// bug in interp
		++name;

	if(!strnicmp(name, "server.", 7))
		return globals.getEntry(name, 0);
	else if(!strnicmp(name, "global/", 7) || !strnicmp(name, "shared/", 7))
	{
		snprintf(buf, sizeof(buf), "global.%s", name + 7);
		return globals.getEntry(buf, 0);
	}
	else if(!strnicmp(name, "driver/", 7))
	{
		snprintf(buf, sizeof(buf), "driver.%s", name + 7);
		return globals.getEntry(buf, 0);
	}

	if(!stricmp(name, SYM_DIGITS))
	{
		dtmf.bin.data[digits] = 0;
		return &dtmf.bin;
	}

	for(i = 0; i < 6; ++i)
	{
		if(!stricmp(name, numbers[i].sym.id))
			break;
	}

	if(i >= 5)
		return ctx->ScriptSymbol::getEntry(name, size);

	if(i < 3)
		time(&now);

	switch(i)
	{
	case 4:
		sprintf(numbers[4].sym.data, "%d", rings);
		break;
	case 3:
		sprintf(numbers[3].sym.data, "%d", digits);
		break;
	case 2:
		if(starttime)
			now -= starttime;
		else
			now = 0;
		snprintf(numbers[2].sym.data, 10, "%02ld:%02ld:%02ld",
			now / 3600, (now / 60) % 60, now % 60);
		break;
	case 1:
		dt = localtime_r(&now, &tbuf);
		sprintf(numbers[1].sym.data, "%04d-%02d-%02d",
			dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
		break;
	case 0:
		dt = localtime_r(&now, &tbuf);
		sprintf(numbers[0].sym.data, "%02d:%02d:%02d",
			dt->tm_hour, dt->tm_min, dt->tm_sec);
	}

	return &numbers[i].sym;
}

void Trunk::initialize(void)
{
	char *e;
	ScriptImage *img = getImage();

	setSymbol(SYM_EXTENSION, 8);
	setSymbol(SYM_EXTENSION, img->getLast("extension"));
	setSymbol(SYM_LANGUAGE, 16);
	e = getenv("BAYONNE_LANGUAGE");
	if(e)
		setSymbol(SYM_LANGUAGE, e);
	else
		setSymbol(SYM_LANGUAGE, img->getLast("language"));
	repSymbol(SYM_LANGUAGE, group->getLast("language"));

	setSymbol(SYM_VOICE, 16);
	e = getenv("BAYONNE_VOICE");
	if(e)
		setSymbol(SYM_VOICE, e);
	else
		setSymbol(SYM_VOICE, img->getLast("voice"));
}

bool Trunk::attach(const char *name)
{
	char buf[65];
	time_t now;
	struct tm *dt;
	struct tm tbuf;
	char buffer[33];
	char *pid, *e;
	trunkmode_t trk = flags.trunk;
	Trunk *parent = NULL;
	TrunkEvent event;
	const char *cp;
	const char *login;
	Symbol *sym;
	aaScript *cmds = aascript;

	if(!name)
		name = keyserver.getIncoming(this);

	setConst(SYM_ROUTE, name);

	// To support multiple seperately compilable script images we have to
	// have base bottom aaScript's for each instance as these hold the
	// "active" script pointer for a given seperately compilable instance.
	// This is done by reviewing the passed script name.  If it is in the
	// form ~xxx[/yyy], then a "user" hosted script app is loaded.  If it
	// is a resolved /pathname, then the associated script project file
	// aaScript image will be set, and it's "main" is referenced to start.

	if(strchr(name, '/') || *name == '~')
	{
		snprintf(buf, sizeof(buf), "%s", name);
		e = strchr(buf, '/');
		if(e)
			*e = 0;
		cmds = aaScript::get(buf);
		if(!cmds)
		{
			errlog("access", "Session=%s", buf);
			return false;

			// Matt B's comments apply here as well
		}
		name = strchr(name, '/');
		if(name)
			++name;
		else
			name = cmds->getLast("start");
		if(!name)
			name = "main";

	}
	setCommand(cmds);

	// This concerns me.  Why are we returning false?
	// What, in fact, is this case?

	// Since we just set TRUNK_MODE_OUTGOING, in our parent
	// stack frame, I'll kill the flags.script case if we
	// are outbound scheduled.

	//Matt B

	if(!running)
		return false;

	exitfd = -1;
	exitmsg[0] = 0;
	getName(buffer);

	if(trk != TRUNK_MODE_OUTGOING)
	{
	    if(flags.script)
	        return false;
	}

	flags.reset = flags.dtmf = flags.offhook = flags.once = flags.onexit = false;
	tgi.pid = 0;
	tgi.dtmf = true;

	counts = 0;
	pid = getSymbol(SYM_PARENT);
	if(pid != NULL)
	{
		parent = Driver::getTrunk(pid, false, driver);
		setAlias(SYM_SOURCE, SYM_PARENT);
	}

	if(parent)
	{
		cp = parent->trunkgid;
		if(cp)
			cp = strchr(cp, '-');
		if(cp && !stricmp(cp, pid))
		{
			setConst(SYM_INFODIGITS, parent->getSymbol(SYM_INFODIGITS));
			setConst(SYM_CALLER, parent->getSymbol(SYM_CALLER));
			setConst(SYM_DIALED, parent->getSymbol(SYM_DIALED));
			setConst(SYM_NAME, parent->getSymbol(SYM_NAME));
			setConst(SYM_CLID, parent->getSymbol(SYM_CLID));
			setConst(SYM_DNID, parent->getSymbol(SYM_DNID));
		}
		else
			setConst(SYM_CALLER, "gone");
	}



	setConst(SYM_VIRTUAL, cmds->getLast("name"));

	setSymbol(SYM_EXITKEY, 1);
	setSymbol(SYM_EXITKEY, "-");

	setSymbol(SYM_FREE, 3);
	setSymbol(SYM_FREE, df_free);
	setSymbol(SYM_USED, 3);
	setSymbol(SYM_USED, df_used);

	setSymbol(SYM_TONE, 16);
	setSymbol(SYM_ANNOTATION, 160);
	setSymbol(SYM_PLAYED, 12);
	setSymbol(SYM_RECORDED, 12);
	setSymbol(SYM_OFFSET, 12);
	setSymbol(SYM_POSITION, 12);
	setSymbol(SYM_CREATED, 20);
	setSymbol(SYM_LOCKFILE, 64);

	setSymbol(SYM_HOME, 64);
	setSymbol(SYM_HOME, "");
	setSymbol(SYM_ERROR, 64);
	setSymbol(SYM_ERROR, "none");

	setSymbol(SYM_EVENTID, 32);
	setSymbol(SYM_EVENTMSG, 64);

	setSymbol(SYM_NOTIFYTEXT, 64);
	setSymbol(SYM_NOTIFYTYPE, 8);


	cp = group->getLast("dialplan");
	if(!cp)
		cp = keyserver.getLast("dialplan");

	setSymbol(SYM_DIALING, 3);
	setSymbol(SYM_DIALING, cp);

	setSymbol(SYM_FORMAT, 8);
	setSymbol(SYM_FORMAT, getDefaultEncoding());

	setSymbol(SYM_PLAYWAIT, 3);
	setSymbol(SYM_PLAYWAIT, "60");

	setSymbol(SYM_TRIM, 8);
	setSymbol(SYM_TRIM, "1200");

	setSymbol(SYM_BASE, 160);
	setSymbol(SYM_BASE, "http://localhost/");

	repSymbol(SYM_VOICE, group->getLast("voice"));

//	setSymbol(SYM_APPL, 16);
//	setSymbol(SYM_APPL, img->getLast("application"));
//	repSymbol(SYM_APPL, group->getLast("application"));

	setSymbol(SYM_VOLUME, 3);
	setSymbol(SYM_VOLUME, group->getLast("volume"));

	setSymbol(SYM_BUFFER, 8);
	setSymbol(SYM_BUFFER, "8000");

	cp = getSymbol(SYM_CALLER);
	if(!parent && cp)
		setAlias(SYM_SOURCE, SYM_CALLER);
	else if(!parent && !cp)
		setConst(SYM_SOURCE, "none");

	setAlias(SYM_TARGET, SYM_DIALED);

	setConst(SYM_NAME, "UNKNOWN");
	setConst(SYM_CALLER, "UNKNOWN");
	setConst(SYM_DIALED,  "none");
	setConst(SYM_INFODIGITS, "00");
	setConst(SYM_CLID, "UNKNOWN");
	setConst(SYM_DNID, "UNKNOWN");
	setConst(SYM_POLICYID, group->getLast("name"));

	snprintf(buf, sizeof(buf), "port/%d", tsid);
	setConst(SYM_PORTID, buf);

	snprintf(buf, sizeof(buf), "%d", tsid);
	setConst(SYM_TSID, buf);

	snprintf(buf, sizeof(buf), "%d", spanid);
	setConst(SYM_SPANID, buf);

	snprintf(buf, sizeof(buf), "%d", cardid);
	setConst(SYM_CARDID, buf);

	setSymbol(SYM_LOGIN, 16);

	login = keyserver.getLogin();
	if(!stricmp(login, "none") || !stricmp(login, "admin"))
		setSymbol(SYM_LOGIN, login);
	else if(!stricmp(login, "port"))
	{
		snprintf(buf, 4, "%03d", id);
		setSymbol(SYM_LOGIN, buf);
	}
	else
		setSymbol(SYM_LOGIN, "none");

	sym = getEntry(SYM_LOGIN, 0);
	sym->flags.readonly = true;

	initSyms();

	sprintf(buf, "%03d", id);
	setConst(SYM_ID, buf);
	setConst(SYM_TRUNKID, buf);

	setConst(SYM_DRIVER, driver->getName());
	sprintf(buf, "%02d", driver->tsid);
	setConst(SYM_DRIVERID, buf);
	sprintf(buf, "%04d", driver->getTrunkCount());
	setConst(SYM_DRIVERSIZE, buf);

//	sprintf(buf, "%d", driver->getTrunkCount());
//	setConst(SYM_PORTS, buf);

	time(&now);

	sprintf(buf, "%s-%03d-%02d%lx", keyserver.getNode(), id, driver->getDriverIndex(), now);
	setConst(SYM_GID, buf);
	setConst(SYM_CALLFWD, "none");
	setSymbol(SYM_JOINDURATION, 11);
	setSymbol(SYM_JOINID, 16);
	setSymbol(SYM_PICKUP, 16);
	setSymbol(SYM_RECALL, 16);
	setSymbol(SYM_TRUNK, 16);
	setSymbol(SYM_STARTID, 16);
	switch(flags.trunk)
	{
	case TRUNK_MODE_INCOMING:
		setConst(SYM_CALLTYPE, "incoming");
		break;
	case TRUNK_MODE_OUTGOING:
		setConst(SYM_CALLTYPE, "outgoing");
	default:
		break;
	}


	if(spanid)
	{
		sprintf(buf, "%d", spanid);
		setConst(SYM_SPAN, buf);
	}

	dt = localtime_r(&now, &tbuf);
	sprintf(buf, "%04d-%02d-%02d",
		dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
	setConst(SYM_STARTDATE, buf);
	sprintf(buf, "%02d:%02d:%02d",
		dt->tm_hour, dt->tm_min, dt->tm_sec);
	setConst(SYM_STARTTIME, buf);

//	setConst(SYM_RELEASE, "1");
//	setConst(SYM_VERSION, cmd->getLast("version"));
//	setConst(SYM_SERVER, cmd->getLast("server"));
//	setConst(SYM_DRIVER, plugins.getDriverName());
//	setConst(SYM_NODE, cmd->getLast("node"));
	setConst(SYM_START, name);
//	setConst(SYM_SCRIPTS, keypaths.getScriptFiles());
//	setConst(SYM_PROMPTS, keypaths.getPromptFiles());

	sprintf(buf, "%d", member);
	setConst(SYM_MEMBER, buf);

	if(!ScriptInterp::attach(name))
	{
		errlog("missing", "Script=%s; Session=%s ", name, cmds->getLast("name"));
		if(parent)
		{
			event.id = TRUNK_CHILD_FAIL;
			parent->postEvent(&event);
		}
		ScriptInterp::purge();
		return false;
	}



	flags.script = true;

	if(parent)
	{
		event.id = TRUNK_CHILD_START;
		event.parm.trunk = this;
		parent->postEvent(&event);
	}

	if(!starttime)
	{
		time(&starttime);
		time(&idletime);
		idle_timer = group->getIdleTime();
	}

	debug->debugState(this, "attach script");
	trunkgid = getSymbol(SYM_GID);

	snprintf(buf, 5, "%s", getSymbol(SYM_CALLTYPE));
	setCalls(buf);
	setField(callrec[tsid].cr_caller, getSymbol(SYM_CLID), 16);
	cp = getSymbol(SYM_DIALER);
	if(!cp)
		cp = getSymbol(SYM_DNID);
	setField(callrec[tsid].cr_dialed, cp, 16);
	setField(callrec[tsid].cr_script, name, 12);
	setField(callrec[tsid].cr_login, getSymbol(SYM_LOGIN), 11);
	return true;
}

void Trunk::detach(void)
{
	Trunk *child = NULL;
	TrunkEvent event;
	char buffer[256];
	char cdr[256];
	char symid[65];
	char *tag, *gid;
	size_t len = 0;

	ctx = this;
	trunkgid = NULL;
	synctimer = exittimer = 0;

	setCalls("exit");

	if(!flags.script || !running)
		return;

	if(tonetmp)
		delete tonetmp;

	tonetmp = NULL;

	++seq;
	setString(cdr, sizeof(cdr), "calls");
	len = 5;

	tag = getSymbol(SYM_PARENT);
	if(tag != NULL)
		child = Driver::getTrunk(tag, false, driver);

        gid = getSymbol(SYM_GID);
        if(gid)
                gid = strchr(gid, '-');

#ifndef	WIN32
	if(exitfd > -1)
	{
		snprintf(buffer, sizeof(buffer),
			"exit %s %s\n", gid, exitmsg);
		::write(exitfd, buffer, strlen(buffer));
		::close(exitfd);
		exitfd = -1; 		
	}
#endif

	if(child)
	{
		child->enterMutex();
		event.id = TRUNK_CHILD_EXIT;
		if(child->postEvent(&event))
		{
			child->setSymbol(SYM_EVENTID, gid);
			child->setSymbol(SYM_EVENTMSG, exitmsg);
		}
		child->leaveMutex();
	}

	tag = (char *)keyserver.getLast("calls");
	if(!tag)
		tag = (char *)keyserver.getLast("cdr");

	if(tag)
	{
		setString(buffer, sizeof(buffer), tag);
		tag = buffer;
	}
	if(tag)
		tag = strtok_r(tag, ",;:| \t\r\n", &gid);
	while(tag)
	{
		if(!strchr(tag, '.'))
		{
			snprintf(symid, sizeof(symid), "session.%s", tag);
			tag = symid;
		}
		tag = getSymbol(tag);
		if(tag)
		{
			snprintf(cdr + len, sizeof(cdr) - len, " %s", tag);
			len = strlen(cdr);
		}
		tag = strtok_r(NULL, ",;:| \t\r\n", &gid);
	}
	cdr[len++] = '\n';
	cdr[len] = 0;
	fifo.control(cdr);	
	dtmf.bin.data[0] = 0;
	digits = 0;
	counts = 0;
	ScriptInterp::detach();
	ScriptSymbol::purge();
	starttime = 0;
	flags.script = false;
	flags.onexit = false;
	flags.listen = flags.bgm = false;
#ifdef	HAVE_TGI
	if(tgi.pid)
		kill(tgi.pid, SIGHUP);
	tgi.pid = 0;
#endif

	if(tonetmp)
	{
		delete tonetmp;
		tonetmp = NULL;
	}

	debug->debugState(this, "detach script");
	setCommand(aascript);
}

void Trunk::stopServices(void)
{
	if(thread)
	{
		if(thread->isExiting())
			delete thread;
	}
	thread = NULL;
}

const char *Trunk::getPrefixPath(void)
{
	const char *prefix = getMember();

        if(!prefix)
                prefix = "";

        if(!stricmp(prefix, "feed"))
                prefix = "memory";
        else
                prefix = getKeyword("prefix");

        if(!prefix)
                return prefix;

        if(!stricmp(prefix, "memory") || !stricmp(prefix, "ram"))
                return keypaths.getLast("tmpfs");

	if(!stricmp(prefix, "tmp"))
		return keypaths.getLast("tmp");

	if(!stricmp(prefix, "cache"))
		return keypaths.getLast("cache");

	if(!stricmp(prefix, "spool"))
		return keypaths.getLast("spool");

        return prefix;
}

timeout_t Trunk::getTimeout(const char *keywd)
{
	if(keywd)
		keywd = getKeyword(keywd);
	if(!keywd)
		keywd = getValue("86400");
	return getSecTimeout(keywd);
}

timeout_t Trunk::getInterdigit(const char *keywd)
{
	ScriptImage *img = getImage();

	if(keywd)
		keywd = getKeyword(keywd);
	if(!keywd)
		keywd = getValue(img->getLast("interdigit"));
	return getSecTimeout(keywd);
}

bool Trunk::hasExitMask(unsigned short mask)
{
	static char *dig = "0123456789*#abcd";
	unsigned count = 0;
	const char *dp;

	if(!mask)
		return false;

	setDTMFDetect(true);
	if(!digits)
		return false;

	while(count < digits)
	{
		dp = strchr(dig, dtmf.bin.data[count]);
		if((mask & (1 << (int)(dp - dig))))
			return true;
		++count;
	}
	return false;
}	

unsigned short Trunk::getExitMask(void)
{
	static char *digits = "0123456789*#abcd";
	unsigned short mask = 0;
	const char *dp, *cp = getKeyword("exit");

	if(!cp || !*cp)
		return 0;

	while(*cp)
	{
		dp = strchr(digits, tolower(*cp));
		++cp;
		if(dp)
			mask |= (1 << (int)(dp - digits));
	}
	return mask;
}

unsigned short Trunk::getDigitMask(const char *cp)
{
	static char *digits = "0123456789*#abcd";
	unsigned short mask = 0;
	const char *dp;

	if(cp)
		cp = getKeyword(cp);

	if(!cp)
		cp = getValue(NULL);

	if(!cp)
		return 0;

	while(*cp)
	{
		dp = strchr(digits, tolower(*cp));
		++cp;
		if(dp)
			mask |= (1 << (int)(dp - digits));
	}
	return mask;
}

bool Trunk::event(const char *evt, bool inheret)
{
	Name::Event *ev;
	size_t len, idx;
	const char *cp;
	bool partial = false;
	char dig, mask;
	char cbuf[8];
	size_t clen = 0;
	bool iflag = inheret;

	if(!isActive())
		return false;

	if(!strnicmp(evt, "route:", 6))
	{
		snprintf(cbuf, sizeof(cbuf), "%s/", getSymbol(SYM_DIALING));
		clen = strlen(cbuf);
		goto plan;
	}

	if(!strnicmp(evt, "digits:", 7))
	{
		snprintf(cbuf, sizeof(cbuf), "%s/", getSymbol(SYM_DIALING));
		clen = strlen(cbuf);
		goto plan;
	}

	goto trap;

plan:
	iflag = false;
	ev = getObject()->events;

	while(ev)
	{
#ifdef	HAVE_REGEX_H
		if(ev->type == '~')
		{
			ev = ev->next;
			continue;
		}
#endif
		len = strchr(evt, ':') - evt + 1;
		cp = ev->name;
		if(strchr(cp, ':'))
		{
			if(!strnicmp(cp, evt, len))
				cp += len;
			else
			{
				ev = ev->next;
				continue;
			}
		}


		if(strchr(cp, '/'))
		{
			if(!strnicmp(cp, cbuf, clen))
				cp += clen;
			else
			{
				ev = ev->next;
				continue;
			}
		}
		idx = 0;
		if(!stricmp(cp, evt + len))
			break;
		while(cp[idx] && evt[idx + len])
		{
			dig = toupper(evt[idx + len]);
			mask = toupper(cp[idx]);
			if(mask == dig)
			{
				++idx;
				continue;
			}
			if(mask == 'X' && dig >= '0' && dig <= '9')
			{
				++idx;
				continue;
			}
			if(mask == 'N' && dig >= '2' && dig <= '9')
			{
				++idx;
				continue;
			}
			if(mask == 'O' && dig >= '2' && dig <= '9')
			{
				++idx;
				--len;
				continue;
			}
			if(mask == 'O' && dig == '1')
			{
				++idx;
				continue;
			}
			if(mask == 'Z' && dig > '0' && dig <= '9')
			{
				++idx;
				--len;
				continue;
			}
			if(mask == 'Z' && dig == '0')
			{
				++idx;
				continue;
			}
			break;
		}
		if(!evt[idx + len])
		{
			if(!strnicmp(evt, "route:", 6))
				break;
			partial = true;
		}
		if(!cp[idx] && !evt[idx + len])
			break;
		ev = ev->next;
	}

	if(ev)
		evt = ev->name;
	else if(!strnicmp(evt, "digits:", 7))
	{
		if(partial)
			evt = "digits:partial";
		else
			evt = "digits:invalid";
	}
	else if(!strnicmp(evt, "route:", 6))
		evt = "route:default"; 

trap:
	return ScriptInterp::event(evt, iflag);
}

void Trunk::trunkError(const char *err)
{
	if(!isActive())
		return;

	if(!err)
		err = getSymbol(SYM_ERROR);
	else
		setSymbol(SYM_ERROR, err);
	ScriptInterp::error(err);
}

bool Trunk::trunkSignal(trunksignal_t signal)
{
	Line *line;

	if(!isActive())
		return false;

	if(signal == TRUNK_SIGNAL_HANGUP)
	{
		if(flags.onexit)
			return false;
	}

	if(!signal)
	{
		advance();
		return true;
	}

	if(signal == TRUNK_SIGNAL_GOTO)
	{
		line = getScript();
		if(line->argc)
                        scrGoto();
		else
			advance();
		return true;
	}

	if(ScriptInterp::signal((unsigned long)(signal) - 1))
	{
		line = getScript();
		if(signal == TRUNK_SIGNAL_HANGUP)
		{
			if(line)
				flags.onexit = true;
		}
		return true;
	}

	return false;
}

bool Trunk::idleHangup(void)
{
	time_t now;

	if(!idle_timer)
		return false;

	time(&now);
	if(now - idletime > idle_timer)
	{
		exit();
		return true;
	}
	return false;
}

timeout_t getTimePosition(const char *opt)
{
	const char *first = NULL;
	const char *second = NULL;
	const char *dot;
	char dotbuf[4];
	size_t len;
	timeout_t timer;

	if(!opt)
		opt = "0";

	first = strchr(opt, ':');
	dot = strchr(opt, '.');

	if(first)
		second = strchr(first + 1, ':');

	if(!first)
		return getSecTimeout(opt);

	if(!dot)
		dot = ".0";

	if(second)
	{
		timer = atoi(opt) * 3600000l;
		timer += atoi(++first) * 60000l;
		timer += atoi(++second) * 1000l;
	}
	else
	{
		timer = atoi(opt) * 60000;
		timer += atoi(++first) * 1000l;
	}

	setString(dotbuf, sizeof(dotbuf), ++dot);
	while((len = strlen(dotbuf)) < 3)
		dotbuf[len++] = '0';
	dotbuf[len] = 0;
	return timer + atoi(dotbuf);
}

timeout_t getMSTimeout(const char *opt)
{
        char *end;
        char decbuf[4];
        long value;
        size_t len;

        if(!opt)
                opt = "0";

	if(strchr(opt, ':'))
		return getTimePosition(opt);

        value = strtol(opt, &end, 10) * 1000;
        if(*end == '.')
        {
		setString(decbuf, sizeof(decbuf), ++end);
                len = strlen(decbuf);
                while(len < 3)
                        decbuf[len++] = '0';
                value += strtol(decbuf, &end, 10);
        }


        switch(*end)
        {
        case 'h':
        case 'H':
                return value * 3600;
        case 'm':
        case 'M':
                if(end[1] == 's' || end[1] == 'S')
                        return value / 1000;
                return value * 60;
        default:
                return value / 1000;
        }
}
		
timeout_t getSecTimeout(const char *opt)
{
	char *end;
	char decbuf[4];
	long value;
	size_t len;

	if(!opt)
		opt = "0";

	if(strchr(opt, ':'))
		return getTimePosition(opt);

	value = strtol(opt, &end, 10) * 1000;
	if(*end == '.')
	{
		setString(decbuf, sizeof(decbuf), ++end);
		len = strlen(decbuf);
		while(len < 3)
			decbuf[len++] = '0';
		value += atol(decbuf);
	}

	switch(*end)
	{
	case 'h':
	case 'H':
		return value * 3600;
	case 'm':
	case 'M':
		if(end[1] == 's' || end[1] == 'S')
			return value / 1000;
		return value * 60;
	default:
		return value;
	}				
}

bool getLogical(const char *str)
{
	if(*str == '.')
		++str;
	switch(*str)
	{
	case '0':
	case 'f':
	case 'F':
	case 'N':
	case 'n':
		return false;
	}
	return true;
}

bool Trunk::ifFeature(ScriptInterp *interp, const char *v)
{
	Driver *drv = ((Trunk *)(interp))->getDriver();

	if(!stricmp(v, "tts") && hasTTS())
		return true;

#ifdef	HAVE_TGI
	if(!stricmp(v, "tgi"))
		return true;
#else
	if(!stricmp(v, "tgi"))
		return false;
#endif

	if(!stricmp(v, "join") && (drv->getCaps() & Driver::capJoin))
		return true;

	if(!stricmp(v, "switch") && (drv->getCaps() & Driver::capSwitch))
		return true;

	if(!stricmp(v, "spans") && (drv->getCaps() & Driver::capSpans))
		return true;

	if(!stricmp(v, "speed") && (drv->getCaps() & Driver::capSpeed))
		return true;

	if(!stricmp(v, "gain") && (drv->getCaps() & Driver::capGain))
		return true;

	if(!stricmp(v, "pitch") && (drv->getCaps() & Driver::capPitch))
		return true;

	if(!stricmp(v, "tts") && hasTTS())
		return true;

#ifdef	HAVE_TGI
	if(!stricmp(v, "tgi"))
		return true;
#else
	if(!stricmp(v, "tgi"))
		return false;
#endif

	if(!stricmp(v, "say") && TTS::ttsFirst)
		return true;

	if(!stricmp(v, "listen") && (drv->getCaps() & Driver::capListen))
		return true;

	if(!strnicmp(v, "conf", 4) && (drv->getCaps() & Driver::capConference))
		return true;

	if(!stricmp(v, "voice") && (((Trunk *)(interp))->getCapabilities() & TRUNK_CAP_VOICE))
		return true;

        if(!stricmp(v, "dial") && (((Trunk *)(interp))->getCapabilities() & TRUNK_CAP_DIAL))
                return true;

        if(!stricmp(v, "fax") && (((Trunk *)(interp))->getCapabilities() & (TRUNK_CAP_SENDFAX|TRUNK_CAP_RECVFAX)))
                return true;

        if(!stricmp(v, "data") && (((Trunk *)(interp))->getCapabilities() & TRUNK_CAP_DATA))
                return true;

        if(!stricmp(v, "station") && (((Trunk *)(interp))->getCapabilities() & TRUNK_CAP_STATION))
                return true;

	return false;
}

bool Trunk::isActiveUser(ScriptInterp *interp, const char *v)
{
	char namebuf[65];
	Symbol *sym;

	snprintf(namebuf, sizeof(namebuf), "%s.password", v);
	sym = globals.getEntry(namebuf, 0);

	if(!sym)
		return false;

	if(!sym->flags.initial)
		return false;

	return true;
}

bool Trunk::ifRinging(ScriptInterp *interp, const char *v)
{
	Trunk *trunk = Driver::getTrunk(v);

	if(!trunk)
		return false;

	if(trunk->isRinging)
		return true;

	if(!trunk->rings)
		return false;

	if(trunk->flags.offhook)
		return false;

	return true;
}

bool Trunk::ifRunning(ScriptInterp *interp, const char *v)
{
	Trunk *trunk = Driver::getTrunk(v);

	if(!trunk)
		return false;

	if(trunk->isActive())
		return true;

	return false;
}

bool Trunk::ifPort(ScriptInterp *interp, const char *v)
{
	Trunk *trunk = Driver::getTrunk(v);
	if(trunk)
		return true;

	return false;
}

bool Trunk::isNode(ScriptInterp *interp, const char *v)
{
	const char *node = keyserver.getNode();
	if(!node)
		return false;

	if(!stricmp(node, v))
		return true;

	return false;
}

bool Trunk::hasDriver(ScriptInterp *interp, const char *v)
{
	if(ost::getDriver(v))
		return true;

	return false;
}

bool Trunk::hasGroup(ScriptInterp *interp, const char *v)
{
	if(TrunkGroup::getGroup(v))
		return true;

	return false;
}

bool Trunk::hasPlugin(ScriptInterp *interp, const char *v)
{
	if(ScriptModule::find(v))
		return true;

	return false;
}

bool Trunk::hasVoice(ScriptInterp *interp, const char *v)
{
	const char *prompts = keypaths.getLast("prompts");
	char buf[256];

	snprintf(buf, sizeof(buf), "%s/%s", prompts, v);
	if(isDir(buf))
		return true;

	return false;
}

bool Trunk::hasVarPrompt(ScriptInterp *interp, const char *v)
{
	ScriptCommand *cmd = interp->getCommand();
	char buf[128], vbuf[256];
	char *ext = "";
	char *cp;
	char *prefix = interp->getKeyword("prefix");
	const char *var = cmd->getLast("datafiles");

	cp = strchr(v, '/');
	if(cp)
		cp = strchr(++cp, '.');
	else
		cp = strchr(v, '.');

	if(!cp)
		ext = interp->getKeyword("extension");

	if(!ext)
		ext = interp->getSymbol(SYM_EXTENSION);

	if(prefix)
		snprintf(buf, sizeof(buf), "%s/%s%s", prefix, v, ext);
	else
		snprintf(buf, sizeof(buf), "%s%s", v, ext);

	if(var)
	{
		snprintf(vbuf, sizeof(vbuf), "%s/%s", vbuf, buf);
		v = vbuf;
	}
	else
		v = buf;

	if(!permitAudioAccess(v, false))
		return false;

	if(isFile(v))
		return true;

	return false;
}

bool Trunk::hasSysPrompt(ScriptInterp *interp, const char *v)
{
	Name *scr = interp->getObject();
	const char *prompts = keypaths.getLast("prompts");
	char buf[256];
	char name[65];
	char *cp;
	char *ext = "";

	snprintf(name, sizeof(name), "%s", scr->name);
	cp = strstr(name, "::");
	if(cp)
		*cp = 0;

	if(strchr(v, '/'))
		return false;

	if(!strchr(v, '.'))
		ext = ".au";

	snprintf(buf, sizeof(buf), "%s/sys/%s/%s%s",
		prompts, name, v, ext);

	if(isFile(buf))
		return true;

	return false;
}

bool Trunk::hasSysVoice(ScriptInterp *interp, const char *v)
{
        const char *prompts = keypaths.getLast("prompts");
        char buf[256];

        snprintf(buf, sizeof(buf), "%s/sys/%s", prompts, v);
        if(isDir(buf))
                return true;

        return false;
}


#ifdef	CCXX_NAMESPACES
}
#endif

