// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "Options.h"
#include "CDDrive.h"
#include "Database.h"
#include "Cdrom.h"

//------------------------------------------------------------------------------
// Create a CDDrive from a path and a polling interval.
//------------------------------------------------------------------------------
CDDrive::CDDrive()
: _track( 0 ), _relTime( 0 ), _desiredTrack( 0 ), _status( Error ), _volume( -1 ), _cd( 0 ), _discStatus( NoDisc )
{
	// Start polling the CDROM drive.
	_timerID = startTimer( options->pollTime() );
	
	emit updateVolume( _volume );
}

//------------------------------------------------------------------------------
// Poll the CDROM drive for its status.
//------------------------------------------------------------------------------
void CDDrive::timerEvent(QTimerEvent*)
{
    int vol;
    
    // Check the audio status of the disc.
    CdromStatus audioStatus;
    int absTime;
    int status = cdrom->status( audioStatus, absTime, vol );

    if ( _cd ) {
        _relTime = absTime - _cd->id().trackStart( _desiredTrack - 1 );
    }
	
    DiscStatus newDiscStatus = NoDisc;
	
    if (status != 0) {
        newDiscStatus = NoDisc;
    }
    if ((status == 0) && (_discStatus == NoDisc)) {
        newDiscStatus = NewDisc;
    }
    if ((status == 0) && (_discStatus == NewDisc)) {
        newDiscStatus = OldDisc;
    }
    if ((status == 0) && (_discStatus == OldDisc)) {
        newDiscStatus = OldDisc;
    }
    
    switch ( newDiscStatus ) {
        case NoDisc:
            if ( _discStatus != NoDisc ) {
                doNoDisc();
            }
            break;
            
        case NewDisc:
            if ( !doNewDisc() ) {
                newDiscStatus = NoDisc;
            }
            if ( _volume != vol ) {
                _volume = vol;
                emit updateVolume( _volume );
            }
            break;
            
        case OldDisc:
            if ( _volume != vol ) {
                _volume = vol;
                emit updateVolume( _volume );
            }
            break;
    }

    _discStatus = newDiscStatus;
    changeStatus( audioStatus, absTime, _relTime );
    emit updateStatus( audioStatus );
}

//------------------------------------------------------------------------------
// Update the drives audio status, and take appropriate action.
//------------------------------------------------------------------------------
void CDDrive::changeStatus( CdromStatus status, int absTime, int relTime )
{
    switch ( status ) {
	 case Error:
		break;

	 case Playing:
		doTime( absTime, relTime );
		break;

	 case Paused:
		break;
		
	 case Completed:
		// Check the repeat mode to see if need to re-play the same track.
		if ( options->repeat() ) {
			if ( ( _cd->program()->isEnd() ) && ( options->shuffle() ) ) {
				_cd->program()->shuffle();
			}
			nextTrack();
		} else {
			if (!_cd->program()->isEnd()) {
				nextTrack();
			} else {
				// We are done playing this disc.
				if ( options->autoEject() ) {
					eject();
				} else {
					if ( options->shuffle() ) {
						_cd->program()->shuffle();
					}
					_desiredTrack = _cd->program()->reset();
					doUpdateTrack();
				}
			}
		}
		break;

	 case Stopped:
		if ( _cd ) {
			doTime( absTime, relTime );
		}
		break;
    }

    _status = status;
}

//------------------------------------------------------------------------------
// Process a previous track command.
//------------------------------------------------------------------------------
void CDDrive::previousTrack()
{
    if ( _discStatus == NoDisc ) return;

    if ( ( ( _status != Paused ) && ( _status != Playing ) ) || ( _relTime < 1 * XcdFrames ) ) {
		_desiredTrack = _cd->program()->previous();
	}
    doUpdateTrack();

	if ( _status == Playing ) {
		cdrom->play( _desiredTrack );
    }
}

//------------------------------------------------------------------------------
// Process a stop command.
//------------------------------------------------------------------------------
void CDDrive::stop()
{
    if ( _discStatus == NoDisc ) return;

    if ( ( _status == Playing ) || ( _status == Paused ) ) {
		cdrom->stop();
		_desiredTrack = _cd->program()->reset();
		doUpdateTrack();
    }
}

//------------------------------------------------------------------------------
// Process a play command.
//------------------------------------------------------------------------------
void CDDrive::play()
{
    if ( _discStatus == NoDisc ) return;
    
    switch (_status) {
	 case Playing:
		cdrom->pause();
		break;
		
	 case Paused:
		cdrom->resume();
		break;
		
	 case Stopped:
		cdrom->play( _desiredTrack );
		break;
		
	 default:
		break;
    }
}

//------------------------------------------------------------------------------
// Process a play track command.  Play a specific track.
//------------------------------------------------------------------------------
void CDDrive::playTrack(int track)
{
    if ( _discStatus == NoDisc ) return;
    
    track = *_cd->program()->ordered().at( track );
	
    // Try to locate the desired track within the current program.
    for (_desiredTrack = _cd->program()->reset(); (_desiredTrack != track) && (!_cd->program()->isEnd()); _desiredTrack = _cd->program()->next());
	
    // If we found it then play it.
    if (_desiredTrack == track) {
        doUpdateTrack();
        cdrom->play( _desiredTrack );
    }
}

//------------------------------------------------------------------------------
// Process a pause command.
//------------------------------------------------------------------------------
void CDDrive::pause()
{
    if ( _discStatus == NoDisc ) return;
    
    switch (_status) {
	 case Playing:
		cdrom->pause();
		break;
		
	 case Paused:
		cdrom->resume();
		break;
		
	 default:
		break;
    }
}

//------------------------------------------------------------------------------
// Process a next track command.
//------------------------------------------------------------------------------
void CDDrive::nextTrack()
{
    if ( _discStatus == NoDisc ) return;
    
    _desiredTrack = _cd->program()->next();
    doUpdateTrack();

	if ( _status == Playing ) {
		cdrom->play( _desiredTrack );
    }
}

//------------------------------------------------------------------------------
// Process an eject command.
//------------------------------------------------------------------------------
void CDDrive::eject()
{
	doNoDisc();
    cdrom->eject();
}

//------------------------------------------------------------------------------
// Increment the volume by one unit.
//------------------------------------------------------------------------------
void CDDrive::incrVolume()
{
    if (_volume < 63) {
        _volume++;
		cdrom->volume( _volume );
		emit updateVolume( _volume );
    }
}

//------------------------------------------------------------------------------
// Decrement the volume by one unit.
//------------------------------------------------------------------------------
void CDDrive::decrVolume()
{
    if (_volume > 0) {
        _volume--;
		cdrom->volume( _volume );
		emit updateVolume( _volume );
    }
}

//------------------------------------------------------------------------------
// Calculate and emit the current time, taking into the account the time mode
// that the user has selected.
//------------------------------------------------------------------------------
void CDDrive::doTime( int absTime, int relTime )
{
    if ( ( _status != Playing ) && ( _status != Paused ) ) {
        absTime = _cd->id().trackStart( _desiredTrack - _cd->id().firstTrack() );
        relTime = 0;
    }
    
    switch ( options->timeMode() ) {
        case TrackElapsed:
            doUpdateTime( relTime );
            break;

        case TrackRemaining:
            doUpdateTime( _cd->id().trackStart( _desiredTrack - _cd->id().firstTrack() + 1 ) - absTime );
            break;

        case DiscElapsed:
            doUpdateTime( absTime - _cd->id().trackStart( 0 ) );
            break;

        case DiscRemaining:
            doUpdateTime( _cd->id().trackStart( _cd->id().numTracks() ) - absTime );
            break;
    }
}

//------------------------------------------------------------------------------
// Configure the CDDrive when no disc is detectd.
//------------------------------------------------------------------------------
void CDDrive::doNoDisc()
{
    if ( _discStatus == NoDisc ) return;

    // Delete any existing disc info.
    if (_cd) {
        delete _cd;
        _cd = 0;
    }
    
	// Reset the audio status.
	_status = Error;
	
    // Mark the disc as absent.
    _discStatus = NoDisc;
    
    // Make the current step invalid.
    changeStep(-1);
    
    // Make the current track invalid.
    _desiredTrack = -1;
    doUpdateTrack();

    // Make the current time invlaid.
    doUpdateTime(-1);

    // Notify everyone else.
    emit noDisc();
}

//------------------------------------------------------------------------------
// Configure the CDDrive when a new disc is detected.
//------------------------------------------------------------------------------
bool CDDrive::doNewDisc()
{
	CompactDiscID* id;
	
	if ( !( id = cdrom->id() ) ) {
		_discStatus = NoDisc;
		return false;
	}

	if ( _cd ) {
		delete _cd;
	}
	_cd = dbase->search( *id );
	delete id;
	
	connect( _cd->program(), SIGNAL( updateStep( int ) ), this, SLOT( changeStep( int ) ) );
	
	if ( options->shuffle() ) {
		_cd->program()->shuffle();
	}
	
	_desiredTrack = _cd->program()->reset();
	doUpdateTrack();

    // Initialize the time display.
    switch ( options->timeMode() ) {
        case TrackElapsed:
            doUpdateTime(0);
            break;

        case TrackRemaining:
            doUpdateTime(_cd->id().trackStart(1) - _cd->id().trackStart(0));
            break;

        case DiscElapsed:
            doUpdateTime(0);
            break;

        case DiscRemaining:
            doUpdateTime(_cd->id().trackStart(_cd->id().numTracks()));
            break;
    }

    // Notify everyone else.
    emit newDisc(_cd);

    // Check for auto play.
    if ( ( options->autoPlay() ) && ( _status != Playing ) && ( _status != Paused ) ) {
        cdrom->play( _desiredTrack );
    }
	
	return true;
}

//------------------------------------------------------------------------------
// Convert the given frame time to minutes and seconds, and emit it.
//------------------------------------------------------------------------------
void CDDrive::doUpdateTime(int frame)
{
    if (frame < 0) {
        emit updateTime(-1, -1);
    } else {
        int minute = frame/XcdSecs/XcdFrames;
        int second = (frame/XcdFrames)%XcdSecs;
        emit updateTime(minute, second);
    }
}

//------------------------------------------------------------------------------
// Inform everyone about the new track.
//------------------------------------------------------------------------------
void CDDrive::doUpdateTrack()
{
	// Make sure the desired track is within the range of tracks on this disc.
    if ((_cd) && (_desiredTrack >= _cd->id().firstTrack()) && (_desiredTrack < (_cd->id().numTracks() + _cd->id().firstTrack()))) {
        emit updateTrack(_desiredTrack, _cd->track(_desiredTrack - _cd->id().firstTrack()));
    } else {
        emit updateTrack(-1, ""); // The desired track is invalid.
    }
}

//------------------------------------------------------------------------------
// Process a change in the program step.
//------------------------------------------------------------------------------
void CDDrive::changeStep(int step)
{
    emit updateStep(step);
}

//------------------------------------------------------------------------------
// Change the poll time preference.
//------------------------------------------------------------------------------
void CDDrive::changePollTime()
{
    killTimer( _timerID );
    _timerID = startTimer( options->pollTime() );
}

//------------------------------------------------------------------------------
// Process a new user-defined program, and begin playing it immediately.
//------------------------------------------------------------------------------
void CDDrive::changeProgram(int length, int tracks[])
{
	// Check for an empty program.
	if (length > 0) {
		_cd->changeProgram( length, tracks );
		connect( _cd->program(), SIGNAL( updateStep( int ) ), this, SLOT( changeStep( int ) ) );
	} else {
		debug("Fix InfoWindow!!!");
	}
	
	changeShuffle();
}

void CDDrive::changeShuffle()
{
	if ( _discStatus == NoDisc ) return;
	
	if ( options->shuffle() ) {
		_cd->program()->shuffle();
	} else {
		_cd->program()->unshuffle();
	}
	_desiredTrack = _cd->program()->reset();
	doUpdateTrack();
	
	cdrom->stop();
	if ( _status == Playing ) {
		cdrom->play( _desiredTrack );
	}
	if ( _status == Paused ) {
		cdrom->play( _desiredTrack );
		cdrom->pause();
	}
}

void CDDrive::changeDbasePath()
{
	if ( _discStatus == NoDisc ) return;

	stop();
	doNoDisc();
}
