// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef _Cdrom_h_
#define _Cdrom_h_

#include <qobject.h>
#include <qstring.h>
#include "CdromBase.h"
#include "CdromType.h"

//: Provide a device-independent interface to an audio CD player.
//. This class is used to indirectly control a CDROM device.  Only a subset
//. of the available audio commands are provided, so that compatibility
//. can be maintained across several different drive types.  The interface
//. is the same for both IDE and SCSI drives.
class Cdrom : public QObject {
	Q_OBJECT
	int        _fd;
	CdromBase* _cdrom;
	QString    _typeConstants[ MaxCdromTypes ];
	QString    _typeNames[ MaxCdromTypes ];
 public:
	//: Initialize the Cdrom object.
	//. There should be only one instance of this object, mainly because there is
	//. a global pointer to it.  It would not be difficult to modify this class
	//. to support multiple instances (read: multiple devices) within the same
	//. application.
	Cdrom();
	
	//: Cleanup the device driver object that we created.
	~Cdrom();
	
	//: Command the CDROM to eject/retract the tray.
	bool eject();
	
	//: Query the table of contents from the CDROM.
	//. If successful, this function returns the number of tracks on the
	//. disc, along with their absolute start times (in frames).  This
	//. information can be used to (hopefully) uniquely identify the disc.
	//. If any error occurs, this function returns 0.
	CompactDiscID* id();
	
	//: Command the CDROM to pause the playing disc.
	bool pause();
	
	//: Command the CDROM to play a single track, from start to finish.
	bool play( int track );
	
	//: Command the CDROM to resume playing the disc.
	bool resume();
	
	//: Query the current audio status of the CDROM, including the current play time, and volume setting.
	bool status( CdromStatus& status, int& absTime, int& volume );
	
	//: Command the CDROM to stop any active play/pause operation.
	bool stop();
	
	//: Command the CDROM to change the volume setting.
	bool volume( int vol );
	
	//: Return the constant string for the given CdromType.
	//. This is the string that is used on the command line and in the
	//. $HOME/.xcdrc file to select the CDROM device driver class.
	const char* typeConstant( int i ) { return (const char*)_typeConstants[i]; }
	
	//: Return the name string for the given CdromType.
	//. This is the string that is displayed on the Preferences window.
	const char* typeName( int i ) { return (const char*)_typeNames[i]; }
 public slots:
	//: Process a change in the CDROM device driver.
	//. This function destroys the old driver object (if any), and creates
	//. a new driver object of the desired type.  Once created, the driver
	//. object is used to initialized the CDROM drive.
	void changeType();
	
	//: Process a change in the CDROM device path.
	//. The open device (if any) is closed, and the new device is opened.
	void changePath();
};

extern Cdrom* cdrom;

#endif
