// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "CompactDiscID.h"

//------------------------------------------------------------------------------
// Create a CompactDiscID from the information off the physical disc.
//------------------------------------------------------------------------------
CompactDiscID::CompactDiscID(int num, int first, int start[]) :
        _numTracks(num),
        _firstTrack(first)
{
    // Allocate the array of track start times (including leadout track).
    _trackStart = new int[_numTracks+1];

    // Copy track start times.
    for (int i = 0; i < _numTracks+1; i++) {
        _trackStart[i] = start[i];
    }
}

//------------------------------------------------------------------------------
// Create a CompactDiscID from another CompactDiscID.
//------------------------------------------------------------------------------
CompactDiscID::CompactDiscID(const CompactDiscID& id) :
        _numTracks(id._numTracks),
        _firstTrack(id._firstTrack)
{
    // Allocate the array of track star times (including leadout track).
    _trackStart = new int[_numTracks+1];

    // Copy track start times.
    for (int i = 0; i < _numTracks+1; i++) {
        _trackStart[i] = id._trackStart[i];
    }
}

//------------------------------------------------------------------------------
// Destroy a CompactDiscID.
//------------------------------------------------------------------------------
CompactDiscID::~CompactDiscID()
{
    delete [] _trackStart;
}

//------------------------------------------------------------------------------
// Determine if two CompactDiscIDs are equal.  Only compare number of tracks
// and track start times.
//------------------------------------------------------------------------------
bool CompactDiscID::operator==(const CompactDiscID& id) const
{
    // Compare number of tracks.
    if (_numTracks != id._numTracks) {
        return 0;
    }

    // Compare track start times.
    for (int i = 0; i < _numTracks; i++) {
        if (_trackStart[i] != id._trackStart[i]) {
            return 0;
        }
    }

    // Compare the leadout track start time.
    if ((_trackStart[_numTracks]/XcdFrames) != (id._trackStart[_numTracks]/XcdFrames)) {
        return 0;
    }

    // They are equal.
    return 1;
}

int CompactDiscID::trackLength( int track ) const
{
    int length = _trackStart[track+1] - _trackStart[track] + 1;

    if ( length >= 0 ) {
        return length;
    } else {
        return 0;
    }
}
